/*
 * Decompiled with CFR 0.152.
 */
package org.openrdf.sail.nativerdf.datastore;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;

public class HashFile {
    private static final int ITEM_SIZE = 8;
    private static final byte[] MAGIC_NUMBER = new byte[]{110, 104, 102};
    private static final byte FILE_FORMAT_VERSION = 1;
    private static final long HEADER_LENGTH = 16L;
    private static final int INIT_BUCKET_COUNT = 64;
    private static final int INIT_BUCKET_SIZE = 8;
    private File file;
    private RandomAccessFile raf;
    private FileChannel fileChannel;
    private boolean forceSync;
    private int bucketCount;
    private int bucketSize;
    private int itemCount;
    private float loadFactor = 0.75f;
    private int recordSize;

    public HashFile(File file) throws IOException {
        this(file, false);
    }

    public HashFile(File file, boolean forceSync) throws IOException {
        boolean created;
        this.file = file;
        this.forceSync = forceSync;
        if (!file.exists() && !(created = file.createNewFile())) {
            throw new IOException("Failed to create file: " + file);
        }
        this.raf = new RandomAccessFile(file, "rw");
        this.fileChannel = this.raf.getChannel();
        if (this.fileChannel.size() == 0L) {
            this.bucketCount = 64;
            this.bucketSize = 8;
            this.itemCount = 0;
            this.recordSize = 8 * this.bucketSize + 4;
            this.writeEmptyBuckets(16L, this.bucketCount);
            this.sync();
        } else {
            this.readFileHeader();
            this.recordSize = 8 * this.bucketSize + 4;
        }
    }

    public File getFile() {
        return this.file;
    }

    public FileChannel getFileChannel() {
        return this.fileChannel;
    }

    public int getBucketCount() {
        return this.bucketCount;
    }

    public int getBucketSize() {
        return this.bucketSize;
    }

    public int getItemCount() {
        return this.itemCount;
    }

    public int getRecordSize() {
        return this.recordSize;
    }

    public IDIterator getIDIterator(int hash) throws IOException {
        return new IDIterator(hash);
    }

    public void storeID(int hash, int id) throws IOException {
        long bucketOffset = this.getBucketOffset(hash);
        this.storeID(bucketOffset, hash, id);
        ++this.itemCount;
        if ((float)this.itemCount >= this.loadFactor * (float)this.bucketCount * (float)this.bucketSize) {
            this.increaseHashTable();
        }
    }

    private void storeID(long bucketOffset, int hash, int id) throws IOException {
        boolean idStored = false;
        ByteBuffer bucket = ByteBuffer.allocate(this.recordSize);
        while (!idStored) {
            this.fileChannel.read(bucket, bucketOffset);
            int slotID = this.findEmptySlotInBucket(bucket);
            if (slotID >= 0) {
                bucket.putInt(8 * slotID, hash);
                bucket.putInt(8 * slotID + 4, id);
                bucket.rewind();
                this.fileChannel.write(bucket, bucketOffset);
                idStored = true;
                continue;
            }
            int overflowID = bucket.getInt(8 * this.bucketSize);
            if (overflowID == 0) {
                overflowID = this.createOverflowBucket();
                bucket.putInt(8 * this.bucketSize, overflowID);
                bucket.rewind();
                this.fileChannel.write(bucket, bucketOffset);
            }
            bucketOffset = this.getOverflowBucketOffset(overflowID);
            bucket.clear();
        }
    }

    public void clear() throws IOException {
        this.fileChannel.truncate(16L + (long)this.bucketCount * (long)this.recordSize);
        this.writeEmptyBuckets(16L, this.bucketCount);
        this.itemCount = 0;
    }

    public void sync() throws IOException {
        this.writeFileHeader();
        if (this.forceSync) {
            this.fileChannel.force(false);
        }
    }

    public void close() throws IOException {
        this.raf.close();
        this.raf = null;
        this.fileChannel = null;
    }

    private RandomAccessFile createEmptyFile(File file) throws IOException {
        boolean created;
        if (!file.exists() && !(created = file.createNewFile())) {
            throw new IOException("Failed to create file " + file);
        }
        RandomAccessFile raf = new RandomAccessFile(file, "rw");
        raf.setLength(0L);
        return raf;
    }

    private void writeFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.put(MAGIC_NUMBER);
        buf.put((byte)1);
        buf.putInt(this.bucketCount);
        buf.putInt(this.bucketSize);
        buf.putInt(this.itemCount);
        buf.rewind();
        this.fileChannel.write(buf, 0L);
    }

    private void readFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        this.fileChannel.read(buf, 0L);
        buf.rewind();
        if ((long)buf.remaining() < 16L) {
            throw new IOException("File too short to be a compatible hash file");
        }
        byte[] magicNumber = new byte[MAGIC_NUMBER.length];
        buf.get(magicNumber);
        byte version = buf.get();
        this.bucketCount = buf.getInt();
        this.bucketSize = buf.getInt();
        this.itemCount = buf.getInt();
        if (!Arrays.equals(MAGIC_NUMBER, magicNumber)) {
            throw new IOException("File doesn't contain compatible hash file data");
        }
        if (version > 1) {
            throw new IOException("Unable to read hash file; it uses a newer file format");
        }
        if (version != 1) {
            throw new IOException("Unable to read hash file; invalid file format version: " + version);
        }
    }

    private long getBucketOffset(int hash) {
        int bucketNo = hash % this.bucketCount;
        if (bucketNo < 0) {
            bucketNo += this.bucketCount;
        }
        return 16L + (long)bucketNo * (long)this.recordSize;
    }

    private long getOverflowBucketOffset(int bucketID) {
        return 16L + ((long)this.bucketCount + (long)bucketID - 1L) * (long)this.recordSize;
    }

    private int createOverflowBucket() throws IOException {
        long offset = this.fileChannel.size();
        this.writeEmptyBuckets(offset, 1);
        return (int)((offset - 16L) / (long)this.recordSize) - this.bucketCount + 1;
    }

    private void writeEmptyBuckets(long fileOffset, int bucketCount) throws IOException {
        ByteBuffer emptyBucket = ByteBuffer.allocate(this.recordSize);
        for (int i = 0; i < bucketCount; ++i) {
            this.fileChannel.write(emptyBucket, fileOffset + (long)i * (long)this.recordSize);
            emptyBucket.rewind();
        }
    }

    private int findEmptySlotInBucket(ByteBuffer bucket) {
        for (int slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
            if (bucket.getInt(8 * slotNo + 4) != 0) continue;
            return slotNo;
        }
        return -1;
    }

    private void increaseHashTable() throws IOException {
        long oldTableSize = 16L + (long)this.bucketCount * (long)this.recordSize;
        long newTableSize = 16L + (long)this.bucketCount * (long)this.recordSize * 2L;
        long oldFileSize = this.fileChannel.size();
        File tmpFile = new File(this.file.getParentFile(), "rehash_" + this.file.getName());
        RandomAccessFile tmpRaf = this.createEmptyFile(tmpFile);
        FileChannel tmpChannel = tmpRaf.getChannel();
        this.fileChannel.transferTo(oldTableSize, oldFileSize, tmpChannel);
        this.writeEmptyBuckets(oldTableSize, this.bucketCount);
        this.bucketCount *= 2;
        this.fileChannel.truncate(newTableSize);
        ByteBuffer bucket = ByteBuffer.allocate(this.recordSize);
        ByteBuffer newBucket = ByteBuffer.allocate(this.recordSize);
        for (long bucketOffset = 16L; bucketOffset < oldTableSize; bucketOffset += (long)this.recordSize) {
            this.fileChannel.read(bucket, bucketOffset);
            boolean bucketChanged = false;
            long newBucketOffset = 0L;
            for (int slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
                int hash;
                long newOffset;
                int id = bucket.getInt(8 * slotNo + 4);
                if (id == 0 || (newOffset = this.getBucketOffset(hash = bucket.getInt(8 * slotNo))) == bucketOffset) continue;
                newBucket.putInt(hash);
                newBucket.putInt(id);
                bucket.putInt(8 * slotNo, 0);
                bucket.putInt(8 * slotNo + 4, 0);
                bucketChanged = true;
                newBucketOffset = newOffset;
            }
            if (bucketChanged) {
                newBucket.flip();
                this.fileChannel.write(newBucket, newBucketOffset);
                newBucket.clear();
            }
            if (bucket.getInt(8 * this.bucketSize) != 0) {
                bucket.putInt(8 * this.bucketSize, 0);
                bucketChanged = true;
            }
            if (bucketChanged) {
                bucket.rewind();
                this.fileChannel.write(bucket, bucketOffset);
            }
            bucket.clear();
        }
        long tmpFileSize = tmpChannel.size();
        for (long bucketOffset = 0L; bucketOffset < tmpFileSize; bucketOffset += (long)this.recordSize) {
            tmpChannel.read(bucket, bucketOffset);
            for (int slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
                int id = bucket.getInt(8 * slotNo + 4);
                if (id == 0) continue;
                int hash = bucket.getInt(8 * slotNo);
                long newBucketOffset = this.getBucketOffset(hash);
                this.storeID(newBucketOffset, hash, id);
                bucket.putInt(8 * slotNo, 0);
                bucket.putInt(8 * slotNo + 4, 0);
            }
            bucket.clear();
        }
        tmpRaf.close();
        tmpFile.delete();
    }

    public void dumpContents(PrintStream out) throws IOException {
        int overflowID;
        int id;
        int hash;
        int slotNo;
        int bucketNo;
        out.println();
        out.println("*** hash file contents ***");
        out.println("_bucketCount=" + this.bucketCount);
        out.println("_bucketSize=" + this.bucketSize);
        out.println("_itemCount=" + this.itemCount);
        ByteBuffer buf = ByteBuffer.allocate(this.recordSize);
        this.fileChannel.position(16L);
        out.println("---Buckets---");
        for (bucketNo = 1; bucketNo <= this.bucketCount; ++bucketNo) {
            buf.clear();
            this.fileChannel.read(buf);
            out.print("Bucket " + bucketNo + ": ");
            for (slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
                hash = buf.getInt(8 * slotNo);
                id = buf.getInt(8 * slotNo + 4);
                if (slotNo > 0) {
                    out.print(" ");
                }
                out.print("[" + this.toHexString(hash) + "," + id + "]");
            }
            overflowID = buf.getInt(8 * this.bucketSize);
            out.println("---> " + overflowID);
        }
        out.println("---Overflow Buckets---");
        bucketNo = 0;
        while (this.fileChannel.position() < this.fileChannel.size()) {
            buf.clear();
            this.fileChannel.read(buf);
            out.print("Bucket " + ++bucketNo + ": ");
            for (slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
                hash = buf.getInt(8 * slotNo);
                id = buf.getInt(8 * slotNo + 4);
                if (slotNo > 0) {
                    out.print(" ");
                }
                out.print("[" + this.toHexString(hash) + "," + id + "]");
            }
            overflowID = buf.getInt(8 * this.bucketSize);
            out.println("---> " + overflowID);
        }
        out.println("*** end of hash file contents ***");
        out.println();
    }

    private String toHexString(int decimal) {
        String hex = Integer.toHexString(decimal);
        StringBuilder result = new StringBuilder(8);
        for (int i = hex.length(); i < 8; ++i) {
            result.append("0");
        }
        result.append(hex);
        return result.toString();
    }

    public static void main(String[] args) throws Exception {
        HashFile hashFile = new HashFile(new File(args[0]));
        hashFile.dumpContents(System.out);
        hashFile.close();
    }

    public class IDIterator {
        private int queryHash;
        private ByteBuffer bucketBuffer;
        private long bucketOffset;
        private int slotNo;

        private IDIterator(int hash) throws IOException {
            this.queryHash = hash;
            this.bucketBuffer = ByteBuffer.allocate(HashFile.this.getRecordSize());
            this.bucketOffset = HashFile.this.getBucketOffset(hash);
            HashFile.this.getFileChannel().read(this.bucketBuffer, this.bucketOffset);
            this.slotNo = -1;
        }

        public int next() throws IOException {
            while (this.bucketBuffer != null) {
                ++this.slotNo;
                while (this.slotNo < HashFile.this.getBucketSize()) {
                    if (this.bucketBuffer.getInt(8 * this.slotNo) == this.queryHash) {
                        return this.bucketBuffer.getInt(8 * this.slotNo + 4);
                    }
                    ++this.slotNo;
                }
                int overflowID = this.bucketBuffer.getInt(8 * HashFile.this.getBucketSize());
                if (overflowID == 0) {
                    this.bucketBuffer = null;
                    this.bucketOffset = 0L;
                    continue;
                }
                this.bucketOffset = HashFile.this.getOverflowBucketOffset(overflowID);
                this.bucketBuffer.clear();
                HashFile.this.getFileChannel().read(this.bucketBuffer, this.bucketOffset);
                this.slotNo = -1;
            }
            return -1;
        }
    }
}

