
/*
 * Copyright 2011 Google Inc.
 *
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */
#include "SkFontHost.h"
#include <math.h>

// define this to use pre-compiled tables for gamma. This is slightly faster,
// and doesn't create any RW global memory, but means we cannot change the
// gamma at runtime.
//#define USE_PREDEFINED_GAMMA_TABLES

#ifndef USE_PREDEFINED_GAMMA_TABLES
    // define this if you want to spew out the "C" code for the tables, given
    // the current values for SK_BLACK_GAMMA and SK_WHITE_GAMMA.
    #define DUMP_GAMMA_TABLESx
#endif

///////////////////////////////////////////////////////////////////////////////

#include "SkGraphics.h"

// declared here, so we can link against it elsewhere
void skia_set_text_gamma(float blackGamma, float whiteGamma);

#ifdef USE_PREDEFINED_GAMMA_TABLES

#include "sk_predefined_gamma.h"

void skia_set_text_gamma(float blackGamma, float whiteGamma) {}

#else   // use writable globals for gamma tables

static void build_power_table(uint8_t table[], float ee) {
//    SkDebugf("------ build_power_table %g\n", ee);
    for (int i = 0; i < 256; i++) {
        float x = i / 255.f;
        //   printf(" %d %g", i, x);
        x = powf(x, ee);
        //   printf(" %g", x);
        int xx = SkScalarRound(SkFloatToScalar(x * 255));
        //   printf(" %d\n", xx);
        table[i] = SkToU8(xx);
    }
}

static bool gGammaIsBuilt;
static uint8_t gBlackGamma[256], gWhiteGamma[256];

static float gBlackGammaCoeff = 1.4f;
static float gWhiteGammaCoeff = 1/1.4f;

void skia_set_text_gamma(float blackGamma, float whiteGamma) {
    gBlackGammaCoeff = blackGamma;
    gWhiteGammaCoeff = whiteGamma;
    gGammaIsBuilt = false;
    SkGraphics::PurgeFontCache();
    build_power_table(gBlackGamma, gBlackGammaCoeff);
    build_power_table(gWhiteGamma, gWhiteGammaCoeff);
}

#ifdef DUMP_GAMMA_TABLES

#include "SkString.h"

static void dump_a_table(const char name[], const uint8_t table[],
                         float gamma) {
    SkDebugf("\n");
    SkDebugf("\/\/ Gamma table for %g\n", gamma);
    SkDebugf("static const uint8_t %s[] = {\n", name);
    for (int y = 0; y < 16; y++) {
        SkString line, tmp;
        for (int x = 0; x < 16; x++) {
            tmp.printf("0x%02X, ", *table++);
            line.append(tmp);
        }
        SkDebugf("    %s\n", line.c_str());
    }
    SkDebugf("};\n");
}

#endif

#endif

///////////////////////////////////////////////////////////////////////////////

void SkFontHost::GetGammaTables(const uint8_t* tables[2]) {
#ifndef USE_PREDEFINED_GAMMA_TABLES
    if (!gGammaIsBuilt) {
        build_power_table(gBlackGamma, gBlackGammaCoeff);
        build_power_table(gWhiteGamma, gWhiteGammaCoeff);
        gGammaIsBuilt = true;

#ifdef DUMP_GAMMA_TABLES
        dump_a_table("gBlackGamma", gBlackGamma, gBlackGammaCoeff);
        dump_a_table("gWhiteGamma", gWhiteGamma, gWhiteGammaCoeff);
#endif
    }
#endif
    tables[0] = gBlackGamma;
    tables[1] = gWhiteGamma;
}

