/*****************************************************************

Copyright (c) 1996-2004 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <unistd.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qstyle.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qwmatrix.h>

#include <kapplication.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kdebug.h>
#include <kdesktopfile.h>
#include <kmimetype.h>
#include <kprocess.h>
#include <krootpixmap.h>
#include <kpixmap.h>
#include <klocale.h>
#include <kio/netaccess.h>
#include <kservice.h>
#include <kurldrag.h>

#include "container_applet.h"
#include "container_button.h"
#include "containerarealayout.h"
#include "dirdrop_mnu.h"
#include "exe_dlg.h"
#include "kicker.h"
#include "kickerSettings.h"
#include "kickertip.h"
#include "paneldrag.h"
#include "pluginmanager.h"

#include "containerarea.h"

// for multihead
extern int kicker_screen_number;

ContainerArea::ContainerArea(KConfig* _c,
                             QWidget* parent,
                             QPopupMenu* opMenu,
                             const char* name)
    : Panner(parent, name),
      _moveAC(0),
      _pos(KPanelExtension::Left),
      _config(_c),
      _dragIndicator(0),
      _dragMoveAC(0),
      _dragMoveOffset(QPoint(0,0)),
      _opMenu(opMenu),
      _rootPixmap(0),
      _useBgTheme(false),
      _bgSet(false),
      m_canAddContainers(true),
      m_immutable(Kicker::the()->isImmutable() || _c->isImmutable()),
      m_layout(0)
{
    setBackgroundOrigin( WidgetOrigin );
    viewport()->setBackgroundOrigin( AncestorOrigin );

    m_contents = new QWidget(viewport());
    m_layout = new ContainerAreaLayout(m_contents);

    // Install an event filter to propagate layout hints coming from
    // m_contents.
    m_contents->installEventFilter(this);

    setAcceptDrops( !Kicker::the()->isImmutable() );
    connect(&_autoScrollTimer, SIGNAL(timeout()), SLOT(autoScroll()));
    connect(kapp, SIGNAL(kdisplayPaletteChanged()), SLOT(setBackgroundTheme()));
}

ContainerArea::~ContainerArea()
{
    // don't emit signals from destructor
    blockSignals( true );
    // clear applets
    removeAllContainers();
}

void ContainerArea::initialize(bool useDefaultConfig)
{
    // do we really need to do this?
    removeAllContainers();

    // restore applet layout or load a default panel layout
    _config->setGroup("General");
    if (_config->hasKey("Applets2"))
    {
        if (_config->groupIsImmutable("General"))
        {
            m_immutable = true;
        }

        m_canAddContainers = !m_immutable &&
                             !_config->entryIsImmutable("Applets2");
        loadContainers(_config->readListEntry("Applets2"));
    }
    else if (useDefaultConfig)
    {
        defaultContainerConfig();
    }
}

void ContainerArea::defaultContainerConfig()
{
    //FIXME: make this use a file template so it isn't hardcoded anymore
    ContainerList containers;

    containers.append(new KMenuButtonContainer(_opMenu, m_contents));
    containers.append(new DesktopButtonContainer(_opMenu, m_contents));

    int dsize;
    if (orientation() == Horizontal)
    {
        dsize = width();
    }
    else
    {
        dsize = height();
    }

    dsize -= 560;
    QStringList buttons;

    QFile f(locate("data", "kicker/default-apps"));
    if (f.open(IO_ReadOnly))
    {
        QTextStream is(&f);

        while (!is.eof())
            buttons << is.readLine();

        f.close();
    }
    else
    {
        buttons << "kde-Home.desktop"
                << "kde-konqbrowser.desktop"
                << "kde-Kontact.desktop"
                << "kde-kword.desktop"
                << "kde-Help.desktop";
    }

    //int size = dsize;
    for (QStringList::ConstIterator it = buttons.begin(); it != buttons.end(); ++it)
    {
        /*size -= 42;
        if (size <= 0)
            break;*/

        BaseContainer *button;
        KService::Ptr service = KService::serviceByStorageId(*it);
        if (!service)
        {
            // look for a special button
            QString s = locate("appdata", *it);
            if (s.isEmpty()) continue;
            QString itExt = (*it).section('/', 1);
            button = new ExtensionButtonContainer(itExt, _opMenu, m_contents);
        }
        else
        {
            button = new ServiceButtonContainer(service, _opMenu, m_contents);
        }

        if (button->isValid())
        {
            containers.append(button);
        }
        else
        {
            delete button;
        }
    }

    PluginManager* manager = PluginManager::the();
    AppletContainer *a = 0;

    // pager applet
    a = manager->createAppletContainer(
        "minipagerapplet.desktop",
        true,
        QString::null,
        _opMenu,
        m_contents);
    containers.append(a);

    // taskbar applet
    a = manager->createAppletContainer(
        "taskbarapplet.desktop",
        true,
        QString::null,
        _opMenu,
        m_contents);
    containers.append(a);

    // system tray applet
    a = manager->createAppletContainer(
        "systemtrayapplet.desktop",
        true,
        QString::null,
        _opMenu,
        m_contents );
    a->setFreeSpace(1);
    containers.append(a);

    // clock applet
    a = manager->createAppletContainer(
        "clockapplet.desktop",
        true,
        QString::null,
        _opMenu,
        m_contents );
    a->setFreeSpace(1);
    containers.append(a);

    for(ContainerIterator it(containers); it.current(); ++it)
    {
         addContainer(it.current());
    }

    saveContainerConfig();
}

void ContainerArea::loadContainers(const QStringList& containers)
{
    // read applet list
    bool badApplets = false;

    // now restore the applets
    QStringList::const_iterator it = containers.begin();
    QStringList::const_iterator itEnd = containers.end();
    for (; it != itEnd; ++it)
    {
        QString appletId(*it);

        // is there a config group for this applet?
        if (!_config->hasGroup(appletId))
        {
            continue;
        }

        KConfigGroup group(_config, appletId.latin1());

        BaseContainer* a = 0;

        int sep = appletId.findRev('_');
        Q_ASSERT(sep != -1);
        QString appletType = appletId.left(sep);

        // create a matching applet container
        if (appletType == "KMenuButton")
            a = new KMenuButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "DesktopButton")
            a = new DesktopButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "WindowListButton")
            a = new WindowListButtonContainer(group, _opMenu, m_contents);
        else if ((appletType == "BookmarksButton") && kapp->authorizeKAction("bookmarks"))
            a = new BookmarksButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "ServiceButton")
            a = new ServiceButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "URLButton")
            a = new URLButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "BrowserButton")
            a = new BrowserButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "ServiceMenuButton")
            a = new ServiceMenuButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "ExecButton")
            a = new NonKDEAppButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "ExtensionButton")
            a = new ExtensionButtonContainer(group, _opMenu, m_contents);
        else if (appletType == "Applet")
        {
            bool immutable = Kicker::the()->isImmutable() ||
                             group.groupIsImmutable() ||
                             group.entryIsImmutable("ConfigFile");
            a = PluginManager::the()->createAppletContainer(
                   group.readPathEntry("DesktopFile"),
                   true, // isStartup
                   group.readPathEntry("ConfigFile"),
                   _opMenu,
                   m_contents,
                   immutable);
        }

        if (a && a->isValid())
        {
            a->setAppletId(appletId);
            a->loadConfiguration(group);
            addContainer(a);
        }
        else
        {
            badApplets = true;
            delete a;
        }
    }

    if (badApplets)
    {
        // since we may have had Bad Applets in our list
        // let's save it again, just in case
        saveContainerConfig();
    }
}

void ContainerArea::saveContainerConfig(bool layoutOnly)
{
    if (!canAddContainers())
    {
        return;
    }

    // Save the applet list
    QStringList alist;
    QLayoutIterator it2 = m_layout->iterator();
    for (; it2.current(); ++it2)
    {
        BaseContainer* a = dynamic_cast<BaseContainer*>(it2.current()->widget());
        if (a)
        {
            KConfigGroup group(_config, a->appletId().latin1());
            a->saveConfiguration(group, layoutOnly);
            alist.append(a->appletId());
        }
    }

    KConfigGroup group( _config, "General" );
    group.writeEntry("Applets2", alist);

    _config->sync();
}

void ContainerArea::removeAllContainers()
{
    while (!m_containers.isEmpty())
    {
        BaseContainer* b = m_containers.first();
        m_containers.removeRef( b );
        delete b;
    }
}

void ContainerArea::configure()
{
    setBackgroundTheme();
    repaint();

    for( ContainerIterator it(m_containers); it.current(); ++it )
    {
        (*it)->configure();
    }
    updateContainersBackground();
}

void ContainerArea::addKMenuButton()
{
    if (!canAddContainers())
    {
        return;
    }

    KMenuButtonContainer *b = new KMenuButtonContainer(_opMenu, m_contents);
    // Set freespace to one since the container will be added at the end.
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addDesktopButton()
{
    if (!canAddContainers())
    {
        return;
    }

    DesktopButtonContainer *b = new DesktopButtonContainer(_opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addWindowListButton()
{
    if (!canAddContainers())
    {
        return;
    }

    WindowListButtonContainer *b = new WindowListButtonContainer(_opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addBookmarksButton()
{
    if (!canAddContainers())
    {
        return;
    }

    BookmarksButtonContainer *b = new BookmarksButtonContainer(_opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addServiceButton(const QString& desktopFile)
{
    if (!canAddContainers())
    {
        return;
    }

    ServiceButtonContainer *b = new ServiceButtonContainer(desktopFile, _opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addURLButton(const QString &url)
{
    if (!canAddContainers())
    {
        return;
    }

    URLButtonContainer *b = new URLButtonContainer(url, _opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addBrowserButton( const QString &startDir, const QString& icon )
{
    if (!canAddContainers())
    {
        return;
    }

    BrowserButtonContainer *b = new BrowserButtonContainer(startDir, _opMenu, icon, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addServiceMenuButton(const QString& relPath)
{
    if (!canAddContainers())
    {
        return;
    }

    ServiceMenuButtonContainer *b = new ServiceMenuButtonContainer(relPath, _opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addNonKDEAppButton(const QString &name, const QString& filePath, const QString &icon,
                                       const QString &cmdLine, bool inTerm)
{
    if (!canAddContainers())
    {
        return;
    }

    NonKDEAppButtonContainer *b = new NonKDEAppButtonContainer(name, filePath, icon, cmdLine, inTerm, _opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

void ContainerArea::addExtensionButton(const QString& df)
{
    if (!canAddContainers())
    {
        return;
    }

    ExtensionButtonContainer *b = new ExtensionButtonContainer(df, _opMenu, m_contents);
    b->setFreeSpace(1);
    addContainer(b, true);
    scrollTo(b);
    saveContainerConfig();
}

AppletContainer* ContainerArea::addApplet(const QString& desktopFile, bool isImmutable)
{
    if (!canAddContainers())
    {
        return 0;
    }

    AppletContainer* a = PluginManager::the()->createAppletContainer(
        desktopFile,
        false,         // not startup
        QString::null, // no config
        _opMenu,
        m_contents,
        isImmutable);

    if (!a || !a->isValid())
    {
        delete a;
        return 0;
    }

    a->setFreeSpace(1);
    addContainer(a, true);
    scrollTo(a);
    saveContainerConfig();

    return a;
}

void ContainerArea::addContainer(BaseContainer* a, bool arrange)
{
    if (!a)
    {
        return;
    }

    if (a->appletId().isNull())
    {
        a->setAppletId(createUniqueId(a->appletType()));
    }

    m_containers.append(a);

    if (arrange)
    {
        if (Kicker::the()->insertionPoint().isNull())
        {
            m_layout->insertIntoFreeSpace(a, QPoint());
        }
        else
        {
            m_layout->insertIntoFreeSpace(a, mapFromGlobal(Kicker::the()->insertionPoint()));
        }
    }
    else
    {
        m_layout->add(a);
    }

    connect(a, SIGNAL(moveme(BaseContainer*)),
            SLOT(startContainerMove(BaseContainer*)));
    connect(a, SIGNAL(removeme(BaseContainer*)),
            SLOT(removeContainer(BaseContainer*)));
    connect(a, SIGNAL(takeme(BaseContainer*)),
            SLOT(takeContainer(BaseContainer*)));
    connect(a, SIGNAL(requestSave()),
            SLOT(slotSaveContainerConfig()));
    connect(a, SIGNAL(maintainFocus(bool)),
            this, SIGNAL(maintainFocus(bool)));

    if (a->inherits("AppletContainer"))
    {
        connect(a, SIGNAL(updateLayout()), SLOT(updateContainersBackground()));
    }

    a->setBackgroundOrigin(AncestorOrigin);
    a->setOrientation(orientation());
    a->setPopupDirection(popupDirection());
    a->configure();
    a->show();
    resizeContents();
}

void ContainerArea::removeContainer(BaseContainer *a)
{
    if (!a || m_immutable)
    {
        return;
    }

    a->slotRemoved(_config);
    m_containers.removeRef(a);
    m_layout->remove(a);
    a->deleteLater();
    a = 0;
    saveContainerConfig(true);
    resizeContents();
}

void ContainerArea::takeContainer(BaseContainer* a)
{
    if (!a)
    {
        return;
    }

    disconnect(a, SIGNAL(moveme(BaseContainer*)),
               this, SLOT(startContainerMove(BaseContainer*)));
    disconnect(a, SIGNAL(removeme(BaseContainer*)),
               this, SLOT(removeContainer(BaseContainer*)));
    disconnect(a, SIGNAL(takeme(BaseContainer*)),
               this, SLOT(takeContainer(BaseContainer*)));
    disconnect(a, SIGNAL(requestSave()),
               this, SLOT(slotSaveContainerConfig()));
    disconnect(a, SIGNAL(maintainFocus(bool)),
               this, SIGNAL(maintainFocus(bool)));

    a->slotRemoved(_config);
    m_containers.removeRef(a);
    m_layout->remove(a);
    saveContainerConfig(true);
    resizeContents();
}

void ContainerArea::resizeContents()
{
    int w = width();
    int h = height();

    if (orientation() == Qt::Horizontal)
    {
        int newWidth = m_layout->widthForHeight(h);
        if (newWidth > w)
        {
            resizeContents(newWidth, h);
        }
        else
        {
            resizeContents(w, h);
        }
    }
    else
    {
        int newHeight = m_layout->heightForWidth(w);

        if (newHeight > h)
        {
            resizeContents(w, newHeight);
        }
        else
        {
            resizeContents(w, h);
        }
    }
}

QString ContainerArea::createUniqueId(const QString& appletType) const
{
    QString idBase = appletType + "_%1";
    QString newId;
    int i = 0;
    bool unique = false;

    while(!unique) {
        i++;
        newId = idBase.arg(i);

        unique = true;
        for( ContainerIterator it(m_containers); it.current() ; ++it ) {
            BaseContainer* b = it.current();
            if (b->appletId() == newId) {
                unique = false;
                break;
            }
        }
    }
    return newId;
}

bool ContainerArea::canAddContainers() const
{
    return m_canAddContainers && Kicker::the()->canAddContainers();
}

void ContainerArea::startContainerMove(BaseContainer *a)
{
    if (!a || m_immutable)
    {
        return;
    }

    _moveAC = a;

    KickerTip::enableTipping(false);
    emit maintainFocus(true);
    setMouseTracking(true);
    grabMouse(sizeAllCursor);

    m_layout->setStretchEnabled(false);
    a->raise();
}

void ContainerArea::mouseReleaseEvent(QMouseEvent *)
{
    if (!_moveAC)
    {
        return;
    }

    // start container move was caled successfuly
    // so we need to complete the move here
    _autoScrollTimer.stop();
    releaseMouse();
    setCursor(arrowCursor);
    setMouseTracking(false);

    _moveAC->completeMoveOperation();
    KickerTip::enableTipping(true);

    _moveAC = 0;

    emit maintainFocus(false);
    m_layout->setStretchEnabled(true);
    updateContainersBackground();
    saveContainerConfig(true);
}

void ContainerArea::mouseMoveEvent(QMouseEvent *ev)
{
    if (!_moveAC) {
        Panner::mouseMoveEvent(ev);
        return;
    }

    if (orientation() == Horizontal)
    {
        int oldX = _moveAC->x() + _moveAC->moveOffset().x();
        int x = ev->pos().x();
        if (ev->state() & ShiftButton)
        {
            m_layout->moveContainerPush(_moveAC, x - oldX);
        }
        else
        {
            m_layout->moveContainerSwitch(_moveAC, x - oldX);
            /* FIXME: Scrolling when the container moves out of the viewport
            bool scroll = false;
            if (rtl)
                if (newPos - 80 <= 0)
                    scroll = true;
            else
                if (newPos + 80 >= (horizontal ? geometry().width()  - moving->geometry().width()
                                               : geometry().height() - moving->geometry().height()))
                    scroll = true;
            [...]
            if (scroll) {
                if (!_autoScrollTimer.isActive())
                    _autoScrollTimer.start(50);

                if (horizontal)
                    scrollBy(dir*10, 0);
                else
                    scrollBy(0, dir*10);
            }
            */
        }
    }
    else
    {
        int oldY = _moveAC->y() + _moveAC->moveOffset().y();
        int y = ev->pos().y();
        if (ev->state() & ShiftButton)
        {
            m_layout->moveContainerPush(_moveAC, y - oldY);
        }
        else
        {
            m_layout->moveContainerSwitch(_moveAC, y - oldY);
            // TODO: Scrolling
        }
    }
}

int ContainerArea::position() const
{
    return static_cast<int>(_pos);
}

KPanelApplet::Direction ContainerArea::popupDirection() const
{
    return KickerLib::positionToDirection(_pos);
}

void ContainerArea::dragEnterEvent(QDragEnterEvent *ev)
{
    bool canAccept = !m_immutable &&
                     (PanelDrag::canDecode(ev) || KURLDrag::canDecode(ev));
    ev->accept(canAccept);

    if (!canAccept)
    {
        return;
    }

    m_layout->setStretchEnabled(false);

    if (!_dragIndicator)
    {
        _dragIndicator = new DragIndicator(this);
    }

    if (orientation() == Horizontal)
    {
        _dragIndicator->setPreferredSize(QSize(height(), height()));
    }
    else
    {
        _dragIndicator->setPreferredSize(QSize(width(), width()));
    }

    _dragMoveOffset = QPoint(_dragIndicator->width()/2,
                             _dragIndicator->height()/2);

    // Find the container before the position of the dragindicator.
    ContainerIterator it(m_containers);
    it.toLast();
    while (it.current())
    {
        BaseContainer* a = it.current();

        if (orientation() == Horizontal &&
                a->x() < ev->pos().x() - _dragMoveOffset.x()
         || orientation() == Vertical &&
                a->y() < ev->pos().y() - _dragMoveOffset.y() )
        {
            _dragMoveAC = a;
            break;
        }

        --it;
    }

    if (orientation() == Horizontal)
    {
        moveDragIndicator((ev->pos() - _dragMoveOffset).x());
    }
    else
    {
        moveDragIndicator((ev->pos() - _dragMoveOffset).y());
    }

    _dragIndicator->show();
    QTimer::singleShot(30000, _dragIndicator, SLOT(hide()));
}

void ContainerArea::dragMoveEvent(QDragMoveEvent* ev)
{
    if (!_dragIndicator)
    {
        return;
    }

    if (orientation() == Horizontal)
    {
        moveDragIndicator((ev->pos() - _dragMoveOffset).x());\
    }
    else
    {
        moveDragIndicator((ev->pos() - _dragMoveOffset).y());
    }
}

void ContainerArea::dragLeaveEvent(QDragLeaveEvent*)
{
    if (_dragIndicator)
    {
        _dragIndicator->hide();
    }
    m_layout->setStretchEnabled(true);
}

void ContainerArea::dropEvent(QDropEvent *ev)
{
    //kdDebug() << "dropEvent()" << endl;
    if (!_dragIndicator)
    {
        // we assume that this is the result of a successful drag enter
        // which means we'll have a _dragIndicator. if for
        // some reason we don't, let's not go down this code path
        return;
    }

    BaseContainer *a = 0;
    if (PanelDrag::decode(ev, &a))
    {
        if (!a)
        {
            _dragIndicator->hide();
            m_layout->setStretchEnabled(true);
            return;
        }

        QObject *parent = ev->source() ? ev->source()->parent() : 0;
        while (parent && (parent != this))
        {
            parent = parent->parent();
        }

        if (parent)
        {
            // Move container a
            if (orientation() == Horizontal)
            {
                int oldX = a->x();
                int x = _dragIndicator->x();
                m_layout->moveContainerSwitch(a, x - oldX);
            }
            else if (orientation() == Vertical)
            {
                int oldY = a->y();
                int y = _dragIndicator->y();
                m_layout->moveContainerSwitch(a, y - oldY);
            }

            _dragIndicator->hide();
            m_layout->setEnabled(true);
            m_layout->setStretchEnabled(true);
            saveContainerConfig(true);
            return;
        }

        // it came from another panel
        Kicker::the()->setInsertionPoint(QPoint(_dragIndicator->x(), _dragIndicator->y()));
        a->reparent(this, 0, QPoint(_dragIndicator->x(), _dragIndicator->y()), true);
        addContainer(a, true);
        Kicker::the()->setInsertionPoint(QPoint());
        m_layout->updateContainerList();
        _dragIndicator->hide();
        m_layout->setStretchEnabled(true);
        saveContainerConfig();
        return;
    }


    // ok, let's try a KURL drag
    KURL::List uriList;
    if (!KURLDrag::decode(ev, uriList))
    {
        _dragIndicator->hide();
        m_layout->setStretchEnabled(true);
        return;
    }

    Kicker::the()->setInsertionPoint(QPoint(_dragIndicator->x(), _dragIndicator->y()));

    KURL::List::ConstIterator it(uriList.begin());
    for (; it != uriList.end(); ++it)
    {
        const KURL &url = *it;

        // Create a new PanelButton for this URL.

        // see if it's a executable or directory
        if (url.protocol() == "programs")
        {
            QString relPath = url.path();
            if (relPath[0] == '/')
            {
                relPath = relPath.right(relPath.length() - 1);
            }
            a = new ServiceMenuButtonContainer(relPath, _opMenu, m_contents);
        }
	else if (url.isLocalFile()) {
	    QFileInfo fi(url.path());
	    if (fi.isDir()) { // directory
		switch (PanelDirDropMenu().exec(mapToGlobal(ev->pos()))) {
		case PanelDirDropMenu::Browser:
		    a = new BrowserButtonContainer(url.path(), _opMenu,
		                   KMimeType::iconForURL(url), m_contents);
		    break;
		case PanelDirDropMenu::Url:
		    a = new URLButtonContainer(url.url(), _opMenu, m_contents);
		    break;
		default: ;
		}
	    }
            else if ( KMimeType::findByURL(url)->name() == "application/x-desktop" ) {
		// a local desktop file being dragged from an external program.
		// Make a copy first.
		KDesktopFile df(url.path());
		KURL newUrl;
		newUrl.setPath(KickerLib::copyDesktopFile(url));
		if (df.readType() == "Link")
		   a = new URLButtonContainer(newUrl.url(), _opMenu, m_contents);
		else
		   a = new ServiceButtonContainer(newUrl.path(), _opMenu, m_contents);
	    }
	    else if (fi.isExecutable()) { // non-KDE executable
		QString pixmapFile;
		KMimeType::pixmapForURL(url, 0, KIcon::Panel, 0,
		                        KIcon::DefaultState, &pixmapFile);
		PanelExeDialog dlg(QString::null, url.path(), pixmapFile,
		                   QString::null, false, 0);
		if (dlg.exec() == QDialog::Accepted) {
		    // KIconloader returns a full path, we only want the name
		    QFileInfo iconfi(dlg.iconPath());
		    a = new NonKDEAppButtonContainer(dlg.title(), dlg.command(),
                                             iconfi.fileName(),
		                             dlg.commandLine(), dlg.useTerminal(),
		                             _opMenu, m_contents);
		}
	    }
	    else // some unknown local file
		a = new URLButtonContainer(url.url(), _opMenu, m_contents);
	}
	else // a internet URL
	    a = new URLButtonContainer(url.url(), _opMenu, m_contents);

	if (!a)
    {
	    _dragIndicator->hide();
        Kicker::the()->setInsertionPoint(QPoint());
        m_layout->setStretchEnabled(true);
	    return;
	}

        // Move the neighbour containers if there isn't enough space
        if (_dragIndicator->size() != _dragIndicator->preferredSize()) {
            int neededSpace;
            int distanceMoved;
            BaseContainer* next;

            if (_dragMoveAC) {
               m_containers.findRef(_dragMoveAC);
                next = m_containers.next();
            } else {
                next = m_containers.first();
            }

            if (orientation() == Horizontal) {
                neededSpace = _dragIndicator->preferredSize().width() - _dragIndicator->width();
                if (_dragMoveAC) {
                    distanceMoved = m_layout->moveContainerPush(_dragMoveAC, -neededSpace/2);
                    _dragIndicator->move(
                         _dragIndicator->x() + distanceMoved,
                         _dragIndicator->y());
                    neededSpace += distanceMoved;
                }
                if (next)
                    neededSpace -= m_layout->moveContainerPush(next, neededSpace);
                if (_dragMoveAC) {
                    distanceMoved = m_layout->moveContainerPush(_dragMoveAC, -neededSpace);
                   _dragIndicator->move(
                       _dragIndicator->x() + distanceMoved,
                       _dragIndicator->y());
                }
            } else {
                neededSpace = _dragIndicator->preferredSize().height() - _dragIndicator->height();
                if (_dragMoveAC) {
                    distanceMoved = m_layout->moveContainerPush(_dragMoveAC, -neededSpace/2);
                    _dragIndicator->move(
                        _dragIndicator->x(),
                        _dragIndicator->y() + distanceMoved);
                    neededSpace += distanceMoved;
                }
                if (next)
                    neededSpace -= m_layout->moveContainerPush(next, neededSpace);
                if (_dragMoveAC) {
                    distanceMoved = m_layout->moveContainerPush(_dragMoveAC, -neededSpace);
                    _dragIndicator->move(
                        _dragIndicator->x(),
                        _dragIndicator->y() + distanceMoved);
                }
            }
        }

        addContainer(a, true);
        m_layout->updateContainerList();
    }

    saveContainerConfig();
    _dragIndicator->hide();
    Kicker::the()->setInsertionPoint(QPoint());
    m_layout->setStretchEnabled(true);
}

bool ContainerArea::eventFilter(QObject* o, QEvent* e)
{
    // Propagate the layout hints which m_contents receives. This way widgets
    // which contain a ContainerArea can react to layout changes of its
    // contents. For example: If an applets grows, the top level widget may
    // want to grow as well.
    if (o == m_contents)
    {
        if (e->type() == QEvent::LayoutHint)
        {
            updateGeometry(); // Posts a new layout hint to our parent.
        }
        return false;
    }

    return Panner::eventFilter(o, e);
}

void ContainerArea::resizeEvent(QResizeEvent *ev)
{
    Panner::resizeEvent(ev);
    setBackgroundTheme();
}

void ContainerArea::viewportResizeEvent(QResizeEvent* ev)
{
    Panner::viewportResizeEvent(ev);
    if (orientation() == Horizontal)
    {
        m_contents->resize(kMax(widthForHeight(ev->size().height()),
                                ev->size().width()),
                           ev->size().height());
    }
    else
    {
        m_contents->resize(ev->size().width(),
                           kMax(heightForWidth(ev->size().width()),
                                ev->size().height()));
    }
    resizeContents(m_contents->width(), m_contents->height());
}

void ContainerArea::setBackgroundTheme()
{
    _bgSet = false;

    if (KickerSettings::transparent())
    {
        if ( !_rootPixmap )
        {
            _rootPixmap = new KRootPixmap(this);
            _rootPixmap->setCustomPainting(true);
            connect(_rootPixmap, SIGNAL(backgroundUpdated(const QPixmap&)),
                    SLOT(updateBackground(const QPixmap&)));
            _rootPixmap->start();
        }
        else
        {
            _rootPixmap->repaint(true);
            _rootPixmap->start();
        }

        double tint = double(KickerSettings::tintValue()) / 100;
        _rootPixmap->setFadeEffect(tint, KickerSettings::tintColor());
        _bgSet = true;
        return;
    }
    else if (_rootPixmap)
    {
        _rootPixmap->stop();
    }

    unsetPalette();

    if (KickerSettings::useBackgroundTheme())
    {
        // by keeping the src image static, we can share it among panels and only
        // reload from disk when it actually changes in the config, not every time we
        // get a resize or configure event
        static QString bgStr;
        static QImage srcImage;
        QString newBgStr = locate("appdata", KickerSettings::backgroundTheme());

        if (bgStr != newBgStr)
        {
            bgStr = newBgStr;
            srcImage.load(bgStr);
        }

        if (!srcImage.isNull())
        {
            QImage bgImage = srcImage;

            if (orientation() == Vertical)
            {
                if (KickerSettings::rotateBackground())
                {
                    QWMatrix matrix;
                    matrix.rotate(position() == KPanelExtension::Left ? 90: 270);
                    bgImage = bgImage.xForm(matrix);
                }

                bgImage = bgImage.scaleWidth( size().width() );
            }
            else
            {
                if (position() == KPanelExtension::Top &&
                    KickerSettings::rotateBackground())
                {
                    QWMatrix matrix;
                    matrix.rotate(180);
                    bgImage = bgImage.xForm(matrix);
                }

                bgImage = bgImage.scaleHeight( size().height() );
            }

            if (KickerSettings::colorizeBackground())
            {
                KickerLib::colorize(bgImage);
            }
            setPaletteBackgroundPixmap(QPixmap(bgImage));
            QTimer::singleShot(0, this, SLOT(updateContainersBackground()));
        }
    }

    _bgSet = true;
}

QRect ContainerArea::availableSpaceFollowing(BaseContainer* a)
{
    QRect availableSpace = rect();
    BaseContainer* b;

    if (a)
    {
        m_containers.findRef(a);
        b = m_containers.next();
    }
    else
        b = m_containers.first();

    if (orientation() == Horizontal)
    {
        if (a)
            availableSpace.setLeft(a->x() + a->width());
        if (b)
            availableSpace.setRight(b->x() - 1);
    }
    else
    {
        if (a)
            availableSpace.setTop(a->y() + a->height());
        if (b)
            availableSpace.setBottom(b->y() - 1);
    }

    return availableSpace;
}

void ContainerArea::moveDragIndicator(int pos)
{
    QRect availableSpace = availableSpaceFollowing(_dragMoveAC);

    // Move _dragIndicator to position pos, restricted by availableSpace.
    // Resize _dragIndicator if necessary.
    if (orientation() == Horizontal)
    {
        if (availableSpace.size().width() <
            _dragIndicator->preferredSize().width())
        {
            _dragIndicator->resize(availableSpace.size());
            _dragIndicator->move(availableSpace.topLeft());
        }
        else
        {
            int newX = pos;
            _dragIndicator->resize(_dragIndicator->preferredSize());
            newX = QMAX(newX, availableSpace.left());
            newX = QMIN(newX,
                availableSpace.right() + 1 - _dragIndicator->width() );
            _dragIndicator->move(newX, availableSpace.top());
        }
    }
    else
    {
        if (availableSpace.size().height() <
            _dragIndicator->preferredSize().height())
        {
            _dragIndicator->resize(availableSpace.size());
            _dragIndicator->move(availableSpace.topLeft());
        }
        else
        {
            int newY = pos;
            _dragIndicator->resize(_dragIndicator->preferredSize());
            newY = QMAX(newY, availableSpace.top());
            newY = QMIN(newY,
                availableSpace.bottom() + 1 - _dragIndicator->height() );
            _dragIndicator->move(availableSpace.left(), newY);
        }
    }
}

void ContainerArea::updateBackground( const QPixmap& pm )
{
    QBrush bgBrush(colorGroup().background(), pm);
    QPalette pal = kapp->palette();
    pal.setBrush(QColorGroup::Background, bgBrush);
    setPalette(pal);

    // because the Pixmap can be smaller as the containerarea
    // we construct a pixmap the same size as we are that every
    // applet or button can use to cut out its background
    _completeBg.resize(width(), height());
    _completeBg.fill(this, 0, 0);

    updateContainersBackground();
}

void ContainerArea::resizeContents(int w, int h)
{
    Panner::resizeContents(w, h);
    m_contents->resize(w, h);
}

void ContainerArea::slotSaveContainerConfig()
{
    saveContainerConfig();
}

void ContainerArea::setPosition(KPanelExtension::Position p)
{
    if (_pos == p)
    {
        return;
    }

    _pos = p;
    Qt::Orientation o = (p == KPanelExtension::Top ||
                         p == KPanelExtension::Bottom) ?
                        Qt::Horizontal : Qt::Vertical;
    bool orientationChanged = (orientation() != o);
    m_layout->setEnabled(false);

    if (orientationChanged)
    {
        setOrientation(o);
        m_layout->setOrientation(o);

        // when we change orientation, we will resize the "width"
        // component down to 0, forcing a resize in resizeContents()
        // when that gets called AFTER we've been moved
        // it's not always safe to do the resize here, as scroll buttons
        // from the panner may get in our way. =/
        if (o == Horizontal)
        {
            resizeContents(0, height());
        }
        else
        {
            resizeContents(width(), 0);
        }

        setBackgroundTheme();
    }

    for (ContainerIterator it(m_containers) ; it.current(); ++it)
    {
        if (orientationChanged)
        {
            (*it)->setOrientation(o);
        }

        (*it)->setPopupDirection( popupDirection() );
    }

    m_layout->setEnabled(true);

    // container extension repaints for us!
    //repaint();
}

void ContainerArea::setAlignment(KPanelExtension::Alignment a)
{
    for (ContainerIterator it(m_containers) ; it.current(); ++it)
    {
        (*it)->setAlignment(a);
    }
}

void ContainerArea::autoScroll()
{
    if(!_moveAC) return;

    if(orientation() == Horizontal) {
        if(_moveAC->pos().x() <= 80)
            scrollBy(-10, 0);
        else if(_moveAC->pos().x() >= width() - _moveAC->width() - 80)
            scrollBy(10, 0);
    }
    else {
        if(_moveAC->pos().y() <= 80)
            scrollBy(0, -10);
        else if(_moveAC->pos().y() >= height() - _moveAC->height() - 80)
            scrollBy(0, 10);
    }
}

void ContainerArea::scrollTo(BaseContainer* b)
{
    if(!b) return;

    int x, y;
    viewportToContents(b->pos().x(), b->pos().y(), x, y);
    ensureVisible(x, y);
}

void ContainerArea::updateContainersBackground()
{
    if (_bgSet)
    {
        if (m_layout)
        {
            m_layout->setEnabled(false);
        }

        for (ContainerIterator it(m_containers); it.current(); ++it)
        {
           it.current()->setBackground();
        }

        if (m_layout)
        {
            m_layout->setEnabled(true);
        }

        // because we blocked updateLayout() calls
        // we have to relayout
        // layoutChildren();
    }
}

ContainerList ContainerArea::containers(const QString& type) const
{
    if (type.isEmpty() || type == "All")
    {
        return m_containers;
    }

    ContainerList list;

    if (type == "Special Button")
    {
        for (ContainerIterator it(m_containers); it.current() ; ++it)
        {
            if (it.current()->appletType() == "KMenuButton" ||
                it.current()->appletType() == "WindowListButton" ||
                it.current()->appletType() == "BookmarksButton" ||
                it.current()->appletType() == "DesktopButton" ||
                it.current()->appletType() == "BrowserButton" ||
                it.current()->appletType() == "ExecButton" ||
                it.current()->appletType() == "ExtensionButton")
            {
                list.append( it.current() );
            }
        }

        return list;
    }

    for (ContainerIterator it(m_containers); it.current() ; ++it)
    {
        if (it.current()->appletType() == type)
        {
            list.append(it.current());
        }
    }

    return list;
}

int ContainerArea::containerCount(const QString& type) const
{
    if (type.isEmpty() || type == "All")
    {
        return m_containers.count();
    }

    int count = 0;
    if (type == "Special Button")
    {
        for (ContainerIterator it(m_containers); it.current() ; ++it)
        {
            if (it.current()->appletType() == "KMenuButton" ||
                it.current()->appletType() == "WindowListButton" ||
                it.current()->appletType() == "BookmarksButton" ||
                it.current()->appletType() == "DesktopButton" ||
                it.current()->appletType() == "BrowserButton" ||
                it.current()->appletType() == "ExecButton" ||
                it.current()->appletType() == "ExtensionButton")
            {
                ++count;
            }
        }

        return count;
    }

    for (ContainerIterator it(m_containers); it.current() ; ++it)
    {
        if (it.current()->appletType() == type)
        {
            ++count;
        }
    }

    return count;
}

void ContainerArea::repaint()
{
    Panner::repaint();
}

const QPixmap* ContainerArea::completeBackgroundPixmap() const
{
    return &_completeBg;
}

int ContainerArea::widthForHeight(int h) const
{
    return m_layout->widthForHeight(h);
}

int ContainerArea::heightForWidth(int w) const
{
    return m_layout->heightForWidth(w);
}


void DragIndicator::paintEvent(QPaintEvent*)
{
    QPainter painter(this);
    QRect rect(0, 0, width(), height());
    style().drawPrimitive( QStyle::PE_FocusRect, &painter, rect, colorGroup(),
                           QStyle::Style_Default, colorGroup().base() );
}

void DragIndicator::mousePressEvent(QMouseEvent*)
{
    hide();
}

#include "containerarea.moc"
