/*
    This file is part of KWeather.
    Copyright (c) 2003 Tobias Koenig <tokoe@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qlineedit.h>

#include <kaboutdata.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kurllabel.h>
#include <kurlrequester.h>

#include "dockwidget.h"
#include "prefdialogdata.h"

#include "kcmweather.h"

extern "C"
{
  KCModule *create_weather( QWidget *parent, const char * )
  {
    return new KCMWeather( parent, "kweather" );
  }
}

KCMWeather::KCMWeather( QWidget *parent, const char *name )
  : KCModule( parent, name )
{
  QVBoxLayout *layout = new QVBoxLayout( this );
  mWidget = new prefsDialogData( this );

  // set the focus to the text entry instead of the config module page selection.
  // note that this dialog is only created once, so this only happens one time.
  mWidget->m_reportLocation->setFocus();

  layout->addWidget( mWidget );

	connect( mWidget->ICAOlink, SIGNAL( leftClickedURL( const QString& ) ),
           SLOT( processURL( const QString& ) ) );
	connect( mWidget->m_enableLog, SIGNAL( toggled( bool ) ),
           SLOT( enableLogWidgets( bool ) ) );
	connect( mWidget->m_viewMode, SIGNAL( released( int ) ),
           SLOT( changeViewMode( int ) ) );
  connect( mWidget->m_reportLocation, SIGNAL( textChanged( const QString& ) ),
           SLOT( reportLocationChanged() ) );
}

void KCMWeather::changeViewMode( int mode )
{
  mViewMode = mode;
  emit changed( true );
}

void KCMWeather::processURL( const QString &url )
{
  kapp->invokeBrowser( url );
}

void KCMWeather::enableLogWidgets( bool value )
{
  mWidget->m_logFile->setEnabled( value );
  mWidget->m_labelLogFile->setEnabled( value );

  emit changed( true );
}

void KCMWeather::reportLocationChanged()
{
  emit changed( true );
}

void KCMWeather::load()
{
  KConfig config( "weather_panelappletrc" );

  config.setGroup( "General Options" );
  bool enabled = config.readBoolEntry( "logging", false );
  mWidget->m_enableLog->setChecked( enabled );
  enableLogWidgets( enabled );

  mWidget->m_logFile->setURL( config.readPathEntry( "log_file_name" ) );
  mWidget->m_reportLocation->setText( config.readEntry( "report_location" ) );
  mWidget->m_viewMode->setButton( config.readNumEntry( "smallview_mode", dockwidget::ShowAll ) );
  changeViewMode( config.readNumEntry( "smallview_mode", dockwidget::ShowAll ) );
  emit changed( false );
}

void KCMWeather::save()
{
  KConfig config( "weather_panelappletrc" );

  config.setGroup( "General Options" );
  config.writeEntry( "logging", mWidget->m_enableLog->isChecked() );
  config.writeEntry( "log_file_name", mWidget->m_logFile->url() );
  config.writeEntry( "report_location", mWidget->m_reportLocation->text() );
  config.writeEntry( "smallview_mode", mViewMode );
  config.sync();

  emit changed( false );
}

void KCMWeather::defaults()
{
  mWidget->m_enableLog->setChecked( false );
  enableLogWidgets( false );

  mWidget->m_logFile->setURL( "" );
  mWidget->m_reportLocation->setText( "" );
  changeViewMode( dockwidget::ShowAll );

  emit changed( true );
}

const KAboutData* KCMWeather::aboutData() const
{
  KAboutData *about = new KAboutData( I18N_NOOP( "kcmweather" ),
                                      I18N_NOOP( "KWeather Configure Dialog" ),
                                      0, 0, KAboutData::License_GPL,
                                      I18N_NOOP( "(c), 2003 Tobias Koenig" ) );

  about->addAuthor( "Tobias Koenig", 0, "tokoe@kde.org" );

  return about;
}

#include "kcmweather.moc"
