/*
 * menu_read.c,v
 * Revision 2.0  1992/04/23  02:52:45  ware
 * First public release.
 *
 * Revision 1.10  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.9  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.8  1992/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.7  1991/12/01  16:14:52  pete
 * Use XoProto().
 *
 * Revision 1.6  1991/08/26  11:58:28  pete
 * Use XoProto() for conditional prototypes.  Working on getting traversals
 * and menus to work more efficiently.  Changed to following naming
 * conventions.
 *
 * Revision 1.5  91/07/19  00:59:55  pete
 * Use shorter file names.  Various speedups.
 *
 * Revision 1.4  1991/06/01  10:03:21  pete
 * Working on menubar
 *
 * Revision 1.3  91/05/29  14:42:16  pete
 * Working on getting the menubar to work.
 *
 * Revision 1.2  1991/05/25  11:03:12  pete
 * Adding external menus
 *
 * Revision 1.1  91/05/23  16:50:43  pete
 * Initial revision
 *
 * Revision 1.1  90/12/19  09:55:25  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <X11/Intrinsic.h>
#include <X11/Xo/XoUnistd.h>
#include "table.h"
#include "hash_fixed.h"
#include <X11/Xo/menu.h>

extern int      hash_str ();

#define TOKEN_EOF	-1
#define TOKEN_NEWLINE	0
#define TOKEN_WORD	1
#define TOKEN_BRACE_OPEN	2
#define TOKEN_BRACE_CLOSE	3
#define TOKEN_MENU	4
#define TOKEN_MNEMONIC	5
#define TOKEN_FUNCTION	6

XoProto (extern int, fgetc, (FILE * fptr));
XoProto (extern int, ungetc, (int ch, FILE * fptr));
XoProto (static XoMenuData, *menu_make, (char *name));
XoProto (static void, menu_add_word, (XoMenuData * m, char *word, int tokentype));
XoProto (static XoMenuData *, menu_make, (char *name));
XoProto (static XoMenuList *, menu_add_to_list, (XoMenuList * menu_list, XoMenuData * menu_current));
XoProto (static void, menu_finish, (XoMenuData *m));
XoProto (static char *, string_new, (char *name));
XoProto (static int, get_word, (FILE * fptr, char *inbuf));
XoProto (static void, m_err, (char *mes));

XoMenuData     *
XoMenuFind (l, name)
	XoMenuList     *l;		/* list of menus */
	char           *name;		/* name of menu to find */
{
	XoMenuData      m;

	if (!l || !name)
		return NULL;
	m.m_name = name;
	return (XoMenuData *) table_find ((Table *) l, (char *) &m);
}

XoMenuList     *
XoMenuReadFile (filename)
	char           *filename;	/* the file to read */
{
	FILE           *fptr;
	char            inbuf[1024];
	int             tokentype;
	int             line;
	int             in_menu;
	int             in_menu_def;
	XoMenuData     *menu_current = NULL;
	XoMenuList     *menu_list = NULL;

	if (!filename || !*filename)
		return ((XoMenuList *) NULL);
	fptr = fopen (filename, "r");
	if (!fptr)
	{
		perror (filename);
		return ((XoMenuList *) NULL);
	}

	line = 1;
	in_menu = in_menu_def = FALSE;
	while ((tokentype = get_word (fptr, inbuf)) != TOKEN_EOF)
	{
		switch (tokentype)
		{
		case TOKEN_NEWLINE:
			++line;
			if (menu_current)
				menu_finish (menu_current);
			break;

		case TOKEN_MENU:
			if (in_menu || in_menu_def)
			{
				m_err ("Menu is a reserved word\n");
			}
			in_menu = TRUE;
			break;

		case TOKEN_MNEMONIC:
		case TOKEN_FUNCTION:
		case TOKEN_WORD:
			if (!in_menu)
			{
				m_err ("Not defining a menu\n");
			}
			else if (!in_menu_def)
			{
				/*
				 * This must be the name of the menu
				 */
				if (menu_current)
				{
					m_err (menu_current->m_name);
					m_err ("Already named this menu\n");
					free (menu_current->m_name);
					free ((char *) menu_current);
				}
				menu_current = menu_make (inbuf);
				menu_list = menu_add_to_list (menu_list, menu_current);
			}
			else
			{
				/*
				 * Insert this into the current list
				 */
				menu_add_word (menu_current, inbuf, tokentype);
			}
			break;
		case TOKEN_BRACE_OPEN:
			if (!in_menu)
			{
				m_err ("Not defining a menu\n");
				break;
			}
			if (!menu_current)
			{
				m_err ("Menu does not have a name\n");
				/*
				 * Make a menu but do not insert it into
				 * menu_list.  This will let us parse the
				 * rest of this menu
				 */
				menu_current = menu_make ("<none>");
			}
			if (in_menu_def)
			{
				m_err ("Second { ignored\n");
			}
			in_menu_def = TRUE;
			break;
		case TOKEN_BRACE_CLOSE:
			menu_finish (menu_current);
			menu_current = NULL;
			in_menu = in_menu_def = FALSE;
			break;
		}
	}
	if (menu_current)
	{
		m_err ("EOF before closing the menu\n");
		menu_finish (menu_current);
	}
	return menu_list;
}

static XoMenuData *
menu_make (name)
	char           *name;
{
	XoMenuData     *m;

	m = (XoMenuData *) malloc (sizeof (XoMenuData));
	if (!m)
	{
		perror ("malloc");
		return NULL;
	}
	m->m_name = string_new (name);
	m->m_current_item = NULL;
	m->m_numitems = 0;
	m->m_list = (XoMenuItem **) malloc (sizeof (XoMenuItem *));
	if (!m->m_list)
	{
		perror ("malloc");
		free ((char *) m);
		return NULL;
	}
	return m;
}

static void
menu_add_word (m, word, tokentype)
	XoMenuData     *m;
	char           *word;
	int             tokentype;
{
	XoMenuItem     *item;

	if (!m || !word || !*word)
		return;
	item = m->m_current_item;
	if (!item)
	{
		item = m->m_current_item =
			(XoMenuItem *) malloc (sizeof (XoMenuItem));
		if (!item)
		{
			perror ("malloc");
			return;
		}
		item->i_label = NULL;
		item->i_mnemonic = NULL;
		item->i_accelerator = NULL;
		item->i_function = NULL;
		item->i_argc = 0;
		item->i_argv = (char **) malloc (sizeof (char *));
	}
	switch (tokentype)
	{
	case TOKEN_WORD:
		if (!item->i_label)
		{
			item->i_label = string_new (word);
		}
		else if (!item->i_accelerator && !item->i_function)
		{
			item->i_accelerator = string_new (word);
		}
		else if (item->i_function)
		{
			++item->i_argc;
			item->i_argv = (char **) realloc ((char *) item->i_argv,
							  sizeof (char *) *
							  (unsigned) item->i_argc);
			item->i_argv[item->i_argc - 1] = string_new (word);
		}
		break;
	case TOKEN_MNEMONIC:
		item->i_mnemonic = string_new (word + 1);
		break;
	case TOKEN_FUNCTION:
		item->i_function = string_new (word + 2);
		break;
	default:
		break;
	}
}

static void
menu_finish (m)
	XoMenuData     *m;
{
	if (!m || !m->m_current_item)
		return;
	++m->m_numitems;
	m->m_list = (XoMenuItem **)
		realloc ((char *) m->m_list,
			 sizeof (XoMenuItem *) * (unsigned) m->m_numitems);
	m->m_list[m->m_numitems - 1] = m->m_current_item;
	m->m_current_item = NULL;
}

#define STATE_SPACE	0
#define STATE_COMMENT	1
#define STATE_STRING	2
#define STATE_WORD	3

static int
get_word (fptr, inbuf)
	FILE           *fptr;
	char           *inbuf;
{
	char           *bufptr;
	int             state;
	int             ch;

	bufptr = inbuf;

	if (!bufptr)
		return EOF;
	state = STATE_SPACE;
	while ((ch = fgetc (fptr)) != EOF)
	{
		switch (state)
		{
		case STATE_SPACE:
			if (ch == '#')
			{
				state = STATE_COMMENT;
			}
			else if (ch == '"')
			{
				state = STATE_STRING;
			}
			else if (ch == '{')
			{
				*inbuf = EOS;
				return TOKEN_BRACE_OPEN;
			}
			else if (ch == '}')
			{
				*inbuf = EOS;
				return TOKEN_BRACE_CLOSE;
			}
			else if (ch == '\n')
			{
				*inbuf = EOS;
				return TOKEN_NEWLINE;
			}
			else if (!isspace (ch))
			{
				*bufptr++ = ch;
				state = STATE_WORD;
			}
			break;
		case STATE_COMMENT:
			if (ch == '\n')
			{
				*bufptr = EOS;
				return TOKEN_NEWLINE;
			}
			break;
		case STATE_STRING:
			if (ch == '"')
			{
				*bufptr++ = EOS;
				return TOKEN_WORD;
			}
			else
			{
				*bufptr++ = ch;
			}
			break;
		case STATE_WORD:
			if (ch == '\n' || ch == '{' || ch == '}' || isspace (ch))
			{
				ungetc (ch, fptr);
				*bufptr++ = EOS;
				if (*inbuf == '_')
				{
					return TOKEN_MNEMONIC;
				}
				else if (inbuf[0] == 'f' && inbuf[1] == '.')
				{
					return TOKEN_FUNCTION;
				}
				else if (strcmp (inbuf, "Menu") == 0)
				{
					return TOKEN_MENU;
				}
				else
				{
					return TOKEN_WORD;
				}
			}
			else
			{
				*bufptr++ = ch;
			}
			break;
		}
	}
	return EOF;
}

static void
menu_item_free (item)
	XoMenuItem     *item;
{
	int             i;

	if (!item)
		return;
	if (item->i_label)
	{
		free (item->i_label);
		item->i_label = NULL;
	}
	if (item->i_mnemonic)
	{
		free (item->i_mnemonic);
		item->i_mnemonic = NULL;
	}
	if (item->i_accelerator)
	{
		free (item->i_accelerator);
		item->i_accelerator = NULL;
	}
	if (item->i_function)
	{
		free (item->i_function);
		item->i_function = NULL;
	}
	for (i = 0; item->i_argv && i < item->i_argc; i++)
	{
		if (item->i_argv[i])
		{
			free (item->i_argv[i]);
			item->i_argv[i] = NULL;
		}
	}
	if (item->i_argv)
	{
		free ((char *) item->i_argv);
		item->i_argv = NULL;
	}
	free ((char *) item);
}

static int
menu_free (d)
	char           *d;
{
	XoMenuData     *menu = (XoMenuData *) d;
	int             i;

	if (!menu)
		return 0;
	if (menu->m_name)
	{
		free (menu->m_name);
		menu->m_name = NULL;
	}
	if (menu->m_current_item)
	{
		free ((char *) menu->m_current_item);
		menu->m_current_item = NULL;
	}
	for (i = 0; menu->m_list && i < menu->m_numitems; i++)
	{
		if (menu->m_list[i])
		{
			menu_item_free (menu->m_list[i]);
			menu->m_list[i] = NULL;	/* I'm just cautious */
		}
	}
	if (menu->m_list)
	{
		menu_item_free (menu->m_list);
		menu->m_list = NULL;
	}
	free ((char *) menu);
	return 0;
}

static int
menu_compare (d1, d2)
	char           *d1;
	char           *d2;
{
	XoMenuData     *m1 = (XoMenuData *) d1;
	XoMenuData     *m2 = (XoMenuData *) d2;

	if (m1 == m2)
		return 0;
	if (!m1 || !m1->m_name)
		return -1;
	if (!m2 || !m2->m_name)
		return 1;
	return strcmp (m1->m_name, m2->m_name);
}

static int
menu_hash (d1)
	char           *d1;
{
	XoMenuData     *m1 = (XoMenuData *) d1;

	if (!m1)
		return 0;
	return hash_str (m1->m_name);
}

static XoMenuList *
menu_add_to_list (menu_list, menu_current)
	XoMenuList     *menu_list;
	XoMenuData     *menu_current;
{
	Table          *t;

	if (!menu_current)
		return menu_list;

	t = (Table *) menu_list;
	if (!t)
	{
		t = table_hash_fixed_create (100, menu_free, menu_compare,
					     menu_hash);
		if (!t)
		{
			m_err ("Unable to create menu table\n");
			return NULL;
		}
	}
	table_add (t, (char *) menu_current);
	return (XoMenuList *) t;
}

static void
m_err (mes)
	char           *mes;
{
	fprintf (stderr, "%s\n", mes);
}

static
char           *
string_new (str)
	char           *str;
{
	char           *ptr;

	if (!str)
		return str;
	if ((ptr = malloc ((unsigned) strlen (str) + 1)) != NULL)
		(void) strcpy (ptr, str);
	return ptr;
}
