/*
 * Row.c,v 2.2 1992/08/11 00:07:24 pete Exp
 * Row.c,v
 * Revision 2.2  1992/08/11  00:07:24  pete
 * Added geometry manager so children can get new size.
 *
 * Revision 2.1  1992/07/11  20:02:41  pete
 * QueryGeometry was using uninitialized fields from proposed.
 *
 * Revision 2.0  1992/04/23  02:51:53  ware
 * First public release.
 *
 * Revision 1.5  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.4  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.3  1992/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.2  1991/09/12  09:49:40  pete
 * Added QueryGeometry method so the MenuBar will layout correctly.
 * Implemented wrapping rows if there is not enough room.
 *
 * Revision 1.1  91/08/30  17:41:14  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xo/RowP.h>
#include <X11/Xo/dbug.h>

#include <X11/Xo/RowRec.h>

XoProto (static void, DoLayout, (Widget gw));
XoProto (static void, CalcSize, (Widget gw, Dimension *width_ret, Dimension *height_ret));
XoProto (static void, CalcHeight, (Widget gw, int new_width, Dimension *height_ret));
XoProto (static void, AskNewSize, (XoRowWidget w, int width, int height, int query));

/*
 *----------------------------------------------------------------------
 * Core Methods
 *----------------------------------------------------------------------
 */

/*
 * Resize -	Do whatever layout is necessary to fix up the size.
 *		All the work is done in DoLayout().
 */
static void
Resize (gw)
	Widget          gw;
{
	DBUG_ENTER ("Row.Resize");
	DoLayout (gw);
	DBUG_VOID_RETURN;
}

static XtGeometryResult
QueryGeometry (gw, proposed, desired)
	Widget          gw;
	XtWidgetGeometry *proposed;
	XtWidgetGeometry *desired;
{
	XoRowWidget     w = (XoRowWidget) gw;

	/*
	 * I'm just being asked my ideal size
	 */
	if (!XoGeoDimFlags (proposed))
	{
		CalcSize (gw, &desired->width, &desired->height);
		desired->request_mode = CWWidth | CWHeight;
		if (desired->width == w->core.width &&
		    desired->height == w->core.height)
			return XtGeometryNo;
		else
			return XtGeometryAlmost;
	}
	/*
	 * Only asking about the width.  Respond by calculating a new height
	 */
	if (XoGeoDimFlags (proposed) == CWWidth)
	{
		Dimension       width;
		Dimension       height;

		CalcSize (gw, &width, &height);
		if (width <= proposed->width && height == gw->core.height)
		{
			return XtGeometryYes;
		}
		else
		{
			/*
			 * So, we want to be wider then our parent want's us
			 * to be.  Lets try to lay our children out in the
			 * shorter space.
			 */
			CalcHeight (gw, (int) proposed->width, &desired->height);
			desired->request_mode = CWHeight;
			return XtGeometryAlmost;
		}
	}
	else if (XoGeoDimFlags (proposed) == CWHeight)
	{
		/*
		 * We can't really do to much with the width since that is
		 * how we try to lay things out, so just say we prefer our
		 * current state.
		 */
		return XtGeometryNo;
	}
	else if (XoGeoDimFlags (proposed))
	{
		/*
		 * We are being told here is your width and height.  What can
		 * we do but say we prefer our current ones?
		 */
		return XtGeometryNo;
	}
	else
	{
		/*
		 * Everything else the parent is free to do as it wants
		 */
		return XtGeometryYes;
	}
}

/*
 *----------------------------------------------------------------------
 * Composite Methods
 *----------------------------------------------------------------------
 */

/*
 * GeometryManager -
 * 	Responds to request for changes in size of position
 *	of children.  (See pp 228 of Asente & Swick).
 *
 *	Requests for new x & y are denied.  Changes in height cause
 *	a new height to be calculated and the parent asked if it
 *	will work.  If not, then a new layout is calculated with
 *	the requested height being used.
 *
 *	Changes in width cause a new maximum width to be calculated.  If
 *	it is different then the parent is asked what it thinks of the
 *	change.  If allowed, then all children are resized to the
 *	same width.  Otherwise the request is denied.
 */

static XtGeometryResult
GeometryManager (gw, request, geo_ret)
	Widget          gw;
	XtWidgetGeometry *request;
	XtWidgetGeometry *geo_ret;
{
	XoRowWidget  	w;		/* this row widget */
	Dimension       new_width;	/* our new width */
	Dimension       new_height;	/* our new height */
	Dimension       old_height;	/* child's old height */
	XtGeometryResult result;	/* parents answer */

	w = (XoRowWidget) XtParent (gw);
	if (!XoGeoDimFlags (request))
		return XtGeometryNo;
	if (XoGeoDimFlags (request) == CWWidth)
	{
		/*
		 * Sure, we'll let it change it's width.
		 */
		if (!(request->request_mode & XtCWQueryOnly))
		{
			XtResizeWidget (gw, request->width,
					gw->core.height,
					gw->core.border_width);
			CalcSize ((Widget) w, &new_width, &new_height);
			AskNewSize (w, (int) new_width,
				      (int) new_height,
				      (int) (request->request_mode & XtCWQueryOnly));

			result = XtGeometryDone;
		}
		else
		{
			result = XtGeometryYes;
		}
	}
	else if (XoGeoDimFlags (request) == CWHeight)
	{
		/*
		 * 1. We set the child's height to this new height.
		 */
		old_height = gw->core.height;
		gw->core.height = request->height;
		/*
		 * 2. Query all the children to determine our "optimum" size
		 * (i.e. CalcSize).
		 */
		CalcSize ((Widget) w, &new_width, &new_height);
		gw->core.height = old_height;
		AskNewSize (w, (int) new_width, (int) new_height,
			     (int) (request->request_mode & XtCWQueryOnly));
		if (gw->core.height == request->height)
			result = XtGeometryDone;
		else
			result = XtGeometryNo;
	}
	else
		result = XtGeometryNo;
	return result;
}


/*
 * ChangeManaged - Called whenever a child is managed or unmanaged.
 *		Calculates desired size, requests that and then
 *		used DoLayout() to place everything.  Note that
 *		DoLayout() returns immediately if the doLayout resource
 *		is False.
 */

static void
ChangeManaged (gw)
	Widget          gw;
{
	XoRowWidget     w = (XoRowWidget) gw;
	Dimension       width;
	Dimension       height;

	DBUG_ENTER ("Row.ChangeManaged");
	CalcSize (gw, &width, &height);
	if (width != w->core.width || height != w->core.height)
	{
		XtMakeResizeRequest (gw, width, height, (Dimension *) NULL,
				     (Dimension *) NULL);
	}
	DoLayout (gw);
	DBUG_VOID_RETURN;
}

/*
 *----------------------------------------------------------------------
 * Row Methods
 *----------------------------------------------------------------------
 */


/*
 *----------------------------------------------------------------------
 * Private Utilities
 *----------------------------------------------------------------------
 */

/*
 * DoLayout -	Arrange the children in a horizontal column each the
 *		same height.  Preserve the width of each child.
 *
 *		If cons_base.do_layout is False, return immediately.
 */

static void
DoLayout (gw)
	Widget          gw;
{
	XoRowWidget     w = (XoRowWidget) gw;
	Widget          c;		/* child */
	int             i;		/* current index */
	Position        x;		/* */
	Position        y;
	int             count;		/* number of children on row */
	Dimension       max_height;	/* highest child on current row */

	DBUG_ENTER ("Row.DoLayout");
	if (!w->cons_base.do_layout)
		DBUG_VOID_RETURN;
	x = 0;
	y = 0;
	max_height = 0;
	count = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		c = w->composite.children[i];
		if (!XtIsManaged (c))
			continue;
		if (x + c->core.width + 2 * c->core.border_width > w->core.width && count)
		{
			DBUG_PRINT ("wrap", ("Wrapping in %s on widget %s",
					     XoName (gw), XoName (c)));
			x = 0;
			y += max_height;
			count = 0;
			max_height = 0;
		}
		if (c->core.height + c->core.border_width > max_height)
			max_height = c->core.height + c->core.border_width;
		/*
		 * FIX: Make this a XtMoveWidget()
		 */
		XtConfigureWidget (c, x, y,
				   c->core.width,
				   c->core.height,
				   c->core.border_width);
		x += c->core.width + 2 * c->core.border_width;
		++count;
	}
	DBUG_VOID_RETURN;
}

/*
 * CalcSize -	Calculate the desired size of this Row.  Does
 *		so by querying each of the managed children for
 *		a desired size.  The maximum height is used as the
 *		height_ret and the width is the sum of the width of the
 *		children plus 2 times the width of each childs border.
 */
static void
CalcSize (gw, width_ret, height_ret)
	Widget          gw;
	Dimension      *width_ret;
	Dimension      *height_ret;
{
	XoRowWidget     w = (XoRowWidget) gw;
	int             i;
	Widget          c;		/* child */
	Dimension       height;
	Dimension       width;
	XtGeometryResult result;	/* unused */
	XtWidgetGeometry geo;

	DBUG_ENTER ("Row.CalcSize");
	width = height = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		c = w->composite.children[i];
		if (!XtIsManaged (c))
			continue;
		result = XtQueryGeometry (c, (XtWidgetGeometry *) NULL, &geo);
		if (geo.height + 2 * geo.border_width > height)
			height = geo.height + 2 * geo.border_width;
		width += geo.width + 2 * geo.border_width;
	}
	if (width_ret)
		*width_ret = width;
	if (height_ret)
		*height_ret = height;
	DBUG_VOID_RETURN;
}

/*
 * Using the given width, calculate what our height should be
 */
static void
CalcHeight (gw, new_width, height_ret)
	Widget          gw;
	int             new_width;
	Dimension      *height_ret;
{
	XoRowWidget     w = (XoRowWidget) gw;
	int             i;
	Widget          c;
	XtWidgetGeometry geo;
	Boolean         first;		/* is this first managed child? */
	int             count;		/* number of children in current row */
	Dimension       width;		/* width of current row */
	Dimension       height;		/* new height */
	Dimension       max_height;	/* heighest child in current row */

	DBUG_ENTER ("Row.CalcHeight");
	first = True;
	height = 0;
	count = 0;
	width = 0;
	max_height = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		c = w->composite.children[i];
		if (!XtIsManaged (c))
			continue;
		(void) XtQueryGeometry (c, (XtWidgetGeometry *) NULL, &geo);
		if (first)
		{
			width = geo.border_width;
			first = False;
		}
		if (width + geo.width + geo.border_width > new_width && count)
		{
			/*
			 * Start a new row
			 */
			DBUG_PRINT ("row", ("Starting a new row with %s",
					    XoName (c)));
			height += max_height;
			max_height = geo.height + 2 * geo.border_width;
			width = geo.border_width;
			/*
			 * Set count to 0 so we always put at least one child
			 * on a row
			 */
			count = 0;
		}
		else if (geo.height + 2 * geo.border_width > max_height)
		{
			max_height = geo.height + 2 * geo.border_width;
			++count;
		}
		else
		{
			++count;
		}
		width += geo.width + geo.border_width;
	}
	height += max_height;
	if (height_ret)
		*height_ret = height;
	DBUG_VOID_RETURN;
}

/*
 * AskNewSize -
 *	Ask our parent for a new height and try several times.
 */

static void
AskNewSize (w, width, height, query)
	XoRowWidget	w;
	int             width;
	int             height;
	int             query;
{
	XtWidgetGeometry request;	/* what we want */
	XtWidgetGeometry reply;		/* what the parent says */
	XtGeometryResult result;
	int             i;

	DBUG_ENTER ("Row.AskNewSize");
	if (w->core.height == height && w->core.width == width)
		return;
	/*
	 * Set this flag to True in case to avoid unnecessary resize
	 */
	w->cons_base.resize_flag = True;
	for (i = 0; i < XO_GEO_NUMREQUEST; i++)
	{
		request.request_mode = CWHeight;
		if (width != w->core.width)
			request.request_mode |= CWWidth;
		request.width = width;
		request.height = height;
		
		/*
		 * Set a flag so we know if the Resize Method has
		 * been called.  It is set to True by Resize().  If
		 * it is still False, then call Resize() by hand.
		 */
		if (query)
		{
			request.request_mode |= XtCWQueryOnly;
		}
		else
		{
			w->cons_base.resize_flag = False;
		}
		result = XtMakeGeometryRequest ((Widget) w, &request, &reply);
		if (result != XtGeometryAlmost)
			break;
		width = reply.width;
		height = reply.height;
	}
	if (!w->cons_base.resize_flag)
	{
		ThisClass(w).core_class.resize ((Widget) w);
	}
	DBUG_VOID_RETURN;
}
