/*
 * output_util.c,v 2.0 1992/04/23 02:48:04 ware Exp
 * output_util.c,v
 * Revision 2.0  1992/04/23  02:48:04  ware
 * First public release.
 *
 * Revision 1.9  1992/02/04  21:21:23  pete
 * Release 44
 *
 * Revision 1.8  1991/08/26  11:34:10  pete
 * Added output_filename() to enforce maximum length on filenames.
 *
 * Revision 1.7  1991/06/29  18:32:20  pete
 * Rework backup scheme.  Instead of always doing the backups, use a
 * temporary file and only do the backup if the two differ.  Also, make
 * the files read-only to hinder direct changes.
 *
 */

#include <stdio.h>
#include <ctype.h>
#include "build.h"
#include INCLUDEFILEOPS
#include "output.h"
#include "str_util.h"

extern char	*rindex ();

static void
output_maketemp (name, temp)
char		*name;
char		*temp;
{
	if (!temp)
		return;
	if (name)
	{
		strcpy (temp, name);
		strcat (temp, ".tmp");
	}
	else
	{
		sprintf (temp, "%d.tmp", getpid ());
	}
}

/*
 * Get the file pointer for a new file.  Insures any old versions of
 * the file are renamed.
 */

FILE *
output_open (name, backupname)
char		*name;
char		*backupname;
{
	FILE		*fptr;		/* the opened file */
	char		buf[4096];	/* path to file */

	if (!name || !*name || !backupname)
	{
		fprintf (stderr, "name is NULL\n");
		return ((FILE *) NULL);
	}
	output_maketemp (name, buf);
	*backupname = '\0';
	if ((fptr = fopen (buf, "w+")) == NULL)
	{
		perror (buf);
	}
	return fptr;
}

int
output_close (fptr, new, old)
FILE		*fptr;
char		*new;			/* newly created file */
char		*old;			/* name of the backup */
{
	char	buf[4096];
	int	diff = FALSE;

	output_maketemp (new, buf);
	if (fclose (fptr) < 0)
	{
		perror (buf);
		(void) unlink (buf);
		return (diff);
	}
	if (access (new, F_OK) != 0)
	{
		if (rename (buf, new) < 0)
		{
			perror (buf);
		}
		else
		{
			chmod (new, 0444);
		}
	}
	else if ((diff = filecmp (buf, new)) != 0)
	{
		if (backup (new, old) < 0)
		{
			perror (new);
		}
		if (rename (buf, new) < 0)
		{
			perror (buf);
		}
		else
		{
			chmod (new, 0444);
		}
	}
	else
	{
		if (unlink (buf) < 0)
			perror (buf);
	}
	return diff;
}

int
backup (name, saved)
char		*name;			/* file to move to a backup */
char		*saved;			/* returned name of the backup */
{
	int	version;		/* the backup version */
	int	status = 0;		/* return status < 0 ==> error */

	for (version = 1;;++version)
	{
		sprintf (saved, "%s.~%d~", name, version);
		/*
		 * If the file does not exist, then that is what this
		 * version will become
		 */
		if (access (saved, F_OK) < 0)
		{
			if (rename (name, saved) < 0)
			{
				status = -1;
			}
			break;
		}
	}
	return status;
}

int
filecmp (f1, f2)
char		*f1;
char		*f2;
{
	FILE	*fp1;
	FILE	*fp2;
	char	buf1[BUFSIZ];
	char	buf2[BUFSIZ];
	int	status;

	if ((fp1 = fopen (f1, "r")) == NULL)
		return 1;
	if ((fp2 = fopen (f2, "r")) == NULL)
	{
		fclose (fp1);
		return 1;
	}
	while (fgets (buf1, sizeof (buf1), fp1) != NULL)
	{
		if (fgets (buf2, sizeof (buf2), fp2) == NULL)
			break;
		if (strcmp (buf1, buf2))
			break;
	}
	if (feof (fp1))
	{
		if (fgets (buf2, sizeof (buf2), fp2) == NULL
		    && feof (fp2))
			status = 0;
		else
			status = 1;
	}
	else
	{
		status = 1;
	}
	(void) fclose (fp1);
	(void) fclose (fp2);
	return status;
}

int
output_goto (fptr, dst, at)
FILE		*fptr;			/* where to write to */
int		dst;			/* column to end up at */
int		at;			/* current column (starting at 1) */
{
	int	i;

	/*
	 * Need to use tabs!
	 */
#define TABSIZE 8
#define NEXTTAB(col)	(col - (col % TABSIZE) + TABSIZE)
	for (i = at; i < dst && NEXTTAB (i) <= dst; i = NEXTTAB(i))
		fputc ('\t', fptr);
	while (i < dst)
	{
		fputc (' ', fptr);
		++i;
	}
	/*
	 * Insure at least one space
	 */
	if (i == at)
	{
		fputc (' ', fptr);
		++i;
	}
	return i;
}

int
output_str (fptr, str, col)
FILE		*fptr;			/* file to write to */
char		*str;			/* string to write */
int		col;			/* current column */
{
	while (*str)
	{
		switch (*str)
		{
		case '\n':
			col = 0;
			break;
		case '\t':
			col = NEXTTAB(col);
			break;
		default:
			++col;
			break;
		}
		fputc (*str, fptr);
		++str;
	}
	return col;
}

void
output_unstylized (name, out)
char		*name;			/* name in stylized format */
char		*out;			/* where to put result */
{
	int	lastupper;		/* if last character was upper case */

	if (!out)
		return;
	if (!name)
	{
		*out = '\0';
		return;
	}
	lastupper = 1;
	while (*name)
	{
		if (isupper (*name) && !lastupper)
		{
			*out++ = '_';
			*out = tolower (*name);
			lastupper = 1;
		}
		else
		{
			lastupper = 0;
			*out = *name;
		}
		out++;
		name++;
	}
}

int
build_dir (dir)
char		*dir;
{
	if (!dir || !*dir)
		return -1;
	if (access (dir, W_OK) == 0)
		return 0;
	if (mkdir (dir, 0777) < 0)
	{
		perror (dir);
		return -1;
	}
	else
	{
		return 0;
	}
}

int
co_file (name, lock)
char		*name;
int		lock;
{
	char	rcsdir[2048];
	char	*file;
	char	rcspath[2048];
	char	command[2048];
	int	status;

	if (!file)
		return 0;
	strcpy (rcsdir, name);
	file = rindex (rcsdir, '/');
	if (file)
		*file++ = '\0';
	else
	{
		file = name;
		strcpy (rcsdir, ".");
	}
	/*
	 * If the file is writable, assume it's already been
	 * checked out from RCS.
	 */
	if (access (name, W_OK) == 0)
		return 0;

	/*
	 * See if there is an RCS file.  First try it without the
	 * RCS/ prefix and then with the RCS/ prefix.
	 */
	sprintf (rcspath, "%s/%s,v", rcsdir, file);
	if (access (rcspath, F_OK) != 0)
	{
		sprintf (rcspath, "%s/RCS/%s,v", rcsdir, file);
		if (access (rcspath, F_OK) != 0)
			return 0;
	}
	sprintf (command, "co %s %s %s", lock?"-l":"", rcspath, name);
	printf ("%s\n", command);
	status = system (command);
	return (status == 0);
}

/*
 * Return a filename of the correct size.  The rule is that it must fit in
 * twelve characters and must include all the suffix.  The name is assumed
 * to be stylized, with each word starting with a capital letter.  As the
 * name is shortened, all but the capital letter is retained.  The last word
 * to be stripped, has enough characters saved to fill out the maximum
 */

#define MAX_LEN	12
#define MAX_FILE	256		/* max characters in a filename */

void
output_filename (file, suffix, out)
char		*file;
char		*suffix;
char		*out;
{
	int		space_left;	/* characters remaining */
	int		wordcount;	/* number of words in file */
	char		**wordlist;	/* list of words in file */
	char		**list;		/* for examing wordlist */
	char		*outptr;	/* pointer into out */

	if (!file || !suffix || !out)
		return;

	wordlist = str_list_stylized (file);
	if (!wordlist)
	{
		strcpy (out, suffix);
		return;
	}
	wordcount = str_list_count (wordlist);

	/*
	 * The first word is special in that it is always copied even
	 * if it is partial.
	 */
	space_left = MAX_LEN - strlen (suffix);
	strncpy (out, wordlist[0], space_left - (wordcount - 1));
	out[space_left - (wordcount - 1)] = '\0';

	outptr = &out[strlen (out)];
	for (list = &wordlist[1]; list && *list; list++)
	{
		space_left = MAX_LEN - strlen (suffix) - strlen (out);
		--wordcount;
		if (strlen (*list) + wordcount - 1 <= space_left)
		{
			strcat (outptr, *list);
			outptr += strlen (*list);
		}
		else
		{
			*outptr++ = *list[0];
			*outptr = '\0';
		}
	}
	str_list_free (wordlist);
	strcat (out, suffix);
}

