
/*

________________________________________________________________

        
        $Id: morphClose.c,v 1.20 1997/01/16 16:05:32 svein Exp $
        Copyright 1992, Blab, UiO
        Image processing lab, Department of Informatics
        University of Oslo
        E-mail: blab@ifi.uio.no
________________________________________________________________
  
  Permission to use, copy, modify and distribute this software and its
  documentation for any purpose and without fee is hereby granted, 
  provided that this copyright notice appear in all copies and that 
  both that copyright notice and this permission notice appear in supporting
  documentation and that the name of B-lab, Department of Informatics or
  University of Oslo not be used in advertising or publicity pertaining 
  to distribution of the software without specific, written prior permission.

  B-LAB DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL B-LAB
  BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
  OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

*/

static char *Id = "$Id: morphClose.c,v 1.20 1997/01/16 16:05:32 svein Exp $, Blab, UiO";

#include <xite/includes.h>
#include <xite/biff.h>
#include <stdlib.h>
#include <xite/blab.h>
#include <xite/convert.h>
#include <xite/message.h>
#include <xite/morph.h>
#include <xite/readarg.h>



/*F:morphClose*

________________________________________________________________

		morphClose
________________________________________________________________

Name:		morphClose  - morphological grayscale close on a band.

Syntax:         | #include <xite/morph.h>
		|
                | int morphClose( IBAND inband, IBAND outband,
                |    ISS_BAND B );

Description:    'morphClose' calculates the morphological operation
                close on the band 'inband', with a structuring
                element 'B'. The result is sent to the band spesified
		in 'outband'. The actual calculation is performed with the
		expression:

		| outband = morphErode(morphDilate(inband))

Restrictions:   'inband' and 'outband' must have pixel type unsigned byte,
                'B' must have pixel type signed short.

See also:       morphClose(1), morphOpen(3), morphDilate(3), morphErode(3)

Return value:   | 0  :  OK
                | 1  :  Bad input pixel type
		| 2  :  Bad output pixel type
		| 3  :  Bad pixel type for structuring element

Author:		Martin Torpe Lie, BLAB, Ifi, UiO

Id: 		$Id: morphClose.c,v 1.20 1997/01/16 16:05:32 svein Exp $
________________________________________________________________

*/

#ifndef FUNCPROTO
int morphClose(inband, outband, B)
IBAND inband, outband;
ISS_BAND B;
#else /* FUNCPROTO */
int morphClose(IBAND inband, IBAND outband, ISS_BAND B)
#endif /* FUNCPROTO */
{
  int status;
  IBAND tmpband;

  tmpband = Imake_band(Iu_byte_typ, Ixsize(inband), Iysize(inband));
  status = morphDilate(inband, tmpband, B);
  if (status) return(Error(status, "morphOpen: Error in morphDilate\n"));
  status = morphErode(tmpband, outband, B);
  if (status) return(Error(status, "morphOpen: Error in morphErode\n"));
  Idel_band(&tmpband);

  return(0);
}



/*P:morphClose*

________________________________________________________________

		morphClose
________________________________________________________________

Name:		morphClose - morphological grayscale close on an image.

Syntax:		morphClose <-a | -b> <struct_element> <inimage> <outimage>

Description:    'morphClose' calculates the morphological operation
                Close, on the image 'inimage', with a structuring
                element given by 'struct_element' (in combination with
                option '-a' or '-b').  The result is sent to the image
		spesified in 'outimage'.

		The actual calculation is performed with the
		expression:

		| outband = morphErode(morphDilate(inband))

Restrictions:	Input images must be of the following types:
                
                &inimage
                Unsigned byte.

                &struct_element
                With option '-a': Ascii characters (numbers) describing the
                structuring element. With option '-b': BIFF image with pixel
                type signed short.

Options:        &-a struct_element
                'struct_element' is an ascii file which contains the
                structuring element. See ascii2biff(1) for file format.

                &-b struct_element
                'struct_element' is a BIFF image which contains the
                structuring element.

See also:	morphClose(3), morphDilate(1), morphErode(1), morphOpen(1),
                ascii2biff(1)

Return value:   | 0 : OK
                | 1 : Usage message
		| 2 : Illegal number of arguments
		| 3 : No structuring element

Author:		Martin Torpe Lie, BLAB, Ifi, UiO

Examples:       morphClose ~blab/img/mona.img mona.img -a filter.txt
                morphClose ~blab/img/mona.img mona.img -b filter.img

Id: 		$Id: morphClose.c,v 1.20 1997/01/16 16:05:32 svein Exp $
________________________________________________________________

*/

#ifdef MAIN

#ifndef FUNCPROTO
int main(argc, argv)
int argc;
char *argv[];
#else /* FUNCPROTO */
int main(int argc, char **argv)
#endif /* FUNCPROTO */
{
  IMAGE inimage, outimage;
  ISS_IMAGE B;
  ISS_BAND B_band;
  int bn;
  char *args;
  char *a,*b;

  InitMessage(&argc,argv, xite_app_std_usage_text(
    "Usage: %s <inimage> <outimage> -a <ascii struct elem>\n\
       or: %s <inimage> <outimage> -b <biff struct elem>\n"));

  Iset_message(TRUE);

  if (argc == 1) Usage(1, NULL);
  args = argvOptions(argc, argv);

  a = read_switch(&argc, argv, "-a", 1, NULL);
  b = read_switch(&argc, argv, "-b", 1, NULL);

  if (argc NE 3) Usage(2,"Illegal number of arguments.\n");

  inimage  = Iread_image(argv[1]);

  if (a) ascii2biff((IBAND *)&B_band, a, Is_short_typ);
  else if (b) {
    B = (ISS_IMAGE) Iread_image(b);
    B_band = B[1];
  } else Usage(3, "No structuring element.\n");

  outimage = Icopy_init(inimage);
  for (bn=1; bn LE Inbands(inimage); bn++)
    morphClose(inimage[bn], outimage[bn], B_band);

  Ihistory(outimage, argv[0], args);
  Iwrite_image(outimage, argv[2]);

  return(0);
}

#endif





