/*
 * Utility Functions to implement consistent error logging mechanisms.
 * Specification requires that this work on VMS and SunOS (UNIX).  Tested
 * on ULTRIX and AIX as well.
 */

#ifndef	lint
    static char afsid[]	= "$__Header$";
    static char rcsid[]	= "$Id: uderrmsg.c,v 1.9 1991/07/26 20:46:47 steve Exp $";
#endif	/* lint not defined */


/*LINTLIBRARY*/

#include "udposix.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include "uderrmsg.h"

#include <errno.h>
#ifdef ERRNO_MISSING
    extern int errno;
#endif

extern char	*udstrdup();


/*
 * These subroutines emit no messages unless UD_VERBOSE bit is on.
 * These subroutines call exit() when UD_FATAL bit is on.
 */
static int	ud_error_options	= (UD_FATAL | UD_VERBOSE);


/*
 * Set to the name of the application by the interface routine.
 */
static char	*ud_routine_name	= "<unknown>";


/*
 * Print a time-stamp.
 */

    char*
udtime_stamp()
{
    static char	buf[32];
    time_t	tim	= time((time_t*)NULL);

    if (tim == -1) {
	(void)strcpy(buf, "<time unavailable>");
    } else {
	buf[sizeof(buf)-1]	= 0;
	(void)strftime(buf, sizeof(buf)-1, "%y-%m-%d %H:%M:%S %Z", 
	    localtime(&tim));
    }

    return buf;
}


/*
 * Print the header of a line (i.e. timp-stamp and application-name).
 */

    static void
print_header()
{
    if (isatty(2)) {
	(void)fprintf(stderr, "%s: ", ud_routine_name);
    } else {
	(void)fprintf(stderr, "(%s) %s: ", udtime_stamp(), ud_routine_name);
    }
}


/*
 * Set the mode-bits of this module.
 */

    int
uderrmode(mode)
    const int	mode;
{
    int		oldmode	= ud_error_options;

    ud_error_options	= mode;

    return oldmode;
}


/*
 * Set the name of the application.
 */

    char*
uderrname(name)
    const char	*name;
{
    char	*oldname	= ud_routine_name;

    if (name != NULL)
	ud_routine_name	= udstrdup(name);

    return oldname;
}


#ifdef vms
/*
 * On the vms system, when a system error occurs which is not
 * mapped into the unix styled errno values, errno is set EVMSERR
 * and a VMS error code is set in vaxc$errno.
 * This routine prints the systems message associated with status return
 * from a system services call.
 */

#include <descrip.h>
#include <ssdef.h>

    static int
vmserror1(int status)
{
    short	msglen;
    char	msgbuf[256];
    $DESCRIPTOR(message, msgbuf);
    register	ret	= SYS$GETMSG(status, &msglen, &message, 15, 0);
    
    switch(ret) {
    case SS$_BUFFEROVF:
    case SS$_NORMAL:
	(void)fprintf(stderr, "%.*s\n", msglen, msgbuf);
	break;
    default:
	break;
    }

    return ret;
}
#endif /* vms */


/*
 * Log SYSTEM errors
 * Use where you would want to call perror(3).
 * Calling sequence is
 *	uderror(format, arg1, arg2,...)
 * with zero or more args of types compatible with the associated format
 * specifiers.  For example:
 *         uderror("shutting down");
 *	   uderror("can't open %s", file_name);
 *         uderror("process %d in state %s",pid,state);
 */

    void
#ifdef UD_STDARG
uderror(const char *fmt, ...)
#else	/* !UD_STDARG */
    /*VARARGS1*/
uderror(fmt, va_alist)
    const char	*fmt;
    va_dcl
#endif /* !UD_STDARG */
{
    if (ud_error_options & UD_VERBOSE) {
	va_list	args;
	int	errnum	= errno;		/* save real one */
	char	*errstr	= strerror(errnum);

	print_header();
	UD_VA_START(args, fmt);
	(void) vfprintf(stderr, fmt, args);
	va_end(args);

#ifdef vms
	if (errnum == EVMSERR) {
	    (void) fputc(': ', stderr);
	    (void) vmserror1(vaxc$errno);
	} else
#endif /* vms */
	    if (errstr == NULL) {
		(void) fputc('\n', stderr);
	    } else {
		(void) fprintf(stderr, ": %s (errno=%d)\n", errstr, errnum);
	    }

	(void) fflush(stderr);			/* make log files current */

	errno	= errnum;
    }

    if (ud_error_options & UD_FATAL)
	exit(ud_error_options);
}


/*
 * Like uderror above, but doesn't check for system error.
 * Use for logging error conditions which are not system errors.
 * Calling sequence is
 *	udadvise(format, arg1, arg2,...)
 * with zero or more args of types compatible with the associated format
 * specifiers.  For example:
 *         udadvise("just advice");
 *         udadvise("%d is not a valid id", id);
 */

    void
#ifdef UD_STDARG
udadvise(const char *fmt,...)
#else /* !UD_STDARG */
    /*VARARGS1*/
udadvise(fmt, va_alist)
    const char	*fmt;
    va_dcl
#endif /* !UD_STDARG */
{
    va_list	args;

    if (ud_error_options & UD_VERBOSE) {
	print_header();
	UD_VA_START(args ,fmt);
	(void)vfprintf(stderr,fmt,args);
	va_end(args);
	(void)fputc('\n',stderr);
	(void)fflush(stderr);	/* to ensure log files are current */
    }

    if (ud_error_options & UD_FATAL)
	exit(ud_error_options);
}
