/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * Match T's and corners to find best set of continuations for an edge map
 */
#include <math.h>
#include <stdio.h>
#include <malloc.h>
#include <hvision.h>
#include "vector.h"
#include "edgemap.h"
#include "region.h"

#ifndef NDEBUG
#define DEBUG(x) {if (TraceLevel > 0) {printf x ; fflush(stdout);}}
#else
#define DEBUG(x)
#endif

char    Usage[] = "\
usage: %s [-n nu] [-a alpha] [-b beta] [-k kbest] [-l ratio]\n\
	  [-s sigma] [-i image] [-o nconf] [-X] [edgemap_file]\n\
\n\
   Adds continuations to an edge map at corners and T-junctions.\n\
\n\
	-i	input image file.  Default is edgemap_file with `.e*'\n\
		changed to `.im'\n\
				   /\n\
	-n, -a	specify const. for \\ (nu + alpha k^2) ds\n\
				   /\n\
	-b	beta * change in angle is penalty at corners\n\
		default: nu = 1, alpha = 1, beta = Nu*MAX(Height,Width)*2/Pi\n\
	-k	use only the kbest best potential mates for each endpoint\n\
		default 5, maximum 10\n\
	-l	Continuation length ratio (default 5)\n\
		continuations must not exceed this many times the length of\n\
		either side being continued\n\
	-s	blur with a gaussian of std dev sigma to get\n\
		average pixel values near endpoints\n\
		default 2.5\n\
	-o	Number of final output configurations (1..100, default 1)\n\
		to save in edge map and region map files.\n\
	-X	Debug run: put all non-excluded completions in `.c-<kbest>'.\n\
\n\
   The algorithm pairs up endpoints in a way that should minimize\n\
   length plus curvature squared over all continuations, with a\n\
   penalty for unmatched endpoints.\n\
\n\
   Puts result in a file with the same name as the input\n\
   but with the extension `.c' or `.c<n>'.\n";

/* parameters */
int	DebugRun = FALSE;
real    Sigma;

#define DEFAULT_SIGMA 2.5
real    Nu;			/* length penalty */

#define DEFAULT_NU 1
real    Alpha;			/* curvature square penalty */

#define DEFAULT_ALPHA 1
real    Beta;			/* at corners, angle penalty */

#define DEFAULT_BETA (Nu*MAX(Height,Width)*2/M_PI)	/* 90 deg turn=long line */
int     NOutputConf;		/* # output configurations to save */

#define DEFAULT_NCONF 1
real    MaxContinLengthRatio;	/* length of contin must be < this * length
				 * of either side */

#define DEFAULT_CONTIN_RATIO 5
int     Kbest;

#define DEFAULT_KBEST 5

char   *ImageFile;

#define	ESSENTIALLY_ZERO 1.0e-6

/* globals */
IMAGE  *InputImage;
real    MinPixel, MaxPixel;	/* the input image */
int     TraceLevel;

#define MAXJUNC 100		/* incomplete junctions */
#define MAXPAIR 100		/* max pairs in a complete pairing */
#define MAXPAIRING 10		/* total no. of complete pairings to keep */
#define MAXKBEST 10		/* no more than the 7 best candidates for
				 * each endpt */
#define MAXCAND 300		/* max endpts to keep info about */

typedef short EndPt;		/* ((contour number)<<1 | end) (HEAD=0,
				 * TAIL=1) */

#define	EndPoint(c,e)	((EndPt) ((c)<<1)|(e))
#define	ContourOf(ep)	((ep)>>1)
#define	EndOf(ep)	((ep)&1)

typedef struct Pairing {
    real    penalty;

    /* the endpoints currently paired */
    short   npair;
    struct pair {
	EndPt   ep1;		/* the two endpoints to join */
	EndPt   ep2;		/* --use EndPoint(contour, end) */
    }       pairs[MAXPAIR];
} Pairing;
int     NBestPairings;
Pairing BestPairings[MAXPAIRING];

typedef struct Relax {

    /* the endpoints that may still be paired */
    real    contourpenalty;	/* total for continuation contours only */

    short   npairable;
    struct pairable {
	EndPt   master;		/* the lower-numbered endpoint */
	short   nmate;
	EndPt   mates[MAXKBEST];/* the Kbest mates, (each > master) */
	real    penalties[MAXKBEST];	/* & associated contin penalties */
    }       pairables[MAXCAND];

    char    ContoursAt[MAXJUNC];/* # contours that meet at each junction */
} Relax;

Pairing CurrentPairing;

/* junctions that originally had 2 or 3 contours; for corner penalties */
short   NJuncsToComplete;
short   JuncsToComplete[MAXJUNC];

/* Junction number given endpoint number, or -1 if boundary */
short  *JunctionOf;		/* JunctionOf[ep] is junction number */
char   *CurrentlyPaired;	/* CurrentlyPaired[ep] is TRUE if it's paired */

#define HUGE_PENALTY 1.0e10	/* impossibly (?) large penalty */

/* disqualification counters for endpoint pair matchings */
int     n_disq_toolong;
int     n_disq_brightness;
int     n_disq_boundary;
int     n_disq_ordering;
int     n_mates_using_kbest;

/* prototypes */

#ifdef ANSI
int     main(int, char **);
int     OutputPairing(struct Pairing *, char *, char *);
int     GetRegionBeyond(int, int);
int     Contin(void);
real    AnglePenalty(double, double);
real    ContinPenalty(int, int, int, int);
vector *IntersectSegments(vector, vector, vector, vector);
real    AngleFix(double);
int     FindBestPairings(struct Relax, int);
int     DeleteMate(struct Relax *, int, int);
int     DeleteMaster(struct Relax *, int);
int     PropogateConstraints(struct Relax *, int);
int     DefiesOrderRelations(int, int, int, int, int);
int     ComputeCurrentPairingPenalty(struct Relax *);
int     DBPrintCurrentPairing(void);
int     SavePairings(void);
int     compair(struct Pairing *, struct Pairing *);
int     FillContourFields(void);

#else				/* ANSI */

int     main();
int     OutputPairing();
int     GetRegionBeyond();
int     Contin();
real    AnglePenalty();
real    ContinPenalty();
vector *IntersectSegments();
real    AngleFix();
int     FindBestPairings();
int     DeleteMate();
int     DeleteMaster();
int     PropogateConstraints();
int     DefiesOrderRelations();
int     ComputeCurrentPairingPenalty();
int     DBPrintCurrentPairing();
int     SavePairings();
int     compair();
int     FillContourFields();

/* from elast.c */

real    InitialElastLength();
real    BuildElast();
real    ContourEnergy();
real    ElastPrior();

#endif				/* ANSI */


/* ---------- the program ------------ */
main(argc, argv)
int     argc;
char  **argv;
{
    char    c;
    static char imagename[512];
    int     i;

    extern char *optarg;
    extern int optind;

    PgmName = argv[0];

    while ((c = getopt(argc, argv, "a:n:l:o:b:s:i:rx:k:X")) != -1) {

	switch (c) {

	case '?':		/* getopt() arg not in list */
	    fprintf(stderr, Usage, PgmName);
	    exit(1);

	case 's':
	    Sigma = atof(optarg);
	    break;
	case 'i':
	    ImageFile = optarg;
	    break;
	case 'l':
	    MaxContinLengthRatio = atof(optarg);
	    break;
	case 'o':
	    NOutputConf = atoi(optarg);
	    break;
	case 'a':
	    Alpha = atof(optarg);
	    break;
	case 'n':
	    Nu = atof(optarg);
	    break;
	case 'b':
	    Beta = atof(optarg);
	    break;
	case 'x':
	    TraceLevel = atoi(optarg);
	    break;
	case 'k':
	    Kbest = atoi(optarg);
	    break;
	case 'X':
	    DebugRun = TRUE;
	    break;

	}
    }

    if (optind < argc) {
	CurrentFile = argv[optind];
	if (freopen(CurrentFile, "r", stdin) == NULL) {
	    fprintf(stderr, "Can't open %s for input\n", CurrentFile);
	    exit(1);
	}
    }
    if (ImageFile == NULL) {
	ImageFile = imagename;
	strcpy(imagename, FileNameNoNumber(CurrentFile, ".im"));
    }
    if (!ReadContours(stdin))
	exit(Status);

    if ((InputImage = hvReadImage(ImageFile)) == NULL) {
	fprintf(stderr, Usage, PgmName);
	exit(1);
    }
    if (InputImage->pixeltype != REALPIX)
	hvConvert(InputImage, InputImage, REALPIX);

    hvMinMax(InputImage, 0, &MinPixel, &MaxPixel);

    /* set parameters */
    if (Sigma == 0)
	Sigma = DEFAULT_SIGMA;

    if (Nu == 0 && Alpha == 0) {
	Nu = DEFAULT_NU;
	Alpha = DEFAULT_ALPHA;
    }
    if (Beta == 0)
	Beta = DEFAULT_BETA;

    if (NOutputConf == 0)
	NOutputConf = DEFAULT_NCONF;

    if (MaxContinLengthRatio == 0)
	MaxContinLengthRatio = DEFAULT_CONTIN_RATIO;

    if (Kbest == 0)
	Kbest = DEFAULT_KBEST;

    /* fill length, ksq, endtan and endavg for each contour */
    for (i = 0; i < NContours; i++)
	FillFields(i, Sigma);

    TraceRegions();
    SetContourRegions();

    Contin();

    exit(Status);
}


OutputPairing(P, ext, expl)
Pairing *P;
char   *ext, *expl;
{
    char   *name = FileNameNoNumber(CurrentFile, ext);
    int     i;
    FILE   *f;

    if ((f = fopen(name, "w")) == NULL) {
	fprintf(stderr,
		"%s: Can't open `%s' for output\n",
		PgmName, name);
	Status = 4;
    } else {
	fprintf(f, "# %s\n", expl);
	for (i = 0; i < P->npair; i++) {
	    fprintf(f, "%d %d %d %d\n",
		    ContourOf(P->pairs[i].ep1),
		    EndOf(P->pairs[i].ep1),
		    ContourOf(P->pairs[i].ep2),
		    EndOf(P->pairs[i].ep2));
	}
	fclose(f);
	fprintf(stderr,
		"%s: wrote continuation pairing to `%s'\n",
		PgmName, name);
    }
}

/*
 * A "continuation list" has a penalty measure and a set of endpoint pairs
 * that are to be connected up.
 * 
 * Contin() builds the best MAXPAIRING legal continuation lists and sorts them
 * in order of increasing penalty.
 * 
 * Strategy:
 * 
 * Put the Kbest best mates for each endpoint in a list indexed by the
 * lower-numbered endpoint of each pair. The lower-numbered endpoint is the
 * "master", and with it we store a list of potential "mates". (An endpoint
 * number is twice the contour number plus zero for the head, or one for the
 * tail.)
 * 
 * The algorithm is defined recursively by
 * 
 * a) If there are no pairable pairs (end condition) compute pairing penalty and
 * keep pairing if it's good. Return.
 * 
 * b) Choose one pairable pair at random, and delete it from the potential
 * pairing list.  Then
 * 
 * b.1) recurse as if nothing happened, and after that,
 * 
 * b.2) put it in the current pairing, propogate its constraints, and recurse.
 * 
 */


/*
 * the region on the left/right side of a contour as we head along the
 * contour starting from end end
 */
#define GetLeftRegion(c,end) Contours[c].reg[(end==HEAD ? LEFT : RIGHT)]
#define GetRightRegion(c,end) Contours[c].reg[(end==HEAD ? RIGHT : LEFT)]

/* the region beyond end end of contour c */
int
GetRegionBeyond(c, end)
int     c, end;
{
    PointList *p = Contours[c].p;
    int index = EndIndex(c, end);

    return RegionFromRay(p->x[index], p->y[index], Contours[c].endtan[end]);
}
short  *LeftRegion, *RightRegion, *RegionBeyond;

/*-----------------  continuations ---------------- */
Contin()
{
    int     c1, c2, end1, end2, i;
    int     j1, j2;
    Relax   relax;		/* settle down */
    struct pairable *pa;

    /* JunctionOf[endpoint] is junction number or -1 if on boundary */
    JunctionOf = (short *) calloc(NContours * 2, sizeof (short));
    for (i = 0; i < NJunctions; i++) {
	int     j;
	int	bound = FALSE;

	for (j = 0; j < Junctions[i].n; j++) {
	    if (Contours[Junctions[i].c[j].cno].p->boundary)
		bound = TRUE;
	}

	for (j = 0; j < Junctions[i].n; j++) {
	    int ep = EndPoint(Junctions[i].c[j].cno,
	    		      Junctions[i].c[j].end);
	    JunctionOf[ep] = bound ? -1 : i;
	}
    }

    /* CurrentlyPaired[ep] is TRUE if ep has a continuation now */
    CurrentlyPaired = (char *) calloc(NContours * 2, sizeof (char));

    /*
     * LeftRegion[ep] is the region on the left of this contour headed
     * outward from this end; etc.
     */
    LeftRegion = (short *) calloc(NContours * 2, sizeof (short));
    RightRegion = (short *) calloc(NContours * 2, sizeof (short));
    RegionBeyond = (short *) calloc(NContours * 2, sizeof (short));

    for (c1 = 0; c1 < NContours; c1++) {
	for (end1 = HEAD; end1 <= TAIL; end1++) {
	    EndPt   ep = EndPoint(c1, end1);

	    if (Contours[c1].p->boundary)
		continue;
	    LeftRegion[ep] = GetLeftRegion(c1, end1);
	    RightRegion[ep] = GetRightRegion(c1, end1);
	    RegionBeyond[ep] = GetRegionBeyond(c1, end1);
	}
    }

    /*
     * Find penalties for each pair of different ends of contours
     */
    relax.npairable = 0;
    pa = &relax.pairables[0];

    n_disq_toolong = 0;
    n_disq_brightness = 0;
    n_disq_ordering = 0;
    n_disq_boundary = 0;
    n_mates_using_kbest = 0;

    for (c1 = 0; c1 < NContours; c1++)
	for (end1 = HEAD; end1 <= TAIL; end1++) {

	    EndPt   ep1 = EndPoint(c1, end1);

	    j1 = JunctionOf[ep1];
	    if (j1 == -1) {
		n_disq_boundary++;
		continue;
	    }

	    pa->master = ep1;
	    pa->nmate = 0;

	    /* keep the Kbest best for c1,end1 */
	    for (c2 = c1; c2 < NContours; c2++)
		for (end2 = HEAD; end2 <= TAIL; end2++) {
		    int     supplant;
		    real    penalty;
		    EndPt   ep2 = EndPoint(c2, end2);

		    if (c1 == c2 && end1 >= end2)
			continue;

		    j2 = JunctionOf[ep2];
		    if (j2 == -1) {
			n_disq_boundary++;
			continue;
		    }
		    if (j1 == j2)
			continue;

		    penalty = ContinPenalty(c1, end1, c2, end2);
		    if (penalty == HUGE_PENALTY)
			continue;

		    /* if we haven't got K yet, just add it */
		    if (pa->nmate < Kbest) {
			supplant = pa->nmate++;
			goto addmate;
		    }
		    /* see which one it supplants */
		    supplant = -1;
		    for (i = 0; i < pa->nmate; i++) {
			if (supplant == -1) {

			    /* if penalty is lower than any so far */
			    if (pa->penalties[i] > penalty)
				supplant = i;

			} else if (pa->penalties[i]
				   > pa->penalties[supplant]) {
			    /* even worse penalty than current supplant */
			    supplant = i;
			}
		    }
		    if (supplant == -1)
			continue;

	    addmate:
		    pa->mates[supplant] = ep2;
		    pa->penalties[supplant] = penalty;
		}

	    if (pa->nmate > 0) {
		n_mates_using_kbest += pa->nmate;
		pa++;
		relax.npairable++;
		if (relax.npairable >= MAXCAND) {
		    fprintf(stderr,
			 "Sorry, can't take more than %d pairable endpts\n",
			    MAXCAND);
		    exit(3);
		}
	    }
	}

    fprintf(stderr, "Of approximately %d endpoint pairings:\n",
	    2 * NContours * (NContours - 1));
    fprintf(stderr, "  %d boundary, %d too long, %d colors didn't match;\n",
	    n_disq_boundary, n_disq_toolong, n_disq_brightness);
    fprintf(stderr, "  %d remain using %d mates max for each endpoint.\n",
	    n_mates_using_kbest, Kbest);

    if (DebugRun) {
	PrintAllPairs(&relax);
	exit(Status);
    }

    NBestPairings = 0;
    NJuncsToComplete = 0;
    relax.contourpenalty = 0.0;	/* for continuations only, not corners */

    /*
     * Find junctions w/2 or 3 contours, for computing corner penalties
     */
    for (i = 0; i < NJunctions; i++) {
	relax.ContoursAt[i] = Junctions[i].n;
	if (Junctions[i].n == 2 || Junctions[i].n == 3) {

	    /* boundary junctions don't count */
	    if (JunctionOf[EndPoint(Junctions[i].c[0].cno,
				    Junctions[i].c[0].end)] == -1)
		continue;

	    /*
	     * bona-fide completable junction (sounds like a dog treat)
	     */
	    if (NJuncsToComplete >= MAXJUNC) {
		fprintf(stderr,
			"Too many junctions to complete (%d)--giving up\n",
			NJuncsToComplete);
		return;
	    }
	    JuncsToComplete[NJuncsToComplete++] = i;
	}
    }

    DEBUG(("%d junctions may have completions\n", NJuncsToComplete));

    /*
     * Recursively ...
     */
    fprintf(stderr, "Finding best pairings ...\n");
    FindBestPairings(relax, 0);

    free(RegionBeyond);
    free(RightRegion);
    free(LeftRegion);
    free(CurrentlyPaired);
    free(JunctionOf);

    fprintf(stderr, "Disqualified %d configurations by ordering relations\n",
	    n_disq_ordering);

    /*
     * Write files for each output configuration
     */
    SavePairings();
}

/***
  ContinPenalty -- penalty for continuation from end1 of c1 to end2 of c2

	      /
	----o'		same endpoints get angle penalty

	----o   o-----  different endpoints get total energy of continuation
			length+curv^2, but no angle penalty

	 a         c
	----o	o----	where intensities a and b are sufficiently different,
	 b  	   d	then a<b must match c<d, or infinite penalty.

*/
real
AnglePenalty(theta1, theta2)
real    theta1, theta2;
{
    return Beta * fabs(AngleFix(AngleFix(theta2) - AngleFix(theta1)));
}

real
ContinPenalty(c1, end1, c2, end2)
int     c1, end1, c2, end2;
{
    PointList *p1 = Contours[c1].p;
    PointList *p2 = Contours[c2].p;
    static PointList *pelast;
    int     i1 = EndIndex(c1, end1);
    int     i2 = EndIndex(c2, end2);
    real    x1 = p1->x[i1], y1 = p1->y[i1], theta1 = Contours[c1].endtan[end1];
    real    x2 = p2->x[i2], y2 = p2->y[i2], theta2 = Contours[c2].endtan[end2];
    real    pixelrange = fabs(MaxPixel - MinPixel);
    real    tolerance = 0.2 * pixelrange;
    real    a, b, c, d, energy, length;
    int     adiffb, cdiffd;

    if (c1 == c2 && end1 == end2)
	return HUGE_PENALTY;

    /* angle theta2 must be u-turned to point in the direction of travel */
    theta2 = AngleFix(theta2 + M_PI);

    /* two endpoints at the same junction */
    if (Distance(x1, y1, x2, y2) <= ESSENTIALLY_ZERO)
	return AnglePenalty(theta1, theta2);

#ifdef ADD_ALL_ELASTICA
    return AddElastica(x1, y1, theta1, x2, y2, theta2, Alpha, Nu);
#endif

    /*
     * disqualify pairs which DO NOT satisfy either: (see diagram above) 1.
     * a~c or b~d  (within 5% of pixel range) 2.  (a<b and c<d)  OR  (a>b and
     * c>d) Inequalities require at least 5% of pixel range.
     */
    a = Contours[c1].endavg[end1][LEFT];
    b = Contours[c1].endavg[end1][RIGHT];

    c = Contours[c2].endavg[end2][RIGHT];	/* note must switch right &
						 * left */
    d = Contours[c2].endavg[end2][LEFT];

    adiffb = (fabs(a - b) > tolerance);
    cdiffd = (fabs(c - d) > tolerance);

    if (			/* 1.  a=c or b=d OK */
	(fabs(a - c) > tolerance && fabs(b - d) > tolerance) &&

    /* 2.  (a<b and c<d)  OR  (a>b and c>d) OK */
	(adiffb && cdiffd && (a - b) * (c - d) < 0)) {

	DEBUG(("%d/%d<->%d/%d end avg a=%g b=%g c=%g d=%g; ignoring\n",
	       c1, end1, c2, end2, a, b, c, d));
	n_disq_brightness++;
	return HUGE_PENALTY;
    }

    /*
     * Disqualify elastica longer than MaxContinLengthRatio times either of
     * its continued sides  (should be both? or either?)
     */
    length = InitialElastLength(x1, y1, theta1, x2, y2, theta2, Alpha, Nu);
    if (length > MaxContinLengthRatio * Contours[c1].length
	|| length > MaxContinLengthRatio * Contours[c2].length) {
	DEBUG(("%d/%d<->%d/%d elastica length %g > %g times one of (%g,%g); ignoring\n",
	       c1, end1, c2, end2, length, MaxContinLengthRatio,
	       Contours[c1].length, Contours[c2].length));
	n_disq_toolong++;
	return HUGE_PENALTY;
    }
    /* compute the energy term for an elastica */
    if (pelast == NULL) {
	pelast = NewPointList(50);	/* why not? */
	pelast->n = 50;
    }
    /* (pity to build the elastica just to get the energy) */
    energy = BuildElast(pelast, x1, y1, theta1, x2, y2, theta2, Alpha, Nu);

    return energy;
}

/*
 * Returns a pointer to a static vector containing the intersection of two
 * segments or NULL if they do not truly intersect, or if they are parallel.
 * Takes four points specifying the two lines; cowboy style no checks for
 * singularities etc.
 */
vector *
IntersectSegments(head1, tail1, head2, tail2)
vector  head1, tail1, head2, tail2;
{
    vector  d1, d2, dhead;
    static vector intersection;
    real    denom;

    d1 = v_difference(head1, tail1);	/* tail1 - head1 */
    d2 = v_difference(head2, tail2);
    dhead = v_difference(head1, head2);

    denom = v_dot(d1, v_perp(d2));

    if (denom == 0) {

	/* they're parallel */
	return NULL;
    } else {

	/* line intersection is at head1 + t1 d1 */
	real    t1 = v_dot(dhead, v_perp(d2)) / denom;
	real    t2 = v_dot(dhead, v_perp(d1)) / denom;

	if (t1 < 0 || t1 > 1 || t2 < 0 || t2 > 1)
	    return NULL;

	intersection = v_sum(head1, v_scalar_mult(t1, d1));

	return &intersection;
    }
}

/*
 * return an angle between -pi and pi
 */
real
AngleFix(a)
real    a;
{
    while (a > M_PI)
	a -= 2 * M_PI;
    while (a < -M_PI)
	a += 2 * M_PI;

    return a;
}

/*
 * recursively collect pairings
 */
FindBestPairings(relax, level)
Relax   relax;
int     level;
{
    int     i, pairno, mateno;
    EndPt   master, mate;
    int     worsti;
    real    worstp;
    real    thispenalty;
    struct pairable *pa;

    if (relax.npairable == 0) {
	/* evaluate this as a complete pairing */

	ComputeCurrentPairingPenalty(&relax);

	if (NBestPairings < MAXPAIRING)
	    /* not full yet */
	    BestPairings[NBestPairings++] = CurrentPairing;
	else {

	    /* replace worst pairing with current if better */
	    worsti = 0;
	    worstp = BestPairings[worsti].penalty;
	    for (i = 1; i < NBestPairings; i++) {
		if (BestPairings[i].penalty > worstp) {
		    worstp = BestPairings[i].penalty;
		    worsti = i;
		}
	    }
	    if (CurrentPairing.penalty < worstp) {
		BestPairings[worsti] = CurrentPairing;
		DEBUG(("%.*sreplacing penalty %g\n",
		       level, "                             ",
		       worstp));
	    }
	}
	return;
    }
    /* choose a random pairable pair */
    i = random();
    pairno = i % relax.npairable;
    pa = &relax.pairables[pairno];
    if (pa->nmate <= 0) {
	fprintf(stderr,
		"error .. Somebody doesn't have a mate .. error\n");
	abort();
    }
    /* random mate for the master */
    mateno = (i >> 8) % pa->nmate;
    master = pa->master;
    mate = pa->mates[mateno];
    thispenalty = pa->penalties[mateno];	/* save continuation penalty */

    /* first, trash the pair and recurse */
    DeleteMate(&relax, pairno, mateno);
    FindBestPairings(relax, level + 1);

    /*
     * put it in the current pairing, propogate its constraints, and recurse.
     */
    i = CurrentPairing.npair++;
    CurrentPairing.pairs[i].ep1 = master;
    CurrentPairing.pairs[i].ep2 = mate;
    CurrentlyPaired[master] = TRUE;
    CurrentlyPaired[mate] = TRUE;
    PropogateConstraints(&relax, master);
    PropogateConstraints(&relax, mate);
    relax.contourpenalty += thispenalty;	/* add continuation penalty */

    FindBestPairings(relax, level + 1);

    CurrentlyPaired[master] = FALSE;
    CurrentlyPaired[mate] = FALSE;
    CurrentPairing.npair--;
}

/*
 * delete a mate from a pairable.  If it was the last mate, delete the master
 * (the whole pairable), and return TRUE.
 */
DeleteMate(r, pairno, mateno)
Relax  *r;
int     pairno, mateno;
{
    struct pairable *pa = &r->pairables[pairno];
    int     i;

    if (pa->nmate == 1) {
	DeleteMaster(r, pairno);
	return TRUE;
    }
    /* delete the mate from the mates of pa */
    for (i = mateno; i < pa->nmate - 1; i++) {
	pa->mates[i] = pa->mates[i + 1];
	pa->penalties[i] = pa->penalties[i + 1];
    }
    pa->nmate--;
    return FALSE;
}

DeleteMaster(r, pairno)
Relax  *r;
int     pairno;
{
    int     i;

    for (i = pairno; i < r->npairable - 1; i++)
	r->pairables[i] = r->pairables[i + 1];
    r->npairable--;
}

PropogateConstraints(r, ep)
Relax  *r;
int     ep;
{
    int     n, jno, wast;
    int     pairno;
    int     f, b1, b2;

    /*
     * ep has been completed to somewhere (doesn't matter).
     * 
     * Delete all pairs with ep; if ep was a master, delete it.
     * 
     * Mark the junction containing ep as having one more contour.  If ep was
     * the fourth, delete all potential pairings that would add to that
     * junction.
     * 
     * If ep belonged to a real T, note the two order relations imposed by this
     * interpretation of the T, and delete all potential pairings that would
     * violate it.
     */
    jno = JunctionOf[ep];
    n = ++r->ContoursAt[jno];

    wast = (n == 4 && Junctions[jno].n == 3);

    if (wast) {
	/* two order relations dictated by this interp. of the T */
	f = RegionBeyond[ep];
	b1 = RightRegion[ep];
	b2 = LeftRegion[ep];
    }
    for (pairno = 0; pairno < r->npairable; pairno++) {
	struct pairable *pa = &r->pairables[pairno];
	int     mateno;

	if (pa->master == ep
	    || (n == 4 && JunctionOf[pa->master] == jno)) {
	    DeleteMaster(r, pairno);
	    pairno--;
	    continue;
	}
	for (mateno = 0; mateno < pa->nmate; mateno++) {
	    int     mate = pa->mates[mateno];

	    if (mate == ep ||
		(n == 4 && JunctionOf[mate] == jno) ||
		(wast && DefiesOrderRelations(pa->master, mate,
					      f, b1, b2))) {
		if (DeleteMate(r, pairno, mateno)) {
		    /* deleted master as well */
		    pairno--;
		    break;
		}
		mateno--;
	    }
	}
    }
}

/*
 * return TRUE if order relations are inconsistent with the addition of the
 * continuation from master to mate, FALSE if OK.
 * 
 * NOTE:  only triple points in the *original* give ordering information when
 * completed.  We leave the ordering implied by completions of corners as an
 * exercise.
 */

DefiesOrderRelations(master, mate, f, b1, b2)
int     master, mate;
int     f, b1, b2;
{
    int     jno1 = JunctionOf[master];
    int     jno2 = JunctionOf[mate];

    /* true T-junctions */
    int     cc1 = (Junctions[jno1].n == 3);
    int     cc2 = (Junctions[jno2].n == 3);

    /* check consistency with completion of T at junc 1 */
    if (cc1) {
	int     B1 = RightRegion[master];
	int     B2 = LeftRegion[master];
	int     F = RegionBeyond[master];

	if ((b1 == F || b2 == F) && (f == B1 || f == B2)) {
	    n_disq_ordering++;
	    return TRUE;	/* canned! */
	}
    }
    if (cc2) {
	int     B1 = RightRegion[mate];
	int     B2 = LeftRegion[mate];
	int     F = RegionBeyond[mate];

	if ((b1 == F || b2 == F) && (f == B1 || f == B2)) {
	    n_disq_ordering++;
	    return TRUE;	/* canned! */
	}
    }
    return FALSE;		/* consistent */
}


ComputeCurrentPairingPenalty(r)
Relax  *r;
{

    /*
     * total penalty = sum of contour penalties (= r->contourpenalty) + sum
     * of junction penalties
     */
    Pairing *P = &CurrentPairing;
    real    penalty = r->contourpenalty;
    int     i, j;
    EndPt   cons[4];
    real    tangents[4];

    /* add junction penalties */
    for (j = 0; j < NJuncsToComplete; j++) {

	int     jno = JuncsToComplete[j];
	Junction *J = &Junctions[jno];
	int     jn = J->n;
	int     n = r->ContoursAt[jno];

	if (n == 4 && jn == 2)
	    continue;		/* completed corner gets zero penalty */

	for (i = 0; i < jn; i++) {
	    cons[i] = EndPoint(J->c[i].cno, J->c[i].end);
	    tangents[i] =
		Contours[J->c[i].cno].endtan[J->c[i].end];
	}

	if (n == jn) {
	    penalty += AnglePenalty(tangents[0],
				    tangents[1] + M_PI);
	    if (jn == 3) {
		/* uncompleted Y gets 3 angle penalties */
		penalty += AnglePenalty(tangents[1],
					tangents[2] + M_PI)
		    + AnglePenalty(tangents[0],
				   tangents[2] + M_PI);
	    }
	    continue;
	}
	/* here, n = 1+jn */
	if (jn == 2) {
	    /* a corner completed to a T */
	    if (CurrentlyPaired[cons[0]])
		penalty += AnglePenalty(tangents[0],
					tangents[1] + M_PI)
		    + AnglePenalty(tangents[1],
				   tangents[0]);
	    else
		penalty += AnglePenalty(tangents[1],
					tangents[0] + M_PI)
		    + AnglePenalty(tangents[0],
				   tangents[1]);
	    continue;
	}
	/* A completed T */
	if (CurrentlyPaired[cons[2]])
	    penalty += AnglePenalty(tangents[0], tangents[1] + M_PI);
	else if (CurrentlyPaired[cons[0]])
	    penalty += AnglePenalty(tangents[1], tangents[2] + M_PI);
	else
	    penalty += AnglePenalty(tangents[0], tangents[2] + M_PI);
    }
    P->penalty = penalty;

#ifndef NDEBUG
    if (TraceLevel > 0)
	DBPrintCurrentPairing();
#endif				/* NDEBUG */
}


#ifndef NDEBUG
DBPrintCurrentPairing()
{
    int     i;
    Pairing *P = &CurrentPairing;

    fprintf(stderr, "%g=", P->penalty);
    for (i = 0; i < P->npair; i++)
	fprintf(stderr, "%d/%d<->%d/%d ",
		ContourOf(P->pairs[i].ep1),
		EndOf(P->pairs[i].ep1),
		ContourOf(P->pairs[i].ep2),
		EndOf(P->pairs[i].ep2));
    fprintf(stderr, "\n");
}

#endif				/* NDEBUG */

SavePairings()
{
    int     compair();
    int     i, j, pairing;
    char    ext[10];
    char    expl[256];

    fprintf(stderr, "%d best pairings\n", NBestPairings);

    qsort(BestPairings, NBestPairings, sizeof (Pairing), compair);

    /* Report best pairings */
    for (i = 0; i < NBestPairings; i++) {
	fprintf(stderr, "penalty=%g: ", BestPairings[i].penalty);
	for (j = 0; j < BestPairings[i].npair; j++) {
	    fprintf(stderr, "%d/%d<->%d/%d ",
		    ContourOf(BestPairings[i].pairs[j].ep1),
		    EndOf(BestPairings[i].pairs[j].ep1),
		    ContourOf(BestPairings[i].pairs[j].ep2),
		    EndOf(BestPairings[i].pairs[j].ep2));
	}
	fprintf(stderr, "\n");
    }


    /* now output all NBestPairings'worth */

    for (pairing = 0; pairing < MIN(NBestPairings, NOutputConf); pairing++) {

	Pairing *p = &BestPairings[pairing];

	sprintf(ext, ".c%d", pairing + 1);
	sprintf(expl, "Continuation choice #%d with nu=%g alpha=%g beta=%g",
		pairing + 1, Nu, Alpha, Beta);
	OutputPairing(p, ext, expl);
    }
}

/* compare two pairings */
compair(p1, p2)
Pairing *p1, *p2;
{
    if (p1->penalty < p2->penalty)
	return -1;
    if (p1->penalty == p2->penalty)
	return 0;
    return 1;
}

/* for debugging */
PrintAllPairs(r)
Relax *r;
{
    char   *name, ext[10];
    int     i;
    FILE   *f;

    sprintf(ext, ".c-%d", Kbest);

    name = FileNameNoNumber(CurrentFile, ext);
    if ((f = fopen(name, "w")) == NULL) {
	fprintf(stderr,
		"%s: Can't open `%s' for output\n",
		PgmName, name);
	Status = 4;
    } else {
	fprintf(f, "# %d best contin's for each contour endpoint of %s\n",
		Kbest, CurrentFile);

	for (i = 0; i < r->npairable; i++) {
	    struct pairable *pa = &r->pairables[i];
	    int     j;

	    for (j = 0; j < pa->nmate; j++)
	        fprintf(f, "%d %d %d %d\n",
		        ContourOf(pa->master),
		        EndOf(pa->master),
		        ContourOf(pa->mates[j]),
		        EndOf(pa->mates[j]));
	}
	fclose(f);
	fprintf(stderr,
		"%s: wrote all continuations to `%s'\n",
		PgmName, name);
    }
}
