/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            LIB
* MODULE:		vipspat.c - Library of basic image processing routines
* REVISION:             3.11
* AUTHOR:               PK/DH
* CREATION DATE:        24 Nov 1991
* REVISION DATE:	5/10/94
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.11
* REVISION DATE:	9 May 1994
* COMMENT:		Fixed Dilate_Image so that mask elements are considered
*                       part of the structuring element if they are > 0.
* BY:                   PK
*
* REVISION:		3.10
* REVISION DATE:	14 April 1994
* COMMENT:		Added Add_Image.
                        Sqrt_Image and Sqr_Image now also work for BYTE, SHORT and
                        LONG types.
                        Fixed to compile cleanly with all warnings.
* BY:			PK
*
* REVISION:		3.9
* REVISION DATE:	5 April 1994
* COMMENT:		Added extra calls to Image_OK.
                        Diff_Image and Subtract Image work for BYTE, SHORT, LONG
                        and FLOAT types.
                        Added FConvolve_Image.
* BY:			PK
*
* REVISION:		3.8
* REVISION DATE:	21 March 1994
* COMMENT:		Removed vector.h
* BY:			CFF
*
* REVISION:		3.7
* REVISION DATE:	03 March 1994
* COMMENT:		Added fn's. from PK's binotrans.c
* BY:			CFF
*
* REVISION:		3.6
* REVISION DATE:	25 February 1994
* COMMENT:		Added calls to Image_OK
* BY:			CFF
*
* REVISION:		3.4
* REVISION:		3.5
* REVISION DATE:	18 Nov 1993
* COMMENT:		Fixed up Read_Int_Mask()
* BY:			CFF
*
* REVISION:		3.4
* REVISION DATE:	16 Sept 1993
* COMMENT:		Fixed up for GENERIC build
* BY:			CFF
*
* REVISION:		3.3
* REVISION DATE:	16 August 1993
* COMMENT:		Fixed up for DEC build
* BY:			CFF
*
* REVISION:		3.2
* REVISION DATE:	15 June 1993
* COMMENT:		Fixed memcpy args 1&2 on line(s) 1921 & 1922
* BY:			CFF
*
* REVISION:		3.1
* REVISION DATE:	9 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:
* REVISION DATE:	24 Jan 1992
* COMMENT:
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)vipspat.c	3.11 5/10/94";

#endif



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <math.h>
#include <malloc.h>
#include <values.h>

#include "vip.h"
#include "vipiofn.h"
#include "vipspatfn.h"
#include "vipcomplexfn.h"
#include "misc.h"
#include "miscfn.h"
#include "vectorfn.h"

/*-   Free_Float_Mask  ------------------------------------------------

Function to free memory allocated for a FLOAT_MASK structure.

--------------------------------------------------------------------*/

void    Free_Float_Mask(mask)
FLOAT_MASK *mask;
{
    if (mask == NULL)
	return;			/* already free */

    Free_Matrix((char **) mask->m);
    free((char *) mask);
    return;
}

/*-  Allocate_Float_Mask  ---------------------------------------------

Function to allocate space for a FLOAT_MASK structure.

--------------------------------------------------------------------*/

FLOAT_MASK *Allocate_Float_Mask(rows, cols)
int     rows, cols;
{
    FLOAT_MASK *mask;

    mask = (FLOAT_MASK *) malloc(sizeof(FLOAT_MASK));
    if (!mask)
	VIP_Error_Msg("Allocate_Float_Mask: Unable to allocate space");

    mask->m = (float **) Allocate_Matrix(rows, cols, sizeof(float));
    if (!mask->m) {
	VIP_Error_Msg("Allocate_Float_Mask: Unable to allocate space");
	return (NULL);
    }

    mask->idcode = FLOAT_MASK_ID;
    mask->cols = cols;
    mask->rows = rows;
    mask->xsymmetry = NOT_SYMMETRIC;
    mask->ysymmetry = NOT_SYMMETRIC;
    return (mask);
}

/*- Bilinear_Interpolate -------------------------------------------

Function to perform bilinear interpolation of grey values in an image.

input: image to interpolate data from.
       x,y - real valued location of desired interpolated grey level

Function returns the interpolated grey value.

-------------------------------------------------------------------*/

double  Bilinear_Interpolate(IMAGE * image, double x, double y)
{
    double  g[4];
    int     xmin, xmax, ymin, ymax;
    double  hfrac, vfrac, upperavg, loweravg;

/* find integer locations around desired 'come from' location */

    if (!Image_OK(image)) {
	VIP_Error_Msg("Bilinear_Interpolate: received invalid image");
	return (0);
    }


    xmin = floor(x);
    xmax = xmin + 1;
    ymin = floor(y);
    ymax = ymin + 1;

    if (xmin >= 0 && xmax < image->cols &&
	ymin >= 0 && ymax < image->rows) {
	switch (image->type) {
	case BYTETYPE:
	    g[0] = image->i.c[ymin][xmin];
	    g[1] = image->i.c[ymin][xmax];
	    g[2] = image->i.c[ymax][xmin];
	    g[3] = image->i.c[ymax][xmax];
	    break;

	case SHORTTYPE:
	    g[0] = image->i.s[ymin][xmin];
	    g[1] = image->i.s[ymin][xmax];
	    g[2] = image->i.s[ymax][xmin];
	    g[3] = image->i.s[ymax][xmax];
	    break;

	case LONGTYPE:
	    g[0] = image->i.l[ymin][xmin];
	    g[1] = image->i.l[ymin][xmax];
	    g[2] = image->i.l[ymax][xmin];
	    g[3] = image->i.l[ymax][xmax];
	    break;

	case FLOATTYPE:
	    g[0] = image->i.f[ymin][xmin];
	    g[1] = image->i.f[ymin][xmax];
	    g[2] = image->i.f[ymax][xmin];
	    g[3] = image->i.f[ymax][xmax];
	    break;

	case DOUBLETYPE:
	    g[0] = image->i.d[ymin][xmin];
	    g[1] = image->i.d[ymin][xmax];
	    g[2] = image->i.d[ymax][xmin];
	    g[3] = image->i.d[ymax][xmax];
	    break;

	default:
	    VIP_Error_Msg("Bilinear_Interpolate: illegal image type");
	    return (0.0);

	}
    }
    else {
	g[0] = g[1] = g[2] = g[3] = 0.0;
    }

    hfrac = x - (double) xmin;
    vfrac = y - (double) ymin;
    upperavg = g[0] + hfrac * (g[1] - g[0]);
    loweravg = g[2] + hfrac * (g[3] - g[2]);

    return (upperavg + vfrac * (loweravg - upperavg));

}

/*- Image_Mean ------------------------------------------------------

Calculate the mean (average) values of pixel values of an image.
The function returns ERROR if any errors are encountered, OK otherwise.
For RGB and HSI images the mean value of each component is calculated
individually, for COMPLEX images the mean value of the real and
imaginary components are computed and mean3 is not used.  Mean2 and
mean3 are not used for images of other types.

--------------------------------------------------------------------*/

int     Image_Mean(IMAGE *im, double *mean1, double *mean2, double *mean3)
{
    register int r, c;

    *mean1 = *mean2 = *mean3 = 0.0;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Mean: received invalid image");
	return (ERROR);
    }

    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		*mean1 += (double) im->i.c[r][c];
	break;
    case SHORTTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		*mean1 += (double) im->i.s[r][c];
	break;
    case LONGTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		*mean1 += (double) im->i.l[r][c];
	break;
    case FLOATTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		*mean1 += (double) im->i.f[r][c];
	break;
    case DOUBLETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		*mean1 += (double) im->i.d[r][c];
	break;
    case COMPLEXTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		*mean1 += (double) im->i.cx[r][c].r;	/* real */
		*mean2 += (double) im->i.cx[r][c].i;	/* imaginary */
	    }
	break;
    case RGBTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		*mean1 += (double) im->i.rgb[r][c][0];	/* red */
		*mean2 += (double) im->i.rgb[r][c][1];	/* green */
		*mean3 += (double) im->i.rgb[r][c][2];	/* blue */
	    }
	break;
    case HSITYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		*mean1 += (double) im->i.hsi[r][c][0];	/* H */
		*mean2 += (double) im->i.hsi[r][c][1];	/* S */
		*mean3 += (double) im->i.hsi[r][c][2];	/* I */
	    }
	break;
    default:
	VIP_Error_Msg("Image_Mean: unknown image type");
	return (ERROR);
    }
    *mean1 /= (double) im->rows * (double) im->cols;
    *mean2 /= (double) im->rows * (double) im->cols;
    *mean3 /= (double) im->rows * (double) im->cols;

    return (OK);
}


/*- Image_Min_Max ---------------------------------------------------

Calculate the minimum and maximum pixel values of an image.
The function returns ERROR if any errors are encountered, OK otherwise.
For RGB and HSI images the minimum and maximum values of each component
are calculated individually, for COMPLEX images the minimum and maximum
values of the real and imaginary components are computed and min3
and max3 are not used.  Min2, min3, max2, and max3 are not used for
images of other types.

--------------------------------------------------------------------*/

int     Image_Min_Max(im, min1, max1, min2, max2, min3, max3)
IMAGE  *im;
double *min1, *max1, *min2, *max2, *min3, *max3;
{
    register int r, c;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Min_Max: received invalid image");
	return (ERROR);
    }
    switch (im->type) {
    case BYTETYPE:
	*min1 = *max1 = (double) im->i.c[0][0];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.c[r][c] < *min1)
		    *min1 = im->i.c[r][c];
		else if (im->i.c[r][c] > *max1)
		    *max1 = im->i.c[r][c];
	break;
    case SHORTTYPE:
	*min1 = *max1 = im->i.s[0][0];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.s[r][c] < *min1)
		    *min1 = im->i.s[r][c];
		else if (im->i.s[r][c] > *max1)
		    *max1 = im->i.s[r][c];
	break;
    case LONGTYPE:
	*min1 = *max1 = im->i.l[0][0];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.l[r][c] < *min1)
		    *min1 = im->i.l[r][c];
		else if (im->i.l[r][c] > *max1)
		    *max1 = im->i.l[r][c];
	break;
    case FLOATTYPE:
	*min1 = *max1 = im->i.f[0][0];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.f[r][c] < *min1)
		    *min1 = im->i.f[r][c];
		else if (im->i.f[r][c] > *max1)
		    *max1 = im->i.f[r][c];
	break;
    case DOUBLETYPE:
	*min1 = *max1 = im->i.d[0][0];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.d[r][c] < *min1)
		    *min1 = im->i.d[r][c];
		else if (im->i.d[r][c] > *max1)
		    *max1 = im->i.d[r][c];
	break;
    case COMPLEXTYPE:
	*min1 = *max1 = im->i.cx[0][0].r;
	*min2 = *max2 = im->i.cx[0][0].i;
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		if (im->i.cx[r][c].r < *min1)
		    *min1 = im->i.cx[r][c].r;
		else if (im->i.cx[r][c].r > *max1)
		    *max1 = im->i.cx[r][c].r;
		if (im->i.cx[r][c].i < *min2)
		    *min2 = im->i.cx[r][c].i;
		else if (im->i.cx[r][c].i > *max2)
		    *max2 = im->i.cx[r][c].i;
	    }
	break;
    case RGBTYPE:
	*min1 = *max1 = im->i.rgb[0][0][0];
	*min2 = *max2 = im->i.rgb[0][0][1];
	*min3 = *max3 = im->i.rgb[0][0][2];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		if (im->i.rgb[r][c][0] < *min1)
		    *min1 = im->i.rgb[r][c][0];
		else if (im->i.rgb[r][c][0] > *max1)
		    *max1 = im->i.rgb[r][c][0];
		if (im->i.rgb[r][c][1] < *min2)
		    *min2 = im->i.rgb[r][c][1];
		else if (im->i.rgb[r][c][1] > *max2)
		    *max2 = im->i.rgb[r][c][1];
		if (im->i.rgb[r][c][2] < *min3)
		    *min3 = im->i.rgb[r][c][2];
		else if (im->i.rgb[r][c][2] > *max3)
		    *max3 = im->i.rgb[r][c][2];
	    }
	break;
    case HSITYPE:
	*min1 = *max1 = im->i.hsi[0][0][0];
	*min2 = *max2 = im->i.hsi[0][0][1];
	*min3 = *max3 = im->i.hsi[0][0][2];
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		if (im->i.hsi[r][c][0] < *min1)
		    *min1 = im->i.hsi[r][c][0];
		else if (im->i.hsi[r][c][0] > *max1)
		    *max1 = im->i.hsi[r][c][0];
		if (im->i.hsi[r][c][1] < *min2)
		    *min2 = im->i.hsi[r][c][1];
		else if (im->i.hsi[r][c][1] > *max2)
		    *max2 = im->i.hsi[r][c][1];
		if (im->i.hsi[r][c][2] < *min3)
		    *min3 = im->i.hsi[r][c][2];
		else if (im->i.hsi[r][c][2] > *max3)
		    *max3 = im->i.hsi[r][c][2];
	    }
	break;
    default:
	VIP_Error_Msg("Image_Min_Max: unknown image type");
	return (ERROR);
    }
    return (OK);
}


/*- Image_Variance --------------------------------------------------

Return the variance of pixel values of an image.
The function returns ERROR if any errors are encountered, OK otherwise.
Note that the function Image_Mean must be called prior the calling
of this function so that the means pixels of the image can be
calculated.
For RGB and HSI images the variance of each component is calculated
independently, for COMPLEX images the variance of the real and
imaginary components are computed and var3 is not used.  Var2 and var3
are not used for images of other types.

--------------------------------------------------------------------*/

int     Image_Variance(im, mean1, mean2, mean3, var1, var2, var3)
IMAGE  *im;
double  mean1, mean2, mean3, *var1, *var2, *var3;
{
    register int r, c;
    double  temp;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Variance: Input image is NULL");
	return (ERROR);
    }

    *var1 = *var2 = *var3 = 0.0;
    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.c[r][c] - mean1;
		*var1 += temp * temp;
	    }
	break;
    case SHORTTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.s[r][c] - mean1;
		*var1 += temp * temp;
	    }
	break;
    case LONGTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.l[r][c] - mean1;
		*var1 += temp * temp;
	    }
	break;
    case FLOATTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.f[r][c] - mean1;
		*var1 += temp * temp;
	    }
	break;
    case DOUBLETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.d[r][c] - mean1;
		*var1 += temp * temp;
	    }
	break;
    case COMPLEXTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.cx[r][c].r - mean1;
		*var1 += temp * temp;
		temp = (double) im->i.cx[r][c].i - mean2;
		*var2 += temp * temp;
	    }
	break;
    case RGBTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.rgb[r][c][0] - mean1;
		*var1 += temp * temp;
		temp = (double) im->i.rgb[r][c][1] - mean2;
		*var2 += temp * temp;
		temp = (double) im->i.rgb[r][c][2] - mean3;
		*var3 += temp * temp;
	    }
	break;
    case HSITYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		temp = (double) im->i.hsi[r][c][0] - mean1;
		*var1 += temp * temp;
		temp = (double) im->i.hsi[r][c][1] - mean2;
		*var2 += temp * temp;
		temp = (double) im->i.hsi[r][c][2] - mean3;
		*var3 += temp * temp;
	    }
	break;
    default:
	VIP_Error_Msg("Image_Variance: unknown image type");
	return (ERROR);
    }

    *var1 /= (double) (im->rows * im->cols);
    *var2 /= (double) (im->rows * im->cols);
    *var3 /= (double) (im->rows * im->cols);
    return (OK);
}


/*- Image_Count -----------------------------------------------------

This function is very similar to Histogram_Image except that it
covers cases for RGBTYPE and HSITYPE as well.
sum1, sum2, and sum3 should be arrays of at least 256 elements.

--------------------------------------------------------------------*/

int     Image_Count(im, sum1, sum2, sum3)
IMAGE  *im;
int     sum1[], sum2[], sum3[];
{
    register int r, c;

    (void) memset(sum1, 0, 256 * sizeof(int));
    (void) memset(sum2, 0, 256 * sizeof(int));
    (void) memset(sum3, 0, 256 * sizeof(int));

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Count: Input image is NULL");
	return (ERROR);
    }

    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		sum1[im->i.c[r][c]]++;
	break;
    case RGBTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		sum1[im->i.rgb[r][c][0]]++;	/* red */
		sum2[im->i.rgb[r][c][1]]++;	/* green */
		sum3[im->i.rgb[r][c][2]]++;	/* blue */
	    }
	break;
    case HSITYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		sum1[im->i.hsi[r][c][0]]++;	/* H */
		sum2[im->i.hsi[r][c][1]]++;	/* S */
		sum3[im->i.hsi[r][c][2]]++;	/* I */
	    }
	break;
    default:
	VIP_Error_Msg("Image_Count: can only processed images of BYTETYPE, RGBTYPE, and HSITYPE");
	return (ERROR);
    }
    return (OK);
}


/*-  Histogram_Image  -----------------------------------------------

Function to generate histogram of grey levels in an image

--------------------------------------------------------------------*/

int     Histogram_Image(image, histogram)
IMAGE  *image;
int     histogram[];
{
    int     i, j;
    unsigned char *cptr;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Histogram_Image: received invalid image");
	return (ERROR);
    }

    for (i = 0; i < 256; i++)	/* initialize histogram array to 0 */
	histogram[i] = 0;

    /* now scan through image and generate histogarm */

    switch (image->type) {

    case BYTETYPE:

	for (i = 0; i < image->rows; i++) {
	    for (j = 0, cptr = image->i.c[i]; j < image->cols; j++, cptr++)
		++histogram[(int) *cptr];
	}
	break;

    default:
	VIP_Error_Msg("Histogram_Image: Only able to process BYTE images");
	break;

    }
    return (OK);
}

/*-  Thresh_Image  --------------------------------------------------

Function to threshold an image

--------------------------------------------------------------------*/

IMAGE  *Thresh_Image(image, threshold)
IMAGE  *image;
int     threshold;
{
    IMAGE  *out_image;
    unsigned char *cptr, *outcptr;
    int     i, j;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Thresh_Image:  received invalid image");
	return (NULL);
    }

    out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, image->type);

    switch (image->type) {

    case BYTETYPE:

	if (threshold < 0 OR threshold > 255) {
	    VIP_Error_Msg("Threshold must be between 0 and 255\n ");
	    return (NULL);
	}
	for (i = 0; i < image->rows; i++) {
	    for (j = 0, cptr = image->i.c[i], outcptr = out_image->i.c[i];
		j < image->cols; j++, cptr++, outcptr++) {
		if (*cptr < threshold)
		    *outcptr = 0;
		else
		    *outcptr = 255;
	    }
	}
	break;

    default:
	VIP_Error_Msg("Thresh_Image: Only able to process BYTE images\n");
	break;

    }
    return (out_image);
}


/*- Floor_Thresh_Image ----------------------------------------------

Pixel values lower than the given threshold are set to zeros but
pixel values higher than it are unchanged.  This function
performs the opposite operation as that in function Ceiling_Thresh_Image.

--------------------------------------------------------------------*/

IMAGE  *Floor_Thresh_Image(im, thres)
IMAGE  *im;
int     thres;
{
    register int r, c;
    IMAGE  *outim;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Floor_Thresh_Image:  received invalid image");
	return (NULL);
    }


    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, BYTETYPE))) {
	VIP_Error_Msg("Floor_Thresh_Image: out of memory");
	return (NULL);
    }

    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.c[r][c] < thres)
		    outim->i.c[r][c] = 0;
		else
		    outim->i.c[r][c] = im->i.c[r][c];
	break;
    default:
	VIP_Error_Msg("Floor_Thresh_Image: can only process images of BYTETYPE");
	Free_Image(outim);
	outim = NULL;
    }
    return (outim);
}


/*- Ceiling_Thresh_Image --------------------------------------------

Pixel values higher than a given threshold value are set to zeros
whereas pixel values lower than it are unchanged.  This function
performs the opposite operation as that in function Floor_Thresh_Image.

--------------------------------------------------------------------*/

IMAGE  *Ceiling_Thresh_Image(im, thres)
IMAGE  *im;
int     thres;
{
    register int r, c;
    IMAGE  *outim;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Ceiling_Thresh_Image:  received invalid image");
	return (NULL);
    }


    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, BYTETYPE))) {
	VIP_Error_Msg("Ceiling_Thresh_Image: out of memory");
	return (NULL);
    }

    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		if (im->i.c[r][c] > thres)
		    outim->i.c[r][c] = 0;
		else
		    outim->i.c[r][c] = im->i.c[r][c];
	break;
    default:
	VIP_Error_Msg("Ceiling_Thresh_Image: can only process images of BYTETYPE");
	Free_Image(outim);
	outim = NULL;
    }
    return (outim);
}


/*- Gaussian_1D -----------------------------------------------------

Produce a one-dimensional Gaussian floating point image.

--------------------------------------------------------------------*/

IMAGE  *Gaussian_1D(omega)
int     omega;
{
    register int x;
    double  sigma, sigma2, C, K, sum;
    int     rad;
    IMAGE  *f;

    rad = omega >> 1;
    if (rad < 1)
	return (NULL);

    sigma = omega / 7.2;
    sigma2 = sigma * sigma;
    if (!(f = (IMAGE *) Allocate_Image(0, 0, 1, 2 * rad + 1, FLOATTYPE))) {
	VIP_Error_Msg("Gaussian_1D: out of memory");
	return (NULL);
    }

    C = 1 / (sigma * SQRT_2PI);
    for (x = -rad; x <= rad; x++) {
	K = x * x / (2 * sigma2);
	f->i.f[0][rad + x] = C * exp(-K);
    }

    sum = 0.0;
    for (x = 0; x < f->cols; x++)
	sum += f->i.f[0][x];
    for (x = 0; x < f->cols; x++)
	f->i.f[0][x] /= sum;

    return (f);
}


/*- Gaussian_2D -----------------------------------------------------

Produce a two-dimensional Gaussian floating point image.

--------------------------------------------------------------------*/

IMAGE  *Gaussian_2D(omega)
int     omega;
{
    register int y, x;
    double  sigma, sigma2, C, K, r2, sum;
    int     rad;
    IMAGE  *f;

    rad = omega >> 1;
    if (rad < 1)
	return (NULL);

    sigma = omega / (TWO_ROOT2 * 3.6);
    sigma2 = sigma * sigma;
    if (!(f = (IMAGE *) Allocate_Image(0, 0, 2 * rad + 1, 2 * rad + 1, FLOATTYPE))) {
	VIP_Error_Msg("Gaussian_2D: out of memory");
	return (NULL);
    }

    C = 1 / (2 * PI * sigma2);
    for (y = -rad; y <= rad; y++)
	for (x = -rad; x <= rad; x++) {
	    r2 = x * x + y * y;
	    K = r2 / (2 * sigma2);
	    f->i.f[rad - y][rad + x] = C * exp(-K);
	}

    sum = 0.0;
    for (y = 0; y < f->rows; y++)
	for (x = 0; x < f->cols; x++)
	    sum += f->i.f[y][x];
    for (y = 0; y < f->rows; y++)
	for (x = 0; x < f->cols; x++)
	    f->i.f[y][x] /= sum;

    return (f);
}


/*- Laplacian_Gaussian_1D ------------------------------------------

Produces a one-dimensional Laplacian of a Gaussian floating point image.

--------------------------------------------------------------------*/

IMAGE  *Laplacian_Gaussian_1D(omega)
int     omega;
{
    register int x;
    double  sigma, sigma2, C, K;
    int     rad;
    IMAGE  *f;

    rad = 1.8 * omega;
    if (rad < 1)
	return (NULL);

    sigma = omega / 2.0;
    sigma2 = sigma * sigma;
    if (!(f = (IMAGE *) Allocate_Image(0, 0, 1, 2 * rad + 1, FLOATTYPE))) {
	VIP_Error_Msg("Laplacian_Gaussian_1D: out of memory");
	return (NULL);
    }

    C = 1 / (sigma2 * sigma2 * sigma / SQRT_2PI);
    for (x = -rad; x <= rad; x++) {
	K = x * x / (2 * sigma2);
	f->i.f[0][x + rad] = (sigma2 - x * x) * C * exp(-K);
    }

    return (f);
}


/*- Laplacian_Gaussian_2D -------------------------------------------

Produces a two-dimensional Laplacian of a Gaussian floating point image.

--------------------------------------------------------------------*/

IMAGE  *Laplacian_Gaussian_2D(omega)
int     omega;
{
    register int y, x;
    double  sigma, sigma2, C, K, r2;
    int     rad;
    IMAGE  *f;

    rad = 1.8 * omega;
    if (rad < 1)
	return (NULL);

    sigma = omega / TWO_ROOT2;
    sigma2 = sigma * sigma;
    if (!(f = (IMAGE *) Allocate_Image(0, 0, 2 * rad + 1, 2 * rad + 1, FLOATTYPE))) {
	VIP_Error_Msg("Laplacian_Gaussian_2D: out of memory");
	return (NULL);
    }

    C = 1 / (2 * PI * sigma2 * sigma2 * sigma2);
    for (y = -rad; y <= rad; y++)
	for (x = -rad; x <= rad; x++) {
	    r2 = x * x + y * y;
	    K = r2 / (2 * sigma2);
	    f->i.f[rad - y][x + rad] = exp(-K) * C * (2 * sigma2 - r2);
	}

    return (f);
}


/*- Logarithmic_Transform_Image -------------------------------------

Return the logarithmic transform of an image.  That is,
the following operation is carried out for each pixel:
	output_image[row][column] = log(1 + input_image[row][column]

--------------------------------------------------------------------*/

IMAGE  *Logarithmic_Transform_Image(inmat)
IMAGE  *inmat;
{
    int     r, c;
    IMAGE  *outmat;

    if (!Image_OK(inmat)) {
	VIP_Error_Msg("Logarithmic_Transform_Image:  received invalid image");
	return (NULL);
    }



    if (!(outmat = (IMAGE *) Allocate_Image(0, 0, inmat->rows, inmat->cols, FLOATTYPE))) {
	VIP_Error_Msg("Logarithmic_Transform_Image: out of memory");
	return (NULL);
    }
    Copy_Header(inmat, outmat);
    outmat->type = FLOATTYPE;
    switch (inmat->type) {
    case BYTETYPE:
	for (r = inmat->rows - 1; r >= 0; r--)
	    for (c = inmat->cols - 1; c >= 0; c--)
		outmat->i.f[r][c] = (float) (log(1.0 + inmat->i.c[r][c]));
	break;
    case SHORTTYPE:
	for (r = inmat->rows - 1; r >= 0; r--)
	    for (c = inmat->cols - 1; c >= 0; c--) {
		if (inmat->i.s[r][c] + 1.0 <= 0.0)
		    VIP_Error_Msg("Logarithmic_Transform_Image: pixel value < 0");
		else
		    outmat->i.f[r][c] = (float) (log(1.0 + inmat->i.s[r][c]));
	    }
	break;
    case LONGTYPE:
	for (r = inmat->rows - 1; r >= 0; r--)
	    for (c = inmat->cols - 1; c >= 0; c--) {
		if (inmat->i.l[r][c] + 1.0 <= 0.0)
		    VIP_Error_Msg("Logarithmic_Transform_Image: pixel value < 0");
		else
		    outmat->i.f[r][c] = (float) (log(1.0 + inmat->i.l[r][c]));
	    }
	break;
    case FLOATTYPE:
	for (r = inmat->rows - 1; r >= 0; r--)
	    for (c = inmat->cols - 1; c >= 0; c--) {
		if (inmat->i.f[r][c] + 1.0 <= 0.0)
		    VIP_Error_Msg("Logarithmic_Transform_Image: pixel value < 0");
		else
		    outmat->i.f[r][c] = (float) (log(1.0 + inmat->i.f[r][c]));
	    }
	break;
    case DOUBLETYPE:
	for (r = inmat->rows - 1; r >= 0; r--)
	    for (c = inmat->cols - 1; c >= 0; c--) {
		if (inmat->i.d[r][c] + 1.0 <= 0.0)
		    VIP_Error_Msg("Logarithmic_Transform_Image: pixel value < 0");
		else
		    outmat->i.f[r][c] = (float) (log(1.0 + inmat->i.d[r][c]));
	    }
	break;
    default:
	VIP_Error_Msg("Logarithmic_Transform_Image: can only process BYTETYPE, FLOATTYPE, DOUBLETYPE images");
	Free_Image(outmat);
	outmat = NULL;
    }
    return (outmat);
}


/*-  Convolve_Image  ------------------------------------------------

Convolution function

This function performs 2D convolution with a mask of integers.  To
achieve precision the integer mask values can be scaled up by a rescale
factor.  The final convolution result is then divided by this rescale
factor to correct the result.

For example: to perform a 3x3 average, the mask values would be 1 and the
rescaling parameter would be 9.

--------------------------------------------------------------------*/

IMAGE  *Convolve_Image(image, mask)
IMAGE  *image;
INT_MASK *mask;
{
    IMAGE  *out_image;
    int     i, j, row, col;
    int     result, rescale, roff, coff;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Convolve_Image: received invalid image");
	return (NULL);
    }

    rescale = (mask->rescale != 1);
    roff = mask->rows / 2;	/* calculate offset to centre of mask */
    coff = mask->cols / 2;

    switch (image->type) {

    case BYTETYPE:
	out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	    image->rows, image->cols, image->type);

	for (row = 0; row < image->rows - mask->rows; row++) {
	    for (col = 0; col < image->cols - mask->cols; col++) {

		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++) {
		    for (j = 0; j < mask->cols; j++) {
			result += mask->m[i][j] * image->i.c[row + i][col + j];
		    }
		}

		if (rescale)	/* rescale the result */
		    out_image->i.c[row + roff][col + coff] = result / mask->rescale;
		else
		    out_image->i.c[row + roff][col + coff] = result;

	    }			/* for each column */
	}			/* for each row */
	break;

    case SHORTTYPE:
	out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	    image->rows, image->cols, image->type);

	for (row = 0; row < image->rows - mask->rows; row++) {
	    for (col = 0; col < image->cols - mask->cols; col++) {

		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++) {
		    for (j = 0; j < mask->cols; j++) {
			result += mask->m[i][j] * image->i.s[row + i][col + j];
		    }
		}

		if (rescale)	/* rescale the result */
		    out_image->i.s[row + roff][col + coff] = result / mask->rescale;
		else
		    out_image->i.s[row + roff][col + coff] = result;

	    }			/* for each column */
	}			/* for each row */
	break;

    case LONGTYPE:
	out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	    image->rows, image->cols, image->type);

	for (row = 0; row < image->rows - mask->rows; row++) {
	    for (col = 0; col < image->cols - mask->cols; col++) {

		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++) {
		    for (j = 0; j < mask->cols; j++) {
			result += mask->m[i][j] * image->i.l[row + i][col + j];
		    }
		}

		if (rescale)	/* rescale the result */
		    out_image->i.l[row + roff][col + coff] = result / mask->rescale;
		else
		    out_image->i.l[row + roff][col + coff] = result;

	    }			/* for each column */
	}			/* for each row */
	break;


    default:
	VIP_Error_Msg("Convolve_Image: Only able to process BYTE, SHORT and LONG images\n");
	return (NULL);
    }

    return (out_image);
}


/*-  FConvolve_Image  ------------------------------------------------

Floating Point Convolution Function

This function performs a 2D convolution with a mask of floats.
A pointer to a FLOAT image is returned.

Code needs to be enhanced to exploit symmetry in masks.

--------------------------------------------------------------------*/

IMAGE  *FConvolve_Image(IMAGE * image, FLOAT_MASK * mask)
{
    IMAGE  *out_image;
    int     i, j, row, col;
    int     roff, coff, rowmax, colmax;
    float   result;

    if (!Image_OK(image)) {
	VIP_Error_Msg("FConvolve_Image: Received invalid image");
	return (NULL);
    }

    if (!mask) {
	VIP_Error_Msg("FConvolve_Image: Received NULL mask");
	return (NULL);
    }


    roff = mask->rows / 2;	/* calculate offset to centre of mask */
    coff = mask->cols / 2;
    rowmax = image->rows - mask->rows;
    colmax = image->cols - mask->cols;

    out_image = (IMAGE *) Allocate_Image(image->umin,
	image->vmin, image->rows, image->cols, FLOATTYPE);
    if (!out_image)
	return (NULL);

    switch (image->type) {

    case BYTETYPE:
	for (row = 0; row < rowmax; row++) {
	    for (col = 0; col < colmax; col++) {
		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++)
		    for (j = 0; j < mask->cols; j++)
			result += mask->m[i][j] * (float) image->i.c[row + i][col + j];

		out_image->i.f[row + roff][col + coff] = result;
	    }
	}
	break;

    case SHORTTYPE:
	for (row = 0; row < rowmax; row++) {
	    for (col = 0; col < colmax; col++) {
		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++)
		    for (j = 0; j < mask->cols; j++)
			result += mask->m[i][j] * (float) image->i.s[row + i][col + j];

		out_image->i.f[row + roff][col + coff] = result;
	    }
	}
	break;

    case LONGTYPE:
	for (row = 0; row < rowmax; row++) {
	    for (col = 0; col < colmax; col++) {
		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++)
		    for (j = 0; j < mask->cols; j++)
			result += mask->m[i][j] * (float) image->i.l[row + i][col + j];

		out_image->i.f[row + roff][col + coff] = result;
	    }
	}
	break;

    case FLOATTYPE:
	for (row = 0; row < rowmax; row++) {
	    for (col = 0; col < colmax; col++) {
		result = 0;	/* initialise convolution result to 0 */

		for (i = 0; i < mask->rows; i++)
		    for (j = 0; j < mask->cols; j++)
			result += mask->m[i][j] * (float) image->i.f[row + i][col + j];

		out_image->i.f[row + roff][col + coff] = result;
	    }
	}
	break;

    default:
	VIP_Error_Msg("FConvolve_Image: Only able to process BYTE, SHORT, LONG and FLOAT images\n");
	Free_Image(out_image);
	return (NULL);
    }				/* switch */

    return (out_image);
}


/*-  Diff_Image  -----------------------------------------------------

Function to take the absolute difference of two images.  Difference is
amplified by gain.

--------------------------------------------------------------------*/

IMAGE  *Diff_Image(image1, image2, gain)
IMAGE  *image1, *image2;
int     gain;
{
    IMAGE  *out_image;
    int     umin, vmin, cols, rows;
    int     i, j, result;


    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Diff_Image:  received invalid image");
	return (NULL);
    }


    if (image1->type != image2->type) {
	VIP_Error_Msg("Diff_Image: Attempt to get difference between different types of images");
	return (NULL);
    }
    /* set bounds of result to the largest common window in both images */

    if (Common_Window(image1, image2, &umin, &vmin, &rows, &cols) == 0) {
	VIP_Error_Msg("Images have no common area ");
	return (NULL);
    }

    out_image = (IMAGE *) Allocate_Image(umin, vmin, rows, cols, image1->type);

    switch (image1->type) {

    case BYTETYPE:
	for (i = 0; i < rows; i++) {
	    for (j = 0; j < cols; j++) {
		result = gain * abs((int) image1->i.c[i][j] - (int) image2->i.c[i][j]);
		if (result > 255)
		    out_image->i.c[i][j] = 255;
		else if (result < 0)
		    out_image->i.c[i][j] = 0;
		else
		    out_image->i.c[i][j] = result;
	    }
	}
	break;


    case SHORTTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.s[i][j] =
		    gain * abs((int) image1->i.s[i][j] - (int) image2->i.s[i][j]);
	break;


    case LONGTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.l[i][j] =
		    gain * abs((int) image1->i.l[i][j] - (int) image2->i.l[i][j]);
	break;


    case FLOATTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.f[i][j] =
		    (float) gain *fabs((double) image1->i.f[i][j] - (double) image2->i.f[i][j]);

	break;



    default:
	VIP_Error_Msg("Diff_Image: Only able to process BYTE, SHORT, LONG and FLOAT images");
	return (NULL);
    }
    return (out_image);
}


/*-  Subtract_Image  ------------------------------------------------

Function to subtract two images. Image2 is subtracted from image1, the
result is multiplied by 'gain' and shifted by 'offset'.  BYTE images
have their results clamped 0-255 otherwise no checks for overflow are
made.

--------------------------------------------------------------------*/

IMAGE  *Subtract_Image(image1, image2, offset, gain)
IMAGE  *image1, *image2;
int     offset, gain;
{
    IMAGE  *out_image;
    int     umin, vmin, cols, rows;
    int     i, j, result;

    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Subtract_Image: received invalid image");
	return (NULL);
    }


    /* set bounds of result to the largest common window in both images */

    if (Common_Window(image1, image2, &umin, &vmin, &rows, &cols) == 0) {
	VIP_Error_Msg("Images have no common area");
	return (NULL);
    }

    out_image = (IMAGE *) Allocate_Image(umin, vmin, rows, cols, image1->type);

    switch (image1->type) {

    case BYTETYPE:
	for (i = 0; i < rows; i++) {
	    for (j = 0; j < cols; j++) {
		result = gain * (image1->i.c[i][j] - image2->i.c[i][j]) + offset;
		if (result > 255)
		    out_image->i.c[i][j] = 255;
		else if (result < 0)
		    out_image->i.c[i][j] = 0;
		else
		    out_image->i.c[i][j] = result;
	    }
	}
	break;


    case SHORTTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.s[i][j] =
		    gain * (image1->i.s[i][j] - image2->i.s[i][j]) + offset;

	break;


    case LONGTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.l[i][j] =
		    gain * (image1->i.l[i][j] - image2->i.l[i][j]) + offset;

	break;

    case FLOATTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.f[i][j] =
		    (float) gain *(image1->i.f[i][j] - image2->i.f[i][j]) + (float) offset;

	break;


    default:
	VIP_Error_Msg("Subtract_Image: Only able to process BYTE, SHORT, LONG and FLOAT images");
	return (NULL);
    }
    return (out_image);
}


/*-  Add_Image  ------------------------------------------------

Function to add two images. BYTE images have their results clamped 0-255
otherwise no checks for overflow are made.

--------------------------------------------------------------------*/

IMAGE  *Add_Image(image1, image2)
IMAGE  *image1, *image2;
{
    IMAGE  *out_image;
    int     umin, vmin, cols, rows;
    int     i, j, result;

    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Add_Image: Received invalid image");
	return (NULL);
    }

    /* set bounds of result to the largest common window in both images */

    if (Common_Window(image1, image2, &umin, &vmin, &rows, &cols) == 0) {
	VIP_Error_Msg("Images have no common area");
	return (NULL);
    }

    out_image = (IMAGE *) Allocate_Image(umin, vmin, rows, cols, image1->type);

    switch (image1->type) {

    case BYTETYPE:
	for (i = 0; i < rows; i++) {
	    for (j = 0; j < cols; j++) {
		result = image1->i.c[i][j] + image2->i.c[i][j];
		if (result > 255)
		    out_image->i.c[i][j] = 255;
		else if (result < 0)
		    out_image->i.c[i][j] = 0;
		else
		    out_image->i.c[i][j] = result;
	    }
	}
	break;


    case SHORTTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.s[i][j] = image1->i.s[i][j] + image2->i.s[i][j];

	break;


    case LONGTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.l[i][j] = image1->i.l[i][j] + image2->i.l[i][j];

	break;

    case FLOATTYPE:
	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++)
		out_image->i.f[i][j] = image1->i.f[i][j] + image2->i.f[i][j];

	break;


    default:
	VIP_Error_Msg("Add_Image: Only able to process BYTE, SHORT, LONG and FLOAT images");
	return (NULL);
    }
    return (out_image);
}


/*-  Scale_Image ----------------------------------------------------

Function to pixel-wise scale an image by a given factor.

--------------------------------------------------------------------*/

IMAGE  *Scale_Image(image, factor)
IMAGE  *image;
float   factor;
{
    register int r, c;
    IMAGE  *out_image;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Scale_Image: received invalid image");
	return (NULL);
    }


    out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, image->type);

    switch (out_image->type) {
    case BYTETYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.c[r][c] = (unsigned char) ((float) image->i.c[r][c] * factor);
	break;
    case SHORTTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.s[r][c] = (short) ((float) image->i.s[r][c] * factor);
	break;
    case LONGTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.l[r][c] = (long) ((float) image->i.l[r][c] * factor);
	break;
    case FLOATTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.f[r][c] = image->i.f[r][c] * factor;
	break;
    case DOUBLETYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.d[r][c] = image->i.d[r][c] * factor;
	break;
    case COMPLEXTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++) {
		out_image->i.cx[r][c].r = image->i.cx[r][c].r * factor;
		out_image->i.cx[r][c].i = image->i.cx[r][c].i * factor;
	    }
	break;
    default:
	VIP_Error_Msg("Scale_Image: Only able to process BYTE, SHORT, LONG, FLOAT, DOUBLE, and COMPLEX images");
    }
    return (out_image);
}

/*-  Offset_Image ----------------------------------------------------

Function to pixel-wise offset an image by a given factor.

--------------------------------------------------------------------*/

IMAGE  *Offset_Image(image, factor)
IMAGE  *image;
float   factor;
{
    register int r, c;
    IMAGE  *out_image;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Offset_Image: received invalid image");
	return (NULL);
    }

    out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	image->rows, image->cols, image->type);

    switch (out_image->type) {
    case BYTETYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.c[r][c] = (unsigned char) ((float) image->i.c[r][c] + factor);
	break;
    case SHORTTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.s[r][c] = (short) ((float) image->i.s[r][c] + factor);
	break;
    case LONGTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.l[r][c] = (long) ((float) image->i.l[r][c] + factor);
	break;
    case FLOATTYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.f[r][c] = image->i.f[r][c] + factor;
	break;
    case DOUBLETYPE:
	for (r = 0; r < image->rows; r++)
	    for (c = 0; c < image->cols; c++)
		out_image->i.d[r][c] = image->i.d[r][c] + factor;
	break;

    default:
	VIP_Error_Msg("Offset_Image: Only able to process BYTE, SHORT, LONG, FLOAT, and DOUBLE images");
    }
    return (out_image);
}

/*-  DeRes_Image  ---------------------------------------------------

Function to reduce the resolution of an image.

--------------------------------------------------------------------*/

IMAGE  *DeRes_Image(image, DeRes)
IMAGE  *image;
int     DeRes;
{
    IMAGE  *out_image;
    int     cols, rows;
    int     i, j, k, l;
    int     sum, DeResSqrd;


    if (!Image_OK(image)) {
	VIP_Error_Msg("DeRes_Image: received invalid image");
	return (NULL);
    }

    if (Out_Of_Range_i(DeRes, 1, MIN(image->rows, image->cols),
	    "DeRes_Image: Illegal de-resolution factor"))
	return (NULL);

    DeResSqrd = DeRes * DeRes;

    cols = image->cols / DeRes;	/* No of de-resolved cols and rows */
    rows = image->rows / DeRes;

    out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
	rows * DeRes, cols * DeRes, image->type);
    if (out_image == NULL)
	return (NULL);

    out_image->resolution = image->resolution * DeRes;

    switch (image->type) {

    case BYTETYPE:

	for (i = 0; i < rows; i++)
	    for (j = 0; j < cols; j++) {
		sum = 0;
		for (k = i * DeRes; k < (i + 1) * DeRes; k++) {
		    for (l = j * DeRes; l < (j + 1) * DeRes; l++) {
			sum += image->i.c[k][l];
		    }
		}
		sum /= DeResSqrd;
		for (k = i * DeRes; k < (i + 1) * DeRes; k++) {
		    for (l = j * DeRes; l < (j + 1) * DeRes; l++) {
			out_image->i.c[k][l] = sum;
		    }
		}
	    }
	break;

    default:
	VIP_Error_Msg("DeRes_Image: Only able to process BYTE images");
	return (NULL);
    }

    return (out_image);
}


/*-  Res_Image  -----------------------------------------------------

Function to increase the resolution of an image.  The input image
should be of BYTETYPE.  If the third argument is non-zero (apply
the Gaussian smoothing process to the input image while increasing
image resolution), and the resolutin factor is 2, the image returned
is of FLOATTYPE, otherwise the image returned is of BYTETYPE.


--------------------------------------------------------------------*/

IMAGE  *Res_Image(image, Res, average)
IMAGE  *image;
int     Res, average;
{
    IMAGE  *out_image;
    int     cols, rows;
    int     i, j, k, l, m, n;
    float   r, c, mr, mc;
    static float gmask7[7][7] = {
	{0.0000000009, 0.0000001841, 0.0000044016, 0.0000126788, 0.0000044016,
	0.0000001841, 0.0000000009},
	{0.0000001841, 0.0000365212, 0.0008728562, 0.0025142505, 0.0008728562,
	0.0000365212, 0.0000001841},
	{0.0000044016, 0.0008728562, 0.0208612438, 0.0600905343, 0.0208612438,
	0.0008728562, 0.0000044016},
	{0.0000126788, 0.0025142505, 0.0600905343, 0.1730899810, 0.0600905343,
	0.0025142505, 0.0000126788},
	{0.0000044016, 0.0008728562, 0.0208612438, 0.0600905343, 0.0208612438,
	0.0008728562, 0.0000044016},
	{0.0000001841, 0.0000365212, 0.0008728562, 0.0025142505, 0.0008728562,
	0.0000365212, 0.0000001841},
	{0.0000000009, 0.0000001841, 0.0000044016, 0.0000126788, 0.0000044016,
	0.0000001841, 0.0000000009}
    };

    if (!Image_OK(image)) {
	VIP_Error_Msg("Res_Image: received invalid image");
	return (NULL);
    }



    if (Out_Of_Range_i(Res, 1, MAXINT, "Res_Image: Illegal resolution factor"))
	return (NULL);

    switch (image->type) {

    case BYTETYPE:
	cols = image->cols * Res;	/* No of resolved cols and rows */
	rows = image->rows * Res;

	if (average && Res == 2)
	    out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
		rows, cols, FLOATTYPE);
	else
	    out_image = (IMAGE *) Allocate_Image(image->umin, image->vmin,
		rows, cols, image->type);
	if (out_image == NULL)
	    return (NULL);

	out_image->resolution = image->resolution * Res;

	if (average && Res == 2) {
	    /*
	     * apply Gaussian operation while increasing resolution to
	     * eliminate the "blocking effect" -- slow process.
	     */
	    for (r = 2.0; r < image->rows - 2; r += 0.5)
		for (c = 2.0; c < image->cols - 2; c += 0.5) {
		    out_image->i.f[(int) (2 * r)][(int) (2 * c)] = 0.0;
		    for (mr = -1.5, k = 0; mr <= 1.5; mr += 0.5, k++)
			for (mc = -1.5, l = 0; mc <= 1.5; mc += 0.5, l++) {
			    i = mr + r;
			    j = mc + c;
			    if (i == (float) ((int) i) &&
				j == (float) ((int) j)) {
				out_image->i.f[(int) (2 * r)][(int) (2 * c)] +=
				    image->i.c[(int) i][(int) j] *
				    gmask7[k][l];
			    }
			}
		}
	}
	else if ((average && Res != 2) || !average) {
	    if (average && Res != 2)
		VIP_Error_Msg("The resolution factor can only be 2 if Gaussian smoothing is specified");
	    for (i = k = 0; i < rows; i += Res, k++)
		for (j = l = 0; j < cols; j += Res, l++)
		    for (m = Res - 1; m >= 0; m--)
			for (n = Res - 1; n >= 0; n--)
			    out_image->i.c[i + m][j + n] = image->i.c[k][l];
	}
	break;

    default:
	VIP_Error_Msg("Res_Image: Only able to process BYTE images");
	return (NULL);
    }

    return (out_image);
}


/*- Sqrt_Image ------------------------------------------------------

Takes the square root of each pixel in an image.

--------------------------------------------------------------------*/

IMAGE  *Sqrt_Image(im)
IMAGE  *im;
{
    register int r, c;
    IMAGE  *outim;
    float  *fin, *fout;
    double *din, *dout;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Sqrt_Image: received invalid image");
	return (NULL);
    }

    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, im->type))) {
	VIP_Error_Msg("Image_Sqrt: out of memory");
	return (NULL);
    }
    Copy_Header(im, outim);

    switch (im->type) {

    case BYTETYPE:
	for (r = 0; r < im->rows; r++)
	    for (c = 0; c < im->cols; c++)
		/* no need to check for -ve values as type is unsigned */
		outim->i.c[r][c] = sqrt((double) im->i.c[r][c]);

	break;

    case SHORTTYPE:
	for (r = 0; r < im->rows; r++)
	    for (c = 0; c < im->cols; c++)
		if (im->i.s[r][c] < 0) {
		    VIP_Error_Msg("Image_Sqrt: Warning - image contains negative pixel values");
		    outim->i.s[r][c] = 0;
		}
		else
		    outim->i.s[r][c] = sqrt((double) im->i.s[r][c]);

	break;

    case LONGTYPE:
	for (r = 0; r < im->rows; r++)
	    for (c = 0; c < im->cols; c++)
		if (im->i.l[r][c] < 0) {
		    VIP_Error_Msg("Image_Sqrt: Warning - image contains negative pixel values");
		    outim->i.l[r][c] = 0;
		}
		else
		    outim->i.l[r][c] = sqrt((double) im->i.l[r][c]);

	break;


    case FLOATTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1,
		fin = &im->i.f[r][c], fout = &outim->i.f[r][c];
		c >= 0; c--, fin--, fout--)
		if (*fin < 0.0) {
		    VIP_Error_Msg("Image_Sqrt: Warning - image contains negative pixel values");
		    *fout = 0.0;
		}
		else
		    *fout = sqrt(*fin);
	break;

    case DOUBLETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1,
		din = &im->i.d[r][c], dout = &outim->i.d[r][c];
		c >= 0; c--, din--, dout--)
		if (*din < 0.0) {
		    VIP_Error_Msg("Image_Sqrt: Warning - image contains negative pixel values");
		    *dout = 0.0;
		}
		else
		    *dout = sqrt(*din);
	break;

    default:
	VIP_Error_Msg("Image_Sqrt: can only process BYTE, SHORT, LONG, FLOAT and DOUBLETYPE images");
	Free_Image(outim);
	return (NULL);
    }
    return (outim);
}


/*- Sqr_Image -------------------------------------------------------

Squares each pixel in an image.  For BYTE images the result is clamped
to 0-255, otherwise no checks for overflow are made.

--------------------------------------------------------------------*/

IMAGE  *Sqr_Image(im)
IMAGE  *im;
{
    register int r, c;
    IMAGE  *outim;
    float  *fin, *fout, result;
    double *din, *dout;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Sqr_Image: received invalid image");
	return (NULL);
    }

    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, im->type))) {
	VIP_Error_Msg("Image_Sqr: out of memory");
	return (NULL);
    }
    Copy_Header(im, outim);

    switch (im->type) {

    case BYTETYPE:
	for (r = 0; r < im->rows; r++) {
	    for (c = 0; c < im->cols; c++) {
		result = SQR(im->i.c[r][c]);
		if (result > 255)
		    outim->i.c[r][c] = 255;
		else
		    outim->i.c[r][c] = result;
	    }
	}
	break;

    case SHORTTYPE:
	for (r = 0; r < im->rows; r++)
	    for (c = 0; c < im->cols; c++)
		outim->i.s[r][c] = SQR(im->i.s[r][c]);
	break;

    case LONGTYPE:
	for (r = 0; r < im->rows; r++)
	    for (c = 0; c < im->cols; c++)
		outim->i.l[r][c] = SQR(im->i.l[r][c]);
	break;

    case FLOATTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1,
		fin = &im->i.f[r][c], fout = &outim->i.f[r][c];
		c >= 0; c--, fin--, fout--)
		*fout = (*fin) * (*fin);
	break;

    case DOUBLETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1,
		din = &im->i.d[r][c], dout = &outim->i.d[r][c];
		c >= 0; c--, din--, dout--)
		*dout = (*din) * (*din);
	break;


    default:
	VIP_Error_Msg("Image_Sqr: can only process BYTE, SHORT, LONG, FLOAT and DOUBLETYPE images");
	Free_Image(outim);
	return (NULL);
    }
    return (outim);
}


/*- Knn_Pixel_Comp --------------------------------------------------

For comparison of pixel values in function Knn_Image.
Returns -1, 0, or 1 depending upon if *a <, =, or > *b.

--------------------------------------------------------------------*/

static int Knn_Pixel_Comp(a, b)
int    *a, *b;
{
    int     diff;

    diff = ABS(*a) - ABS(*b);
    if (diff < 0)
	return (-1);
    else if (diff == 0)
	return (0);
    return (1);
}


/*- Knn_Image -------------------------------------------------------

Returns a K Nearest Neighbour Smoothing image.
The size of the neighbourhood used in the smoothing process is
(2*n+1) x (2*n+1).  K is the number of neighbours to be selected
in the smoothing process.

--------------------------------------------------------------------*/

IMAGE  *Knn_Image(im, n, k)
IMAGE  *im;
int     n, k;
{
    register int rr, cc, r, c, kk;
    int    *neigh = NULL, sum, siz, sizint, nr, nc;
    float   count;
    IMAGE  *mat;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Knn_Image: received invalid image");
	return (NULL);
    }

    /* allocate space for the output image */
    if (!(mat = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, FLOATTYPE))) {
	VIP_Error_Msg("Knn_Image: out of memory");
	return (NULL);
    }
    Copy_Header(im, mat);
    mat->type = FLOATTYPE;

    /* allocate space for k neighbours */
    siz = (n << 1) + 1;
    siz *= siz;
    sizint = sizeof(int);
    if (!(neigh = (int *) malloc(siz * sizint))) {
	Free_Image(mat);
	VIP_Error_Msg("Knn_Image: out of memory");
	return (NULL);
    }

    count = (float) k;
    nr = im->rows;
    nc = im->cols;
    switch (im->type) {
    case BYTETYPE:
	for (rr = n; rr < nr - n; rr++)
	    for (cc = n; cc < nc - n; cc++) {
		sum = kk = 0;
		for (r = -n; r <= n; r++)
		    for (c = -n; c <= n; c++) {
			if (r == 0 && c == 0)
			    continue;
			neigh[kk++] = (int) im->i.c[rr + r][cc + c] - (int) im->i.c[rr][cc];
		    }
		qsort(neigh, siz, sizint, Knn_Pixel_Comp);
		for (kk = 0; kk < k; kk++)
		    sum += neigh[kk] + (int) im->i.c[rr][cc];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case SHORTTYPE:
	for (rr = n; rr < nr - n; rr++)
	    for (cc = n; cc < nc - n; cc++) {
		sum = kk = 0;
		for (r = -n; r <= n; r++)
		    for (c = -n; c <= n; c++) {
			if (r == 0 && c == 0)
			    continue;
			neigh[kk++] = (int) im->i.s[rr + r][cc + c] - (int) im->i.s[rr][cc];
		    }
		qsort(neigh, siz, sizint, Knn_Pixel_Comp);
		for (kk = 0; kk < k; kk++)
		    sum += neigh[kk] + (int) im->i.s[rr][cc];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case LONGTYPE:
	for (rr = n; rr < nr - n; rr++)
	    for (cc = n; cc < nc - n; cc++) {
		sum = kk = 0;
		for (r = -n; r <= n; r++)
		    for (c = -n; c <= n; c++) {
			if (r == 0 && c == 0)
			    continue;
			neigh[kk++] = (int) im->i.l[rr + r][cc + c] - (int) im->i.l[rr][cc];
		    }
		qsort(neigh, siz, sizint, Knn_Pixel_Comp);
		for (kk = 0; kk < k; kk++)
		    sum += neigh[kk] + (int) im->i.l[rr][cc];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case FLOATTYPE:
	for (rr = n; rr < nr - n; rr++)
	    for (cc = n; cc < nc - n; cc++) {
		sum = kk = 0;
		for (r = -n; r <= n; r++)
		    for (c = -n; c <= n; c++) {
			if (r == 0 && c == 0)
			    continue;
			neigh[kk++] = (int) im->i.f[rr + r][cc + c] - (int) im->i.f[rr][cc];
		    }
		qsort(neigh, siz, sizint, Knn_Pixel_Comp);
		for (kk = 0; kk < k; kk++)
		    sum += neigh[kk] + (int) im->i.f[rr][cc];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case DOUBLETYPE:
	for (rr = n; rr < nr - n; rr++)
	    for (cc = n; cc < nc - n; cc++) {
		sum = kk = 0;
		for (r = -n; r <= n; r++)
		    for (c = -n; c <= n; c++) {
			if (r == 0 && c == 0)
			    continue;
			neigh[kk++] = (int) im->i.d[rr + r][cc + c] - (int) im->i.d[rr][cc];
		    }
		qsort(neigh, siz, sizint, Knn_Pixel_Comp);
		for (kk = 0; kk < k; kk++)
		    sum += neigh[kk] + (int) im->i.d[rr][cc];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    default:
	VIP_Error_Msg("Knn_Image: can only process images of BYTETYPE, SHORTTYPE, LONGTYPE, FLOATTYPE, and DOUBLETYPE");
	Free_Image(mat);
	mat = NULL;
    }
    free((char *) neigh);

    return (mat);
}


/*- Snn_Image -------------------------------------------------------

Returns a Symmetric Nearest Neighbour Smoothing image.
The size of the neighbourhood used in the smoothing process is
nbhsiz x nbhsize.

--------------------------------------------------------------------*/

IMAGE  *Snn_Image(im, nbhsiz)
IMAGE  *im;
int     nbhsiz;

/* symmetric nearest neighbour smoothing. */
{
    register int rr, cc, r, c;
    int     sum;
    float   count;
    IMAGE  *mat = NULL;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Snn_Image: input image is NULL");
	return (NULL);
    }
    if (!(mat = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, FLOATTYPE))) {
	VIP_Error_Msg("out of memory");
	return (NULL);
    }

    Copy_Header(im, mat);
    mat->type = FLOATTYPE;
    count = 2.0 * nbhsiz * (nbhsiz + 1);

    switch (im->type) {
    case BYTETYPE:
	for (rr = nbhsiz; rr < im->rows - nbhsiz; rr++)
	    for (cc = nbhsiz; cc < im->cols - nbhsiz; cc++) {
		sum = 0;
		for (r = -nbhsiz; r <= -1; r++)
		    for (c = -nbhsiz; c <= nbhsiz; c++)
			if (fabs(im->i.c[rr][cc] - im->i.c[rr + r][cc + c]) <
			    fabs(im->i.c[rr][cc] - im->i.c[rr - r][cc - c]))
			    sum += im->i.c[rr + r][cc + c];
			else
			    sum += im->i.c[rr - r][cc - c];
		for (c = -nbhsiz; c < 0; c++)
		    if (fabs(im->i.c[rr][cc] - im->i.c[rr][cc + c]) <
			fabs(im->i.c[rr][cc] - im->i.c[rr][cc - c]))
			sum += im->i.c[rr][cc + c];
		    else
			sum += im->i.c[rr][cc - c];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case SHORTTYPE:
	for (rr = nbhsiz; rr < im->rows - nbhsiz; rr++)
	    for (cc = nbhsiz; cc < im->cols - nbhsiz; cc++) {
		sum = 0;
		for (r = -nbhsiz; r <= -1; r++)
		    for (c = -nbhsiz; c <= nbhsiz; c++)
			if (fabs(im->i.s[rr][cc] - im->i.s[rr + r][cc + c]) <
			    fabs(im->i.s[rr][cc] - im->i.s[rr - r][cc - c]))
			    sum += im->i.s[rr + r][cc + c];
			else
			    sum += im->i.s[rr - r][cc - c];
		for (c = -nbhsiz; c < 0; c++)
		    if (fabs(im->i.s[rr][cc] - im->i.s[rr][cc + c]) <
			fabs(im->i.s[rr][cc] - im->i.s[rr][cc - c]))
			sum += im->i.s[rr][cc + c];
		    else
			sum += im->i.s[rr][cc - c];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case LONGTYPE:
	for (rr = nbhsiz; rr < im->rows - nbhsiz; rr++)
	    for (cc = nbhsiz; cc < im->cols - nbhsiz; cc++) {
		sum = 0;
		for (r = -nbhsiz; r <= -1; r++)
		    for (c = -nbhsiz; c <= nbhsiz; c++)
			if (fabs(im->i.l[rr][cc] - im->i.l[rr + r][cc + c]) <
			    fabs(im->i.l[rr][cc] - im->i.l[rr - r][cc - c]))
			    sum += im->i.l[rr + r][cc + c];
			else
			    sum += im->i.l[rr - r][cc - c];
		for (c = -nbhsiz; c < 0; c++)
		    if (fabs(im->i.l[rr][cc] - im->i.l[rr][cc + c]) <
			fabs(im->i.l[rr][cc] - im->i.l[rr][cc - c]))
			sum += im->i.l[rr][cc + c];
		    else
			sum += im->i.l[rr][cc - c];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case FLOATTYPE:
	for (rr = nbhsiz; rr < im->rows - nbhsiz; rr++)
	    for (cc = nbhsiz; cc < im->cols - nbhsiz; cc++) {
		sum = 0;
		for (r = -nbhsiz; r <= -1; r++)
		    for (c = -nbhsiz; c <= nbhsiz; c++)
			if (fabs(im->i.f[rr][cc] - im->i.f[rr + r][cc + c]) <
			    fabs(im->i.f[rr][cc] - im->i.f[rr - r][cc - c]))
			    sum += im->i.f[rr + r][cc + c];
			else
			    sum += im->i.f[rr - r][cc - c];
		for (c = -nbhsiz; c < 0; c++)
		    if (fabs(im->i.f[rr][cc] - im->i.f[rr][cc + c]) <
			fabs(im->i.f[rr][cc] - im->i.f[rr][cc - c]))
			sum += im->i.f[rr][cc + c];
		    else
			sum += im->i.f[rr][cc - c];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    case DOUBLETYPE:
	for (rr = nbhsiz; rr < im->rows - nbhsiz; rr++)
	    for (cc = nbhsiz; cc < im->cols - nbhsiz; cc++) {
		sum = 0;
		for (r = -nbhsiz; r <= -1; r++)
		    for (c = -nbhsiz; c <= nbhsiz; c++)
			if (fabs(im->i.d[rr][cc] - im->i.d[rr + r][cc + c]) <
			    fabs(im->i.d[rr][cc] - im->i.d[rr - r][cc - c]))
			    sum += im->i.d[rr + r][cc + c];
			else
			    sum += im->i.d[rr - r][cc - c];
		for (c = -nbhsiz; c < 0; c++)
		    if (fabs(im->i.d[rr][cc] - im->i.d[rr][cc + c]) <
			fabs(im->i.d[rr][cc] - im->i.d[rr][cc - c]))
			sum += im->i.d[rr][cc + c];
		    else
			sum += im->i.d[rr][cc - c];
		mat->i.f[rr][cc] = (float) sum / count;
	    }
	break;
    default:
	VIP_Error_Msg("Snn_Image: can only process images of BYTETYPE, SHORTTYPE, LONGTYPE, FLOATTYPE, and DOUBLETYPE");
	Free_Image(mat);
	return (NULL);
    }
    return (mat);
}


/*- Clip_Image ------------------------------------------------------

Return a sub-image from a larger image.  All the arguments are
assumed to be within the image boundary.

--------------------------------------------------------------------*/

IMAGE  *Clip_Image(im, orgr, orgc, nrows, ncols)
IMAGE  *im;
int     orgr, orgc, nrows, ncols;
{
    register int r, subr;
    IMAGE  *subim;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Clip_Image: received invalid image");
	return (NULL);
    }


    if (!(subim = (IMAGE *) Allocate_Image(orgr, orgc, nrows, ncols, im->type))) {
	VIP_Error_Msg("Clip_Image: out of memory");
	return (NULL);
    }

    Copy_Header(im, subim);
    subim->rows = nrows;
    subim->cols = ncols;
    subim->umin = orgr;
    subim->vmin = orgc;

    switch (im->type) {
    case BYTETYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.c[subr], &im->i.c[r][orgc], sizeof(unsigned char) * ncols);
	break;
    case SHORTTYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.s[subr], &im->i.s[r][orgc], sizeof(short) * ncols);
	break;
    case LONGTYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.l[subr], &im->i.l[r][orgc], sizeof(long) * ncols);
	break;
    case FLOATTYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.f[subr], &im->i.f[r][orgc], sizeof(float) * ncols);
	break;
    case DOUBLETYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.d[subr], &im->i.d[r][orgc], sizeof(double) * ncols);
	break;
    case COMPLEXTYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.cx[subr], &im->i.cx[r][orgc], sizeof(COMPLEX) * ncols);
	break;
    case RGBTYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.rgb[subr], im->i.rgb[r][orgc], sizeof(RGB) * ncols);
	break;
    case HSITYPE:
	for (subr = 0, r = orgr; subr < nrows; subr++, r++)
	    memcpy(subim->i.hsi[subr], im->i.hsi[r][orgc], sizeof(HSI) * ncols);
	break;
    default:
	VIP_Error_Msg("Clip_Image: unknown image type");
	Free_Image(subim);
	subim = NULL;
    }

    return (subim);
}


/*- Image_Row_Reflect -----------------------------------------------

Return an image that is a mirror reflection horizontally of a given
image.

--------------------------------------------------------------------*/

IMAGE  *Image_Row_Reflect(im)
IMAGE  *im;
{
    register int r;
    int     nr, nr_half, nc;
    IMAGE  *outim;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Row_Reflect: received invalid image");
	return (NULL);
    }


    nr = im->rows - 1;
    nc = im->cols;
    nr_half = im->rows / 2 - 1;

    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, im->type))) {
	VIP_Error_Msg("Image_Column_Reflect: out of memory");
	return (NULL);
    }
    Copy_Header(im, outim);

    switch (im->type) {
    case BYTETYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.c[r], im->i.c[nr - r], sizeof(unsigned char) * nc);
	    memcpy(outim->i.c[nr - r], im->i.c[r], sizeof(unsigned char) * nc);
	}
	break;
    case SHORTTYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.s[r], im->i.s[nr - r], sizeof(short) * nc);
	    memcpy(outim->i.s[nr - r], im->i.s[r], sizeof(short) * nc);
	}
	break;
    case LONGTYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.l[r], im->i.l[nr - r], sizeof(long) * nc);
	    memcpy(outim->i.l[nr - r], im->i.l[r], sizeof(long) * nc);
	}
	break;
    case FLOATTYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.f[r], im->i.f[nr - r], sizeof(float) * nc);
	    memcpy(outim->i.f[nr - r], im->i.f[r], sizeof(float) * nc);
	}
	break;
    case DOUBLETYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.d[r], im->i.d[nr - r], sizeof(double) * nc);
	    memcpy(outim->i.d[nr - r], im->i.d[r], sizeof(double) * nc);
	}
	break;
    case COMPLEXTYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.cx[r], im->i.cx[nr - r], sizeof(COMPLEX) * nc);
	    memcpy(outim->i.cx[nr - r], im->i.cx[r], sizeof(COMPLEX) * nc);
	}
	break;
    case RGBTYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.rgb[r], im->i.rgb[nr - r], sizeof(RGB) * nc);
	    memcpy(outim->i.rgb[nr - r], im->i.rgb[r], sizeof(RGB) * nc);
	}
	break;
    case HSITYPE:
	for (r = nr_half; r >= 0; r--) {
	    memcpy(outim->i.hsi[r], im->i.hsi[nr - r], sizeof(HSI) * nc);
	    memcpy(outim->i.hsi[nr - r], im->i.hsi[r], sizeof(HSI) * nc);
	}
	break;
    }
    return (outim);
}


/*- Image_Column_Reflect --------------------------------------------

Return an image that is a reflection about the central column axis
of a given image.

--------------------------------------------------------------------*/

IMAGE  *Image_Column_Reflect(im)
IMAGE  *im;

{
    register int r, c;
    int     nc_half, nc;
    IMAGE  *outim;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Column_Reflect: received invalid image");
	return (NULL);
    }


    nc_half = im->cols / 2;
    nc = im->cols - 1;

    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, im->type))) {
	VIP_Error_Msg("Image_Column_Reflect: out of memory");
	return (NULL);
    }
    Copy_Header(im, outim);

    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		outim->i.c[r][c] = im->i.c[r][nc - c];
		outim->i.c[r][nc - c] = im->i.c[r][c];
	    }
	break;
    case SHORTTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		outim->i.s[r][c] = im->i.s[r][nc - c];
		outim->i.s[r][nc - c] = im->i.s[r][c];
	    }
	break;
    case LONGTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		outim->i.l[r][c] = im->i.l[r][nc - c];
		outim->i.l[r][nc - c] = im->i.l[r][c];
	    }
	break;
    case FLOATTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		outim->i.f[r][c] = im->i.f[r][nc - c];
		outim->i.f[r][nc - c] = im->i.f[r][c];
	    }
	break;
    case DOUBLETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		outim->i.d[r][c] = im->i.d[r][nc - c];
		outim->i.d[r][nc - c] = im->i.d[r][c];
	    }
	break;
    case COMPLEXTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		(void) memcpy(&outim->i.cx[r][c], &im->i.cx[r][nc - c], sizeof(struct struct_vip_COMPLEX));
		memcpy(&outim->i.cx[r][nc - c], &im->i.cx[r][c], sizeof(struct struct_vip_COMPLEX));
	    }
	break;
    case RGBTYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		memcpy(outim->i.rgb[r][c], im->i.rgb[r][nc - c], sizeof(RGB));
		memcpy(outim->i.rgb[r][nc - c], im->i.rgb[r][c], sizeof(RGB));
	    }
	break;
    case HSITYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = nc_half - 1; c >= 0; c--) {
		memcpy(outim->i.hsi[r][c], im->i.hsi[r][nc - c], sizeof(HSI));
		memcpy(outim->i.hsi[r][nc - c], im->i.hsi[r][c], sizeof(HSI));
	    }
	break;
    }
    return (outim);
}


/*-  Find_Spot  -----------------------------------------------------

Function to find the centroid of a blob within a window in an image
returns No of spots found (0 or 1)

--------------------------------------------------------------------*/

int     Find_Spot(image, umin, vmin, umax, vmax, thresh, bw, cu, cv)
IMAGE  *image;
int     umin, vmin, umax, vmax;	/* search area */
int     thresh;			/* threshold   */
int     bw;			/* bw = 1 for white blob on black, = 0 for
				 * black on white */
int    *cu, *cv;		/* coords of blob centre */
{
    int     u, v, found;
    int     sumu = 0, sumv = 0, area = 0;
    char    message[80];

    if (!Image_OK(image)) {
	VIP_Error_Msg("Find_Spot: received invalid image");
	return (ERROR);
    }

    if (bw != 1 AND bw != 0) {
	VIP_Error_Msg("Spot detection bw must be 0 or 1 ");
	return (ERROR);
    }
    if ((umin > umax) OR(vmin > vmax)
	OR Out_Of_Range_i(umin, image->umin, image->umin + image->cols, " ")
	OR Out_Of_Range_i(vmin, image->vmin, image->vmin + image->rows, " ")) {
	VIP_Error_Msg("Find_Spot: Illegal search area parameters");
	return (ERROR);
    }
    switch (image->type) {

    case BYTETYPE:

	for (u = umin; u <= umax; u++) {
	    for (v = vmin; v <= vmax; v++) {
		if ((bw == 1) AND(PIXEL_B(image, u, v) > thresh)) {
		    sumu += u;
		    sumv += v;
		    area++;
		}
		else if ((bw == 0) AND(PIXEL_B(image, u, v) < thresh)) {
		    sumu += u;
		    sumv += v;
		    area++;
		}
	    }
	}
	break;

    default:
	VIP_Error_Msg("Find_Spot: Can only process BYTE images");
	return (0);
    }

    if (area == 0) {
	VIP_Error_Msg("No blob found");
	found = 0;
	*cu = (umin + umax) / 2;/* set blob positition to the centre of */
	*cv = (vmin + vmax) / 2;/* the search area */
    }
    else {
	found = 1;
	*cu = sumu / area;	/* we have got something */
	*cv = sumv / area;
	(void) sprintf(message, "Blob area = %d", area);
	VIP_Error_Msg(message);
    }

#ifdef CRAP
    if (DISPLAY OR DEBUG) {	/* display image and centre of spot */
	fg_setind(255);
	fg_rect(umin + ucorr, vmin + vcorr, umax + ucorr, vmax + vcorr);
	/* outline search area */
	if (found)
	    DrawX(*cu + ucorr, *cv + vcorr, 20);
    }
#endif

    return (found);
}



/*-  Wind_Int   -----------------------------------------------------

Function to calculate average intensity in a window

--------------------------------------------------------------------*/

int     Wind_Int(image, umin, vmin, umax, vmax, avint)
IMAGE  *image;
int     umin, vmin, umax, vmax;	/* search area (in absolute (framegrabber)
				 * coordinates */
int    *avint;			/* average intensity */
{
    int     u, v, usize, vsize;
    float   intsum;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Wind_Int: received invalid image");
	return (ERROR);
    }


    if (umin > umax) {
	u = umin;
	umin = umax;
	umax = u;
    }
    if (vmin > vmax) {
	v = vmin;
	vmin = vmax;
	vmax = v;
    }


    if (umin < image->umin || vmin < image->vmin
	|| umax > (image->cols + image->umin) || vmax > (image->rows + image->vmin)) {
	VIP_Error_Msg("Wind_Int: Window falls outside image bounds");
	return (0);
    }

    usize = umax - umin + 1;
    vsize = vmax - vmin + 1;

    intsum = 0;

    switch (image->type) {

    case BYTETYPE:

	for (u = umin; u <= umax; u++) {
	    for (v = vmin; v <= vmax; v++) {
		intsum += PIXEL_B(image, u, v);
	    }
	}
	break;

    default:
	VIP_Error_Msg("Wind_Int: Only able to process BYTE images");
	return (ERROR);
    }

    *avint = (int) (intsum / usize / vsize);

#ifdef CRAP
    if (DISPLAY OR DEBUG) {	/* display image and centre of spot */
	fg_setind(255);
	fg_rect(umin + ucorr, vmin + vcorr, umax + ucorr, vmax + vcorr);
	/* outline search area */
    }
#endif

    return (OK);
}


/*-  Allocate_Int_Mask  ---------------------------------------------

Function to allocate space for an INT_MASK structure.

--------------------------------------------------------------------*/

INT_MASK *Allocate_Int_Mask(rows, cols)
int     rows, cols;
{

    INT_MASK *mask;
    int     i;

    mask = (INT_MASK *) malloc(sizeof(INT_MASK));
    if (!mask)
	VIP_Error_Msg("Allocate_Int_Mask: Unable to allocate space");
    mask->m = (l_array *) malloc(rows * sizeof(l_array));
    for (i = 0; i < rows; i++) {
	mask->m[i] = (int32 *) malloc(cols * sizeof(long));
	if (!mask->m[i])
	    VIP_Error_Msg("Allocate_Int_Mask: Unable to allocate space");
    }
    mask->idcode = INT_MASK_ID;
/*    mask->version = VERSION;*/
    mask->cols = cols;
    mask->rows = rows;
    return (mask);
}


/*-   Free_Int_Mask  ------------------------------------------------

Function to free memory allocated for an INT_MASK structure.

--------------------------------------------------------------------*/

void    Free_Int_Mask(mask)
INT_MASK *mask;
{
    int     i;

    if (mask == NULL)
	return;			/* already free */

    for (i = 0; i < mask->rows; i++)
	free((char *) mask->m[i]);
    free((char *) mask->m);
    free((char *) mask);
    return;
}


/*-  Read_Int_Mask  ---------------------------------------------------

Function for reading a mask file

Expected file format:
Blank lines can used anywhere.
Comment lines start with a '*'.

Data format is simply 2D array of integer mask values (For morphological
masks a zero value indicates a non active element.
The optional 'rescale' parameter is used  when the integer mask values
have been scaled up to provide extra precision.  After the convolution
has been performed the result can be scaled down by this parameter to
normalise the result.  If the rescale value is omitted it is assumed to be 1.
The mask need not be square but should be rectangular, eg.

* 3 x 3 Sobel mask for vertical edges
rescale 1
-1 0 1
-2 0 2
-1 0 1


----------------------------------------------------------------------*/

INT_MASK *Read_Int_Mask(filename)
char   *filename;
{
    INT_MASK *mask;
    char    line[80], message[80], *identifier, *parameters;
    FILE   *datafile;
    int     m, h, i, j, rows, cols, *dptr, *dptr_top, rescale;
    char    tempstr[10];

    dptr = (int *) malloc(1024 * sizeof(int));
    dptr_top = dptr;

    datafile = fopen(filename, "r");

    if (datafile == NULL) {
	(void) sprintf(message, "Read_Int_Mask: Could not open mask file %s ", filename);
	VIP_Error_Msg(message);
	return (NULL);
    }
    rescale = 1;		/* default value  */
    rows = cols = 0;

    while (fgets(line, 80, datafile) != NULL) {
	identifier = (char *) leftjust(line);
	parameters = (char *) strchr(line, ' ');	/* stuff after the first
							 * blank */

	if (strlen(identifier) == 0 ||
	    strncmp(identifier, "*", 1) == 0 ||
	    strncmp(identifier, "\n", 1) == 0) {
	    ;			/* Blank line or comment */

	}
	else if (strncasecmp(identifier, "rescale", 7) == 0) {
	    (void) sscanf(parameters, "%d", &rescale);

	}
	else {			/* read in the mask */
	    i = 0;

	    for (h = 0, m = 0, tempstr[0] = '\0';;) {
		tempstr[h] = (char) identifier[m];
		m++;

		if (tempstr[h] == '\n') {
		    tempstr[h] = '\0';
		    if ((strlen(tempstr) > 0)) {
			*dptr = atoi(tempstr);
			dptr++;
			i++;
		    }
		    tempstr[0] = '\0';
		    h = 0;
		    break;
		}

		if (tempstr[h] == ' ') {
		    tempstr[h] = '\0';
		    if ((strlen(tempstr) > 0)) {
			*dptr = atoi(tempstr);
			dptr++;
			i++;
		    }
		    tempstr[0] = '\0';
		    h = 0;
		}
		else {
		    h++;
		}
	    }

	    rows++;
	    cols = i;
	    if (rows * cols > 1024) {
		VIP_Error_Msg("Read_Int_Mask: File too large!");
		return (NULL);
	    }
	}

    }				/* while not EOF */

    dptr = dptr_top;

    mask = Allocate_Int_Mask(rows, cols);
    mask->rescale = rescale;

    for (i = 0; i < rows; i++) {
	for (j = 0; j < cols; j++) {
	    mask->m[i][j] = *dptr;
	    dptr++;
	}
    }
    free(dptr_top);
    (void) fclose(datafile);
    return (mask);
}


/*- Write_Int_Mask --------------------------------------------------

Function to write a INT_MASK structure to a file.

--------------------------------------------------------------------*/

int     Write_Int_Mask(mask, filename, comment)
INT_MASK *mask;
char   *filename, *comment;
{
    FILE   *maskfile;
    int     i, j;
    char    message[80];

    maskfile = fopen(filename, "w");
    if (!maskfile) {
	(void) sprintf(message, "Write_Int_Mask: Unable to open %s", filename);
	VIP_Error_Msg(message);
	return (ERROR);
    }
    (void) fprintf(maskfile, "* %s\n", comment);
    (void) fprintf(maskfile, "rescale %d\n", mask->rescale);
    for (i = 0; i < mask->rows; i++) {
	for (j = 0; j < mask->cols; j++)
	    (void) fprintf(maskfile, "%5d ", mask->m[i][j]);
	(void) fprintf(maskfile, "\n");
    }
    (void) fclose(maskfile);
    return (OK);
}


/*- Erode_Image -----------------------------------------------------

Function to perform morphological erosion.  An INT_MASK structure is
used to specify the structuring element.  Non-zero values indicate
active elements.

--------------------------------------------------------------------*/

IMAGE  *Erode_Image(inimage, mask)
IMAGE  *inimage;
INT_MASK *mask;
{
    IMAGE  *outimage;
    int     h, w, i, j, k, l, rowoffset, coloffset, min, value;
    int     rows, cols;


    if (!Image_OK(inimage)) {
	VIP_Error_Msg("Erode_Image: received invalid image");
	return (NULL);
    }


    h = inimage->rows;
    w = inimage->cols;

    rowoffset = mask->rows / 2;
    coloffset = mask->cols / 2;
    rows = h - mask->rows;
    cols = w - mask->cols;

    outimage = (IMAGE *) Allocate_Image(inimage->umin, inimage->vmin,
	inimage->rows, inimage->cols, inimage->type);

    switch (inimage->type) {

    case BYTETYPE:

	for (i = 0; i < rows; i++) {	/* for each row in image */
	    for (j = 0; j < cols; j++) {	/* for each column */
		min = 255;
		for (k = 0; k < mask->rows; k++) {
		    for (l = 0; l < mask->cols; l++) {
			if (mask->m[k][l]) {
			    value = inimage->i.c[i + k][j + l];
			    if (value < min)
				min = value;
			}
		    }
		}
		outimage->i.c[i + rowoffset][j + coloffset] = min;
	    }
	}
	break;

    default:
	VIP_Error_Msg("erode: Only able to process BYTE images");
	return (NULL);
    }
    return (outimage);
}


/*- Dilate_Image ----------------------------------------------------

Function to perform morphological dilation using a mask specified in an
INT_MASK structure.

--------------------------------------------------------------------*/

IMAGE  *Dilate_Image(inimage, mask)
IMAGE  *inimage;
INT_MASK *mask;
{
    IMAGE  *outimage;
    int     h, w, i, j, k, l, rowoffset, coloffset, max, value, rows, cols;

    if (!Image_OK(inimage)) {
	VIP_Error_Msg("Dilate_Image: received invalid image");
	return (NULL);
    }


    h = inimage->rows;
    w = inimage->cols;

    rowoffset = mask->rows / 2;
    coloffset = mask->cols / 2;
    rows = h - mask->rows;
    cols = w - mask->cols;

    outimage = (IMAGE *) Allocate_Image(inimage->umin, inimage->vmin,
	inimage->rows, inimage->cols, inimage->type);

    switch (inimage->type) {

    case BYTETYPE:

	for (i = 0; i < rows; i++) {	/* for each row in image */
	    for (j = 0; j < cols; j++) {	/* for each column */

		max = 0;
		for (k = 0; k < mask->rows; k++) {
		    for (l = 0; l < mask->cols; l++) {
			if (mask->m[k][l]) {
			    value = inimage->i.c[i + k][j + l];
			    if (value > max)
				max = value;
			}
		    }
		}
		outimage->i.c[i + rowoffset][j + coloffset] = max;
	    }
	}
	break;

    default:
	VIP_Error_Msg("dilate: Only able to process BYTE images");
	return (NULL);
    }
    return (outimage);
}


/* Function for the Local Energy Model ************************************** */


/*- Read_Energy_Filter ----------------------------------------------

Read from disk an odd and even filters for the Local Energey feature
detector.  The function returns 1 for successful read, 0 otherwise.

--------------------------------------------------------------------*/

int     Read_Energy_Filter(filename, Hmask, Fmask, mask_size)
char   *filename;
float **Hmask, **Fmask;
int    *mask_size;
{
    register int i;
    FILE   *fp;

    if (!(fp = fopen(filename, "r"))) {
	VIP_Error_Msg("Read_Energy_Filter: Error! could not open file");
	return (0);
    }

    if (fscanf(fp, "%d\n", mask_size) != 1) {
	VIP_Error_Msg("Read_Energy_Filter: Error! short read");
	(void) fclose(fp);
	return (0);
    }

    *Hmask = (float *) Allocate_FArray(*mask_size);
    *Fmask = (float *) Allocate_FArray(*mask_size);
    if (!(*Hmask) || !(*Fmask)) {
	VIP_Error_Msg("Read_Energy_Filter: out of memory");
	(void) fclose(fp);
	return (0);
    }

    /* Read even filter F from disk */
    for (i = 0; i < *mask_size; i++)
	if (fscanf(fp, " %f", &(*Fmask)[i]) != 1) {
	    VIP_Error_Msg("Read_Energy_Filter: short read");
	    (void) fclose(fp);
	    return (0);
	}

    /* Read odd filter H from disk */
    for (i = 0; i < *mask_size; i++)
	if (fscanf(fp, " %f", &(*Hmask)[i]) != 1) {
	    VIP_Error_Msg("Read_Energy_Filter: short read");
	    (void) fclose(fp);
	    return (0);
	}
    (void) fclose(fp);
    return (1);
}


/*- Apply_Local_Energy ----------------------------------------------

Apply the Local Energy feature detector to an image.  The process
involves convolving the given image with an odd and an even masks
of a given size.  The argument 'threshold' is for the non-maxima
suppression process.

The function returns the convolution output.

--------------------------------------------------------------------*/

IMAGE  *Apply_Local_Energy(im, Hmask, Fmask, mask_size, threshold)
IMAGE  *im;
float  *Hmask, *Fmask;
int     mask_size;
float   threshold;
{
    float  *F, *H, *E;
    int     i, r, c, rad;
    int     norow, nocol, siz;
    IMAGE  *edge;

    if (!im || !Hmask || !Fmask) {
	VIP_Error_Msg("Apply_Local_Energy: recieved invalid image or mask");
	return (NULL);
    }

    norow = im->rows;
    nocol = im->cols;
    siz = (norow > nocol ? norow : nocol);
    F = (float *) Allocate_FArray(siz);
    H = (float *) Allocate_FArray(siz);
    E = (float *) Allocate_FArray(siz);
    siz *= sizeof(float);

    if (!E || !F || !H) {
	VIP_Error_Msg("Apply_Local_Energy: out of memory");
	return (NULL);
    }

    if (!(edge = (IMAGE *) Allocate_Image(0, 0, norow, nocol, FLOATTYPE))) {
	VIP_Error_Msg("Apply_Local_Energy: out of memory");
	return (NULL);
    }
    rad = mask_size / 2;
    /* Perform horizontal convolution */
    if (VIP_STD_ERR)
	(void) fprintf(stderr, "horizontal convolution, row number:     ");
    for (r = rad; r < norow - rad; r++) {	/* Process each line */
	if (VIP_STD_ERR)
	    (void) fprintf(stderr, "\b\b\b%3d", r);

	(void) memset(F, 0, siz);
	(void) memset(H, 0, siz);
	(void) memset(E, 0, siz);

	/* Do convolution */
	for (c = rad; c < nocol - rad; c++) {
	    /* Convolve even mask for column domain */
	    for (i = 0; i < mask_size; i++)
		F[c] += im->i.c[r][c - rad + i] * Fmask[i];

	    /* Square even mask for column domain */
	    E[c] = F[c] * F[c];

	    /* Convolve odd mask for column domain */
	    for (i = 0; i < mask_size; i++)
		H[c] += im->i.c[r][c - rad + i] * Hmask[i];

	    /* Square odd mask for column domain */
	    E[c] += H[c] * H[c];
	}

	/* Find local maxima in energy function */
	for (c = rad + 2; c < nocol - rad - 1; c++) {
	    if ((E[c] - E[c - 1] > threshold) &&
		(E[c] - E[c + 1] > threshold))
		edge->i.f[r][c] = (E[c] - E[c - 1] +
		    E[c] - E[c + 1]) / 2.0;
	    else if ((E[c] == E[c + 1]) &&
		    (E[c] - E[c - 1] > threshold) &&
		(E[c + 1] - E[c + 2] > threshold))
		edge->i.f[r][c] = (E[c] - E[c - 1] +
		    E[c + 1] - E[c + 2]) / 2.0;
	    if ((fabs(E[c] - E[c + 1]) < .1) &&
		(E[c] - E[c - 1] > threshold) &&
		(fabs(E[c + 1] - E[c + 2]) < .1))
		edge->i.f[r][c] = (E[c] - E[c - 1] +
		    E[c] - E[c + 1]) / 2.0;
	    if ((fabs(E[c] - E[c - 1]) < .1) &&
		(E[c] - E[c + 1] > threshold) &&
		(fabs(E[c - 1] - E[c - 2]) < .1))
		edge->i.f[r][c] = (E[c] - E[c - 1] +
		    E[c] - E[c + 1]) / 2.0;
	}
    }


    /* Perform vertical convolution */
    if (VIP_STD_ERR)
	(void) fprintf(stderr, "\nvertical convolution, col number:     ");
    for (c = rad; c < nocol - rad; c++) {	/* Process each column */
	if (VIP_STD_ERR)
	    (void) fprintf(stderr, "\b\b\b%3d", c);

	(void) memset(F, 0, siz);
	(void) memset(H, 0, siz);
	(void) memset(E, 0, siz);

	/* Do convolution */
	for (r = rad; r < norow - rad; r++) {
	    /* Convolve even mask for row domain */
	    for (i = 0; i < mask_size; i++)
		F[r] += im->i.c[r - rad + i][c] * Fmask[i];

	    /* Square even mask for column domain */
	    E[r] = F[r] * F[r];

	    /* Convolve odd mask for column domain */
	    for (i = 0; i < mask_size; i++)	/* Do convolution */
		H[r] += im->i.c[r - rad + i][c] * Hmask[i];

	    /* Square odd mask for column domain */
	    E[r] += H[r] * H[r];
	}

	/* Find local maxima in energy function */
	for (r = rad + 2; r < norow - rad - 1; r++) {
	    if ((E[r] - E[r - 1] > threshold) &&
		(E[r] - E[r + 1] > threshold))
		edge->i.f[r][c] += (E[r] - E[r - 1] +
		    E[r] - E[r + 1]) / 2.0;
	    else if ((E[r] == E[r + 1]) &&
		    (E[r] - E[r - 1] > threshold) &&
		(E[r + 1] - E[r + 2] > threshold))
		edge->i.f[r][c] += (E[r] - E[r - 1] +
		    E[r + 1] - E[r + 2]) / 2.0;
	    if ((fabs(E[r] - E[r + 1]) < .1) &&
		(E[r] - E[r - 1] > threshold) &&
		(fabs(E[r + 1] - E[r + 2]) < .1))
		edge->i.f[r][c] += (E[r] - E[r - 1] +
		    E[r] - E[r + 1]) / 2.0;
	    if ((fabs(E[r] - E[r - 1]) < .1) &&
		(E[r] - E[r + 1] > threshold) &&
		(fabs(E[r - 1] - E[r - 2]) < .1))
		edge->i.f[r][c] += (E[r] - E[r - 1] +
		    E[r] - E[r + 1]) / 2.0;

	}
    }
    return (edge);
}



/*- Average_Image ---------------------------------------------------

Function to take average of two images.

--------------------------------------------------------------------*/

IMAGE  *Average_Image(image1, image2)
IMAGE  *image1, *image2;
{
    IMAGE  *outimage;
    int     umin, vmin, rows, cols, i, j;

    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Average_Image: received invalid image");
	return (NULL);
    }



    if (image1->type != image2->type) {
	VIP_Error_Msg("average: Images of different types");
	return (NULL);
    }
    if (!Common_Window(image1, image2, &umin, &vmin, &rows, &cols)) {
	VIP_Error_Msg("average: No common area between images");
	return (NULL);
    }
    if (!(outimage = (IMAGE *) Allocate_Image(umin, vmin, rows, cols, image1->type))) {
	VIP_Error_Msg("average: Could not allocate output image");
	return (NULL);
    }
    switch (image1->type) {

    case BYTETYPE:

	for (i = 0; i < rows; i++) {	/* for each row in image */
	    for (j = 0; j < cols; j++) {	/* for each column */
		outimage->i.c[i][j] =
		    (get_pixel_B(image1, i + umin, j + vmin)
		    + get_pixel_B(image2, i + umin, j + vmin)) / 2;
	    }
	}
	break;

    default:
	VIP_Error_Msg("average: Can only process BYTE images");
	return (NULL);

    }

    return (outimage);
}

int     get_pixel_B(img, um, vm)
IMAGE  *img;
int     um;
int     vm;
{
    return (img->i.c[(um) - img->umin][(vm) - img->vmin]);
}

/*- Rotate_Float_Mask ------------------------------------------

Function to generate rotated versions of a reference mask

input: inmask - reference mask
       angle  - rotation angle in radians

Returns rotated version of inmask.

---------------------------------------------------------------*/

FLOAT_MASK *Rotate_Float_Mask(FLOAT_MASK * inmask, double angle)
{
    FLOAT_MASK *mask, *tmask, *outmask;
    int     i, j, size, mid, firstrow, firstcol;
    float   rot[2][2];
    int     found;
    double  x, y, g[4];
    int     xmin, xmax, ymin, ymax;
    double  hfrac, vfrac, upperavg, loweravg;


    if (!inmask) {
	VIP_Error_Msg("Rotate_Float_Mask: recieved NULL mask");
	return (NULL);
    }

    if (!(inmask->rows & 1) || !(inmask->cols & 1)) {
	VIP_Error_Msg("Rotate_Float_Mask: mask being rotated must be of odd size");
	return (NULL);
    }

/* calculate initial size of output mask as sqrt(2) * max dimension */


    size = sqrt(2.0) * (float) (MAX(inmask->rows, inmask->cols)) + 0.5;
    if (!(size & 1))
	size++;			/* ensure size is odd */

    mid = (size - 1) / 2;	/* co-ord of middle of mask */

    mask = (FLOAT_MASK *) Allocate_Float_Mask(size, size);
    tmask = (FLOAT_MASK *) Allocate_Float_Mask(size, size);

    for (i = 0; i < size; i++)
	for (j = 0; j < size; j++)
	    mask->m[i][j] = tmask->m[i][j] = 0.0;

    for (i = 0; i < inmask->rows; i++)
	for (j = 0; j < inmask->cols; j++) {
	    tmask->m[i + (size - inmask->rows) / 2][j + (size - inmask->cols) / 2] = inmask->m[i][j];

	}

/* set up rotation matrix */

    rot[0][0] = rot[1][1] = cos(angle);
    rot[0][1] = sin(angle);
    rot[1][0] = -sin(angle);

    for (i = 0; i < size; i++) {
	for (j = 0; j < size; j++) {

/* calculate location in source mask that corresponds to each position in
rotated mask */

	    x = rot[0][0] * (float) (i - mid) + rot[0][1] * (float) (j - mid)
		+ (float) mid;
	    y = rot[1][0] * (float) (i - mid) + rot[1][1] * (float) (j - mid)
		+ (float) mid;

/* find integer locations around desired 'come from' location */

	    xmin = floor(x);
	    xmax = xmin + 1;
	    ymin = floor(y);
	    ymax = ymin + 1;

	    g[0] = g[1] = g[2] = g[3] = 0.0;

	    if (xmin >= 0 && xmin < size && ymin >= 0 && ymin < size)
		g[0] = tmask->m[xmin][ymin];

	    if (xmax >= 0 && xmax < size && ymin >= 0 && ymin < size)
		g[1] = tmask->m[xmax][ymin];

	    if (xmin >= 0 && xmin < size && ymax >= 0 && ymax < size)
		g[2] = tmask->m[xmin][ymax];

	    if (xmax >= 0 && xmax < size && ymax >= 0 && ymax < size)
		g[3] = tmask->m[xmax][ymax];

	    hfrac = x - (double) xmin;
	    vfrac = y - (double) ymin;

	    upperavg = g[0] + hfrac * (g[1] - g[0]);
	    loweravg = g[2] + hfrac * (g[3] - g[2]);

	    mask->m[i][j] = upperavg + vfrac * (loweravg - upperavg);


	}
    }

/* Remove rows and columns of mask that have only zero values,
   it is assumed the mask is symmetric */

    firstcol = firstrow = 0;

    found = FALSE;
    for (i = 0; (i < size) && !found; i++)
	for (j = 0; j < size; j++)
	    if (mask->m[i][j] != 0.0) {
		firstrow = i;
		found = TRUE;
		break;
	    }

    found = FALSE;
    for (j = 0; (j < size) && !found; j++)
	for (i = 0; i < size; i++)
	    if (mask->m[i][j] != 0.0) {
		firstcol = j;
		found = TRUE;
		break;
	    }


    outmask = (FLOAT_MASK *) Allocate_Float_Mask(size - 2 * firstrow, size - 2 * firstcol);

    for (i = 0; i < outmask->rows; i++)
	for (j = 0; j < outmask->cols; j++)
	    outmask->m[i][j] = mask->m[firstrow + i][firstcol + j];

    Free_Float_Mask(mask);
    Free_Float_Mask(tmask);
    return (outmask);

}

/*- Rotate_Image ------------------------------------------

Function to rotate an image.  Bilinear interpolation is used
to calculate the grey values.

input: inimage - input image
       angle  - rotation angle in radians

Returns rotated version of image

---------------------------------------------------------------*/

IMAGE  *Rotate_Image(IMAGE * inimage, double angle)
{
    IMAGE  *image;
    int     row, col, midrow, midcol;
    float   rot[2][2];
    double  x, y;


    if (!Image_OK(inimage)) {
	VIP_Error_Msg("Rotate_Image: received invalid image");
	return (NULL);
    }


    midrow = inimage->rows / 2;
    midcol = inimage->cols / 2;

    if (inimage->type != BYTETYPE &&
	inimage->type != SHORTTYPE &&
	inimage->type != LONGTYPE &&
	inimage->type != FLOATTYPE) {
	VIP_Error_Msg("Rotate_Image: can only operate on BYTE, SHORT, LONG and FLOAT images");
	return (NULL);
    }
    image = (IMAGE *) Allocate_Image(0, 0, inimage->rows, inimage->cols, inimage->type);

/* set up rotation matrix */

    rot[0][0] = rot[1][1] = cos(angle);
    rot[0][1] = sin(angle);
    rot[1][0] = -sin(angle);

    for (row = 0; row < inimage->rows; row++) {
	for (col = 0; col < inimage->cols; col++) {

/* calculate location in source image that corresponds to each position in the
rotated image */

	    x = rot[0][0] * (float) (col - midcol) +
		rot[0][1] * (float) (row - midrow) + (float) midcol;
	    y = rot[1][0] * (float) (col - midcol) +
		rot[1][1] * (float) (row - midrow) + (float) midrow;

	    switch (inimage->type) {
	    case BYTETYPE:
		image->i.c[row][col] = (int) Bilinear_Interpolate(inimage, x, y);
		break;

	    case SHORTTYPE:
		image->i.s[row][col] = (short) Bilinear_Interpolate(inimage, x, y);
		break;

	    case LONGTYPE:
		image->i.l[row][col] = (long) Bilinear_Interpolate(inimage, x, y);
		break;

	    case FLOATTYPE:
		image->i.f[row][col] = (float) Bilinear_Interpolate(inimage, x, y);
		break;
	    }
	}
    }
    return (image);

    return (image);
}


/*- XYScale_Image  --------------------------------------------------

Function to scale an image in x and y directions.

------------------------------------------------------------------*/

IMAGE  *XYScale_Image(IMAGE * image, float xscale, float yscale)
{
    IMAGE  *outimage;
    double  x, y;
    int     row, col, rows, cols;

    if (!Image_OK(image)) {
	VIP_Error_Msg("XYScale_Image: received invalid image");
	return (NULL);
    }


    rows = image->rows * yscale;
    cols = image->cols * xscale;

    if (image->type != BYTETYPE &&
	image->type != SHORTTYPE &&
	image->type != LONGTYPE &&
	image->type != FLOATTYPE) {
	VIP_Error_Msg("Scale_Image: can only operate on BYTE, SHORT, LONG and FLOAT images");
	return (NULL);
    }
    outimage = (IMAGE *) Allocate_Image(0, 0, rows, cols, image->type);

    if (!outimage) {
	VIP_Error_Msg("Scale_Image unable to allocate memory");
	return (NULL);
    }

    for (row = 0; row < rows; row++) {
	for (col = 0; col < cols; col++) {
	    x = (float) col / xscale;
	    y = (float) row / yscale;

	    switch (image->type) {
	    case BYTETYPE:
		outimage->i.c[row][col] = (int) Bilinear_Interpolate(image, x, y);
		break;

	    case SHORTTYPE:
		outimage->i.s[row][col] = (short) Bilinear_Interpolate(image, x, y);
		break;

	    case LONGTYPE:
		outimage->i.l[row][col] = (long) Bilinear_Interpolate(image, x, y);
		break;

	    case FLOATTYPE:
		outimage->i.f[row][col] = (float) Bilinear_Interpolate(image, x, y);
		break;
	    }
	}

    }

    return (outimage);
}


/*- Scale_Float_Mask --------------------------------------------------

Function to scale values in a FLOAT_MASK

----------------------------------------------------------------------*/

void    Scale_Float_Mask(FLOAT_MASK * mask, float scale)
{
    int     i, j;

    if (!mask) {
	VIP_Error_Msg("Scale_Float_Mask: received NULL mask");
	return;
    }

    for (i = 0; i < mask->rows; i++)
	for (j = 0; j < mask->cols; j++)
	    mask->m[i][j] *= scale;

    return;
}




/*- Float_Mask2Image ------------------------------------------------


Copy a FLOAT_MASK to a FLOAT image of size rows x cols so that one can
use its FFT for convolution with an image.

If shift is 0 the mask is placed around the 'corners' of the image
in the default FFT data arrangement.

If shift is 1 the mask is centred in the image.

--------------------------------------------------------------------*/

IMAGE  *Float_Mask2Image(FLOAT_MASK * mask, int rows, int cols, int shift)
{
    int     mrow, mcol, row, col, rowoffset, coloffset;
    IMAGE  *image;

    if (!mask) {
	VIP_Error_Msg("Float_Mask2Image: received NULL mask");
	return (NULL);
    }


    if ((mask->rows > rows) || (mask->cols > cols)) {
	VIP_Error_Msg("Mask2Image: Mask is larger than desired image\n");
	return (NULL);
    }

    image = Allocate_Image(0, 0, rows, cols, FLOATTYPE);
    if (!image)
	return (NULL);

    Set_Image(image, 0.0);

    if (shift == 0) {		/* spread mask around the corners in default
				 * FFT form */

/* bottom right hand quadrant of mask */

	for (mrow = mask->rows / 2, row = 0; mrow < mask->rows; mrow++, row++)
	    for (mcol = mask->cols / 2, col = 0; mcol < mask->cols; mcol++, col++)
		image->i.f[row][col] = mask->m[mrow][mcol];

/* top left  quadrant of mask */

	for (mrow = 0, row = image->rows - mask->rows / 2; mrow < mask->rows / 2; mrow++, row++)
	    for (mcol = 0, col = image->cols - mask->cols / 2; mcol < mask->cols / 2; mcol++, col++)
		image->i.f[row][col] = mask->m[mrow][mcol];

/* top right quadrant of mask */

	for (mrow = 0, row = image->rows - mask->rows / 2; mrow < mask->rows / 2; mrow++, row++)
	    for (mcol = mask->cols / 2, col = 0; mcol < mask->cols; mcol++, col++)
		image->i.f[row][col] = mask->m[mrow][mcol];

/* bottom left  quadrant of mask */

	for (mrow = mask->rows / 2, row = 0; mrow < mask->rows; mrow++, row++)
	    for (mcol = 0, col = image->cols - mask->cols / 2; mcol < mask->cols / 2; mcol++, col++)
		image->i.f[row][col] = mask->m[mrow][mcol];

    }
    else if (shift == 1) {	/* place the mask in the centre */

	rowoffset = image->rows / 2 - mask->rows / 2;
	coloffset = image->cols / 2 - mask->cols / 2;

	for (row = 0; row < mask->rows; row++)
	    for (col = 0; col < mask->cols; col++)
		image->i.f[rowoffset + row][coloffset + col] = mask->m[row][col];

    }
    else {			/* error */
	VIP_Error_Msg("Mask2Image: Illegal shift value");
	Free_Image(image);
	return (NULL);
    }

    return (image);
}


/*- Instantaneous_Freq  ----------------------------------------

Function to calculate an estimate of the instantaneous frequency
at a point in a data array using the continuous wavelet transform.

inputs:  A   - array of data points.
         x   - point of interest in the array.
         sinmask - array of sinusoidal Morlet wavelets.
         cosmask - array of cosinusoidal Morlet wavelets.
         nfreq   - No of frequencies/wavelets to analyse.
         minfreq } - frequencies corresponding to first and last
         maxfreq }   wavelets in wavelet array

Function returns weighted average frequency.

*Function used to return frequency corresponding to wavelet mask that
responded most strongly.

-----------------------------------------------------------------*/

float   Instantaneous_Freq(float A[], int x, FLOAT_MASK * sinmask[], FLOAT_MASK * cosmask[], int nfreq
           ,int Noct, float freq[])
{
    int     f, i, cols, offset;
    int     index = 0;
    float   sinresult, cosresult, result, maxresult, instfreq;
    static  count = 0;
    float   average, mags;

    maxresult = 0;
    average = mags = 0.0;

    for (f = 0; f < nfreq; f++) {
	sinresult = cosresult = 0.0;

	cols = sinmask[f]->cols;/* assume sin and cos masks the same */
	offset = x - cols / 2;	/* where to start convolving from */
	if (offset < 0)
	    offset = 0;

/* can't test for too large offset - don't know how many elements */

	for (i = 0; i < cols; i++) {
	    sinresult += A[offset + i] * sinmask[f]->m[0][i];
	    cosresult += A[offset + i] * cosmask[f]->m[0][i];
	}
	freq[f] = result = SQR(sinresult) + SQR(cosresult);

	mags += result;
	average += result * (float) f;

	if (result > maxresult) {
	    maxresult = result;
	    index = f;
	}
    }

    if (mags > 0.1) {
	index = (int) (average / mags);
	instfreq = 1.0 / pow(2.0, (double) index / (double) Noct + 1.0);
    }
    else {
	instfreq = 1.0;
    }
    if (VIP_VERBOSE) {
	printf("%5d %8.3f %8.3f %4d %4d\n", index, maxresult, instfreq, Noct, nfreq);
	count++;
    }
    return (instfreq);
}


/*-  Correlate  -------------------------------------------

Function returns the offset that maximises the correlation
between two data arrays.

inputs: A - data array of floats
        B - data array of floats
        n - No of data values

returns offset of B relative to A that maximizes correlation.

----------------------------------------------------------*/

int     Correlate(float A[], float B[], int n)
{
    int     i, offset, bestoffset = 0;
    double  result, maxresult;

    maxresult = 0.0;

/* search in the range +/- half array length */

    for (offset = -n / 2; offset < n / 2; offset++) {
	result = 0.0;
	for (i = 0; i < n; i++) {
	    if ((i >= 0) && (i < n) && ((i - offset) >= 0) && ((i - offset) < n))
		result += A[i] * B[i - offset];
	}
	if (result > maxresult) {
	    maxresult = result;
	    bestoffset = offset;
	}
    }

/*printf("%d %lf\n",bestoffset,maxresult); */

    return (bestoffset);
}

/*- Log_Image ------------------------------------------------------

Function to take the natural log of an image.  Input images are
converted to FLOATTYPE, the natural log taken, and a FLOATTYPE
image is returned.  Note that 1 is added to each pixel value to
avoid taking the log of 0.

-----------------------------------------------------------------*/

IMAGE  *Log_Image(IMAGE * image)
{
    int     row, col;
    IMAGE  *fimage = NULL, *outimage = NULL;


    if (!Image_OK(image)) {
	VIP_Error_Msg("Log_Image: received invalid image");
	return (NULL);
    }

    if (image->type != FLOATTYPE)
	fimage = (IMAGE *) Convert2Float_Image(image);
    else
	fimage = image;


    outimage = (IMAGE *) Allocate_Image(0, 0, image->rows, image->cols, FLOATTYPE);

    for (row = 0; row < image->rows; row++)
	for (col = 0; col < image->cols; col++)
	    outimage->i.f[row][col] = log((double) (fimage->i.f[row][col] + 1.0));


    Free_Image(fimage);

    return (outimage);
}

/*- Exp_Image ------------------------------------------------------

Function to take the natural exponent of an image.  Input images are
converted to FLOATTYPE, the natural exponent taken, and a FLOATTYPE
image is returned.

-----------------------------------------------------------------*/

IMAGE  *Exp_Image(IMAGE * image)
{
    int     row, col;
    IMAGE  *fimage, *outimage;
    int     FREE = FALSE;


    if (!Image_OK(image)) {
	VIP_Error_Msg("Exp_Image: received invalid image");
	return (NULL);
    }

    if (image->type != FLOATTYPE) {
	fimage = (IMAGE *) Convert2Float_Image(image);
	FREE = TRUE;
    }
    else {
	fimage = image;
	FREE = FALSE;
    }

    outimage = (IMAGE *) Allocate_Image(0, 0, image->rows, image->cols, FLOATTYPE);

    for (row = 0; row < image->rows; row++)
	for (col = 0; col < image->cols; col++)
	    outimage->i.f[row][col] = exp((double) fimage->i.f[row][col]);

    if (FREE)
	Free_Image(fimage);

    return (outimage);
}

/*-----------------------------------------------------------------*/

int     closestpower(int x)
{
    int     y = 1;

    while (y < x)
	y *= 2;

    return (y);
}

/*-----------------------------------------------------------------*/

int     Closest_Power_Of_2(int x)
{
    int     y = 1;

    while (y < x)		/* Keep doubling until we are >= x */
	y = y << 1;

    if ((y - x) < (x - (y / 2)))/* Find closest power of 2 */
	return (y);
    else
	return (y / 2);
}

/*-----------------------------------------------------------------*/

int     Power_Of_2_Greater_Or_Equal(int x)
{
    int     y = 1;

    while (y < x)
	y *= 2;

    return (y);
}

/*- Image_Line ----------------------------------------------------

Function to draw a line in an image

------------------------------------------------------------------*/


void    Image_Line(IMAGE * image, int us, int vs, int ue, int ve, int grey)
{
    int     i, u, v, tmp, steps;
    double  fu, fv, du, dv;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Image_Line: received invalid image");
	return;
    }


    if (image->type != BYTETYPE) {
	VIP_Error_Msg("Image_Line: Image must be BYTETYPE");
	return;
    }


/* make sure points are ordered conveniently */

    if (ue < us) {		/* swap the points */
	tmp = us;
	us = ue;
	ue = tmp;
	tmp = vs;
	vs = ve;
	ve = tmp;
    }


    steps = MAX(abs(ve - vs), abs(ue - us));
    if (steps == 0)
	steps = 1;

    du = (double) (ue - us) / (double) steps;
    dv = (double) (ve - vs) / (double) steps;

    fu = us;
    fv = vs;
    u = (int) fu;
    v = (int) fv;
    if (u >= 0 && u < image->cols && v >= 0 && v < image->rows)
	image->i.c[v][u] = grey;

    for (i = 1; i <= steps; i++) {
	fu += du;
	fv += dv;
	u = (int) fu;
	v = (int) fv;
	if (u >= 0 && u < image->cols && v >= 0 && v < image->rows)
	    image->i.c[v][u] = grey;

    }

    return;
}

/*- Image_Point ----------------------------------------------------

Function to draw a point in an image

------------------------------------------------------------------*/

void    Image_Point(IMAGE * image, int u, int v, int size, int grey)
{
    int     i, j, halfsize;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Image_Point: received invalid image");
	return;
    }

    if (image->type != BYTETYPE) {
	VIP_Error_Msg("Image_Point: Image must be BYTETYPE");
	return;
    }

    halfsize = size / 2;

    for (i = u - halfsize; i <= u + halfsize; i++) {
	for (j = v - halfsize; j <= v + halfsize; j++) {

	    if (i >= 0 && i < image->cols && j >= 0 && j < image->rows)
		image->i.c[j][i] = grey;

	}
    }
    return;
}

/*- Point_Convolve -----------------------------------------

Function returns the result of convolving a FLOAT MASK with an
image at a specified point.

--------------------------------------------------------------*/

float   Point_Convolve(IMAGE * image, int uc, int vc, FLOAT_MASK * mask)
{
    register int i;
    register unsigned char *ip;
    register float *mp;
    register float result;

    int     umin, umax, vmin, vmax, v, mv;
    int     mskumin, mskvmin;
    int     steps;


    if (!Image_OK(image)) {
	VIP_Error_Msg("Point_Convolve: received invalid image");
	return (0.0);
    }


/* set up limits to cope with bounds of image */

    umin = MAX(0, uc - (mask->cols / 2));
    umax = MIN((image->cols - 1), uc + (mask->cols / 2));
    vmin = MAX(0, vc - (mask->rows / 2));
    vmax = MIN((image->rows - 1), vc + (mask->rows / 2));
    mskumin = umin - (uc - (mask->cols / 2));
    mskvmin = vmin - (vc - (mask->rows / 2));
    steps = umax - umin + 1;

    result = 0.0;

    for (v = vmin, mv = mskvmin; v <= vmax; v++, mv++) {

	ip = &image->i.c[v][umin];
	mp = &mask->m[mv][mskumin];
	for (i = steps; i; i--)
	    result += *ip++ * *mp++;

/*
  for( u = umin, mu = mskumin; u <= umax; u++, mu++)  {
     result += image->i.c[v][u]*mask->m[mv][mu];
}
*/
    }
    return (result);
}


/*- Accumulate_Add_Image --------------------------------------------

Function to perform an accumulation addition of an image to another.
Elements in image Ac are incremented by the corresponding values in
image.  There are no checks for numerical overflow.

-------------------------------------------------------------------*/

void    Accumulate_Add_Image(IMAGE * Ac, IMAGE * image)
{
    register int row, col;

    if (!Image_OK(Ac) || !Image_OK(image)) {
	VIP_Error_Msg("Accumulate_Add_Image: received invalid image");
	return;
    }

    if (Ac->rows != image->rows ||
	Ac->cols != image->cols) {
	VIP_Error_Msg("Accumulate_Add_Image: the images are of different sizes");
	return;
    }

    if (Ac->type != image->type) {
	VIP_Error_Msg("Accumulate_Add_Image: the images are of different types");
	return;
    }

    switch (Ac->type) {

    case BYTETYPE:
	for (row = 0; row < Ac->rows; row++)
	    for (col = 0; col < Ac->cols; col++)
		Ac->i.c[row][col] += image->i.c[row][col];
	break;

    case SHORTTYPE:
	for (row = 0; row < Ac->rows; row++)
	    for (col = 0; col < Ac->cols; col++)
		Ac->i.s[row][col] += image->i.s[row][col];
	break;

    case LONGTYPE:
	for (row = 0; row < Ac->rows; row++)
	    for (col = 0; col < Ac->cols; col++)
		Ac->i.l[row][col] += image->i.l[row][col];
	break;

    case FLOATTYPE:
	for (row = 0; row < Ac->rows; row++)
	    for (col = 0; col < Ac->cols; col++)
		Ac->i.f[row][col] += image->i.f[row][col];
	break;

    case DOUBLETYPE:
	for (row = 0; row < Ac->rows; row++)
	    for (col = 0; col < Ac->cols; col++)
		Ac->i.d[row][col] += image->i.d[row][col];
	break;

    case COMPLEXTYPE:
	for (row = 0; row < Ac->rows; row++)
	    for (col = 0; col < Ac->cols; col++) {
		Ac->i.cx[row][col].r += image->i.cx[row][col].r;
		Ac->i.cx[row][col].i += image->i.cx[row][col].i;
	    }
	break;

    default:
	VIP_Error_Msg("Accumulate_Add_Image: Can only operate on BYTE, SHORT, LONG, FLOAT, DOUBLE and COMPLEX images");
	return;
	break;
    }

    return;
}

/*- Multiply_Image -------------------------------------------------

Function to pixelwise multiply two images together.  There are no
checks for numerical overflow.

------------------------------------------------------------------*/

IMAGE  *Multiply_Image(IMAGE * image1, IMAGE * image2)
{
    register int row, col;
    IMAGE  *image;

    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Multiply_Image: received invalid image");
	return (NULL);
    }

    if (image1->rows != image2->rows ||
	image1->cols != image2->cols) {
	VIP_Error_Msg("Multiply_Image: the images are of different sizes");
	return (NULL);
    }


    if (image1->type == FLOATTYPE && image2->type == COMPLEXTYPE) {
	image = (IMAGE *) Allocate_Image(0, 0, image2->rows, image2->cols, image2->type);
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++) {
		image->i.cx[row][col].r = image1->i.f[row][col] * image2->i.cx[row][col].r;
		image->i.cx[row][col].i = image1->i.f[row][col] * image2->i.cx[row][col].i;
	    }
	return (image);
    }

    else if (image1->type == COMPLEXTYPE && image2->type == FLOATTYPE) {
	image = (IMAGE *) Allocate_Image(0, 0, image1->rows, image1->cols, image1->type);
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++) {
		image->i.cx[row][col].r = image1->i.cx[row][col].r * image2->i.f[row][col];
		image->i.cx[row][col].i = image1->i.cx[row][col].i * image2->i.f[row][col];
	    }
	return (image);
    }

    else if (image1->type != image2->type) {
	VIP_Error_Msg("Multiply_Image: the images are of different types");
	return (NULL);
    }

    else {

	image = (IMAGE *) Allocate_Image(0, 0, image1->rows, image1->cols, image1->type);

	switch (image->type) {

	case BYTETYPE:
	    for (row = 0; row < image->rows; row++)
		for (col = 0; col < image->cols; col++)
		    image->i.c[row][col] = image1->i.c[row][col] * image2->i.c[row][col];
	    break;

	case SHORTTYPE:
	    for (row = 0; row < image->rows; row++)
		for (col = 0; col < image->cols; col++)
		    image->i.s[row][col] = image1->i.s[row][col] * image2->i.s[row][col];
	    break;

	case LONGTYPE:
	    for (row = 0; row < image->rows; row++)
		for (col = 0; col < image->cols; col++)
		    image->i.l[row][col] = image1->i.l[row][col] * image2->i.l[row][col];
	    break;

	case FLOATTYPE:
	    for (row = 0; row < image->rows; row++)
		for (col = 0; col < image->cols; col++)
		    image->i.f[row][col] = image1->i.f[row][col] * image2->i.f[row][col];
	    break;

	case DOUBLETYPE:
	    for (row = 0; row < image->rows; row++)
		for (col = 0; col < image->cols; col++)
		    image->i.d[row][col] = image1->i.d[row][col] * image2->i.d[row][col];
	    break;

	case COMPLEXTYPE:
	    for (row = 0; row < image->rows; row++)
		for (col = 0; col < image->cols; col++)
		    Complex_Mult(&image1->i.cx[row][col], &image2->i.cx[row][col], &image->i.cx[row][col]);
	    break;

	default:
	    VIP_Error_Msg("Multiply_Image: Can only operate on BYTE SHORT LONG FLOAT DOUBLE and COMPLEX images");
	    Free_Image(image);
	    return (NULL);
	    break;
	}
	return (image);
    }
}

/*- Divide_Image -------------------------------------------------

Function to pixelwise divide two images.  There are no
checks for numerical overflow.

image = image1/image2

------------------------------------------------------------------*/

IMAGE  *Divide_Image(IMAGE * image1, IMAGE * image2)
{
    register int row, col;
    IMAGE  *image;


    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Divide_Image: received invalid image");
	return (NULL);
    }


    if (image1->rows != image2->rows ||
	image1->cols != image2->cols) {
	VIP_Error_Msg("Divide_Image: the images are of different sizes");
	return (NULL);
    }

    if (image1->type != image2->type) {
	VIP_Error_Msg("Divide_Image: the images are of different types");
	return (NULL);
    }

    image = (IMAGE *) Allocate_Image(0, 0, image1->rows, image2->cols, image1->type);

    switch (image->type) {

    case BYTETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.c[row][col] = image1->i.c[row][col] / image2->i.c[row][col];
	break;

    case SHORTTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.s[row][col] = image1->i.s[row][col] / image2->i.s[row][col];
	break;

    case LONGTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.l[row][col] = image1->i.l[row][col] / image2->i.l[row][col];
	break;

    case FLOATTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.f[row][col] = image1->i.f[row][col] / image2->i.f[row][col];
	break;

    case DOUBLETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.d[row][col] = image1->i.d[row][col] / image2->i.d[row][col];
	break;

    case COMPLEXTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		Complex_Div(&image1->i.cx[row][col], &image2->i.cx[row][col], &image->i.cx[row][col]);
	break;

    default:
	VIP_Error_Msg("Divide_Image: Can only operate on BYTE SHORT LONG FLOAT \
DOUBLE and COMPLEX images");
	Free_Image(image);
	return (NULL);
	break;
    }
    return (image);
}


/*- Squared_Image_Difference --------------------------------------------------

Function returns a FLOAT image that is the squared difference between
two input images.

-----------------------------------------------------------------------------*/

IMAGE  *Squared_Image_Difference(IMAGE * image1, IMAGE * image2)
{
    register int row, col;
    IMAGE  *image;


    if (!Image_OK(image1) || !Image_OK(image2)) {
	VIP_Error_Msg("Squared_Image_Difference: received invalid image");
	return (NULL);
    }


    if (image1->rows != image2->rows ||
	image1->cols != image2->cols) {
	VIP_Error_Msg("Squared_Image_Difference: the images are of different sizes");
	return (NULL);
    }

    if (image1->type != image2->type) {
	VIP_Error_Msg("Squared_Image_Difference: the images are of different types");
	return (NULL);
    }
    image = (IMAGE *) Allocate_Image(0, 0, image1->rows, image2->cols, FLOATTYPE);

    switch (image1->type) {

    case BYTETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.f[row][col] = SQR(image1->i.c[row][col] - image2->i.c[row][col]);
	break;

    case SHORTTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.f[row][col] = SQR(image1->i.s[row][col] - image2->i.s[row][col]);
	break;

    case LONGTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.f[row][col] = SQR(image1->i.l[row][col] - image2->i.l[row][col]);
	break;

    case FLOATTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.f[row][col] = SQR(image1->i.f[row][col] - image2->i.f[row][col]);
	break;

    case DOUBLETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		image->i.f[row][col] = SQR(image1->i.d[row][col] - image2->i.d[row][col]);
	break;

    default:
	VIP_Error_Msg("Squared_Image_Difference: Can only operate on BYTE SHORT LONG FLOAT \
DOUBLE images");
	Free_Image(image);
	return (NULL);
	break;
    }
    return (image);
}

/*- Make_Butterworth_Filter ----------------------------------------------

Function to create a Butterworth High or Low-Pass filter.  It is created in
the form of a FLOAT image of the specified size which can then be
used to scale the FFT of an image.

Note: The filter is constructed assuming that the FFT of the image
has not had the zero frequency point shifted to the centre.

----------------------------------------------------------------------*/

IMAGE  *Make_Butterworth_Filter(int rows, int cols, double CutOff, int order, char *type)
{
    int     lc, rc, tr, br;
    int     rowson2, colson2;
    double  radius, n;
    IMAGE  *image;


    image = (IMAGE *) Allocate_Image(0, 0, rows, cols, FLOATTYPE);
    rowson2 = rows / 2;
    colson2 = cols / 2;
    n = order * 2;

/* High-Pass Filter */

    if (strncmp(type, "high", 4) == 0) {

/* zero frequency filter value */

	image->i.f[0][0] = 0.0;

/* calculate top row */

	for (lc = 1, rc = cols - 1; lc <= colson2; lc++, rc--) {
	    radius = lc;
	    image->i.f[0][lc] = image->i.f[0][rc] = 1.0 / (1.0 + 0.414 * pow(radius / CutOff, n));
	}

/* calculate left column of filter */

	for (tr = 1, br = rows - 1; tr <= rowson2; tr++, br--) {
	    radius = tr;
	    image->i.f[tr][0] = image->i.f[br][0] = 1.0 / (1.0 + 0.414 * pow(radius / CutOff, n));
	}

/* now calculate the rest */

	for (tr = 1, br = rows - 1; tr <= rowson2; tr++, br--) {
	    for (lc = 1, rc = cols - 1; lc <= colson2; lc++, rc--) {
		radius = sqrt((double) (tr * tr + lc * lc));
		image->i.f[tr][lc] = image->i.f[tr][rc] = image->i.f[br][lc] = image->i.f[br][rc]
		    = 1.0 / (1.0 + 0.414 * pow(radius / CutOff, n));
	    }
	}

/* Low-Pass Filter */

    }
    else if (strncmp(type, "low", 3) == 0) {

/* zero frequency filter value */

	image->i.f[0][0] = 1.0;

/* calculate top row */

	for (lc = 1, rc = cols - 1; lc <= colson2; lc++, rc--) {
	    radius = lc;
	    image->i.f[0][lc] = image->i.f[0][rc] = 1.0 / (1.0 + 0.414 * pow(CutOff / radius, n));
	}

/* calculate left column of filter */

	for (tr = 1, br = rows - 1; tr <= rowson2; tr++, br--) {
	    radius = tr;
	    image->i.f[tr][0] = image->i.f[br][0] = 1.0 / (1.0 + 0.414 * pow(CutOff / radius, n));
	}

/* now calculate the rest */

	for (tr = 1, br = rows - 1; tr <= rowson2; tr++, br--) {
	    for (lc = 1, rc = cols - 1; lc <= colson2; lc++, rc--) {
		radius = sqrt((double) (tr * tr + lc * lc));
		image->i.f[tr][lc] = image->i.f[tr][rc] = image->i.f[br][lc] = image->i.f[br][rc]
		    = 1.0 / (1.0 + 0.414 * pow(CutOff / radius, n));
	    }
	}

    }
    else {

	VIP_Error_Msg("Make_Butterworth_Filter: type must be high or low");
	Free_Image(image);
	return (NULL);

    }


    return (image);
}


/*- Make_Homomorphic_Filter  ----------------------------------------------

Function to create a Butterworth-like Homomorphic filter.  It is created in
the form of a FLOAT image of the specified size which can then be
used to scale the FFT of an image.

Note: The filter is constructed assuming that the FFT of the image
has not had its origin shifted to the centre.

No checks are made on the bounds of HighGain and LowGain but for
'sensible' results LowGain should be between 0 and 1 and
HighGain should be greater than 1.  A starting point would be
to use a LowGain of 0.5 and a HighGain of 2.0

Reference: Gonzalez and Woods "Digital Image Processing"
page 214.

----------------------------------------------------------------------*/

IMAGE  *Make_Homomorphic_Filter(int rows, int cols, double CutOff,
            double LowGain, double HighGain, int order)
{
    int     lc, rc, tr, br;
    int     rowson2, colson2;
    double  CutOffRadius, radius, n;
    IMAGE  *image;

    if (CutOff < 0.0 || CutOff > 0.5) {
	VIP_Error_Msg("Make_Homomorphic_Filter: CutOff frequency must be in range 0 - 0.5");
	return (NULL);
    }

    image = (IMAGE *) Allocate_Image(0, 0, rows, cols, FLOATTYPE);
    rowson2 = rows / 2;
    colson2 = cols / 2;
    n = order * 2;
    CutOffRadius = CutOff * MAX(rows, cols);

/* zero frequency filter value */

    image->i.f[0][0] = LowGain;

/* calculate top row */

    for (lc = 1, rc = cols - 1; lc <= colson2; lc++, rc--) {
	radius = lc;
	image->i.f[0][lc] = image->i.f[0][rc]
	    = (HighGain - LowGain) / (1.0 + 0.414 * pow(CutOffRadius / radius, n)) + LowGain;
    }

/* calculate left column of filter */

    for (tr = 1, br = rows - 1; tr <= rowson2; tr++, br--) {
	radius = tr;
	image->i.f[tr][0] = image->i.f[br][0]
	    = (HighGain - LowGain) / (1.0 + 0.414 * pow(CutOffRadius / radius, n)) + LowGain;
    }

/* now calculate the rest */

    for (tr = 1, br = rows - 1; tr <= rowson2; tr++, br--) {
	for (lc = 1, rc = cols - 1; lc <= colson2; lc++, rc--) {
	    radius = sqrt((double) (tr * tr + lc * lc));
	    image->i.f[tr][lc] = image->i.f[tr][rc] = image->i.f[br][lc] = image->i.f[br][rc]
		= (HighGain - LowGain) / (1.0 + 0.414 * pow(CutOffRadius / radius, n)) + LowGain;
	}
    }

    return (image);
}

/*- Image_Mean_StdDev -----------------------------------------------

Function to calculate image mean and standard deviation.

--------------------------------------------------------------------*/

int     Image_Mean_StdDev(IMAGE * image, double *mean, double *StdDev)
{
    int     row, col;
    double  N;


    if (!Image_OK(image)) {
	VIP_Error_Msg("Image_Mean_StdDev: received invalid image");
	return (0);
    }


    *mean = *StdDev = 0.0;
    N = (double) (image->rows * image->cols);

/* First calculate mean */

    switch (image->type) {

    case BYTETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*mean += image->i.c[row][col];
	break;

    case SHORTTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*mean += image->i.s[row][col];
	break;

    case LONGTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*mean += image->i.l[row][col];
	break;

    case FLOATTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*mean += image->i.f[row][col];
	break;

    case DOUBLETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*mean += image->i.c[row][col];
	break;

    default:
	VIP_Error_Msg("Image_Mean_StdDev: Can only process BYTE SHORT LONG FLOAT DOUBLE images");
	return (0);
    }				/* switch */

    *mean /= N;

/* Now calculate standard deviation */

    switch (image->type) {

    case BYTETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*StdDev += SQR(*mean - (double) image->i.c[row][col]);
	break;
    case SHORTTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*StdDev += SQR(*mean - (double) image->i.s[row][col]);
	break;
    case LONGTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*StdDev += SQR(*mean - (double) image->i.l[row][col]);
	break;
    case FLOATTYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*StdDev += SQR(*mean - (double) image->i.f[row][col]);
	break;
    case DOUBLETYPE:
	for (row = 0; row < image->rows; row++)
	    for (col = 0; col < image->cols; col++)
		*StdDev += SQR(*mean - (double) image->i.d[row][col]);

    }				/* switch */

    *StdDev /= N;
    *StdDev = sqrt(*StdDev);

    return (1);
}


/*- Truncate_Image_Histogram --------------------------------------------

Function to modify an image so that specified percentages of the
extremes of its histograms are truncated.  Values in the image are
clamped to lie within this range resulting in some saturation in the
image. The aim is to remove outlying pixel values from an image so
that its displayed appearance is improved.

input: image        - pointer to image to be modified
       lowerpercent - percentage to be cut from the bottom of the histogram
       higherpercent - percentage to be cut from the top of the histogram

Typically setting lowerpercent and higherpercent to 5 will be useful.

--------------------------------------------------------------------------*/

IMAGE  *Truncate_Image_Histogram(IMAGE * image, int lowerpercent, int higherpercent)
{
    IMAGE  *bimage = NULL, *fimage = NULL, *outimage = NULL;
    int     i, row, col;
    double  min, max, min1, min2, min3, max1, max2, max3;
    int     bFREE = FALSE, fFREE = FALSE;
    int     histogram[256];
    float   N, acc;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Truncate_Image_Histogram: received invalid image");
	return (NULL);
    }


    Image_Min_Max(image, &min1, &max1, &min2, &max2, &min3, &max3);
    N = image->rows * image->cols;

    if (image->type == BYTETYPE) {
	bimage = image;
	bFREE = FALSE;
    }
    else {			/* construct BYTETYPE image - note we first
				 * construct a FLOAT image because only
				 * Float2Byte rescales values to range 0-255
				 * (Convert2Byte_Image does not)  */

	if (image->type == FLOATTYPE) {
	    fimage = image;
	    fFREE = FALSE;
	}
	else {
	    fimage = (IMAGE *) Convert2Float_Image(image);
	    if (!fimage) {
		VIP_Error_Msg("Truncate_Histogram: image conversion failure");
		return (NULL);
	    }
	}
	bimage = (IMAGE *) Float2Byte_Image(fimage);
	bFREE = TRUE;
    }

    outimage = (IMAGE *) Allocate_Image(0, 0, image->rows, image->cols, image->type);

    Histogram_Image(bimage, histogram);

    acc = 0.0;
    i = 0;
    while ((acc / N) * 100.0 < lowerpercent && i < 255) {
	acc += histogram[i];
	i++;
    }

    min = min1 + (float) i *(max1 - min1) / 256.0;

    acc = 0.0;
    i = 255;
    while ((acc / N) * 100.0 < higherpercent && i > 0) {
	acc += histogram[i];
	i--;
    }

    max = max1 - (float) (255 - i) * (max1 - min1) / 256.0;

    switch (image->type) {

    case BYTETYPE:
	for (row = 0; row < image->rows; row++) {
	    for (col = 0; col < image->cols; col++) {
		if (image->i.c[row][col] < min)
		    outimage->i.c[row][col] = min;
		else if (image->i.c[row][col] > max)
		    outimage->i.c[row][col] = max;
		else
		    outimage->i.c[row][col] = image->i.c[row][col];
	    }
	}
	break;

    case SHORTTYPE:
	for (row = 0; row < image->rows; row++) {
	    for (col = 0; col < image->cols; col++) {
		if (image->i.s[row][col] < min)
		    outimage->i.s[row][col] = min;
		else if (image->i.s[row][col] > max)
		    outimage->i.s[row][col] = max;
		else
		    outimage->i.s[row][col] = image->i.s[row][col];
	    }
	}
	break;

    case LONGTYPE:
	for (row = 0; row < image->rows; row++) {
	    for (col = 0; col < image->cols; col++) {
		if (image->i.l[row][col] < min)
		    outimage->i.l[row][col] = min;
		else if (image->i.l[row][col] > max)
		    outimage->i.l[row][col] = max;
		else
		    outimage->i.l[row][col] = image->i.l[row][col];
	    }
	}
	break;

    case FLOATTYPE:
	for (row = 0; row < image->rows; row++) {
	    for (col = 0; col < image->cols; col++) {
		if (image->i.f[row][col] < min)
		    outimage->i.f[row][col] = min;
		else if (image->i.f[row][col] > max)
		    outimage->i.f[row][col] = max;
		else
		    outimage->i.f[row][col] = image->i.f[row][col];
	    }
	}
	break;

    case DOUBLETYPE:
	for (row = 0; row < image->rows; row++) {
	    for (col = 0; col < image->cols; col++) {
		if (image->i.d[row][col] < min)
		    outimage->i.d[row][col] = min;
		else if (image->i.d[row][col] > max)
		    outimage->i.d[row][col] = max;
		else
		    outimage->i.d[row][col] = image->i.d[row][col];
	    }
	}
	break;

    default:
	VIP_Error_Msg("Truncate_Image: can only operate on BYTE SHORT, LONG, FLOAT and DOUBLE images");
	Free_Image(outimage);
	outimage = NULL;

    }				/* switch */

    if (fFREE)
	Free_Image(fimage);
    if (bFREE)
	Free_Image(bimage);
    return (outimage);
}


/*- Distance_Transform ------------------------------------------

Function computes a distance transform for a binary image.  It is
assumed that objects pixels in the image have a value of 0.

Function returns a SHORTTYPE image with distances away from the
objects calculated using the distance relationship shown below.

       -------------
       | 7 | 5 | 7 |
       -------------
       | 5 | x | 5 |
       -------------
       | 7 | 5 | 7 |
       -------------

Peter Kovesi  March 1994.

----------------------------------------------------------------*/

IMAGE  *Distance_Transform(IMAGE * inimage)
{
    int     row, col;
    IMAGE  *outim;

    if (!Image_OK(inimage)) {
	VIP_Error_Msg("Distance_Transform: received invalid image");
	return (NULL);
    }

    if (inimage->type != BYTETYPE) {
	VIP_Error_Msg("Distance_Transform: can only handle BYTE images");
	return (NULL);
    }


/* initialize output image to have values 0 on object pixles
   and a value of MAXSHORT elsewhere */

    outim = Allocate_Image(0, 0, inimage->rows, inimage->cols, SHORTTYPE);

    for (row = 0; row < inimage->rows; row++)
	for (col = 0; col < inimage->cols; col++)
	    outim->i.s[row][col] = inimage->i.c[row][col] == 0 ? 0 : MAXSHORT;

/* Do a forward pass through the image */

    for (row = 1; row < inimage->rows; row++)
	for (col = 1; col < inimage->cols - 1; col++)
	    outim->i.s[row][col] =
		MIN(
		MIN(
		    MIN(outim->i.s[row][col], outim->i.s[row][col - 1] + 5),
		    MIN(outim->i.s[row - 1][col - 1] + 7, outim->i.s[row - 1][col] + 5)
		), outim->i.s[row - 1][col + 1] + 7);

/* Then a backward pass to complete */

    for (row = inimage->rows - 2; row >= 0; row--)
	for (col = inimage->cols - 2; col > 0; col--)
	    outim->i.s[row][col] =
		MIN(
		MIN(
		    MIN(outim->i.s[row][col], outim->i.s[row][col + 1] + 5),
		    MIN(outim->i.s[row + 1][col + 1] + 7, outim->i.s[row + 1][col] + 5)
		), outim->i.s[row + 1][col - 1] + 7);

    return (outim);
}
