/**
 ** sipp - SImple Polygon Processor
 **
 **  A general 3d graphic package
 **
 **  Copyright Jonas Yngvesson  (jonas-y@isy.liu.se) 1988/89/90/91
 **            Inge Wallin      (ingwa@isy.liu.se)         1990/91
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 1, or any later version.
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 ** You can receive a copy of the GNU General Public License from the
 ** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **/

/**
 ** torus.c - Creating a torus as a sipp object.
 **/

#include <xalloca.h>
#include <math.h>

#include <sipp.h>


static void
arr_rot(arr1, arr2, len, angle)
    Vector arr1[];
    Vector arr2[];
    int    len;
    double angle;
{
    int    i;
    double sa, ca;

    sa = sin(angle);
    ca = cos(angle);
    for (i = 0; i < len; i++) {
        arr2[i].x = arr1[i].x * ca - arr1[i].y * sa;
        arr2[i].y = arr1[i].x * sa + arr1[i].y * ca;
        arr2[i].z = arr1[i].z;
    }
}


static void
push_band(arr1, arr2, len)
    Vector arr1[];
    Vector arr2[];
    int    len;
{
    int i, j;

    for (i = 0; i < len; i++) {
        j = (i + 1) % len;
        vertex_tx_push(arr1[i].x, arr1[i].y, arr1[i].z, 
                       arr1[i].x, arr1[i].y, arr1[i].z);
        vertex_tx_push(arr2[i].x, arr2[i].y, arr2[i].z, 
                       arr2[i].x, arr2[i].y, arr2[i].z);
        vertex_tx_push(arr2[j].x, arr2[j].y, arr2[j].z, 
                       arr2[j].x, arr2[j].y, arr2[j].z);
        vertex_tx_push(arr1[j].x, arr1[j].y, arr1[j].z, 
                       arr1[j].x, arr1[j].y, arr1[j].z);
        polygon_push();
    }
}



Object *
sipp_torus(bigradius, smallradius, res1, res2, surface, shader)
    double  bigradius;      /* Radius of the ring */
    double  smallradius;    /* Radius of the "tube" */
    int     res1;           /* Number of polygons around the ring */
    int     res2;           /* Number of polygons around the tube */
    void   *surface;
    Shader *shader;
{
    Object *torus;
    Vector *arr1;
    Vector *arr2;
    Vector *tmp;
    int     i;

    /* Create two arrays to hold vertices around the tube */
    arr1 = (Vector *)alloca(res2 * sizeof(Vector));
    arr2 = (Vector *)alloca(res2 * sizeof(Vector));

    for (i = 0; i < res2; i++) {
        arr1[i].x = bigradius + smallradius * cos(i * 2.0 * M_PI / res2);
        arr1[i].y = 0.0;
        arr1[i].z = smallradius * sin(i * 2.0 * M_PI / res2);
    }

    /* Sweep out the torus by rotating the two perimeters */
    /* defined in arr1 and arr2. */
    for (i = 0; i < res1; i++) {
        arr_rot(arr1, arr2, res2, 2.0 * M_PI / (double)res1);
        push_band(arr1, arr2, res2);
        tmp = arr1;
        arr1 = arr2;
        arr2 = tmp;
    }

    torus = object_create();
    object_add_surface(torus, surface_create(surface, shader));

    return torus;
}

