/*
 * This include file defines the different shaders availiable in
 * sipp. Each shader is defined by a structure containing the necessary
 * parameters to describe how a surface should be shaded with that 
 * particular shader, and the extern declaration of the shader function 
 * itself.
 */


#ifndef _SHADERS_H
#define _SHADERS_H


#include <sipp.h>



/* 
 * Surface description for the bozo shader.
 */
typedef struct {
    Color   *colors;
    int      no_of_cols;
    double   ambient;
    double   specular;
    double   c3;
    double   scale;        /* Scale the texture by this value */
} Bozo_desc;
    


/*
 * Surface description used by the marble shader. marble_shader
 * creates a solid texture (using noise & turbulence) that
 * simulates marble.
 */
typedef struct {
    double   ambient;
    double   specular;
    double   c3;
    double   scale;        /* Scale the marble texture by this value */
    Color    base;         /* "Base" color of the surface */
    Color    strip;        /* Color of the "stripes" in the marble */
} Marble_desc;



/*
 * Surface description used by the granite shader. granite_shader
 * creates a solid texture (using noise) that mixes two colors
 * to simulate granite.
 */
typedef struct {
    double   ambient;
    double   specular;
    double   c3;
    double   scale;        /* Scale the texture by this value */
    Color    col1;         /* The two color components */
    Color    col2;         /*                          */
} Granite_desc;



/*
 * Mask shader. This shader is SUN specific since it uses
 * a pixrect. It projects the pixrect on the x-y plane (in
 * texture coordinates). When a surface is shaded it checks
 * if the x, y coordinate in the pixrect is zero and calls
 * one of two other shaders depending of the outcome of that 
 * test.
 */
typedef struct {
    Shader *fg_shader;          /* Shader to call if mask(x, y) != 0 */
    void   *fg_surface;         /* Surface description for fg_shader */
    Shader *bg_shader;          /* Shader to call if mask(x, y) == 0 */
    void   *bg_surface;         /* Surface description for bg_shader */
    void   *mask;               /* Pointer to mask image */
    bool  (*pixel_test)();      /* Function that tests a pixel value */
    int     x0, y0;             /* Where to put origo in the mask image */
    int     xsize, ysize;       /* Size of the mask image */
    double  xscale, yscale;     /* Scale the mask image with these values */
} Mask_desc;


/*
 * Surface description for the bumpy_shader(). This shader 
 * fiddles with the surface normals of a surface so the surface 
 * looks bumpy.
 */

typedef struct {
    Shader *shader;
    void   *surface;
    double scale;
    bool   bumpflag;
    bool   holeflag;
} Bumpy_desc;


/*
 * Declarations of the actual shading functions.
 */
extern void marble_shader();
extern void granite_shader();
extern void bozo_shader();
extern void mask_shader();
extern void bumpy_shader();
extern void planet_shader();

#endif /* _SHADERS_H */
