/*********************************************************/
/*                                                       */
/*                      lzw_decomp.c                     */
/*                                                       */
/*     Data Decompression Routine using LZW algorithm    */
/*                                                       */
/*                                                       */
/*      This routine can be called as follows:           */
/*                                                       */
/*          lzw_decomp(inbuf, ibsize)                    */
/*              unsigned char inbuf[];                   */
/*              int *ibsize;                              */
/*          {                                            */
/*          }                                            */
/*                                                       */
/*      Note: uncompress.c is the LZW implementation on  */
/*            the SUN system.                            */
/*                                                       */
/*                            LBL 7/12/88                */
/*                                                       */
/*********************************************************/
#include <stdio.h>
#include <math.h>
#include <scry_image.h>

#define	min(a,b)	((a>b) ? b : a)

#define BITS 12

#if BITS == 16
# define HSIZE	69001		/* 95% occupancy */
#endif
#if BITS == 15
# define HSIZE	35023		/* 94% occupancy */
#endif
#if BITS == 14
# define HSIZE	18013		/* 91% occupancy */
#endif
#if BITS == 13
# define HSIZE	9001		/* 91% occupancy */
#endif
#if BITS <= 12
# define HSIZE	5003		/* 80% occupancy */
#endif

#if BITS > 15
typedef long int	code_int;
#else
typedef int		code_int;
#endif

typedef long int	count_int;
typedef	unsigned char	char_type;

#define BIT_MASK	0x1f
#define BLOCK_MASK	0x80
#define INIT_BITS       9	 /* initial number of bits/code        */

#undef vax

int n_bits;    	                 /* number of bits/code                */
int maxbits = BITS;              /* user settable max # bits/code      */
code_int maxcode;                /* maximum code, given n_bits         */
code_int maxmaxcode = 1 << BITS; /* should NEVER generate this code    */
# define MAXCODE(n_bits)	((1 << (n_bits)) - 1)

count_int htab [HSIZE];          /* hashing table                      */
unsigned short codetab [HSIZE];  /* code table                         */

code_int free_ent = 0;

#define htabof(i)	htab[i]
#define codetabof(i)	codetab[i]

int block_compress = BLOCK_MASK;
int clear_flg = 0;
#define FIRST	257	         /* first free entry                   */
#define	CLEAR	256	         /* table clear output code            */

char_type rmask[9] = {0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff};
int in_count = 0;	 /* length of input                    */

#define tab_prefixof(i)	codetabof(i)
#define tab_suffixof(i)	((char_type *)(htab))[i]
#define de_stack		((char_type *)&tab_suffixof(1<<BITS))




int lzw_output_size ;

unsigned char *outbuf ;

lzw_decomp(inbuf, ibsize) 
unsigned char inbuf[];     /* input buffer       */
int *ibsize;                /* input buffer size  */
{
    register char_type *stackp; /* internal buffer pointer */
    register int finchar;       /* next char in the stream */
    register code_int code, oldcode, incode; /* codes      */
    register out_count;         /* output stream counter   */
    code_int getcode();         /* function getcode()      */
    int i ;

    outbuf = (unsigned char *) calloc((S_image_info.s_height*S_image_info.s_width),sizeof(char)) ;
    in_count = 2;          /* skip the first two magic codes */
    out_count = 0;         /* set output stream counter to 0 */

    maxcode = MAXCODE(n_bits = INIT_BITS); /* max code value */
    maxbits = inbuf[in_count++];       /* max # code of bits */
    block_compress = maxbits & BLOCK_MASK; /* block comp     */
    maxbits &= BIT_MASK;           /* recover # of code bits */
    maxmaxcode = 1 << maxbits;     /* set max code value     */


    /*
     * As above, initialize the first 256 entries in the table.
     */
    for ( code = 255; code >= 0; code-- ) 
    {
	tab_prefixof(code) = 0;
	tab_suffixof(code) = (char_type)code;
    }
    free_ent = ((block_compress) ? FIRST : 256 );

    finchar = oldcode = getcode(inbuf, *ibsize);/* get first code       */
    if(oldcode == -1)	           /* EOF already?                     */
	return;			   /* Get out of here                  */
    /*putc((char)finchar, outfp);	*/   /* first code must be 8 bits = char */
    outbuf[out_count++] = (char)finchar;
    stackp = de_stack;

    /* loop to get codes and to reconstruct strigs                     */
    while ( (code = getcode(inbuf, *ibsize)) > -1 ) 
    {
	if ( (code == CLEAR) && block_compress ) 
	{
	    for ( code = 255; code >= 0; code-- )
		tab_prefixof(code) = 0;
	    clear_flg = 1;
	    free_ent = FIRST - 1;
	    if (code = getcode(inbuf, *ibsize) == -1 )/* O,untimely death! */
		break;
	}
	incode = code;
	/*
	 * Special case for KwKwK string.
	 */
	if ( code >= free_ent ) 
	{
            *stackp++ = finchar;
	    code = oldcode;
	}

	/*
	 * Generate output characters in reverse order
	 */
	while ( code >= 256 ) 
	{
	    *stackp++ = tab_suffixof(code);
	    code = tab_prefixof(code);
	}
	*stackp++ = finchar = tab_suffixof(code);

	/*
	 * And put them out in forward order
	 */
	do
           /*
	    putc((char)*--stackp, outfp);
           */
            outbuf[out_count++] = (char)*--stackp;
	while ( stackp > de_stack );

	/*
	 * Generate the new entry.
	 */
	if ( (code=free_ent) < maxmaxcode ) 
	{
	    tab_prefixof(code) = (unsigned short)oldcode;
	    tab_suffixof(code) = finchar;
	    free_ent = code+1;
	} 
	/*
	 * Remember previous code.
	 */
	oldcode = incode;
    }
    for (i = 0 ; i < out_count ; i++)
        inbuf[i] = outbuf[i] ;
    *ibsize = out_count;
    free(outbuf) ;
}





code_int
getcode(inbuf, ibsize) 
unsigned char inbuf[];
int ibsize;
{
    /*
     * On the VAX, it is important to have the register declarations
     * in exactly the order given, or the asm will break.
     */
    register code_int code;
    static int offset = 0, size = 0;
    static char_type buf[BITS];
    register int r_off, bits;
    register char_type *bp = buf;
    int i;

    if ( clear_flg > 0 || offset >= size || free_ent > maxcode ) {
	/*
	 * If the next entry will be too big for the current code
	 * size, then we must increase the size.  This implies reading
	 * a new buffer full, too.
	 */
	if ( free_ent > maxcode ) {
	    n_bits++;
	    if ( n_bits == maxbits )
		maxcode = maxmaxcode;	/* won't get any bigger now */
	    else
		maxcode = MAXCODE(n_bits);
	}
	if ( clear_flg > 0) {
    	    maxcode = MAXCODE (n_bits = INIT_BITS);
	    clear_flg = 0;
	}
	for (i=0; i<n_bits; i++)
	{
	    buf[i] = inbuf[in_count++];
        }
	size = n_bits;
        if (in_count >= ibsize)
	   size = ibsize - (in_count - n_bits);
	if ( size <= 0 )
        {
	    return -1;			/* end of file */
	}
	offset = 0;
	/* Round size down to integral number of codes */
	size = (size << 3) - (n_bits - 1);
    }
    r_off = offset;
    bits = n_bits;
	bp += (r_off >> 3);
	r_off &= 7;
	code = (*bp++ >> r_off);
	bits -= (8 - r_off);
	r_off = 8 - r_off;		/* now, offset into code word */
	if ( bits >= 8 )
	{
	    code |= *bp++ << r_off;
	    r_off += 8;
	    bits -= 8;
	}
	/* high order bits. */
	code |= (*bp & rmask[bits]) << r_off;
    offset += n_bits;

    return code;
}

