/* data.c      Distribution 1.1   89/3/30   Scry */

/*   The Scry system is copyright (C) 1988, 1989, Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof.     */

/*	data.c		arrays useful in the tessellation algorithm
 * This is the result of a lot of laborious work in determining how
 * to get 256 different patterns out of 14 basic ones.  There are
 * a number of different arrays that are created and filled out in
 * here, most of them actually useful at some point or another.
 */
#include "march.h"

int       initialization_done = 0;

typedef u_char Vertices;	/* Required to hold vertex data: 8 bits */

#define VX 01			/* Vertex values for directions */
#define VY 02			/* Vn = BIT(En) */
#define VZ 04
#define XY 02			/* Common bit place to faces */
#define XZ 01
#define YZ 00

/* These are all the basic data.  From there we'll have to rotate them.  Figure out the
 * minimum number of rotations of each pattern we'll need to reach the optimum.
 * There's a maximum of 4*4*4 rotations, which is the number of ways we can turn
 * it with my algorithms.  One is no rotation whatsoever.
 */

/*	Edges are 12 bits, an int, and vertices are 8 bits, an unsigned char.
 */

#define VERTS(w,x,y,z) (BIT(w) | BIT(x) | BIT(y) | BIT(z))
Vertices  vertpats[15] = {VERTS(-1, -1, -1, -1),	/* empty cube */
    VERTS(0, -1, -1, -1),	/* one vertex */
    VERTS(0, 1, -1, -1),	/* one edge */
    VERTS(0, 3, -1, -1),	/* two corners */
    VERTS(0, 7, -1, -1),	/* two corners */
    VERTS(1, 5, 4, -1),
    VERTS(0, 1, 7, -1),		/* edge and two corners */
    VERTS(2, 1, 7, -1),		/* three corners */
    VERTS(0, 1, 5, 4),		/* half a cube, square */
    VERTS(0, 4, 5, 6),		/* half a cube, hex */
    VERTS(0, 2, 5, 7),
    VERTS(0, 4, 5, 7),
    VERTS(2, 4, 5, 1),
    VERTS(0, 3, 5, 6),		/* four corners */
VERTS(1, 5, 4, 6)};


#define TRIAG(x,y,z) (BIT(x) | BIT(y) | BIT(z))

Edges     edgepats[15][MAXTRIANGLES] = {
    {0, 0, 0, 0},		/* 0 */
    {TRIAG(0, 1, 2), 0, 0, 0},	/* 1 */
    {TRIAG(1, 2, 5), TRIAG(2, 10, 5), 0, 0},	/* 2 */
    {TRIAG(0, 1, 2), TRIAG(4, 5, 6), 0, 0},	/* 3 */
    {TRIAG(0, 1, 2), TRIAG(6, 9, 12), 0, 0},	/* 4 */
    {TRIAG(0, 2, 13), TRIAG(0, 13, 9), TRIAG(0, 9, 5), 0},	/* 5 */
    {TRIAG(1, 2, 5), TRIAG(2, 10, 5), TRIAG(6, 9, 12), 0},	/* 6 */
    {TRIAG(1, 4, 14), TRIAG(0, 5, 10), TRIAG(6, 9, 12), 0},	/* 7 */
    {TRIAG(1, 5, 9), TRIAG(1, 13, 9), 0, 0},	/* 8 */
    {TRIAG(1, 0, 14), TRIAG(0, 9, 10), TRIAG(9, 12, 14), TRIAG(14, 0, 9)},	/* 9 */
    {TRIAG(0, 2, 14), TRIAG(0, 14, 4), TRIAG(8, 10, 6), TRIAG(8, 12, 6)},	/* 10 */
    {TRIAG(0, 10, 6), TRIAG(0, 1, 13), TRIAG(13, 12, 6), TRIAG(0, 13, 6)},	/* 11 */
    {TRIAG(0, 2, 13), TRIAG(0, 13, 9), TRIAG(0, 9, 5), TRIAG(4, 1, 14)},	/* 12 */
    {TRIAG(0, 1, 2), TRIAG(4, 5, 6), TRIAG(8, 9, 10), TRIAG(12, 13, 14)},	/* 13 */
    {TRIAG(0, 2, 14), TRIAG(14, 12, 9), TRIAG(0, 14, 9), TRIAG(0, 9, 5)}	/* 14 */
};

/* These are the Lorensen and Cline patterns.  I produced equally
 * valid ones with my own algorithm, but these are here just for
 * authenticity's sake.
 */
/*	{ TRIAG(0,2,5),	TRIAG(2,13,5),	TRIAG(13,5,9),	0 },			/* 5 */
/*	{ TRIAG(1,14,12),TRIAG(0,1,12),	TRIAG(0,12,9),	TRIAG(0,9,10) }, 	/* 9 */
/*	{ TRIAG(1,13,0),TRIAG(0,13,6),	TRIAG(12,13,6),	TRIAG(0,10,6) }, 	/* 11 */
/*	{ TRIAG(1,4,14),TRIAG(2,0,5),	TRIAG(2,13,5),	TRIAG(5,13,9) }, 	/* 12 */
/*	{ TRIAG(2,14,0),TRIAG(14,0,9),	TRIAG(14,9,12),	TRIAG(0,9,5) }		/* 14 */

Edges     cubedges[256][MAXTRIANGLES];	/* This will contain all the
					 * edgepoint data. */
 /* The index into the array will be the vertex data. */
Vectors   numtri[256];		/* Number of triangles per cube */
Vectors   edgecorns[256];	/* Number of edges per corner */
Vectors   numedg[256];		/* Number of edges per cube */
Vectors   edgecornsi[256];	/* i corners, for the YZ face: 0,1,2,8,13 */
Vectors   edgecornsj[256];	/* j corners, for the XZ face: 0,1,2,4,14 */
Vectors   edgecornsk[256];	/* k corners, for the XY face: 0,1,2,5,10 */
Vectors   edgecornst[256];	/* top of th' cube */
#ifdef PARANOIA
Vectors   pattern[256];		/* Which pattern it is: 0-14 */
Vectors   morebits[256];	/* Are there 5 or more bits active? */
#endif

/* There are 12 edges.  These are the X,Y,Z coordinates of them, with 2 = 1 unit of
 * the unit cube, 1 = .5 unit of the unit cube, and 0 = no distance.  This is what
 * sink() reads.  My method of referencing them leaves a few holes, so there have
 * to be 15.
 */
Vectors   edgenums[15][3] = {{1, 0, 0},	/* 000 (0:x) */
{0, 1, 0},			/* 001 (0:y) */
{0, 0, 1},			/* 002 (0:z) */
{0, 0, 0},			/* Unused */
{1, 2, 0},			/* 004 (3:x) */
{2, 1, 0},			/* 005 (3:y) */
{2, 2, 1},			/* 006 (3:z) */
{0, 0, 0},			/* Unused */
{1, 0, 2},			/* 010 (5:x) */
{2, 1, 2},			/* 011 (5:y) */
{2, 0, 1},			/* 012 (5:z) */
{0, 0, 0},			/* Unused */
{1, 2, 2},			/* 014 (6:x) */
{0, 1, 2},			/* 015 (6:y) */
{0, 2, 1} /* 016 (6:z) */ };


Vectors   revedges[3][3][3] = {	/* Index for reconversion */
    {{-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}},	/* -1's to make sure */
    {{-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}},	/* we don't screw up */
{{-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}};

init_marchcubes()
{				/* Initialise the marching cubes! */
    int       i, j;

    for (i = 0; i < 15; i++)	/* Set up revedges[][][] for use */
	revedges[edgenums[i][0]][edgenums[i][1]][edgenums[i][2]] = i;
    revedges[0][0][0] = -1;

#ifdef PARANOIA
    for (i = 0; i < 256; i++) {
	for (j = 0; j < 8; j++)
	    if (i & BIT(j))
		morebits[i]++;
	if (morebits[i] > 4)
	    morebits[i] = 1;
	else
	    morebits[i] = 0;
    }
#endif

    for (i = 0; i < 256; i++)
	for (j = 0; j < 4; j++)
	    cubedges[i][j] = 0;

    (void) rotate_cubes();

    initialization_done = 1;
}

/* The rotate routines provide the XOR mask to turn your current vert into one
 * that has been rotated num right angles counterclockwise in the indicated plane.
 */
#define CMPBIT(n,x,y) (!!(n & BIT(x)) == !!(n & BIT(y)))

#define ROTATE_YZ(vert,num) ((num == 2) ? (VY | VZ) : (CMPBIT(vert,EY,EZ) ? \
	((num == 3) ? VZ : (num ? VY : 0)) : ((num == 3) ? VY : (num ? VZ : 0))))
#define ROTATE_XY(vert,num) ((num == 2) ? (VX | VY) : (CMPBIT(vert,EX,EY) ? \
	((num == 3) ? VY : (num ? VX : 0)) : ((num == 3) ? VX : (num ? VY : 0))))
#define ROTATE_XZ(vert,num) ((num == 2) ? (VX | VZ) : (CMPBIT(vert,EX,EZ) ? \
	((num == 3) ? VX : (num ? VZ : 0)) : ((num == 3) ? VZ : (num ? VX : 0))))

int       pat[2][4] = {{0, 1, 2, 1},
{0, 0, 2, 2}};

#define LEFT(edge,axis) ((axis == EZ) ? (edge & (3 << 2)) >> 2 : (edge & (3 << 4)) >> 4)
#define RIGHT(edge,axis) ((axis == EX) ? (edge & (3 << 2)) >> 2 : (edge & 3))
#define LSHIFT(axis,num) ((axis == EZ) ? num << 2 : num << 4)
#define RSHIFT(axis,num) (axis ? num : num << 2)
#define AXIS(axis,edge) (edge & (3 << (axis*2)))

rotate(edge, axis, num)		/* This is macroizeable */
    int       edge, axis, num;
{
    int       index, newedge = 0, parity, inc, y;

    if (num == 0)
	return (edge);		/* Save some CPU */

    y = (axis == EY);

/* Find the index into the pattern that rotating edges follow; if the axis' pattern is
 * 01, then it'll follow the second one in pat, otherwise it will follow the first.
 */
    if (edge & BIT(axis * 2)) {
	parity = 1;		/* 0 2 2 0 */
	if (LEFT(edge, axis) == 2)
	    if (RIGHT(edge, axis) == 2)
		index = (y ? 3 : 2);
	    else
		index = (y ? 2 : 3);
	else if (RIGHT(edge, axis) == 2)
	    index = (y ? 0 : 1);
	else
	    index = (y ? 1 : 0);
    } else {			/* 1 2 1 0 */
	parity = 0;
	switch (LEFT(edge, axis)) {
	case 0:
	    index = 0;
	    break;
	case 1:
	    if (RIGHT(edge, axis))
		index = (y ? 3 : 1);
	    else
		index = (y ? 1 : 3);
	    break;
	case 2:
	    index = 2;
	    break;
	}
    }
    if (axis == EY)
	inc = -1;
    else
	inc = 1;

    /* Actual bit shifting */
    newedge |= LSHIFT(axis, pat[parity][(index + num) & 3]);
    newedge |= RSHIFT(axis, pat[parity][(index + num + inc) & 3]);
    newedge |= AXIS(axis, edge);
    return (newedge);
}

#define ETOC(n) (edgenums[n][0] | (edgenums[n][1] << 2) | (edgenums[n][2] << 4))
#define CTOE(n) BIT(revedges[n&3][(n&(3<<2))>>2][(n&(3<<4))>>4])
/* ETOC converts an edge number to a set of EZEYEX bit coordinates, while
 * CTOE reverses this process.
 */


/* rotedges will take the old and new patterns of a rotated cube, along with
 * the number of 90 degree blocks of rotation the cube has undergone, and
 * fill in the edges array for npat's index.  Have the cube rotation program
 * check to see if that one has already been used- perhaps a 256 bit temporary
 * variable, perhaps actually looking in the table.  This is all integer math,
 * though quite hairy, so it should only take a few seconds of setup at the most.
 */
rotedges(xr, yr, zr, onum, npat)/* npat is the bitmap because we're storing
				 * them that */
    int       xr, yr, zr, onum;
    int       npat;		/* way in the big array; in the preliminary
				 * one, */
{				/* we just use indexes. */
    int       i, j, e0, e1, e2, e3;

    for (i = 0; i < 4; i++) {
	cubedges[npat][i] = 0;
	if (edgepats[onum][i]) {
	    for (j = 0; j < 15; j++) {
		if (edgepats[onum][i] & BIT(j)) {
		    e0 = ETOC(j);
		    e1 = rotate(e0, EX, xr);
		    e2 = rotate(e1, EY, yr);
		    e3 = rotate(e2, EZ, zr);
		    cubedges[npat][i] |= CTOE(e3);
		}
	    }
	} else
	    return;
    }
}

rotcube(xr, yr, zr, onum)
    int       xr, yr, zr, onum;
{
    int       i, aftx, afty, aftz, final = 0;

    for (i = 0; i < 8; i++)
	if (vertpats[onum] & BIT(i)) {
	    aftx = i ^ ROTATE_YZ(i, xr);
	    afty = aftx ^ ROTATE_XZ(aftx, yr);
	    aftz = afty ^ ROTATE_XY(afty, zr);
	    final |= BIT(aftz);
	}
#ifdef PARANOIA
    pattern[final] = onum;
    pattern[final ^ 0xFF] = onum;
#endif

    if (cubedges[final][0] == 0)/* Only bother for things we haven't done
				 * before */
	rotedges(xr, yr, zr, onum, final);
}

int 
rotate_cubes()
{
    int       h, i, j, k, tmp, patt;

    rotcube(0, 0, 0, 0);	/* There's only 1 pattern for 0 */
    for (h = 1; h < 15; h++)	/* Account for another ~127 cubes */
	for (i = 0; i < 4; i++)
	    for (j = 0; j < 4; j++)
		for (k = 0; k < 4; k++)
		    rotcube(i, j, k, h);
    for (i = 0; i < 255; i++)	/* Take care of the rest that have >4
				 * vertices */
	if (cubedges[i ^ 0xFF][0]);
	else
	    for (j = 0; j < 4; j++)
		cubedges[i ^ 0xFF][j] = cubedges[i][j];

    for (i = 0; i < 256; i++) {	/* Get data for predictive usage */
	tmp = 0;
	for (j = 0; j < 4; j++)
	    if (cubedges[i][j])
		tmp++;
	    else
		break;
	numtri[i] = tmp;

	tmp = 0;
	patt = (cubedges[i][0] | cubedges[i][1] | cubedges[i][2] | cubedges[i][3]);
	for (j = 0; j < 3; j++) {
	    if (patt & BIT(j))
		tmp++;
	}
	edgecorns[i] = tmp;

	tmp = 0;
	for (j = 0; j < 15; j++) {
	    if (patt & BIT(j))
		tmp++;
	}
	numedg[i] = tmp;

	edgecornsi[i] = ((patt & BIT(5)) ? 1 : 0) + ((patt & BIT(10)) ? 1 : 0);
	edgecornsj[i] = ((patt & BIT(4)) ? 1 : 0) + ((patt & BIT(14)) ? 1 : 0);
	edgecornsk[i] = ((patt & BIT(8)) ? 1 : 0) + ((patt & BIT(13)) ? 1 : 0);
	edgecornst[i] = edgecorns[i] + edgecornsi[i];
    }

    return (0);
}
