
/*  mask.c       Brian Tierney, LBL       8/90
 *    for use with segal
 *
 *  These routines are used for automatic mask computation
 *     (threshold and polygon mask)
 */

#include "segal.h"
extern view_win_objects *view_win;
extern segal_win_objects *segal_win;
extern pixedit_win_objects *edit_win;

#define Realloc(x,y,z) (z *)realloc((char *)x,(unsigned)(y * sizeof(z)))

#define MAX_POINTS 100
#define FVAL  10		/* value in buffer which indicates reached by
				 * fill */
#define PLYVAL 20		/* value in buffer which indicates edge of
				 * polygon */
#define CROSS_SIZE 10

typedef struct cross {
    int       x, y;		/* center of cross */
    int       l1x1, l1y1, l1x2, l1y2;	/* endpoints of line 1 */
    int       l2x1, l2y1, l2x2, l2y2;	/* endpoints of line 2 */
}         CROSS;

CROSS     cross_list[MAX_POINTS];
static int num_cross = 0;

typedef struct s_item {
    short     i, j;
}         STACK_ITEM;

STACK_ITEM *stack;

int       sp;			/* global stack pointer */
int       stack_size;
#define STACK_CHUNK 10000

/**************************************************************/
void
cross_proc(event)
    Event    *event;
{
    void      move_cross(), add_cross();
    static int cnum;
    int       x, y;

    x = event_x(event);
    y = event_y(event);

    if (event_id(event) == LOC_DRAG) {
	if (x - CROSS_SIZE < cross_list[cnum].x &&
	    x + CROSS_SIZE > cross_list[cnum].x &&
	    y - CROSS_SIZE < cross_list[cnum].y &&
	    y + CROSS_SIZE > cross_list[cnum].y)
	    move_cross(cnum, x, y);
	return;
    }
    if (cnum = cross_nearby(x, y))
	move_cross(cnum, x, y);
    else {
	if (num_cross >= MAX_POINTS - 1) {
	    fprintf(stderr, "Error: Maximum number of polygon vertices exceeded. \n");
	    return;
	}
	add_cross(num_cross, x, y);
	num_cross++;
    }
}

/**************************************************************/
void
add_cross(cnum, x, y)
    int       cnum, x, y;
{
    int       nx1, nx2, ny1, ny2;
    void      draw_cross();

    cross_list[cnum].x = x;
    cross_list[cnum].y = y;

    nx1 = x - CROSS_SIZE;
    nx2 = x + CROSS_SIZE;
    ny1 = y - CROSS_SIZE;
    ny2 = y + CROSS_SIZE;
    if (nx1 < 0)
	nx1 = 0;
    if (ny1 < 0)
	ny1 = 0;
    if (nx2 > segal.cols - 1)
	nx2 = segal.cols - 1;
    if (ny2 > segal.rows - 1)
	ny2 = segal.rows - 1;

    cross_list[cnum].l1x1 = cross_list[cnum].l2x1 = nx1;
    cross_list[cnum].l1x2 = cross_list[cnum].l2x2 = nx2;
    cross_list[cnum].l1y1 = cross_list[cnum].l2y2 = ny1;
    cross_list[cnum].l1y2 = cross_list[cnum].l2y1 = ny2;

    draw_cross(cross_list[cnum]);
}

/**************************************************************/
void
draw_cross(cross)
    CROSS     cross;
{
    XSetForeground(display, gc, standout);

    XDrawLine(display, view_xid, gc, cross.l1x1, cross.l1y1,
	      cross.l1x2, cross.l1y2);

    XDrawLine(display, view_xid, gc, cross.l2x1, cross.l2y1,
	      cross.l2x2, cross.l2y2);
}

/**************************************************************/
int
cross_nearby(x, y)
    int       x, y;
{
    int       i;
    static int near = CROSS_SIZE + 10;

    for (i = 0; i < num_cross; i++) {
	if (x - near < cross_list[i].x &&
	    x + near > cross_list[i].x &&
	    y - near < cross_list[i].y &&
	    y + near > cross_list[i].y)
	    return (i);
    }
    return (0);
}

/**************************************************************/
void
move_cross(cnum, x, y)
    int       cnum, x, y;
{
    void      erase_cross();

    erase_cross(cross_list[cnum]);
    add_cross(cnum, x, y);
}

/**************************************************************/
void
erase_cross(cross)
    CROSS     cross;
{
    int       sx, sy, size;

    sx = cross.l1x1 - 3;
    sy = cross.l1y1 - 3;
    if (sx < 0)
	sx = 0;
    if (sy < 0)
	sy = 0;
    size = CROSS_SIZE * 2 + 5;

    /* redraw original image */
    if (segal.display_type == 0 && image != NULL)
	XPutImage(display, view_xid, gc, image, sx, sy,
		  sx, sy, size, size);
    if (segal.display_type == 1 && mask_image != NULL)
	XPutImage(display, view_xid, gc, mask_image, sx, sy,
		  sx, sy, size, size);
    if (segal.display_type == 2 && blend_image != NULL)
	XPutImage(display, view_xid, gc, blend_image, sx, sy,
		  sx, sy, size, size);
}

/**************************************************************/
void
threshold_proc(item, value, event)
    Panel_item item;
    int       value;
    Event    *event;
{
    static int old_val = -1;
    register int i, size;
    register u_char *iptr, *mptr;

    if (himage.fp == NULL)
	return;

    if (value == old_val && segal.changed)
	return;

    size = segal.rows * segal.cols;

    if (verbose)
	fprintf(stderr, "segal: threshold_proc: value: %d\n", value);

    set_watch_cursor();

    if (segal.display_type == 0) {
	segal.display_type = 1;
	xv_set(segal_win->display_type, PANEL_VALUE, 1, NULL);
    }
    iptr = himage.data[0];
    mptr = work_buf[0];

    for (i = 0; i < size; i++) {
	if (iptr[i] >= value)
	    mptr[i] = PVAL;	/* white */
	else
	    mptr[i] = 0;	/* black */
    }

    map_image_to_lut(mptr, mask_image->data, size);

    blend(himage.data[0], work_buf[0], (u_char *) blend_image->data, size);

    image_repaint_proc();
    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == TRUE) {
	zoom();
	edit_repaint_proc();
    }
    old_val = value;
    segal.changed = 1;		/* mask modified flag */

    unset_watch_cursor();

    /* get/keep caret */
    (void) xv_set(segal_win->controls, PANEL_CARET_ITEM,
		  segal_win->thresh_val, NULL);
}

/****************************************************************/
void
poly_proc(item, value)
    Panel_item item;
    int       value;
{
    if (value == 1) {
	segal.poly_flag = 1;
	(void) xv_set(segal_win->poly_mesg,
		    PANEL_LABEL_STRING, "Select points for polygon.", NULL);
	(void) xv_set(segal_win->show_poly_item, PANEL_INACTIVE, FALSE, NULL);
	(void) xv_set(segal_win->poly_fill_item, PANEL_INACTIVE, FALSE, NULL);
	(void) xv_set(segal_win->clear_item, PANEL_INACTIVE, FALSE, NULL);
    } else {
	segal.poly_flag = 0;
	(void) xv_set(segal_win->poly_mesg, PANEL_LABEL_STRING, " ", NULL);
	(void) xv_set(segal_win->show_poly_item, PANEL_INACTIVE, TRUE, NULL);
	(void) xv_set(segal_win->poly_fill_item, PANEL_INACTIVE, TRUE, NULL);
	(void) xv_set(segal_win->clear_item, PANEL_INACTIVE, TRUE, NULL);
    }
    image_repaint_proc();
}

/****************************************************************/
void
show_poly_proc()
{
    void      draw_poly();
    register int i;

    image_repaint_proc();

    /* redraw all crosses */
    for (i = 0; i < num_cross; i++)
	draw_cross(cross_list[i]);

    draw_poly();
}

/****************************************************************/
void
fill_proc()
{
    void      flood_poly();
    int       size;
    register int i, j;

    set_watch_cursor();
    flood_poly();

    /* set all locations reached in fill back to 0 */
    for (i = 0; i < segal.rows; i++)
	for (j = 0; j < segal.cols; j++)
	    if (work_buf[i][j] == FVAL || work_buf[i][j] == PLYVAL)
		work_buf[i][j] = 0;

    size = segal.rows * segal.cols;
    map_image_to_lut(work_buf[0], mask_image->data, size);

    blend(himage.data[0], work_buf[0], (u_char *) blend_image->data, size);

#ifdef DONT_NEED?
    segal.display_type = 2;
    (void) xv_set(segal_win->display_type, PANEL_VALUE, 2, NULL);
#endif

    image_repaint_proc();

    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == TRUE) {
	zoom();
	edit_repaint_proc();
    }
    segal.changed = 1;		/* mask modified flag */
    num_cross = 0;		/* reset cross list */

    unset_watch_cursor();
}

/****************************************************************/
void
clear_proc()
{
    image_repaint_proc();
    num_cross = 0;
}

/****************************************************************/
void
draw_poly()
{
    int       i;
    void      line_fill();

    for (i = 0; i < num_cross; i++) {
	if (i == num_cross - 1)	/* last point: close the polygon */
	    line_fill(work_buf, cross_list[i].x, cross_list[i].y,
		      cross_list[0].x, cross_list[0].y);
	else
	    line_fill(work_buf, cross_list[i].x, cross_list[i].y,
		      cross_list[i + 1].x, cross_list[i + 1].y);
    }
}

/****************************************************************/
void
flood_poly()
{
    stack_size = STACK_CHUNK;
    alloc_stack(stack_size);

    fill(0, 0);

    free((char *) stack);
}

/****************************************************************/

fill(x, y)
    int       x, y;
{
    register int xx, yy;

    sp = 0;			/* initialize stack pointer */
    push(-1, -1);		/* null stack */

    do {
start:
	work_buf[x][y] = FVAL;	/* mark work_buf */

	xx = x + 1;
	if (xx < segal.rows && work_buf[xx][y] != PLYVAL &&
	    work_buf[xx][y] != FVAL) {
	    push(x, y);
	    x++;
	    goto start;
	}
	xx = x - 1;
	if (xx > 0 && work_buf[xx][y] != PLYVAL && work_buf[xx][y] != FVAL) {
	    push(x, y);
	    x--;
	    goto start;
	}
	yy = y - 1;
	if (yy > 0 && work_buf[x][yy] != PLYVAL && work_buf[x][yy] != FVAL) {
	    push(x, y);
	    y--;
	    goto start;
	}
	yy = y + 1;
	if (yy < segal.cols && work_buf[x][yy] != PLYVAL &&
	    work_buf[x][yy] != FVAL) {
	    push(x, y);
	    y++;
	    goto start;
	}
	pop(&x, &y);

    } while (x >= 0);		/* neg x indicates empty stack */
    if (sp != 0)
	fprintf(stderr, "Error: stack not empty \n");

    return;
}

/***************************************************************/
push(i, j)
    int       i, j;
{
    sp++;

    if (sp >= stack_size) {
	stack_size += STACK_CHUNK;
#ifdef DEBUG
	fprintf(stderr, " increasing stack size to %d.. \n", stack_size);
#endif
	if ((stack = Realloc(stack, stack_size, STACK_ITEM)) == NULL)
	    perror("realloc");
    }
    stack[sp].i = i;
    stack[sp].j = j;
}

/***************************************************************/
pop(i, j)
    int      *i, *j;
{
    *i = stack[sp].i;
    *j = stack[sp].j;
    sp--;
}

/***************************************************************/
alloc_stack(st_size)		/* allocation stack for non-recursive
				 * flood-fill alg */
    int       st_size;
{
    if ((stack = Calloc(st_size, STACK_ITEM)) == NULL)
	perror("calloc: stack");
}

/********************************************************************/
void
line_fill(buf, x1, y1, x2, y2)	/* Bresenhams's scan conversion algorithm */
    u_char  **buf;
    int       x1, y1, x2, y2;

/* this code adapted from:   Digital Line Drawing
 *                           by Paul Heckbert
 * from "Graphics Gems", Academic Press, 1990
 */
{
    int       d, x, y, ax, ay, sx, sy, dx, dy;

    /* absolute value of a */
#ifndef ABS
#define ABS(a)          (((a)<0) ? -(a) : (a))
#endif

    /* take binary sign of a, either -1, or 1 if >= 0 */
#define SGN(a)          (((a)<0) ? -1 : 1)

    if (x1 == x2 && y1 == y2) {
	return;
    }
    dx = x2 - x1;
    ax = ABS(dx) << 1;
    sx = SGN(dx);

    dy = y2 - y1;
    ay = ABS(dy) << 1;
    sy = SGN(dy);

    x = x1;
    y = y1;
    if (ax > ay) {		/* x dominant */
	d = ay - (ax >> 1);
	for (;;) {
	    buf[y][x] = PLYVAL;	/* label buf with PLYVAL */
	    XDrawPoint(display, view_xid, gc, x, y);
	    if (x == x2)
		return;
	    if (d >= 0) {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    } else {			/* y dominant */
	d = ax - (ay >> 1);
	for (;;) {
	    buf[y][x] = PLYVAL;	/* label buf with PLYVAL */
	    XDrawPoint(display, view_xid, gc, x, y);
	    if (y == y2)
		return;
	    if (d >= 0) {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
}
