/*
 *  load_save.c   routines for loading and saving of images   -Brian Tierney
 */

#include "segal.h"

/* these need to be global */
extern pixedit_win_objects *edit_win;
extern view_win_objects *view_win;
extern segal_win_objects *segal_win;

/* Note: may want to add memory mapping to this some day
     for greater speed on multi-framed images */

struct header ihd, mhd;	/* image and mask hips header */

void      write_mask_frame();

/**********************************************************/
int
load_image()
{
/* returns: -1 if error,
            0 if file already loaded, and
            1 if successful load */

    char     *fname = NULL;
    static char prev_fname[80];
    void      segal_info_init(), allocate_image_data();
    int       fd;

    fname = (char *) xv_get(segal_win->image1, PANEL_VALUE, NULL);

    if (fname == NULL || strlen(fname) == 0)	/* check for blank name */
	return (-1);
    if (strcmp(fname, prev_fname) == 0)	/* only do if new file name */
	return (0);

    fprintf(stderr, " Loading image %s ... \n", fname);

    himage.fp = fopen(fname, "r");
    if (himage.fp == NULL) {
	fprintf(stderr, "\n error opening file %s \n\n", fname);
	return (-1);
    }
    fd = fileno(himage.fp);
    fread_header(fd, &ihd);
    himage.start = ftell(himage.fp);

    if (ihd.pixel_format != PFBYTE) {
	fprintf(stderr, "input sequence must be byte \n");
	return (-1);
    }
    fprintf(stderr, "image size: %d x %d, %d frame(s) \n",
	    ihd.rows, ihd.cols, ihd.num_frame);

    strcpy(himage.filename, fname);

    /* free and reallocate if size changes */
    if (segal.rows != ihd.rows || segal.cols != ihd.cols) {
	allocate_image_data(ihd.rows, ihd.cols);
    }
    if (read_2d_byte_array(himage.fp, himage.data, ihd.rows, ihd.cols) == -1)
	return (-1);

    /* intialize segal_info structure */
    segal_info_init(ihd);
    segal.display_type = (int) xv_get(segal_win->display_type, PANEL_VALUE, 0);

    strcpy(prev_fname, fname);
    return (1);
}


/**************************************************************/
int
load_mask(item)
    Panel_item item;
{
/* returns: -1 if error,
            0 if file already loaded, and
            1 if successful load */

    char     *fname = NULL;
    static char prev_fname[80];
    int       fd;
    void      segal_info_init();

    fname = (char *) xv_get(segal_win->image2, PANEL_VALUE, NULL);

    if (fname == NULL || strlen(fname) == 0)	/* check for blank name */
	return (-1);
    if (strcmp(fname, prev_fname) == 0)	/* only do if new file name */
	return (0);

    fprintf(stderr, " Loading mask %s ... \n", fname);

    hmask.fp = fopen(fname, "r+");	/* read/write */
    if (hmask.fp == NULL)
	return (-1);

    fd = fileno(hmask.fp);
    fread_header(fd, &mhd);
    hmask.start = ftell(hmask.fp);

    if (mhd.pixel_format != PFBYTE) {
	fprintf(stderr, "input sequence must be byte \n");
	return (-1);
    }
    if (verbose)
	fprintf(stderr, "mask size: %d x %d, %d frame(s) \n",
		mhd.rows, mhd.cols, mhd.num_frame);

    strcpy(hmask.filename, fname);

    if (himage.fp == NULL) {	/* if no image file */
	/* free and reallocate if size changes */
	if (segal.rows != mhd.rows || segal.cols != mhd.cols)
	    allocate_image_data(mhd.rows, mhd.cols);
    }
    if (read_2d_byte_array(hmask.fp, hmask.data, mhd.rows, mhd.cols) == -1)
	return (-1);

    if (himage.fp != NULL) {
	if (segal.rows != mhd.rows || segal.cols != mhd.cols ||
	    segal.frames != mhd.num_frame) {
	    fprintf(stderr, "Error: image and mask must be the same size!! \n");
	    segal.display_type = 0;
	    (void) xv_set(segal_win->display_type, PANEL_VALUE, 0, NULL);
	    return (-1);
	}
    }
    /* intialize segal_info structure if not done in load_image() */
    if (himage.fp == NULL) {
	segal_info_init(mhd);

	segal.display_type = 1;	/* must be set to mask */
	(void) xv_set(segal_win->display_type, PANEL_VALUE, 1, NULL);

    }
    bcopy((char *) hmask.data[0], (char *) work_buf[0],
	  segal.rows * segal.cols);

    strcpy(prev_fname, fname);
    return (1);
}

/**************************************************************/
static void
allocate_image_data(rows, cols)
    int       rows, cols;
{
    u_char  **alloc_2d_byte_array();

    if (segal.rows > 0) {	/* so don't do the 1st time through */
	free_2d_byte_array(himage.data);
	free_2d_byte_array(hmask.data);
	free_2d_byte_array(work_buf);
	himage.data = hmask.data = work_buf = NULL;
    }
    if (image != NULL) {
	XDestroyImage(image);
	image = NULL;
    }
    if (mask_image != NULL) {
	XDestroyImage(mask_image);
	mask_image = NULL;
    }
    if (blend_image != NULL) {
	XDestroyImage(blend_image);
	blend_image = NULL;
    }
    himage.data = alloc_2d_byte_array(rows, cols);

    hmask.data = alloc_2d_byte_array(rows, cols);

    work_buf = alloc_2d_byte_array(rows, cols);
}

/*************************************************************/
void
segal_info_init(hd)
struct header hd;
{				/* initializes globel structure of info used
				 * everywhere */
    char      segal_mesg[30];

    segal.rows = hd.rows;
    segal.cols = hd.cols;
    segal.frames = hd.num_frame;
    segal.curr_frame = 1;
    segal.poly_flag = 0;
    segal.changed = 0;
    segal.slider1 = (int) xv_get(segal_win->bg_slider, PANEL_VALUE, 0);
    segal.slider2 = 100 - (int) xv_get(segal_win->fg_slider, PANEL_VALUE, 0);
    make_blend_lut();

    sprintf(segal_mesg, "Image Size: %d x %d",
	    segal.rows, segal.cols);
    (void) xv_set(segal_win->size_mesg, PANEL_LABEL_STRING,
		  segal_mesg, NULL);
    sprintf(segal_mesg, "Current Frame: %d/%d",
	    segal.curr_frame, hd.num_frame);
    (void) xv_set(segal_win->frame_mesg, PANEL_LABEL_STRING,
		  segal_mesg, NULL);

}

/****************************************************/
void
step_up_proc(item, event)	/* load next frame */
    Panel_item item;
    Event    *event;
{
    char      segal_mesg[30];
    long      offset;

    if (segal.curr_frame < segal.frames) {
	set_watch_cursor();

	/* write mask frame */
	if (hmask.fp != NULL && segal.changed != 0) {
	    write_mask_frame(1);
	}
	segal.curr_frame++;
	sprintf(segal_mesg, "Current Frame: %d/%d",
		segal.curr_frame, segal.frames);
	(void) xv_set(segal_win->frame_mesg, PANEL_LABEL_STRING,
		      segal_mesg, NULL);

	if (himage.fp != NULL) {
	    offset = himage.start + ((segal.curr_frame - 1) *
				     (segal.rows * segal.cols));
	    if (fseek(himage.fp, offset, 0) == -1)
		perror("fseek");
#ifdef DEBUG
	    fprintf(stderr, " start of data at: %d,  this frame at: %d, \n",
		    himage.start, offset);
#endif
	    if (verbose)
		fprintf(stderr, " Loading frame %d ... \n", segal.curr_frame);
	    if (read_2d_byte_array(himage.fp, himage.data,
				   segal.rows, segal.cols) == -1)
		return;
	}
	if (hmask.fp != NULL) {
	    offset = hmask.start + ((segal.curr_frame - 1) *
				    (segal.rows * segal.cols));
	    if (fseek(hmask.fp, offset, 0) == -1)
		perror("fseek");
	    if (read_2d_byte_array(hmask.fp, hmask.data,
				   segal.rows, segal.cols) == -1) {
		hmask.fp = NULL;
		return;
	    }
	    bcopy((char *) hmask.data[0], (char *) work_buf[0],
		  segal.rows * segal.cols);
	}
	map_images();

	if (himage.data != NULL || hmask.data != NULL)
	    image_repaint_proc();

	if ((int) xv_get(view_win->win, XV_SHOW, NULL) == TRUE) {
	    zoom();
	    edit_repaint_proc();
	}
	unset_watch_cursor();
    }
}

/****************************************************/
void
step_down_proc(item, event)	/* load previous frame */
    Panel_item item;
    Event    *event;
{
    long      offset;
    char      segal_mesg[30];

    if (segal.curr_frame > 1) {
	set_watch_cursor();

	/* write mask frame */
	if (hmask.fp != NULL && segal.changed != 0) {
	    write_mask_frame(1);
	}
	segal.curr_frame--;
	sprintf(segal_mesg, "Current Frame: %d/%d",
		segal.curr_frame, segal.frames);
	(void) xv_set(segal_win->frame_mesg, PANEL_LABEL_STRING,
		      segal_mesg, NULL);

	if (himage.fp != NULL) {
	    offset = himage.start + ((segal.curr_frame - 1) *
				     (segal.rows * segal.cols));
	    if (fseek(himage.fp, offset, 0) == -1)
		perror("fseek");
	    if (verbose)
		fprintf(stderr, " Loading frame %d ... \n", segal.curr_frame);
	    if (read_2d_byte_array(himage.fp, himage.data,
				   segal.rows, segal.cols) == -1)
		return;
	}
	if (hmask.fp != NULL) {
	    offset = hmask.start + ((segal.curr_frame - 1) *
				    (segal.rows * segal.cols));
	    if (fseek(hmask.fp, offset, 0) == -1)
		perror("fseek");
	    if (read_2d_byte_array(hmask.fp, hmask.data,
				   segal.rows, segal.cols) == -1) {
		hmask.fp = NULL;
		return;
	    }
	    bcopy((char *) hmask.data[0], (char *) work_buf[0],
		  segal.rows * segal.cols);
	}
	map_images();

	if (himage.data != NULL || hmask.data != NULL)
	    image_repaint_proc();

	if ((int) xv_get(view_win->win, XV_SHOW, NULL) == TRUE) {
	    zoom();
	    edit_repaint_proc();
	}
	unset_watch_cursor();
    }
}

/*****************************************************************/
void
mask_save_proc(item, event)	/* save current frame (SAVE button) */
    Panel_item item;
    Event    *event;
{
    char     *fname;

    if (verbose)
	fputs("pixedit: save_proc\n", stderr);

    fname = (char *) xv_get(segal_win->image2, PANEL_VALUE, NULL);

    if (hmask.fp == NULL)	/* mask not yet loaded */
	create_mask_file(item, fname);
    else {
	if (strcmp(fname, hmask.filename) == 0)	/* check if new file name */
	    write_mask_frame(0);
	else
	    create_mask_file(item, fname);
    }
}

/**********************************************************/
int
create_mask_file(item, fname)
    Panel_item item;
    char     *fname;
{
    void      write_blank_frames(), write_mask_header();

    if (check_valid_mask_name(item, fname) < 0)
	return (-1);

    fprintf(stderr, "creating mask file: %s \n", fname);

    /* create new file */
    if ((hmask.fp = fopen(fname, "w+")) == NULL) {
	fprintf(stderr, "\n error opening file %s \n\n", fname);
	return (-1);
    }
    write_mask_header();

    if (segal.frames > 1)
	write_blank_frames();

    write_mask_frame(0);

    strcpy(hmask.filename, fname);

    return (0);
}

/**********************************************************/
void
write_mask_frame(step_flag)
    int       step_flag;
{				/* called from step_up, step_down, and
				 * mask_save_proc */
    long      offset;

    set_watch_cursor();

    offset = hmask.start + ((segal.curr_frame - 1) *
			    (segal.rows * segal.cols));

    if (offset <= 0) {
	fprintf(stderr, "Error: invalid offset for write \n");
	return;
    }
    if (fseek(hmask.fp, offset, 0) == -1)
	perror("fseek");
    if (verbose)
	fprintf(stderr, " start: %d, frame size: %d \n", hmask.start,
		segal.rows * segal.cols);
    fprintf(stderr, " writing mask frame %d, offset: %d, to file %s... \n",
	    segal.curr_frame, offset, hmask.filename);
    write_2d_byte_array(hmask.fp, work_buf, segal.rows, segal.cols);

    segal.changed = 0;

    if (!step_flag)		/* indicates came from step_up or step_down
				 * proc */
	bcopy((char *) work_buf[0], (char *) hmask.data[0],
	      segal.rows * segal.cols);

    unset_watch_cursor();
}

/*******************************************************/
void
write_blank_frames()
{				/* creates mask image file that is all zeros */
    int       i;
    char     *buf;

    buf = Calloc(segal.rows * segal.cols, char);

    fprintf(stderr, " Creating blank mask frames... \n");

    for (i = 0; i < segal.frames; i++)
	Fwrite(buf, sizeof(char), segal.rows * segal.cols, hmask.fp);

    cfree((char *) buf);

}

/*********************************************************/
int
check_valid_mask_name(item, fname)
    Panel_item item;
    char     *fname;
{
    int       result;
    Panel     panel = (Panel) xv_get(item, PANEL_PARENT_PANEL);

    if (fname == NULL || (strlen(fname) == 0)) {
	result = notice_prompt(panel, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       "Error: must enter mask file name!", NULL,
			       NOTICE_BUTTON_YES, "OK",
			       NULL);
	return (-1);
    }
    if (strcmp(fname, himage.filename) == 0) {
	result = notice_prompt(panel, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       "Error: Mask file name must be different",
			       "than the image file name!", NULL,
			       NOTICE_BUTTON_YES, "OK",
			       NULL);
	return (-1);
    }

    if ((hmask.fp = fopen(fname, "r")) != NULL) {	/* check if exists */
	result = notice_prompt(panel, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       "Overwrite existing file?", NULL,
			       NOTICE_BUTTON_YES, "Yes",
			       NOTICE_BUTTON_NO, "No",
			       NULL);

	fclose(hmask.fp);
	if (result == NOTICE_NO)
	    return (-1);
    }
    return (0);
}

/*********************************************************/
void
write_mask_header()
{
    int       fd;

    if (hmask.fp == NULL) {
	fprintf(stderr, "ERROR: attempting to write header to un-opened file \n");
	return;
    }
    if (himage.fp != NULL)
	mhd = ihd;     /* add to image header */
    else
	init_header(&mhd, "segal", "", segal.frames, "", segal.rows, segal.cols,
		    8, 0, PFBYTE, "");

    fd = fileno(hmask.fp);
    update_header(&mhd, ac, av);
    fwrite_header(fd, &mhd);
    hmask.start = ftell(hmask.fp);
}
