/* $Id: ps.c,v 1.22 90/06/30 16:26:06 pturner Exp Locker: pturner $
 *
 * driver for postscript printer
 *
 * courtesy of:
 *
 * Jim Hudgens
 * hudgens@ray.met.fsu.edu
 *
 * Further modifications by,
 * Ole Holm Nielsen
 * ohnielse@ltf.dth.dk
 *
 */

#include <stdio.h>
#include "externs.h"

extern char version[];
extern double charsize;
extern double devcharsize;
extern int ptofile;
extern char printstr[];

/*
 * printer control string
 */
#ifndef PS_PRSTR
char ps_prstr[128] = "/usr/ucb/lpr -h";

#else
char ps_prstr[128] = PS_PRSTR;

#endif

/* postscript page at scale = 0.25 */

#define PSXMIN 150
#define PSXMAX 2200
#define PSYMIN 150
#define PSYMAX 3200
#define DXPS 2050
#define DYPS 3050
#define CHARS 1.5

#define PSXMINP 150
#define PSXMAXP 2200
#define PSYMINP 550
#define PSYMAXP 2700
#define DXPSP 2050
#define DYPSP 2050
#define CHARSP 1.25

#define MINCOLOR 0
#define MAXCOLOR 9
#define MAXLINESTYLE 14

#define PORTRAIT 0
#define LANDSCAPE 1

static int psxmin = PSXMIN;
static int psxmax = PSXMAX;
static int psymin = PSYMIN;
static int psymax = PSYMAX;
static int psdx = DXPS;
static int psdy = DYPS;
static int pscolor = -1;
static int psdmode;
static int psfont = 0;
static double pscharsize = 1.5;
static int pslinestyle;
static char *fname;
static int orientflag = PORTRAIT;
#define MAXPATHLEN 1000		/* MAXPATHLEN points in a path between strokes*/
static int pathlength = 0;
static int x_current = 99999, y_current = 99999;
double xconv(), yconv();
static FILE *psout;

static void stroke()
{
    if (pathlength) {
         fprintf(psout, "stroke\n");
         x_current = 99999;
         y_current = 99999;
         pathlength = 0;
    }
}

int pssetmode(mode)
    int mode;
{
    char tbuf[128];
    char *mktemp();

    if (mode % 2) {
	if (!ptofile) {
	    fname = mktemp("/usr/tmp/XXXXXX");
	} else {
	    fname = printstr;
	}
	if ((psout = fopen(fname, "w")) == NULL) {
	    return 0;
	}
    }
    switch (mode) {
        case 1:			/* PS landscape */
	    orientflag = LANDSCAPE;
	    pscharsize = CHARS;
	    psxmin = PSXMIN;
	    psxmax = PSXMAX;
	    psymin = PSYMIN;
	    psymax = PSYMAX;
	    psdx = DXPS;
	    psdy = DYPS;
	    break;
        case 3:			/* PS portrait */
	    orientflag = PORTRAIT;
	    pscharsize = CHARSP;
	    psxmin = PSXMINP;
	    psxmax = PSXMAXP;
	    psymin = PSYMINP;
	    psymax = PSYMAXP;
	    psdx = DXPSP;
	    psdy = DYPSP;
	    break;
        case 2:
        case 4:
            stroke();
	    fprintf(psout, "showpage\n");
	    fclose(psout);
	    if (!ptofile) {
	        sprintf(tbuf, "%s %s", ps_prstr, fname);
	        system(tbuf);
	        unlink(fname);
	    }
	    orientflag = PORTRAIT;
	    break;
    }
}

void drawps(x2, y2, mode)
    int x2, y2, mode;
{
    register int xtmp, ytmp;

    if (x2 < 0 || y2 < 0)	/* Eliminate garbage on output */
	return;

    if (orientflag == LANDSCAPE) {
	xtmp = y2;
	ytmp = psymax - x2;
    } else {
	xtmp = x2;
	ytmp = y2;
    }

    if (mode)
        fprintf(psout, "%d %d l\n", xtmp, ytmp);		/* lineto */
    else {
        /* Avoid excessive moveto's generated by grtool */
        if (xtmp == x_current && ytmp == y_current)
            return;
        fprintf(psout, "%d %d m\n", xtmp, ytmp);		/* moveto */
    }
    pathlength++;
    x_current = xtmp;
    y_current = ytmp;

    /* Printers have some maximum number of points in a path.
       See PostScript Language Reference Manual (Red book), p. 261.
       Hence the fix that follows */

    if (pathlength > MAXPATHLEN) {
        fprintf(psout, "%% Fix for MAXPATHLEN\n");
        stroke();
    	fprintf(psout, "%d %d m\n", xtmp, ytmp);	/* moveto */
    }
}

int xconvps(x)
    double x;
{
    if (orientflag == LANDSCAPE) {
	return ((int) (psymin + psdy * xconv(x)));
    } else {
	return ((int) (psxmin + psdx * xconv(x)));
    }
}

int yconvps(y)
    double y;
{
    if (orientflag == LANDSCAPE) {
	return ((int) (psxmin + psdx * yconv(y)));
    } else {
	return ((int) (psymin + psdy * yconv(y)));
    }
}

void pssetfont(n)
    int n;
{
    hselectfont(psfont = n);
}

int pssetcolor(c)
    int c;
{
    if (c != pscolor) {
        stroke();
	c = c % MAXCOLOR;
	fprintf(psout, "%d setlinewidth\n", 2 * (c - 1) + 1);
    }
    pscolor = c;
    return c;
}

void psdrawtic(x, y, dir, updown)
    int x, y, dir, updown;
{
    switch (dir) {
    case 0:
	switch (updown) {
	case 0:
	    drawps(x, y, 0);
	    drawps(x, y + devxticl, 1);
	    break;
	case 1:
	    drawps(x, y, 0);
	    drawps(x, y - devxticl, 1);
	    break;
	}
	break;
    case 1:
	switch (updown) {
	case 0:
	    drawps(x, y, 0);
	    drawps(x + devyticl, y, 1);
	    break;
	case 1:
	    drawps(x, y, 0);
	    drawps(x - devyticl, y, 1);
	    break;
	}
	break;
    }
}

int pssetlinestyle(style)
    int style;
{
    if (style == pslinestyle)
	return (pslinestyle);
    stroke();

    switch (style) {
        case 1:			/* solid */
	    fprintf(psout, "[] 0 setdash\n");
	    break;
        case 2:			/* dotted */
	    fprintf(psout, "[12 40] 0 setdash\n");
	    break;
        case 3:			/* long dash */
	    fprintf(psout, "[40 20] 0 setdash\n");
	    break;
        case 4:			/* short dash */
	    fprintf(psout, "[20 20] 0 setdash\n");
	    break;
        case 5:			/* dot-dashed */
	    fprintf(psout, "[40 20 12 20] 0 setdash\n");
	    break;
        }
    return (pslinestyle = style);
}

void dispstrps(x, y, rot, s)
    int x, y, rot;
    char *s;
{
    puthersh(x, y, pscharsize * charsize, rot, pscolor, vector, s);
}

void psleavegraphics()
{
    pssetmode(psdmode + 1);
}

/*           postscript initialization routine  */
psinitgraphics(dmode)
    int dmode;
{

    psdmode = dmode;
    pssetmode(psdmode);
    devconvx = xconvps;
    devconvy = yconvps;
    vector = drawps;
    devwritestr = dispstrps;
    devsetcolor = pssetcolor;
    devsetfont = pssetfont;
    devsetline = pssetlinestyle;
    devdrawtic = psdrawtic;
    devleavegraphics = psleavegraphics;
    devcharsize = pscharsize;
    devxticl = 20;
    devyticl = 20;
    devarrowlength = 20;

    fprintf(psout, "%%! PostScript\n");
    fprintf(psout, "%%%%BoundingBox: %d %d %d %d\n",
        PSXMINP * 72 / 300, PSYMINP * 72 / 300,
	PSXMAXP * 72 / 300, PSYMAXP * 72 / 300);
    fprintf(psout, "%%%% Creator: %s\n", version);

    fprintf(psout, "/m {moveto} bind def\n");
    fprintf(psout, "/l {lineto} bind def\n");

    fprintf(psout, "0.25 0.25 scale\n");
    fprintf(psout, "1 setlinecap\n");
    pssetcolor(1);
    setfont(2);
    setcolor(1);
    setlinestyle(0);
}
