/*

    compile with:

  	cc -o sawrs sawrs2.0.c -lm [-lc_s]

------------------------------------------------------------------------------

  	sawrs
 
	(sawgrass for Rayshade)

 	Copyright (C) 1992, Lawrence K. Coffin.
	All rights reserved.
 
	This software may be freely copied, modified, and redistributed
	provided that this copyright notice is preserved on all copies.
 
	You may not distribute this software, in whole or in part, as part of
	any commercial product without the express consent of the authors.
  
	There is no warranty or other guarantee of fitness of this software
	for any purpose.  It is provided solely "as is".

   Useage:

        Useage: sawrs gravity numblades numsegments > file.obj

                gravity changes the ammount of arc in the blade:
                        ~14 is a good starting point.
                numblades is the number of blades in the tuft.
                numsegments is the number of cones that make up each blade:
                        ~15 is a good start.

	NOTE: prints object to stdout

   This program creates a tuft of "sawgrass" to be included into rayshade
	scenes.  It randomly picks a point on the xy plane within a circle
	with a radius based on the number of blades in the tuft.  The grass
	blades are then formed by cones whose bases are centered along a
	parabola defined by a random stating vector and a random "gravity"
	(-z acceleration).  The resulting object can be used in a scene with
	"object <surface> grasstuft".  The resulting object will be
	proportionaly scaled to fit the tuft between heights of 1 and -1.


	The cylinders that make up the grass are stored in a boundstype structure
	with rad1 the radius at x1,y1,z1 and rad2 the radius at x2,y2,z2.
	The cylinders are then sorted into a nearest neighbor tree using the
	x1,y1,z1 point to determine closest neighbors. (See the hfsow header for more
	detail on the sorting routines)




	Please send all comments, suggestions, and bug reports to:

	lcoffin@clciris.chem.umr.edu

*/


#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#define abs(x)  ((x) < 0 ? -(x) : (x))
#define RADIUS 0.3

typedef struct _bounds {
	float rad1, rad2;
	float x1, y1, z1, x2, y2, z2;
	int picked;
	struct _bounds *left, *right;
} boundstype;

typedef struct _list {
	int number, count;
	boundstype *bounds;
} listtype;


listtype *list;

int sortbyx()
{
	boundstype temp;
	int front, min, i;
	extern listtype *list;

	for (front = 0; front < list->number - 1; front++){
		min = front;
		for (i = front+1; i < list->number; i++){
			if (list->bounds[i].x1 < list->bounds[min].x1){
				min = i;
			}
		}
		temp = list->bounds[front];
		list->bounds[front] = list->bounds[min];
		list->bounds[min] = temp;
	}


}


int sortobjs()
{
	int upper, lower, odd, current, best, numbounds, i;
	float dx, dy, dz, mindx, mindy, mindz;
	float dist, mindist;
	extern listtype *list;
	boundstype *tempbounds;

	if (list->number == 1){
		return;
	}

	tempbounds = list->bounds;
	numbounds = list->number;

	odd = (list->number)%2;
	list->number = (list->number)/2 + odd;


	list->bounds = (boundstype *)malloc(list->number*sizeof(boundstype));
	for (i = 0; i < list->number; i++){
		list->bounds[i].x1 = 0;
		list->bounds[i].y1 = 0;
		list->bounds[i].z1 = 0;
		list->bounds[i].x2 = 0;
		list->bounds[i].y2 = 0;
		list->bounds[i].z2 = 0;
		list->bounds[i].rad1 = 0;
		list->bounds[i].rad2 = 0;
		list->bounds[i].picked = 0;
		list->bounds[i].left = NULL;
		list->bounds[i].right = NULL;

	}

	current = 0;
	
	for (upper = 0; upper < (list->number) - odd; upper++){
		while(tempbounds[current].picked && current < numbounds - odd){
			current++;
		}
		lower = current+1;
		while(tempbounds[lower].picked && lower < numbounds - odd){
			lower++;
		}
		if(lower > numbounds || current > numbounds){
			fprintf(stderr,"current or lower is greater than numbounds\n");
		}
		best = lower;
		dx = (tempbounds[lower].x1) - (tempbounds[current].x1);
		dy = (tempbounds[lower].y1) - (tempbounds[current].y1);
		dz = (tempbounds[lower].z1) - (tempbounds[current].z1);
		mindist = sqrt(dx*dx + dy*dy + dz*dz);
		
		for(lower++ ; lower < numbounds; lower++){
			if (tempbounds[lower].picked == 0){
				dx = (tempbounds[lower].x1) - (tempbounds[current].x1);
				dy = (tempbounds[lower].y1) - (tempbounds[current].y1);
				dz = (tempbounds[lower].z1) - (tempbounds[current].z1);
				dist = sqrt(dx*dx + dy*dy + dz*dz);
				if(dx > mindist){
					break;
				}
				else if (dist < mindist){
					best = lower;
					mindist = dist;
					mindx = dx;
					mindy = dy;
					mindz = dz;
				}
			}
		}
		list->bounds[upper].picked = 0;
		list->bounds[upper].left = &(tempbounds[current]);
		list->bounds[upper].right = &(tempbounds[best]);
		list->bounds[upper].x1 = tempbounds[current].x1 + mindx/2.;
		list->bounds[upper].y1 = tempbounds[current].y1 + mindy/2.;
		list->bounds[upper].z1 = tempbounds[current].z1 + mindz/2.;

		tempbounds[current].picked = 1;
		tempbounds[best].picked = 1;
	}
	if (odd){
		for(current = 0; current < numbounds; current++){
			if(tempbounds[current].picked == 0){
				break;
			}
		}
		list->bounds[upper].left = tempbounds[current].left;
		list->bounds[upper].right = tempbounds[current].right;
		list->bounds[upper].picked = 0;
		list->bounds[upper].x1 = tempbounds[current].x1;
		list->bounds[upper].y1 = tempbounds[current].y1;
		list->bounds[upper].z1 = tempbounds[current].z1;
		list->bounds[upper].x2 = tempbounds[current].x2;
		list->bounds[upper].y2 = tempbounds[current].y2;
		list->bounds[upper].z2 = tempbounds[current].z2;
		list->bounds[upper].rad1 = tempbounds[current].rad1;
		list->bounds[upper].rad2 = tempbounds[current].rad2;
	}
	sortbyx();
	sortobjs();

}

printobjs(boundstype *bounds)
{
	if(bounds->left == NULL){
		printf("\tcone %.4f  %.4f %.4f %.4f  %.4f  %.4f %.4f %.4f\n",
			bounds->rad1,  
			bounds->x1,bounds->y1,bounds->z1,                               /* first base */
			bounds->rad2,
			bounds->x2,bounds->y2,bounds->z2);                              /* second base */
		fflush(stdout);
		free(bounds);
		return;
	}

	printf("grid 1 1 1\n");
	printobjs(bounds->left);
	printobjs(bounds->right);
	printf("end\n");
	free(bounds);
}





main(int argc, char *argv[])
{

	int numblades, maxblades, i;
	int numseg, maxseg;
	float xvel, yvel, zvel, x, y, z;
	float accel;
	float oldx, oldy, oldz;
	float startx, starty;
	float maxalt, minalt;
	extern listtype *list;

	if(argc < 4) {
		fprintf(stderr,"\n\tUseage: %s gravity numblades numsegments\n\n",argv[0]);
		fprintf(stderr,"\t\tgravity changes the ammount of arc in the blade:\n\t\t\t~14 is a good starting point.\n");
		fprintf(stderr,"\t\tnumblades is the number of blades in the tuft.\n");
		fprintf(stderr,"\t\tnumsegments is the number of cones that make up each blade:\n\t\t\t~15 is a good start.\n\n");
		exit(1);
	}

	accel = -(float)atoi(argv[1])/100.0;
	maxblades = atoi(argv[2]);
	maxseg = atoi(argv[3]);

	list = (listtype *)malloc(sizeof(listtype));

	list->number = maxblades*maxseg;

	list->bounds = (boundstype *)malloc((list->number)*sizeof(boundstype));
	for (i = 0; i < list->number; i++){
		list->bounds[i].x1 = 0;
		list->bounds[i].y1 = 0;
		list->bounds[i].z1 = 0;
		list->bounds[i].x2 = 0;
		list->bounds[i].y2 = 0;
		list->bounds[i].z2 = 0;
		list->bounds[i].rad1 = 0;
		list->bounds[i].rad2 = 0;
		list->bounds[i].picked = 0;
		list->bounds[i].left = NULL;
		list->bounds[i].right = NULL;

	}
	list->count = 0;

	srand(time(0));


	maxalt = 0.;
	minalt = 0.;
	numblades = 0;
	while(numblades < maxblades){

	/* randomly pick a starting x and y coordinate */
	startx = (float)maxblades*(float)rand()/(10.0*(float)RAND_MAX) - maxblades/20;
	starty = (float)maxblades*(float)rand()/(10.0*(float)RAND_MAX) - maxblades/20;

		oldx = startx;
		oldy = starty;
		oldz = 0;
		numseg = 1;

		/* randomly pick initial velocities */
		xvel = 2*(float)rand()/(float)RAND_MAX-1;
		yvel = 2*(float)rand()/(float)RAND_MAX-1;
		zvel = (float)rand()/(float)RAND_MAX+numblades%4+2; 
			/* numblades%4+2 gives every 4 blades a different velocity */
		
		while(numseg <= maxseg){
			x = xvel*numseg + startx; /* X = Xi + Vt */
			y = yvel*numseg + starty; /* Y = Yi + Vt */
			z = accel*numseg*numseg + zvel*numseg; /* Z = Zi + Vt + -(G)t^2 (Zi = 0) */

			list->bounds[list->count].rad1 = (RADIUS - RADIUS*((float)(numseg-1)/(float)maxseg));  /* larger radius */
			list->bounds[list->count].rad2 = (RADIUS - RADIUS*((float)(numseg)/(float)maxseg));	/* smaller radius */
			list->bounds[list->count].x1 = oldx;
			list->bounds[list->count].y1 = oldy;
			list->bounds[list->count].z1 = oldz;
			list->bounds[list->count].x2 = x;
			list->bounds[list->count].y2 = y;
			list->bounds[list->count].z2 = z;
			list->bounds[list->count].picked = 0;
			list->bounds[list->count].left = NULL;
			list->bounds[list->count].right = NULL;
			list->count++;

			numseg++;
			if(z > maxalt){  /* keep track of min and max alts */
				maxalt = z;
			}
			else if(z < minalt){
				minalt = z;
			}
			oldx = x;
			oldy = y;
			oldz = z;
		}
		numblades++;
	}

	sortbyx();


	sortobjs();

	printf("name grass_tuft\n"); /* start the un-normalized object */

	printobjs(list->bounds);

	printf("\n"); /* end the un-normalized object */

	printf("\nname grasstuft list\n"); /* start the normalized object */

	if(abs(minalt) > abs(maxalt)) maxalt = abs(minalt);

	/* print out the un-normalized object scaled to fit between z = 1 and -1 */
	printf("\tobject grass_tuft scale (1/%f) (1/%f) (1/%f)\n", maxalt, maxalt, maxalt);
	printf("end\n"); /* end the normalized object */
	exit(0);  /* Bye!! */
}
