#include <RpSoftPullMan.h>
#include <assert.h>

#define CANDIDATE_LIMIT (32)

//
// required for reporting of statistics
//
#ifdef RpSHOWSTATS
#include <iostream.h>
#endif


// ************************************************************************
// Splay trees: a fast associative data structure.
//
// References:
//    Daniel Sleator, sleator@cs.cmu.edu
//    An Implementation of Top-Down Splaying (March 1992)
//    ftp://spade.pc.cs.cmu.edu/usr/sleator/public/
//
// Original:
//    Sleator & Tarjan, Self-Adjusting Binary Search Trees
//    JACM Volume 32, No 3, July 1985, pp 652-686.
//

RpSoftPullMan::MapNode::MapNode( const requestKey& setKey )
{
   reset( setKey );
}


RpSoftPullMan::MapNode::~MapNode( void )
{
   return;
}


void
RpSoftPullMan::MapNode::reset( const requestKey& key )
{
   _pLeft = 0;
   _pRight = 0;
   _key = key;
   return;
}

// ------------------------------------------------------------------------


RpSoftPullMan::Map::Map( void )
{
   _pRoot = 0;
   _uSize = 0;
   return;
}


RpSoftPullMan::Map::~Map( void )
{
   while ( _pRoot )
      erase( _pRoot->_key );

   assert( (_uSize == 0) && ( _pRoot == 0 ) );
   return;
}


void
RpSoftPullMan::Map::insert( RpSoftPullMan::MapNode *pNewNode )
{
   RpSoftPullMan::MapNode		*pOldTree;
   
   pOldTree = _pRoot;
   
   if ( pNewNode )
   {
      pOldTree = splay( pNewNode->_key, pOldTree );
      if ( pOldTree )
      {
	 	 
	 if ( pNewNode->_key < pOldTree->_key )
	 {
            pNewNode->_pLeft = pOldTree->_pLeft;
            pNewNode->_pRight = pOldTree;
            pOldTree->_pLeft = 0;
            _uSize++;
	 } else if ( pNewNode->_key > pOldTree->_key ) {
            pNewNode->_pRight = pOldTree->_pRight;
            pNewNode->_pLeft = pOldTree;
            pOldTree->_pRight = 0;
            _uSize++;
	 }
	 
      } else {
      
         assert( pNewNode->_pLeft == 0 );
         assert( pNewNode->_pRight == 0 );
         _uSize = 1;         
      }

      _pRoot = pNewNode;
   }
   
   return;
}



void
RpSoftPullMan::Map::erase( const requestKey& key )
{
   MapNode *pCandidate;
   
   if ( _pRoot )
   {
      pCandidate = splay( key, _pRoot );
      
      if ( pCandidate->_key == key )
      {
         if (!(pCandidate->_pLeft))
         {
            _pRoot = pCandidate->_pRight;
         } else {
            _pRoot = splay( key, pCandidate->_pLeft );
            _pRoot->_pRight = pCandidate->_pRight;
         }
         
         --_uSize;
         pCandidate->deleteInstance();
      }
      
   }
   
   return;
}


RpSoftPullMan::MapNode *
RpSoftPullMan::Map::find( const requestKey& key )
{
   MapNode *pNewRoot;
   MapNode *pRet = 0;
   
   if ( _pRoot )
   {
      pNewRoot = splay( key, _pRoot );
      if ( key == pNewRoot->_key )      
         pRet = pNewRoot;
      
      _pRoot = pNewRoot;      
   }
   
   return ( pRet );
}



void
RpSoftPullMan::Map::swapContents( RpSoftPullMan::Map& src )
{
   MapNode	*pTemp;
   
   pTemp = _pRoot;
   _pRoot = src._pRoot;
   src._pRoot = pTemp;
   
   _uSize ^= src._uSize;
   src._uSize ^= _uSize;
   _uSize ^= src._uSize;
   
   return;
}


//
// note: requires a (*cmp)(MapNode*,MapNode*) function which
//    returns a negative if the first item is lessthan;.
//
RpSoftPullMan::MapNode *
RpSoftPullMan::Map::min( RpSoftPullMan::MapNode *pRoot, int(*cmp)(RpSoftPullMan::MapNode*,RpSoftPullMan::MapNode*))
{
   MapNode *pRet;
   MapNode *pTemp;
   
   assert( cmp );
   pRet = pRoot;
   if ( pRoot )
   {
      pTemp = min( pRoot->_pLeft, cmp );
      if ( pTemp )
         if ( ((*cmp)( pTemp, pRet )) < 0 )
            pRet = pTemp;
            
      pTemp = min( pRoot->_pRight, cmp );      
      if ( pTemp )
         if ( ((*cmp)( pTemp, pRet )) < 0 )
            pRet = pTemp;
   }
   
   return ( pRet );
}



RpSoftPullMan::MapNode *
RpSoftPullMan::Map::splay( const requestKey& key, MapNode *pTree )
{
   MapNode 	N( key );
   MapNode	*l, *r, *y;
   
   if ( pTree )
   {
      N._pLeft = 0;
      N._pRight = 0;      
      l = r = &N;
      
      while ( -1 )
      {
         if ( key == pTree->_key ) // equal
         {
            break; // while
         } else if ( key < pTree->_key ) { // less than
         
            if (!( pTree->_pLeft ))
               break; // while
               
            if ( key < pTree->_pLeft->_key )
            {
               y = pTree->_pLeft;
               pTree->_pLeft = y->_pRight;
               y->_pRight = pTree;
               pTree = y;
               
               if ( !(pTree->_pLeft ))
                  break; // while
            }
            
            r->_pLeft = pTree;
            r = pTree;
            pTree = pTree->_pLeft;
            
         } else { // greater than
         
            if ( !(pTree->_pRight ))
               break; // while
              
            if ( key > pTree->_pRight->_key )
            {
               y = pTree->_pRight;
               pTree->_pRight = y->_pLeft;
               y->_pLeft = pTree;
               pTree = y;
               if ( !(pTree->_pRight ))
                  break; // while
            }
            
            l->_pRight = pTree;
            l = pTree;
            pTree = pTree->_pRight;
            
         }
         
      } // while
      
      l->_pRight = pTree->_pLeft;
      r->_pLeft = pTree->_pRight;
      pTree->_pLeft = N._pRight;
      pTree->_pRight = N._pLeft;
      
   }
   
   return ( pTree );
}

// ************************************************************************

unsigned
RpSoftPullMan::candidateElement::ca_memmgr_count = 0;

RpSoftPullMan::candidateElement *
RpSoftPullMan::candidateElement::ca_memmgr_ppStore[ CAE_MEMMGR ];


RpSoftPullMan::candidateElement *
RpSoftPullMan::candidateElement::newInstance( const requestKey& key )
{
   candidateElement *pRet = 0;
   
   if ( ca_memmgr_count )
   {
      ca_memmgr_count--;
      pRet = ca_memmgr_ppStore[ ca_memmgr_count ];
      assert( pRet );
      
      pRet->reset( key );      
   } else
      pRet = new candidateElement( key );
   
   return ( pRet );
}



void
RpSoftPullMan::candidateElement::deleteInstance( void )
{
   if ( ca_memmgr_count < CAE_MEMMGR )
   {
      ca_memmgr_ppStore[ ca_memmgr_count ] = this;
      ca_memmgr_count++;
        
   } else
      delete this;
   
   return;
}


RpSoftPullMan::candidateElement *
RpSoftPullMan::candidateElement::newInstance( const candidateElement& src )
{
   candidateElement *pRet;
   
   pRet = newInstance( src._key );
   if ( pRet )
   {
      pRet->priority = src.priority;
      pRet->caller = src.caller;
      pRet->cost = src.cost;
   }
   
   return( pRet );
}


RpSoftPullMan::candidateElement *
RpSoftPullMan::candidateElement::newInstance( const cacheElement& src )
{
   candidateElement	*pRet;
   
   pRet = newInstance( src._key );
   if ( pRet )
   {
      pRet->priority = src.priority;
      pRet->caller = src.caller;
      pRet->cost = src.cost;
   }
   
   return( pRet );
}


// ************************************************************************

unsigned
RpSoftPullMan::cacheElement::ch_memmgr_count = 0;

RpSoftPullMan::cacheElement *
RpSoftPullMan::cacheElement::ch_memmgr_ppStore[ CHE_MEMMGR ];

RpSoftPullMan::cacheElement *
RpSoftPullMan::cacheElement::newInstance( const requestKey& key )
{
   cacheElement	*pRet = 0;
   
   if ( ch_memmgr_count )
   {
      ch_memmgr_count--;
      pRet = ch_memmgr_ppStore[ ch_memmgr_count ];      
      assert( pRet );
      
      pRet->reset( key );      
   } else
      pRet = new cacheElement( key );
      
   return( pRet );
}


void
RpSoftPullMan::cacheElement::deleteInstance( void )
{
   if ( ch_memmgr_count < CHE_MEMMGR )
   {
      ch_memmgr_ppStore[ ch_memmgr_count ] = this;
      ch_memmgr_count++;
      
      if ( pTile )
         pTile->deleteTile();
         
      pTile = 0;
      
   } else
      delete this;
   
   return;
}


RpSoftPullMan::cacheElement *
RpSoftPullMan::cacheElement::newInstance( const candidateElement& src )
{
   cacheElement		*pRet = 0;
   RpImageArea		area;
   
   pRet = newInstance( src._key );
   if ( pRet )
   {
      pRet->priority = src.priority;
      pRet->caller = src.caller;
      pRet->cost = src.cost;

      area = pRet->_key.p->getArea();
      area.z = src._key.z;
      area.c = src._key.c;

      if ( pRet->_key.p )
	 pRet->pTile = RpImageTile::newTile( area, pRet->_key.p->getType() );
      else
	 pRet->pTile = 0;
   }
   
   return( pRet );
}


RpSoftPullMan::cacheElement *
RpSoftPullMan::cacheElement::newInstance( const cacheElement& src )
{
   cacheElement	*pRet = 0;
   
   pRet = newInstance( src._key );
   if ( pRet )
   {
      pRet->priority = src.priority;
      pRet->caller = src.caller;
      pRet->cost = src.cost;

      if ( pRet->_key.p && src.pTile )
      {
	 pRet->pTile = src.pTile;
	 pRet->pTile->registerReference();
      }   
      else
	 pRet->pTile = 0;
   }
   
   return( pRet );
}

// ************************************************************************

unsigned
RpSoftPullMan::topologyElement::te_memmgr_count = 0;

RpSoftPullMan::topologyElement *
RpSoftPullMan::topologyElement::te_memmgr_ppStore[ TE_MEMMGR ];


RpSoftPullMan::topologyElement *
RpSoftPullMan::topologyElement::newInstance( const RpSoftPullMan::requestKey& setKey )
{
   RpSoftPullMan::topologyElement *pRet = 0;
   
   if ( te_memmgr_count )
   {
      te_memmgr_count--;
      pRet = te_memmgr_ppStore[ te_memmgr_count ];
      assert( pRet );
      
      pRet->reset( setKey );
      pRet->cost = 0.0;
      
   } else
      pRet = new topologyElement( setKey );

   return( pRet );
}


void
RpSoftPullMan::topologyElement::deleteInstance( void )
{
   if ( te_memmgr_count < TE_MEMMGR )
   {
      te_memmgr_ppStore[ te_memmgr_count ] = this;
      te_memmgr_count++;
            
   } else
      delete this;
      
   return;
}

// ************************************************************************

//
// post: returns the total bytes used by cached tiles.
//
unsigned long
RpSoftPullMan::getRAMSize( void ) const
{
   return( _getRAMSize( (cacheElement *) _cache.root() ) );
}

//
// post: returns the total bytes used by this cache element
//       and its children.
// note: recursive execution
//
unsigned long
RpSoftPullMan::_getRAMSize( cacheElement * pRoot )
{
   unsigned long	lRet = 0;
   
   if ( pRoot )
   {
      lRet = _getRAMSize( (cacheElement *) pRoot->_pLeft );
      lRet += _getRAMSize( (cacheElement *) pRoot->_pRight );
      if ( pRoot->pTile )
         lRet += pRoot->pTile->getRAMSize();
   }
   
   return( lRet );
}


//
// pre: pCached is a valid element in the cache.
// post: pCached has been removed from the cache.
// post: An equivalent candidate has been added to
//       candidate list.
//
void
RpSoftPullMan::demoteCacheElement( RpSoftPullMan::cacheElement *pCached )
{
   candidateElement	*pCandidate;
   
   if ( pCached )
   {
      pCandidate = candidateElement::newInstance( *pCached );

      if ( pCandidate )
      {
	 _candidates.insert( pCandidate );
	 _cache.erase( pCached->_key );
      }
   }
   
   return;
}


//
// post: Current cache contents do not exceed the given bound.
//      (Cache elements may have been removed.)
// note: If lowest-priority cache element is *gifted* then it is
//     demoted to a candidate element.
//
void
RpSoftPullMan::setRAMBound( unsigned long ulBytes )
{
   cacheElement 	*pRemovable;
   unsigned long	ulUsed;
   
   ulUsed = getRAMSize();
   
   while( ulUsed > ulBytes )
   {
      pRemovable = minCacheElement( (cacheElement *) _cache.root() );
      if ( pRemovable->pTile )
      {
         assert( pRemovable->pTile->getRAMSize() <= ulUsed );
         ulUsed -= pRemovable->pTile->getRAMSize();
      }
         
      if ( pRemovable->priority < 0 )
         demoteCacheElement( pRemovable );
      else
         _cache.erase( pRemovable->_key );
   }
   
   RpImageMan::setRAMBound( ulBytes );
   return;
}


//
// pre: pCaller and area are valid data
//     regarding an incoming user request.
// post: 
//    topology database has been updated to record the user request.
//
void
RpSoftPullMan::registerTopology( RpSampledImage *pCaller, const RpImageArea& area )
{
   float			rCost;
   requestKey			registerMe( pCaller, area );
  
   topologyElement		*pElement;
   
   if ( !_pTopology )
      _pTopology = new Map;
   
   if ( _pTopology )
   {
      assert( pCaller );
      rCost = pCaller->getCostEstimate( area );

      pElement = (topologyElement *) _pTopology->find( registerMe );
      if ( pElement )       
         pElement->cost += rCost;               
      else {
      
         pElement = topologyElement::newInstance( registerMe );
         if ( pElement )
         {
            pElement->cost = rCost;
            _pTopology->insert( pElement );
         }
      }
   }
   
   return;
}


//
// pre: p1 and p2 are valid pointers to topologyElements
// post: returns (-1) iff (*p1) is the worse topology choice.
// note: use with RpSoftPullMan::Map::min() to get the *best* topology element.
//
int
RpSoftPullMan::worstTopology( RpSoftPullMan::MapNode *p1, RpSoftPullMan::MapNode *p2 )
{
   assert( p1 && p2 );
   return ( 
      (((RpSoftPullMan::topologyElement *)p1)->cost > ((RpSoftPullMan::topologyElement *)p2)->cost )
      ? (-1) : ( 0 )
   );
}


//
// post: the topology has been analyzed and the best candidate for
//       caching has been identified.  It has been placed on the
//       candidate list; or its existing entry in the candidate
//       list has been refreshed to reflect its increased importance.
//
void
RpSoftPullMan::submitCandidate( void )
{
   candidateElement		*pRegisterMe;
   
   topologyElement		*pCandidate;
   candidateElement		*pCandidateHit;   
   
   if ( _pTopology )
   {

      //
      // determine best candidate from topology and add it to the candidate list
      // note: the minimum of the worst is the best.
      //
      pCandidate = (topologyElement *) Map::min( (MapNode *) _pTopology->root(), worstTopology );

      if ( pCandidate )
      {
         pCandidateHit = (candidateElement *) _candidates.find( pCandidate->_key );
	 if ( pCandidateHit )
	 {
	    //
	    // update candidate's qualifications
	    // 
	    pCandidateHit->cost += pCandidate->cost;
	    pCandidateHit->caller = _currentCaller;
	    
	    if ( pCandidateHit->priority >= 0 )
	       pCandidateHit->priority = _iPriorityCounter;
	       	    
	 } else {
	 
	    trimCandidates();
	 
	    //
	    // insert new candidate entry
	    //
	    pRegisterMe = candidateElement::newInstance( pCandidate->_key );
	    if ( pRegisterMe )
	    {
	       pRegisterMe->cost = pCandidate->cost;
	       pRegisterMe->priority = _iPriorityCounter;
	       pRegisterMe->caller = _currentCaller;
	       _candidates.insert( pRegisterMe );
	    }
	 }

         clearTopology();

      }
   
   }
   
   return;
}
 

//
// post: the topology database is completely blank.
//      
void
RpSoftPullMan::clearTopology( void )
{      
   if ( _pTopology )
      delete _pTopology;
      
   _pTopology = 0;
   return;
}


//
// post: records topology and caller information about the incoming
//       request.
//
void
RpSoftPullMan::incrRecurseCounter( RpSampledImage *pCaller, const RpImageArea& area )
{
   if (( !_uRecurseCounter )&&( !_iPriorityCounter ))
   {
      _currentCaller = callerID( pCaller, area );
      _currentCaller.c = 0; // drop c to insure conservative caching
   }

   registerTopology( pCaller, area );
   
   _uRecurseCounter++;
   return;
}


//
// post: depending upon the position in the topology, may have submitted
//       a candidate for caching.
//
void
RpSoftPullMan::decrRecurseCounter( void )
{
   _uRecurseCounter--;

   if ( !_uRecurseCounter )
      submitCandidate( );
      
   return;
}


//
// post: alters parameters to determine the best divisors to
//     use in decomposing the given area.
// note: The technique shows preference to square tiles in
//    calculating the decomposition divisors.
//
void
RpSoftPullMan::calcDecomposition( const RpImageArea& area, unsigned& uXDiv, unsigned& uYDiv )
{
   unsigned uXResult, uYResult;
   
   uXDiv = 1;
   uYDiv = 1;
   
   uXResult = area.width;
   uYResult = area.height;

   if( uXResult && uYResult )
      while ( (getTileBound() / uYResult ) < uXResult )
      {
	 if ( uXResult > uYResult )
	 {
            ++uXDiv;
            uXResult = ( area.width / uXDiv );
	 } else {
            ++uYDiv;
            uYResult = ( area.height / uYDiv );
	 }
      }
   
   return;
}


//
// pre: arguments are valid.
// post:
//    on success, tile request has been serviced.
// note:
//    a single tile request may have been decomposed into many
//    tile requests for purposes of limiting memory usage.
//
void
RpSoftPullMan::decomposeRequest( RpImageTile *pRet, RpSampledImage *pCaller )
{   
   unsigned		uXDiv, uYDiv;

   if ( pRet && pCaller )
   {
      calcDecomposition( pRet->getArea(), uXDiv, uYDiv );
      
      //
      // this conditional is an optimization:
      //
      if ( (uXDiv == 1) && (uYDiv == 1) )
         pCaller->fillTile( pRet );         
      else
      {
	 unsigned uLoopX, uLoopY;

	 RpImageTile *pTemp;
	 RpImageArea tempArea;
	 
	 assert( uXDiv && uYDiv );
	 
	 tempArea = pRet->getArea();
	 for( uLoopY=0; uLoopY<uYDiv; uLoopY++ )
	 {
            tempArea.x = pRet->getArea().x;
            tempArea.height = 
               ((( uLoopY + 1 ) * pRet->getArea().height ) / uYDiv ) -
               (( uLoopY * pRet->getArea().height ) / uYDiv );


	    for( uLoopX=0; uLoopX<uXDiv; uLoopX++ )
	    {
	       tempArea.width =
		  ((( uLoopX + 1 ) * pRet->getArea().width ) / uXDiv ) -
		  (( uLoopX * pRet->getArea().width ) / uXDiv );

               pTemp = RpImageTile::newTile( tempArea, pCaller->getType() );
	       if ( pTemp ) {
	          if ( !(pCaller->fillTile( pTemp )) )
	          {
	             assert( pRet->getType() == pTemp->getType() );
		     tilecpy( *pRet, *pTemp, tempArea );
		  }
                  pTemp->deleteTile();
               }

	       tempArea.x += tempArea.width;
	    }

	    tempArea.y += tempArea.height;
	 }
      }
   }
   return;
}

//
// desc: attempts to service the user request from the cache
// post: returns true if the user request was serviced; otherwise false.
//
int
RpSoftPullMan::cacheServiced( RpImageTile *pWriteHere, RpSampledImage *pCaller )
{
   int			iRet = 0;
   cacheElement		*pHit;
   RpFillStrategy	*pFiller;

   if ( pWriteHere && pCaller )
   {
      requestKey key( pCaller, pWriteHere->getArea() );
      
      pHit = (cacheElement *) _cache.find( key );
      
      if ( pHit )
	 if ( pHit->pTile )
	 {
            iRet = -1;
            //
            // register hit characteristics:
            //
            if ( pHit->priority >= 0 )
               pHit->priority = _iPriorityCounter;
            pHit->caller = _currentCaller;
            pHit->cost += 
              (pHit->_key).p->getCostEstimate( pWriteHere->getArea() );

            assert( pHit->pTile->getType() == pWriteHere->getType() );
            assert( pHit->_key.p->getType() == pWriteHere->getType() );
            // 
            // copy pWriteHere <- *(pHit->pTile)
            //
            tilecpy( *pWriteHere, *(pHit->pTile), pWriteHere->getArea() );
            pFiller = pCaller->getFillStrategy();
            if ( pFiller )
               pFiller->fillTile( pCaller, pWriteHere );
	 }
   }
      
   return( iRet );
}


//
// desc: compare between a cache element and a candidate.
// post: returns true if the candidate is the more *important* element.
//
int
RpSoftPullMan::priorityLessThan( const RpSoftPullMan::cacheElement *c1, const RpSoftPullMan::candidateElement *c2 ) const
{
   assert( c1 && c2 );
   return (
     ( c1->priority > c2->priority ) ||
     (( c1->priority == c2->priority ) && ( c1->caller != c2->caller ) && ( c2->caller == _currentCaller ))
   );
}


//
// desc: attempts to recover memory from old cache elements to service
//       the pending promotion of a candidate to a cache element.
// pre: arguments are valid.
// post: returns true if enough RAM has been recovered to promote the
//       candidate to a cache element.
// note: This is computationally expensive.
//         
int
RpSoftPullMan::gotEnoughRAM( candidateElement *pCandidateElement, int iDoFree )
{
   signed long		iAvailable;
   signed long		iRequired;
   Map			removeables;
   cacheElement		*pFlushCacheElement;
   int			iRet = 0;
   
   assert( pCandidateElement );
   assert( pCandidateElement->_key.p );
   
   iRequired =
      pCandidateElement->_key.p->getArea().width *
      pCandidateElement->_key.p->getArea().height *
      bytesize( pCandidateElement->_key.p->getType() );      
      
   iAvailable = getRAMBound() - getRAMSize();
   
   if ( iRequired > iAvailable )
   {
      //
      // while:
      //    add low-priority cache elements to the removables
      //    list until the required amount of RAM can be had.
      //
      pFlushCacheElement = minCacheElement( (cacheElement *) _cache.root() );
      while ( pFlushCacheElement )
      {
         if ( !priorityLessThan( pFlushCacheElement, pCandidateElement ))
             break; // while

         if ( pFlushCacheElement->pTile )
         {
            iAvailable += pFlushCacheElement->pTile->getRAMSize();
            removeables.insert( (MapNode *) cacheElement::newInstance( *pFlushCacheElement ));
         }
         
         _cache.erase( pFlushCacheElement->_key );

         if ( iAvailable < iRequired )
         {
            pFlushCacheElement = minCacheElement( (cacheElement *) _cache.root() );
         } else
            break; // while
            
      } // end while

      if ( iAvailable >= iRequired )
         iRet = -1;
      else
         iRet = 0;
         
      //
      // commit:
      // destruction of removeables structure will free the required RAM
      //      
      if (( iAvailable < iRequired )&& (! iDoFree ) )
      {
         //
         // fallback: move remove-candidates back into the cache.
         //
         pFlushCacheElement = (cacheElement *) removeables.root();
         while ( pFlushCacheElement )
         {
            _cache.insert( (MapNode *) cacheElement::newInstance( *pFlushCacheElement ) );
            removeables.erase( pFlushCacheElement->_key );
            pFlushCacheElement = (cacheElement *) removeables.root();
         }      
      }
      
   } else
      iRet = -1;
      
   return ( iRet );
}

//
// pre: arguement is valid
// post: traverses the given subtree, looking for the lowest-priority
//      cache element
// post: may return a cacheElement pointer or NULL.
// note: recursive execution
// note: uses current state information about the image manager.
//
RpSoftPullMan::cacheElement *
RpSoftPullMan::minCacheElement( RpSoftPullMan::cacheElement *pCacheRoot ) const
{
   cacheElement *pTemp;
   cacheElement *pRet;
   
   pRet = pCacheRoot;   
   if ( pCacheRoot )
   {   
      pTemp = minCacheElement((cacheElement *)pCacheRoot->_pLeft);
      if ( pTemp )
         if (
            ( pTemp->priority > pRet->priority ) ||
            ((pTemp->priority == pRet->priority)&&( pTemp->caller != _currentCaller ) && ( pRet->caller == _currentCaller )) ||
            ((pTemp->priority == pRet->priority)&&( pTemp->caller == pRet->caller ) && (pRet->caller == _currentCaller ) && ( pTemp->cost < pRet->cost ))
            )
            pRet = pTemp;
               
      pTemp = minCacheElement((cacheElement *)pCacheRoot->_pRight);
      if ( pTemp )
         if (
            ( pTemp->priority > pRet->priority ) ||
            ((pTemp->priority == pRet->priority)&&( pTemp->caller != _currentCaller ) && ( pRet->caller == _currentCaller )) ||
            ((pTemp->priority == pRet->priority)&&( pTemp->caller == pRet->caller ) && (pRet->caller == _currentCaller ) && ( pTemp->cost < pRet->cost ))
            )
            pRet = pTemp;               
   }
   
   return ( pRet );
}

//
// pre: arguments are valid
// post:
//    on success, the given candidate becomes cached and its
//    entry on the candidate list is removed.
// note:
//    this method is very computationally expensive and
//    results in indirect recursion.
//
void
RpSoftPullMan::promoteCandidate( candidateElement *pCandidate )
{
   Map		*pHoldTopology;
   unsigned	uHoldRecurse;
   cacheElement *pNewEntry;
   
   //
   // push cache state:
   // (This allows us to calculate this cache element and restore the
   // topology state afterward.)
   //
   pHoldTopology = _pTopology;
   _pTopology = 0;
   uHoldRecurse = _uRecurseCounter;
   _uRecurseCounter = 0;
   _iPriorityCounter++;

   //
   // recover enough RAM here;
   // or drop-out if we don't have enough RAM.
   //
   if ( pCandidate )
      if ( gotEnoughRAM( pCandidate, 0 ) )
      {
	 //
	 // copy data from candidate structure
	 // to cache-element format:
	 //
	 pNewEntry = cacheElement::newInstance( *pCandidate );
	 if ( pNewEntry )
	 {
	    //
	    // recurse for cache tile calculation & promote candidate on success.
	    //
	    if ( pNewEntry->pTile )
	    {
	       assert( (pNewEntry->_key).p );
	       assert( (pNewEntry->_key).p->getType() == pNewEntry->pTile->getType() );
	       
	       //
	       // decompose un-forced cache calculations;
	       // do not decompose forced cache calculations!
	       //
	       if( pCandidate->priority >= 0 )
                  decomposeRequest( pNewEntry->pTile, (pNewEntry->_key).p );
               else
                  (pNewEntry->_key).p->fillTile( pNewEntry->pTile );
               
               if ( gotEnoughRAM( pCandidate, -1 ) )
               {
 	          _cache.insert( pNewEntry );	             
	          _candidates.erase( pCandidate->_key );	       
	       }
	       
	       #ifdef RpSHOWSTATS
	       cout << "SoftPull cache reports: " << _cache.size() << " elements using " << (getRAMSize() >> 10) << " KB\n";
               #endif
               
	    } else
		pNewEntry->deleteInstance();
	 }

      }

   //
   // pop cache state
   //
   _iPriorityCounter--;
   _uRecurseCounter = uHoldRecurse;
   if ( _pTopology )
      delete _pTopology;    
   _pTopology = pHoldTopology;
   
   return;
}


// desc: attempts to service user request from candidate list.
// pre: arguements are valid.
// post: if the user request has been serviced, returns true; false otherwise.
// note: servicing this request may have resulted in discard of existing
//       cache elements iff this candidate was promoted.
//
int
RpSoftPullMan::candidateServiced( RpImageTile *pWriteHere, RpSampledImage *pCaller )
{
   int				iRet = 0;
   candidateElement		*pCandidateHit;
   cacheElement			*pCacheHit;   
   RpFillStrategy		*pFiller;
   
   if ( pWriteHere && pCaller )
   {
      requestKey key( pCaller, pWriteHere->getArea() );   
      
      pCandidateHit = (candidateElement *) _candidates.find( key );
      if ( pCandidateHit )
      {
         //
         // register hit characteristics:
         //
         if ( pCandidateHit->priority >= 0 )
            pCandidateHit->priority = _iPriorityCounter;
         pCandidateHit->caller = _currentCaller;
         pCandidateHit->cost += 
            (pCandidateHit->_key).p->getCostEstimate( pWriteHere->getArea() );
         
         //
         // promote candidate to cache
         //         
         promoteCandidate( pCandidateHit );
         
         // 
         // copy data from cache
         //
         pCacheHit = (cacheElement *) _cache.find( key );
         if ( pCacheHit )
            if ( pCacheHit->pTile )
            {
               iRet = -1;
               
               assert( pCacheHit->pTile->getType() == pWriteHere->getType() );
               assert( pCacheHit->_key.p->getType() == pWriteHere->getType() );               
               
               tilecpy( *pWriteHere, *(pCacheHit->pTile), pWriteHere->getArea() );
               pFiller = pCaller->getFillStrategy();
               if ( pFiller )
          	  pFiller->fillTile( pCaller, pWriteHere );
            }

         
      }
   }
   
   return( iRet );
}


//
// desc: API to the manager
//
RpImageTile *
RpSoftPullMan::newTile( RpSampledImage *pCaller, const RpImageArea& area )
{
   RpImageTile		*pRet = 0;

   if ( pCaller )
   {
      pRet = RpImageTile::newTile( area, pCaller->getType() );
      if ( pRet )
	 if ( !cacheServiced( pRet, pCaller ) )
	    if ( !candidateServiced( pRet, pCaller ))
	    {
               incrRecurseCounter( pCaller, area );      
	       decomposeRequest( pRet, pCaller );
               decrRecurseCounter();
            }
            
   }
   
   return ( pRet );
}



void
RpSoftPullMan::forceCache( RpSampledImage *pSuggestions )
{
   candidateElement	*pCandidate;
   RpImageArea		area;
   
   if ( pSuggestions )
   {
      area = pSuggestions->getArea();
      
      while ( area.z )
      {      
         area.z--;
         
	 while( area.c )
	 {
            area.c--;
            pCandidate = candidateElement::newInstance( requestKey( pSuggestions, area ));
            if ( pCandidate )
            {
               pCandidate->priority = -1;
               pCandidate->cost = 0.0;
               _candidates.insert( pCandidate );
            }
	 }
      }
   }
   
   return;
}


//
// post:
//    deallocates all tile-cache RAM.
//    and clears memory of all but the *gifted* elements
//    in the candidate list, tile cache and topology database.
//
void
RpSoftPullMan::flushCache( void )
{   
   flushCandidateMap();
   flushCacheMap();   
   clearTopology();
         
   return;
}


//
// post: removes all but the *gifted* candidates from the candidate list.
//
void
RpSoftPullMan::flushCandidateMap( void )
{
   Map			tempMap;
   candidateElement	*pLoop;
   
   pLoop = (candidateElement *) _candidates.root();
   while ( pLoop )
   {
      if ( pLoop->priority < 0 )
         tempMap.insert( (MapNode *) candidateElement::newInstance( *pLoop ) );
      
      _candidates.erase( pLoop->_key );
      pLoop = (candidateElement *) _candidates.root();  
   }
   
   assert( _candidates.size() == 0 );   
   _candidates.swapContents( tempMap );   
   return;
}


//
// post: removes all but the *gifted* cache elements from the cache
//       *gifted* cache elements are demoted to candidates in the
//       candidate list.
//
void
RpSoftPullMan::flushCacheMap( void )
{
   cacheElement		*pLoop;
   
   pLoop = (cacheElement *) _cache.root();
   while ( pLoop )
   {
      if ( pLoop->priority < 0 )
         demoteCacheElement( pLoop );
      else
         _cache.erase( pLoop->_key );
         
      pLoop = (cacheElement *) _cache.root();
   }
   
   return;
}


//
// note: this method is intended to bound the number of elements in
//     the candidate list.
// 
//
void
RpSoftPullMan::trimCandidates( void )
{
   candidateElement *pLoop;
   
   while ( _candidates.size() > CANDIDATE_LIMIT )
   {
       //
       // min of minCandidateCosts means we will discard the
       // highest cost element.
       //
       pLoop = (candidateElement *) Map::min( (MapNode *) _candidates.root(), minCandidateCost );
       assert( pLoop );
       assert( _candidates.find( pLoop->_key ));
       
       if ( pLoop->priority < 0 )
          break;
          
       _candidates.erase( pLoop->_key );
   }
   
   return;
}


int
RpSoftPullMan::minCandidateCost( RpSoftPullMan::MapNode *p1, RpSoftPullMan::MapNode *p2 )
{
   candidateElement *pE1;
   candidateElement *pE2;
   int		iRet = 0;
   
   assert( p1 && p2 );
   pE1 = (candidateElement *) p1;
   pE2 = (candidateElement *) p2;   
   
   if (
      ( pE1->priority > pE2->priority ) ||
      ( (pE1->priority == pE2->priority) && ( pE1->cost < pE2->cost) )
   )
      iRet = -1;
      
   return( iRet );
}

// ************************************************************************

RpSoftPullMan::~RpSoftPullMan( void )
{
   if ( _pTopology )
      delete _pTopology;

   return;
}
