#ifndef RpImageMan_CLASS
#define RpImageMan_CLASS

//
// define RpSHOWSTATS for reporting on cache statistics
// #define RpSHOWSTATS
//


// ************************************************************************
// CLASS: RpImageMan, RpManReference
// DESC:
//    All clients should use RpManReference to manipulate the image
//    manager.
// ************************************************************************

#include <RpImageTile.h>

class RpSampledImage;
class RpImageMan {
   public:
   
      // ******************************************************************
      // desc:
      //    Enumeration designates the available image managers:
      //
      enum Type 
      {
         Mini,
         SoftPull
      };
      // ******************************************************************
      
       virtual
       ~RpImageMan( void );
       
      // ------------------------------------------------------------------
      // desc:
      //    Allocates and returns a tile of image data.
      //    Uses RpFillStrategy *after* filling the tile with image data.
      //    May calculate & cache more data area than requested.
      // post:
      //    returns 0 on error.
      //    
      virtual RpImageTile *
      newTile ( RpSampledImage *pCaller, const RpImageArea& area ) = 0;
      
      // ------------------------------------------------------------------
      // desc:
      //    returns the amount of RAM in use.
      //
      virtual unsigned long
      getRAMSize( void ) const = 0;
      
      // ------------------------------------------------------------------
      // desc:
      //    returns the limit of RAM to use.
      //
      unsigned long
      getRAMBound( void ) const;

      // ------------------------------------------------------------------
      // desc:
      //    sets the limit of RAM to use.
      //
      virtual void
      setRAMBound( unsigned long ulBytes );
      
      // ------------------------------------------------------------------
      // desc:
      //    returns the area limit for ~working tiles~
      //
      unsigned long
      getTileBound( void ) const;

      // ------------------------------------------------------------------
      // desc:
      //    sets the area limit for ~working tiles~
      //      
      virtual void
      setTileBound( unsigned long ulTileBound );
      
      // ------------------------------------------------------------------
      // desc:
      //    this method allows application programmers to recommend
      //    caching of a particular image's data.
      // note:
      //    This is just a hint; the manager may or may not use this
      //    information.  Furthermore the hint may not be preserved
      //    if the application changes the manager type.
      //
      virtual void
      forceCache( RpSampledImage *pClient ) = 0;
      
      // ------------------------------------------------------------------
      // desc:
      //    invalidates cached information associated with a specific image
      // post:
      //    No information relating to (*pClient) is currently cached.
      //
      virtual void
      flushCache( RpSampledImage *pClient ) = 0;
      
      // ------------------------------------------------------------------
      // desc:
      //    discards all cached information
      //
      virtual void
      flushCache( void ) = 0;

   protected:
       RpImageMan( unsigned long ulSetRAMBound, unsigned long ulSetTileBound );
       friend class RpManReference;
       
   private:
       unsigned long		_lRAMBound;      
       unsigned long		_lTileBound;
};


// ************************************************************************
// CLASS: RpManReference
// DESC:
//    Developers should instantiate RpManReference objects in order
//    to communicate with the image manager.
//
class RpManReference {
   public:
      RpManReference( void );
      ~RpManReference( void );
      
      // ------------------------------------------------------------------
      // desc:
      //    this is the access method which presents developers with an
      //    interface to the image manager.
      //
      RpImageMan *
      getPointer( void ) { return( _pManager ); }
      
      // ------------------------------------------------------------------
      // desc:
      //    convenience method
      //
      RpImageMan *
      operator->( void ) { return ( getPointer() ); }
      
      // ------------------------------------------------------------------
      // desc:
      //    method for changing the current image manager
      // caveat:
      //    It is a good idea to select the image manager you wish to
      //    use first -- before applying other manager settings.
      // 
      RpImageMan *
      setManagerType( RpImageMan::Type setType );
      
   protected:
      static RpImageMan *	_pManager;
      static unsigned		_uReferences;

      static RpImageMan::Type	_type;
      static unsigned long	_ulRAM;
      static unsigned long	_ulPixels;

      void
      realloc( void );
};


#endif
