/**********************************************************************/
/*                                                                    */
/* walk.c : walk-through routines                                     */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          *
/**********************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fmclient.h>
#include <math.h>
#include <gl/gl.h>
#include <gl/device.h>
#include <sys/types.h>
#include <dirent.h>
#ifdef RELEASE3.2
#include <malloc.h>
#else
#include <stdlib.h>
#endif

#define IRIS4D 1
#include "geo.h"
#include "walk.h"
#include "auto.h"
#include "scene.h"
#include "vcr.h"

/**********************************************************************/
/* Prototypes                                                         */
/**********************************************************************/
void Initialize();
void Orient();
void HandleEvents();
void DoEventsWind();
void DoWind();
void DoEventsPanel();
void DoEventsHelp();
void Draw_Scene();
void Update_Scene();
void initcylmat();
void Draw_Axis();
void InitMovement();
void InitAbsMovement();
void InitRelMovement();

extern void InitFonts();
extern void InitGraphics(); 
extern RelMovement *CreateRelMove();
extern void ParseArgs();
extern void Draw_Mesh();
extern void Init_Scene();
extern char HelpText[];
extern SLogType ReadLog;
extern long WindWid;
extern Matrix viewmat;

/**********************************************************************/
/* Main                                                               */
/**********************************************************************/
void main(argc, argv)
     int argc;
     char *argv[];
{
  System.UseVCR = FALSE;
  System.VCRmode = FULL;
  System.UseRGB = TRUE;
  System.UseZBuf = TRUE;
  System.UseDoubleBuf = TRUE;
  System.gamma = FALSE;
  System.FullScreen = FALSE;
  Option.shadeBW = FALSE;
  Option.RGBscale = 1.0;

  ParseArgs(argc, argv);
  Initialize();
  Draw_Scene();
  HandleEvents();
}

/**********************************************************************/
/* Initialize graphics, queue, materials, fonts, lights               */
/* and movement structures                                            */
/**********************************************************************/
void Initialize() 
{
  printf("Reading scene...\n");
  Init_Scene();
  printf("done\n");
  InitGraphics();
  InitFonts();
  InitMovement();
  InitHelp(HelpText);
  InitQueue();
}

/**********************************************************************/
/* Initialize movement */
/**********************************************************************/
void InitMovement()
{
  /* Set initial Viewer position, and clear any movement */
  InitAbsMovement(&Viewer);
  InitRelMovement(&view);

  /* No movements recorded, and currently not moving */
  Option.log = NOLOG;
  Option.mode = STILL;
}

/**********************************************************************/
/* Clear relative view movement record, and object matrix */
/**********************************************************************/
void InitRelMovement(v)
     RelMovement *v;
{  
  v->rot[0] = v->rot[1] = v->rot[2] = 0.0;
  v->transl[0] = v->transl[1] = v->transl[2] = 0.0;
}

/**********************************************************************/
/* Clear absolute view movement record to default */
/* based on size, position of environment */
/**********************************************************************/
void InitAbsMovement(c)
     AbsMovement *c;
{
  long xsize, ysize;

  c->lookFrom.x = ReadLog.worldbox.max.x - 2.*ENLARGE_WORLD; 
  c->lookFrom.y = ReadLog.worldbox.max.y - 2.*ENLARGE_WORLD; 
  c->lookFrom.z = ReadLog.worldbox.max.z - 2.*ENLARGE_WORLD;
  c->lookAt.x = (ReadLog.worldbox.max.x - ReadLog.worldbox.min.x ) / 2.0;
  c->lookAt.y = (ReadLog.worldbox.max.y - ReadLog.worldbox.min.y ) / 2.0;
  c->lookAt.z = (ReadLog.worldbox.max.z - ReadLog.worldbox.min.z ) / 2.0;
  c->lookUp.x = 0.0; 
  c->lookUp.y = 1.0; 
  c->lookUp.z = 0.0;
  c->fovx = 60;   c->fovy = 60;
  c->near = 0.001; c->far = System.zfar;
  getsize(&xsize, &ysize);
  c->xRes = xsize; c->yRes = ysize;
  c->bank = 0.0;
}

/**********************************************************************/
/* Set viewer's absolute position, get viewing matrix                 */
/**********************************************************************/
void SetViewerPos(viewer)
     AbsMovement *viewer;
{
  winset(WindWid);
  if (Option.debug) {
    printf("\n\t> Camera set at:\n");
    printf("\tfov=%d, aspect=%g, near=%g, far = %g\n",
	   viewer->fovx, System.scrnaspect, viewer->near, viewer->far);
    printf("\tfrom=%g %g %g; at=%g %g %g; bank=%g\n\n",
	   viewer->lookFrom.x, viewer->lookFrom.y, viewer->lookFrom.z,
     viewer->lookAt.x, viewer->lookAt.y, viewer->lookAt.z,	 
	   viewer->bank);
  }

  perspective(viewer->fovx*10, System.scrnaspect, viewer->near, viewer->far);
  lookat(viewer->lookFrom.x, viewer->lookFrom.y, viewer->lookFrom.z,
	 viewer->lookAt.x, viewer->lookAt.y, viewer->lookAt.z,	 
	 viewer->bank);
  mmode(MVIEWING);
  getmatrix(viewmat);
  
  if (Option.debug) printf("Set viewer position\n");
}

/**********************************************************************/
/* Update the scene viewed from current position                      */
/* Note tranformations from viewer are "negated". To prevent too      */
/* sudden movement, use of scaling factors ANGLE and COORD shock      */
/* absorbers                                                          */
/**********************************************************************/
void Update_Scene() 
{
  int i;
  Movement *amoves;                 /* Automated movement */
  RelMovement *rmove;               /* Relative movment */
  char mtype;                       /* Movement type (for automated) */

  if (Option.debug) printf("Updating scene. Mode = %d\n", Option.mode);
  /* if (Option.mode != PREORIENT) */
  InitRelMovement(&view);

  switch (Option.mode) {

  case ORIENT_SPIN: /* User is currently rotating viewer in y */    
    view.rot[Y] = (view.m.x-view.om.x) * ANGLE_SHOCK_ABSORBER;
    Orient();
    if (Option.log) {
      rmove = CreateRelMove(view.rot, view.transl);
      LogRelMove(rmove, &Autoview, MOVEREL_SPIN);
    }
    view.om.x = view.m.x;
    break;

  case ORIENT_ROT: /* User is currently rotating viewer in x,z */    
    view.rot[X] = (view.m.x-view.om.x) * ANGLE_SHOCK_ABSORBER;
    view.rot[Z] = (view.m.y-view.om.y) * ANGLE_SHOCK_ABSORBER;
    Orient();
    if (Option.log) {
      rmove = CreateRelMove(view.rot, view.transl);
      LogRelMove(rmove, &Autoview, MOVEREL_ROT);
    }
    view.om.x = view.m.x;
    view.om.y = view.m.y;
    break;

  case ORIENT_TRANSL: /* User is currently translating viewer */    
    view.transl[X] = (view.m.x-view.om.x) * COORD_SHOCK_ABSORBER;
    view.transl[Y] = (view.m.y-view.om.y) * COORD_SHOCK_ABSORBER;
    Orient();
    if (Option.log) {
      rmove = CreateRelMove(view.rot, view.transl);
      LogRelMove(rmove, &Autoview, MOVEREL_TRANSL);
    }
    view.om.x = view.m.x;
    view.om.y = view.m.y;
    break;

  case ORIENT_ZOOM:  /* User is currently zooming in */
    view.transl[Z] = (view.om.x-view.m.x) * COORD_SHOCK_ABSORBER;
    Orient();
    if (Option.log) {
      rmove = CreateRelMove(view.rot, view.transl);
      LogRelMove(rmove, &Autoview, MOVEREL_ZOOM);
    }
    view.om.x = view.m.x;
    break;

  case ORIENT_SCALE:  /* User is currently scaling objects wrt viewer */    
    view.transl[Z] = (view.m.x-view.om.x);
    Orient(); 
    if (Option.log) {
      rmove = CreateRelMove(view.rot, view.transl);
      LogRelMove(rmove, &Autoview, MOVEREL_SCALE);
    }
    view.om.x = view.m.x;
    break;

  case PREORIENT:              /* Orientation is being read from file */
    amoves = Autoview;
    for (i=0;i<moves_logged;i++) {
      mtype = amoves->mtype;

      if (mtype == MOVEABS) {
	loadmatrix(Identity);
	CopyViewer(&Viewer,amoves->amove);   /* Change eye position */
	Option.mode = ORIENT_ABS;

      } else if ((mtype == MOVEREL_ROT) ||
		 (mtype == MOVEREL_TRANSL) ||
		 (mtype == MOVEREL_ZOOM) ||
		 (mtype == MOVEREL_SCALE) ||
		 (mtype == MOVEREL_SPIN)) {
	CopyViewChg(&view,amoves->rmove);      /* Change transformations */
	if (mtype == MOVEREL_ROT) {
	  Option.mode = ORIENT_ROT; }
	else if (mtype == MOVEREL_TRANSL) {
	  Option.mode = ORIENT_TRANSL; }
	else if (mtype == MOVEREL_ZOOM) {
	  Option.mode = ORIENT_ZOOM; } 
	else if (mtype == MOVEREL_SCALE) { 
	  Option.mode = ORIENT_SCALE; }
	else if (mtype == MOVEREL_SPIN) { 
	  Option.mode = ORIENT_SPIN; 
	}
	
      } else {
	fprintf(stderr,"Update_Scene(): Invalid movement type: PREORIENT\n");
	DoExit(-1);
      }
      Orient();                          /* (Re)orient */
      Draw_Scene();                      /* (Re)draw scene */
      amoves = amoves->next;
    }
    Option.mode = PREORIENT;
    break;
    
  case STILL: /* Viewer is not moving */
    if (Option.debug) printf("Something may be wrong. Not moving.");
    break;

  default:
    fprintf(stderr,"Update_Scene(): Invalid Movement Mode.\n");
    DoExit(-1);
    break;
  }

  Draw_Scene();   /* (Re)draw scene */
}

/**********************************************************************/
/* Rotate in X, Z directions                                          */
/**********************************************************************/
void RotateXZ()
{
  if (Option.debug) printf("Rotating[%g,%g]\n", 
			   view.rot[X], view.rot[Z]);
  rotate(view.rot[X], XDIR);
  rotate(view.rot[Z], ZDIR);
}

/**********************************************************************/
/* Spin = Rotate in Y direction                                       */
/**********************************************************************/
void Spin()
{
  if (Option.debug) printf("Spinning[%g]\n", view.rot[Y]);
  rotate(view.rot[Y], YDIR);
}

/**********************************************************************/
/* Translate in X or Y directions                                     */
/**********************************************************************/
void TranslateXY()
{
  translate(view.transl[X], view.transl[Y], 0.0);
  if (Option.debug) printf("Translating[%f,%f]\n", 
			   view.transl[X], view.transl[Y]);
}

/**********************************************************************/
/* Zoom in by x factor = translation in Z                             */
/**********************************************************************/
void Zoom()
{
  translate(0.0, 0.0, view.transl[Z]);
  if (Option.debug) printf("Zooming...%f\n", view.transl[Z]);
}

/**********************************************************************/
/* Uniformly scale the image                                          */
/**********************************************************************/
void Scale()
{
  double zfactor;
  
  zfactor = 1.0 + view.transl[Z] * 20.0 /* Option.ZoomFactor */ / 
    (float) System.xscrnsize;
  scale(zfactor, zfactor, zfactor);
  if (Option.debug) printf("Scaling...%f\n", view.transl[Z]);
}

/**********************************************************************/
/* Update object transformation matrix based on viewing position, and */
/* movement                                                           */
/**********************************************************************/
void Orient() 
{
  if (Option.debug) printf("Orienting with mode = %d\n", Option.mode);

  /* Change in camera position, set objmat to Identity */
  if (Option.mode == ORIENT_ABS) { 
    pushmatrix(); /* J3 */
    loadmatrix(Identity);
    getmatrix(objmat);
    popmatrix(); /* J3 */
  } 

  else { /* Find movement and set objmat */
    pushmatrix();    
    loadmatrix(Identity);
    
    switch(Option.mode) {
    case ORIENT_ROT:          RotateXZ(); break;
    case ORIENT_SPIN:         Spin(); break;
    case ORIENT_TRANSL:       TranslateXY(); break;
    case ORIENT_ZOOM:         Zoom(); break;
    case ORIENT_SCALE:        Scale(); break;
    default: break;
    }
    multmatrix(objmat); 
    getmatrix(objmat);
    popmatrix();
  }
}

/**********************************************************************/
/* Redraw the scene */
/**********************************************************************/
void Draw_Scene() 
{
  /* Matrix tmp; */

  if (Option.debug) printf("Drawing Patches...\n");

  /* Set projection matrix, clear window and set shademodel */
  pushmatrix(); {
    SetViewerPos(&Viewer); 
    czclear(0x000000, System.zfar);   
    shademodel(Option.shademodel);
  } popmatrix();   

  /* Set view/model matrix and draw objects */
  pushmatrix(); {
    loadmatrix(objmat);
    multmatrix(viewmat);
    Draw_Mesh();
  } popmatrix(); 

  swapbuffers();

  /* if (System.UseVCR) vcr_grab(); */
}


/**********************************************************************/
/* Draw x,y,z, axis */
/**********************************************************************/
/* XYZ axis */
float origin[3] = {0,0,0};
float xaxis[3] = {1,0,0};
float yaxis[3] = {0,1,0};
float zaxis[3] = {0,0,1};

void Draw_Axis()
{
  float c[4];

  c[0] = 1; c[1] = 0; c[2] = 0; c[3] = 0;
  bgnline(); c4f(c); v3f(origin); v3f(xaxis); endline();
  c[0] = 0; c[1] = 1; c[2] = 0; 
  bgnline(); c4f(c); v3f(origin); v3f(yaxis); endline();
  c[0] = 0; c[1] = 0; c[2] = 1; 
  bgnline(); c4f(c); v3f(origin); v3f(zaxis); endline();
}

