/**********************************************************************/
/*                                                                    */
/* auto.c : Routines to handle automatic movement throught a scene    */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          *
/**********************************************************************/
#include <stdio.h>
#include <gl/gl.h>
#include <fmclient.h>
#include <gl/device.h>
#define IRIS4D 1
#include "geo.h"
#include "walk.h"
#include "auto.h"

/**********************************************************************/
/* Create a relative move structure                                   */
/**********************************************************************/
AbsMovement *CreateAbsMove(camera)
     AbsMovement camera;
{
  AbsMovement *amptr;

  if ((amptr = (AbsMovement *) malloc(sizeof(AbsMovement))) == 0) 
    fprintf(stderr,"walk: Insufficient memory to store movement. Sorry.\n");
  amptr->lookFrom = camera.lookFrom;
  amptr->lookAt = camera.lookAt;
  amptr->lookUp = camera.lookUp;
  amptr->fovx = camera.fovx;  amptr->fovy = camera.fovy;
  amptr->near = camera.near;  amptr->far = camera.far;
  amptr->xRes = camera.xRes;  amptr->yRes = camera.yRes; 
  amptr->bank = camera.bank;
  amptr->buffer = (unsigned long *)0;
  /* malloc(camera.xRes*camera.yRes * sizeof(unsigned long)); */
  
  return amptr;
}

/**********************************************************************/
/* Create a relative move structure                                   */
/**********************************************************************/
RelMovement *CreateRelMove(rot, transl)
     double rot[3];
     double transl[3];
{
  RelMovement *rmptr;

  if ((rmptr = (RelMovement *) malloc(sizeof(RelMovement))) == 0) 
    fprintf(stderr,"walk: Insufficient memory to store movement. Sorry.\n");
  rmptr->rot[0] = rot[0];
  rmptr->rot[1] = rot[1];
  rmptr->rot[2] = rot[2];
  rmptr->transl[0] = transl[0];
  rmptr->transl[1] = transl[1];
  rmptr->transl[2] = transl[2];

  return rmptr;
}
  
/**********************************************************************/
/* Output contents of an absolute move to screen */
/**********************************************************************/
void PrintAbsMove(amove)
     AbsMovement *amove;
{
  printf("Move A {\n");

  printf("  lookFrom %g %g %g\n", amove->lookFrom.x, 
	 amove->lookFrom.y, amove->lookFrom.z);
  printf("  lookAt %g %g %g\n", amove->lookAt.x,
	 amove->lookAt.y, amove->lookAt.z);
  printf("  lookUp %g %g %g\n", amove->lookUp.x, 
	 amove->lookUp.y, amove->lookUp.z);
  printf("  fovx %d fovy %d near %g far %g\n",
	 amove->fovx, amove->fovy, amove->near, amove->far);
  printf("  xRes %d yRes %d\n", amove->xRes, amove->yRes);
  printf("  bank %g\n", amove->bank);

  printf("}\n");
}

/**********************************************************************/
/* Output contents of a relative move to screen */
/**********************************************************************/
void PrintRelMove(rmove, mtype)
     RelMovement *rmove;
     int mtype;
{
  switch(mtype) {

  case MOVEREL_ROT: 
    printf("Move R {\n");
    printf("%g %g %g\n", rmove->rot[0],
	   rmove->rot[1], rmove->rot[2]);
    break;

  case MOVEREL_SPIN: 
    printf("Move P {\n");
    printf("%g %g %g\n", rmove->rot[0],
	   rmove->rot[1], rmove->rot[2]);
    break;

  case MOVEREL_TRANSL: 
    printf("Move T {\n");
    printf("%g %g %g\n", rmove->transl[0],
	   rmove->transl[1], rmove->transl[2]);
    break;

  case MOVEREL_SCALE: 
    printf("Move S {\n");
    printf("%g %g %g\n", rmove->transl[0],
	   rmove->transl[1], rmove->transl[2]);
    break;

  case MOVEREL_ZOOM: 
    printf("Move Z {\n");
    printf("%g %g %g\n", rmove->transl[0],
	   rmove->transl[1], rmove->transl[2]);
    break;

  default: 
    fprintf(stderr,"%s: Invalid movement to print: %d\n", ProgName, mtype);
    exit(-1); /* DoExit(-1); */
  }
  printf("}\n");
}

/**********************************************************************/
/* Clear any stored orientations                                      */
/**********************************************************************/
void ClearOrientLog()
{
  if (Option.debug) printf("Clearing log\n");
  if (Autoview) free(Autoview); 
  /* if (Mtail) free(Mtail); */
  moves_logged = 0;
  Option.log = NOLOG;
}

/**********************************************************************/
/* Open movement file                                                 */
/**********************************************************************/
int OpenMoveFile(filename,mode)
     char *filename, *mode;
{
  if (!(movefile = fopen(filename,mode))) {
    fprintf(stderr,"%s: cannot open file %s\n", ProgName, filename);
    return 0;
  }
  return 1;
}

/**********************************************************************/
/* Read movements from file                                           */
/**********************************************************************/
int ReadMoves(filename, moves)
     char *filename;
     Movement **moves;
{
  int num_relmoves = 0;
  int num_absmoves = 0;
  int num_moves = 0;
  int moves_read = 0;
  Movement *mptr, *nmptr;
  int t1,t2;
  float t[4];

  if (!OpenMoveFile(filename,READF)) {
    exit(-1); /* DoExit(-1); */
  } else if (Option.debug) {
    printf("Reading moves from %s\n", filename);
  }

  /* Allocate space for moves */
  fscanf(movefile,"nmoves %d\n", &num_moves);
  if ((mptr = *moves = (Movement *) malloc(num_moves * sizeof(Movement)))
      == 0) {
    fprintf(stderr,"walk: Insufficient memory to load movements. Sorry.\n");
    exit(1);
  }

  /* Read in absolute or relative camera movements */
  while (moves_read < num_moves) {
    fscanf(movefile, "Move %c {\n", &Movetype);
    if (Option.debug) printf("Movetype: %c\n", Movetype);

    mptr->mtype = Movetype;
    switch (Movetype) {

    case MOVEABS: {
      if ((mptr->amove = (AbsMovement *) malloc(sizeof(AbsMovement))) == 0) {
	fprintf(stderr,
		"walk: Insufficient memory to load movement. Sorry.\n");
      exit(1);
      }

      fscanf(movefile, "lookFrom %g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\tlookFrom %g %g %g\n", t[0], t[1], t[2]);
      mptr->amove->lookFrom.x = t[0];
      mptr->amove->lookFrom.y = t[1];
      mptr->amove->lookFrom.z = t[2];

      fscanf(movefile, "lookAt %g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\tlookAt %g %g %g\n", t[0], t[1], t[2]);
      mptr->amove->lookAt.x = t[0];
      mptr->amove->lookAt.y = t[1];
      mptr->amove->lookAt.z = t[2];

      fscanf(movefile, "lookUp %g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\tlookup %g %g %g\n", t[0], t[1], t[2]);
      mptr->amove->lookUp.x = t[0];
      mptr->amove->lookUp.y = t[1];
      mptr->amove->lookUp.z = t[2];

      fscanf(movefile, "fovx %d fovy %d near %g far %g\n",&t1,&t2,&t[0],&t[1]);
      mptr->amove->fovx = t1; mptr->amove->fovy = t2;
      mptr->amove->near = t[0]; mptr->amove->far = t[1];
      if (Option.debug) 
	printf("\tfovx %d fovy %d near %g far %g\n", t1, t2, t[0], t[1]);
      
      fscanf(movefile,"xRes %d yRes %d\n", &t1, &t2);
      mptr->amove->xRes = t1; mptr->amove->yRes = t2;
      if (Option.debug) printf("\txRes %d yRes %d\n", t1, t2);

      fscanf(movefile, "bank %g\n", &t[0]);
      mptr->amove->bank = t[0];
      if (Option.debug) printf("\tbank %g\n", t[0]);

      fscanf(movefile, "}\n");

      /* Allocate buffer for image storage purposes */
      mptr->amove->buffer = (unsigned  long *)0;
      /* malloc(mptr->amove->xRes * mptr->amove->yRes * 
	 sizeof(unsigned long)); */

      num_absmoves++;
      break;
    }

    case MOVEREL_ROT: {
      if ((mptr->rmove = (RelMovement *) malloc(sizeof(RelMovement))) == 0) {
	fprintf(stderr,
		"walk: Insufficient memory to load movement. Sorry.\n");
	exit(1);
      }
      fscanf(movefile,"%g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\t%g, %g, %g\n", t[0], t[1], t[2]);
      mptr->rmove->rot[0] = t[0];
      mptr->rmove->rot[1] = t[1];
      mptr->rmove->rot[2] = t[2];
      fscanf(movefile,"}\n");
      num_relmoves++;
      break;
    }

    case MOVEREL_SPIN: {
      if ((mptr->rmove = (RelMovement *) malloc(sizeof(RelMovement))) == 0) {
	fprintf(stderr,
		"walk: Insufficient memory to load movement. Sorry.\n");
	exit(1);
      }
      fscanf(movefile,"%g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\t%g, %g, %g\n", t[0], t[1], t[2]);
      mptr->rmove->rot[0] = t[0];
      mptr->rmove->rot[1] = t[1];
      mptr->rmove->rot[2] = t[2];
      fscanf(movefile,"}\n");
      num_relmoves++;
      break;
    }

    case MOVEREL_TRANSL: {
      if ((mptr->rmove = (RelMovement *) malloc(sizeof(RelMovement))) == 0) {
	fprintf(stderr,
		"walk: Insufficient memory to load movement. Sorry.\n");
	exit(1);
      }
      fscanf(movefile,"%g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\t%g, %g, %g\n", t[0], t[1], t[2]);
      mptr->rmove->transl[0] = t[0];
      mptr->rmove->transl[1] = t[1];
      mptr->rmove->transl[2] = t[2];
      
      fscanf(movefile,"}\n");
      num_relmoves++;
      break;
    }

    case MOVEREL_SCALE: {
      if ((mptr->rmove = (RelMovement *) malloc(sizeof(RelMovement))) == 0) {
	fprintf(stderr,
		"walk: Insufficient memory to load movement. Sorry.\n");
	exit(1);
      }
      fscanf(movefile,"%g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\t%g, %g, %g\n", t[0], t[1], t[2]);
      mptr->rmove->transl[0] = t[0];
      mptr->rmove->transl[1] = t[1];
      mptr->rmove->transl[2] = t[2];

      fscanf(movefile,"}\n");
      num_relmoves++;
      break;
    }

    case MOVEREL_ZOOM: {
      if ((mptr->rmove = (RelMovement *) malloc(sizeof(RelMovement))) == 0) {
	fprintf(stderr,
		"walk: Insufficient memory to load movement. Sorry.\n");
	exit(1);
      }
      fscanf(movefile,"%g %g %g\n", &t[0], &t[1], &t[2]);
      if (Option.debug) printf("\t%g, %g, %g\n", t[0], t[1], t[2]);
      mptr->rmove->transl[0] = t[0];
      mptr->rmove->transl[1] = t[1];
      mptr->rmove->transl[2] = t[2];

      fscanf(movefile,"}\n");
      num_relmoves++;
      break;
    }
      
    default: {
      fprintf(stderr,"%s: Invalid movement read from %s\n", ProgName,
	      filename);
      exit(-1); /* DoExit(-1); */
      break;
    }
    }
    nmptr = (mptr+1);
    mptr->next = nmptr;
    mptr++;
    moves_read++;
  }
  fclose(movefile);
  if (Option.debug) {
    printf("\n\tNumber of absolute moves   = %d\n", num_absmoves);
    printf("\tNumber of relative moves   = %d\n", num_relmoves);
    printf("\tTotal moves read from file = %d\n\n", num_moves);
  }
  return (num_moves);
}

/**********************************************************************/
/* Log a relative move */
/**********************************************************************/
void LogRelMove(rmove, moves, mtype)
     RelMovement *rmove;
     Movement **moves;
     char mtype;
{
  Movement *nmptr;

  if (moves_logged == 0) {
    if ((Mtail = *moves = nmptr = (Movement *) malloc(sizeof(Movement)))== 0) {
      fprintf(stderr, "walk: Insufficient memory to log movement. Sorry.\n");
      exit(1);
    }
    nmptr->mtype = mtype;
    nmptr->rmove = rmove;
  } else {
    if ((Mtail->next = nmptr = (Movement *) malloc(sizeof(Movement))) == 0) {
      fprintf(stderr, "walk: Insufficient memory to log movement. Sorry.\n");
      exit(1);
    }
    nmptr->mtype = mtype;
    /* printf("mtype = %c\n", mtype); */
    nmptr->rmove = rmove;
    Mtail = nmptr;
  }
  moves_logged++;
}
  
/**********************************************************************/
/* Log an absolute move */
/**********************************************************************/
void LogAbsMove(amove, moves, mtype)
     AbsMovement *amove;
     Movement **moves;
     char mtype;
{
  Movement *nmptr;

  if (moves_logged == 0) {
    if ((Mtail = *moves = nmptr = (Movement *) malloc(sizeof(Movement)))== 0) {
      fprintf(stderr, "walk: Insufficient memory to log movement. Sorry.\n");
      exit(1);
    }
    nmptr->mtype = mtype;
    nmptr->amove = amove;
  } else {
    if ((Mtail->next = nmptr = (Movement *) malloc(sizeof(Movement))) == 0) {
      fprintf(stderr, "walk: Insufficient memory to log movement. Sorry.\n");
      exit(1);
    }
    nmptr->mtype = mtype;
    nmptr->amove = amove;
    Mtail = nmptr;
  }
  moves_logged++;
}

/**********************************************************************/
/* Open logging file for output */
/**********************************************************************/
void InitMoveLog(filename,mode,nmoves)
     char *filename, *mode;
     int nmoves;
{
  if (!OpenMoveFile(filename,mode)) {
    exit(-1); 
  } else if (Option.debug) {
    printf("\nWriting moves to %s\n", filename);
  }
  fprintf(movefile,"%s %d\n", HEAD_NMOVES, nmoves);
}

/**********************************************************************/
/* Write an relative move to file */
/**********************************************************************/
void WriteRelMove(rmove, mtype)
     RelMovement *rmove;
     int mtype;
{
  switch(mtype) {
  case MOVEREL_ROT: 
    fprintf(movefile,"Move R {\n");
    fprintf(movefile,"%g %g %g\n", rmove->rot[0],
	    rmove->rot[1], rmove->rot[2]);
    break;

  case MOVEREL_SPIN: 
    fprintf(movefile,"Move P {\n");
    fprintf(movefile,"%g %g %g\n", rmove->rot[0],
	    rmove->rot[1], rmove->rot[2]);
    break;

  case MOVEREL_TRANSL: 
    fprintf(movefile,"Move T {\n");
    fprintf(movefile,"%g %g %g\n", rmove->transl[0],
	    rmove->transl[1], rmove->transl[2]);
    break;

  case MOVEREL_SCALE: 
    fprintf(movefile,"Move S {\n");
    fprintf(movefile,"%g %g %g\n", rmove->transl[0],
	    rmove->transl[1], rmove->transl[2]);
    break;
    
  case MOVEREL_ZOOM: 
    fprintf(movefile,"Move Z {\n");
    fprintf(movefile,"%g %g %g\n", rmove->transl[0],
	    rmove->transl[1], rmove->transl[2]);
    break;

  default: 
    fprintf(stderr,"%s: Invalid movement to write: %c\n", ProgName, mtype);
    exit(-1); /* DoExit(-1); */
  }
  fprintf(movefile,"}\n");
}

/**********************************************************************/
/* Write an absolute move to file */
/**********************************************************************/
void WriteAbsMove(amove)
     AbsMovement *amove;
{
  fprintf(movefile,"Move A {\n");

  fprintf(movefile, "  lookFrom %g %g %g\n", (amove->lookFrom.x),
	  (amove->lookFrom.y), (amove->lookFrom.z));
  fprintf(movefile, "  lookAt %g %g %g\n", (amove->lookAt.x),
	  (amove->lookAt.y), (amove->lookAt.z));
  fprintf(movefile, "  lookUp %g %g %g\n", (amove->lookUp.x),
	  (amove->lookUp.y), (amove->lookUp.z));
  fprintf(movefile, "  fovx %d fovy %d near %g far %g\n",
	  amove->fovx, amove->fovy, amove->near, amove->far);
  fprintf(movefile, "  xRes %d yRes %d\n", amove->xRes, amove->yRes);
  fprintf(movefile, "  bank %g\n", (amove->bank));

  fprintf(movefile,"}\n");
}

/**********************************************************************/
/* Write movements to file */
/**********************************************************************/
int WriteMoves(filename, moves, nmoves)
     char *filename;
     Movement *moves;
     int nmoves;
{
  int i;

  InitMoveLog(filename,WRITEF,nmoves);
  for(i=0;i<nmoves;i++) {
    if (moves->mtype == MOVEREL_ROT) 
      WriteRelMove(moves->rmove, MOVEREL_ROT);
    else if (moves->mtype == MOVEREL_SPIN) 
      WriteRelMove(moves->rmove, MOVEREL_SPIN);
    else if (moves->mtype == MOVEREL_TRANSL) 
      WriteRelMove(moves->rmove, MOVEREL_TRANSL);
    else if (moves->mtype == MOVEREL_SCALE) 
      WriteRelMove(moves->rmove, MOVEREL_SCALE);
    else if (moves->mtype == MOVEREL_ZOOM) 
      WriteRelMove(moves->rmove, MOVEREL_ZOOM);
    else if (moves->mtype == MOVEABS)
      WriteAbsMove(moves->amove);
    else {
      fprintf(stderr,"Invalid movement type: %c\n",moves->mtype);
      exit(1);
    }
    moves = moves->next;
  }
  fclose(movefile);
  if (Option.debug) printf("Moves written to file = %d\n\n",moves_logged);
}





