/**********************************************************************/
/* struct.h :                                                         */
/*                                                                    */
/* Final structures for radiosity / ray tracing program               */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#ifndef STRUCT_H
#define STRUCT_H 

/**********************************************************************/
/* Types for textures and surface, and image attributes               */
/* Output currently restricted to 3 colour RGB                        */
/**********************************************************************/
#define MAX_SPECTRA_SAMPLES 3
typedef struct { double samples[MAX_SPECTRA_SAMPLES]; } Spectra;
typedef struct { unsigned char r,g,b,a; } ColourChar;
typedef struct { double r, g, b, a; } Colour;

/**********************************************************************/
/* Object types                                                       */
/**********************************************************************/
/* light classifications : note classification is based on given      */
/* tolerances (not used)                                              */
#define LIGHT_DIR	1           /* Directional light (area)       */
#define LIGHT_NONDIR	2           /* Non-directional light (area)   */
#define LIGHT_PT	3           /* Point light source             */

/* types of primitive objects */
#define CONE		1          /* A cone                          */
#define CONE_ID         "cone"
#define CUBE		2          /* A cube                          */
#define CUBE_ID         "cube"
#define CYLINDER	3          /* A cylinder                      */
#define CYLINDER_ID     "cylinder"
#define DISK		4          /* A disk                          */
#define DISK_ID         "disk"
#define SPHERE		5          /* A sphere                        */
#define SPHERE_ID       "sphere"
#define MESH            6          /* A polygonal mesh                */
#define MESH_ID         "mesh"
#define PATCH		7          /* A 4 sided polygon in a mesh     */
#define PATCH_ID        "patch"
#define TRIANGLE	8          /* A 3 sided polygon in a mesh     */
#define TRI_ID         "triangle"

#define LIGHT_SRC	1          /* A primitive is a light source   */
#define NONLIGHT_SRC	2          /* A primitive is not light source */

/* types of nodes in scene */
#define OBJECT		1          /* An object                       */
#define SURFACE		2          /* A surface property              */
#define TEXTURE		3          /* A texture map                   */
#define ALL    		4

#define MAX_PATCH_VTX   4          /* Maximum vertices per patch      */
#define MAX_TRI_VTX     3          /* Number of vertices per triangle */
#define MAX_PATCH_CHILDREN 4       /* Maximum subpatches per patch    */

/* Polygon list */
typedef struct PolyListType {
  struct PolyType *patch;
  struct PolyListType *next;
} PolyList;

typedef struct PolyListHeadType {
  PolyList *front, *back;
  int num_polys;
} PolyListHead;

/* Vertex */
typedef struct VertexType {
  int id;
  char *name;
  Vector pos;            /* vertex coordinates                       */
  PolyList* polylist;    /* List of polygons vertex belongs to       */
  PolyListHead polyhead; /* Info on polygon list                     */
} Vertex;

/* U, V and possible W planes of quad or triangle */
typedef struct PolyUVW {
  Vector Nu, Nv, Nw;    /* normals for above planes                  */
  double Lu, Lv, Lw;    /* length of edges u,v,w                     */
  double du, dv, dw;
} polyUVW;

/* Patches / elements  */
typedef struct PolyType { /* Assume vertices of a patch are on the 
			     same plane */
  int id;               /* Id */
  char *name;
  int class;            /* classification (patch or element)         */
  Vertex *vert[4];      /* vertices                                  */
  Vector normal[4];	/* normals                                   */
  int changingNormal;	/* if TRUE, then normal changes within patch */
  double area;          /* area                                      */
  polyUVW *uvw;         /* u,v,w planes along edges of polygon       */
  int numVert;          /* Number of vertices in patch               */
  struct PolyType *child[4]; /* Subpatches of this patch / element   */
  double d;		/* d of plane on which patch / element 
			   resides */
  int level;            /* subdivision level                         */
  struct MeshType *Mfather; /* Mesh patch belongs to                 */
  struct PolyType *next; /* Next patch in mesh (if a patch)          */
  struct PolyType *Pfather; /* Larger patch or element, this element
			       belongs to */
  PolyList* Links;      /* Form-factor linkages                      */
  PolyListHead polyhead; /* Info on polygon list                     */
  Spectra unshot_B;     /* Unshot radiosity                          */
  Spectra B;            /* Total radiosity = emission originally     */
  Spectra               /* Vertex radiosity                          */
    vtx_B[MAX_PATCH_VTX];     
  Spectra               /* Vertex display radiosity                  */
    vtx_dB[MAX_PATCH_VTX];     
} Polygon;

/* List of vertices */
typedef struct Vertex_listType {  
  Vertex *vtx;              /* List of all vertices                  */
  struct Vertex_listType *next;
} Vertex_list;

/* Mesh */
typedef struct MeshType {
  int id;
  char *name;              /* mesh name                              */
  Polygon *polys;          /* polys in mesh                          */
  int num_polys;           /* number of polys in mesh                */
  struct ObjectType *Ofather;/* Object, mesh belongs to              */
  BoundingBoxType *box;    /* Bounding box for mesh                  */
} Mesh;

/**********************************************************************/
/* Physical properties */
/**********************************************************************/
/* Shading type */
typedef struct {
  Spectra Ka;              /* Ambient term (optional)                 */
  Spectra Kd;              /* Diffuse constant                        */
  Spectra Ks;              /* Specular constant                       */
  Spectra p;               /* Reflectance                             */
  Spectra t;               /* Refraction constant                     */
  Spectra n;               /* Index of refraction                     */
} ShadeType;

/* Radiating energy */
typedef struct {
  Spectra E;                /* Emittance                              */
  Spectra B;                /* Radiance                               */
} Radiance;

/* Transmission type */
typedef struct {
  Spectra trans;           /* Transmission constant                   */
  Spectra Ktrans;          /* Transmissivity                          */
  Spectra blendCol;       
} Transmit;

/* Specular type */
typedef struct {
  Spectra n;               /* Currently Phong model                   */
  Spectra metal;           
} Specularity;

/* Surface Property */
typedef struct {
  int id;
  char *name;
  ShadeType shade;        /* Reflection, refraction funct. parameters */
  Radiance rad;           /* Radiating energy function parameters     */
  Specularity spec;       /* Reflectance function parameters          */
  Transmit *trans;        /* Transmittance function parameters        */
} SurfaceProp;

/* Light specific properties */
typedef struct {
  int id;
  char *name;
  int lightType;          /* Type of light = point/area                */
  double standardDistance;/* distance that light has reasonable intensity */
  Vector shineFrom;	  /* position of light                         */
  double dropoff;	  /* spotlight factor. if 0, then no spotlight */
  double cutoff;	  /* angle at which spot no longer shines      */
  double cosCutoff;	  /* not user defined                          */
  Vector shineAt;	  /* co-ordinate light is pointed at           */
  Vector shineDir;	  /* not user defined                          */
  int maxSamples;
  int usedSamples;
} LightProp;

/* Texture map properties */
typedef struct {
  int id;
  char *name;
  int class;              /* Type of texture                          */
  int resolution;         /* Resolution of texture                    */
  ColourChar *map;        /* texture map                              */
} TextureProp;

/**********************************************************************/
/* The scene */
/**********************************************************************/
typedef struct SphereInfo {          /* Sphere info */
  Vector orig;
  double r;
} SphereInfo;

typedef struct CylInfo {             /* Cylinder info */
  double h;
  double r1, r2;
} CylInfo;

typedef struct CubeInfo {            /* Cube info */
  double length;
} CubeInfo;

typedef struct ConeInfo {            /* Cone info */
  double h;
  double r1, r2;
} ConeInfo;

typedef struct ObjectType {
  int id;
  char *name;
  long rayID;	          /* for stats/etc                           */
  Matrix WToM;	          /* world space to model space              */
  Matrix MToW;	          /* model space to world space              */
  char *primtype;         /* Primitive type (cube, cone, etc)        */
  int primid;             /* Primtive id (CONE, CUBE, etc)           */
  SurfaceProp *surface;   /* Surface properties                      */
  TextureProp *texture;   /* A texture property                      */
  LightProp *light;       /* Light properties                        */
  Mesh *meshes;           /* Associated mesh for primitive           */
  int num_meshes;         /* number of meshes                        */
  char *info;             /* ptr to prim-dependent info,must be cast */
  int (*intersect) ();    /* ptr to intersection routine for prim    */
  BoundingBoxType *box;   /* Bounding box for object                 */
  int in_facingprim;      /* Is this an in facing primitive ?        */
} Objectt;                /* IRIS has Object type already...         */

/* The scene */
typedef struct SceneType {
  Objectt *objects;
  TextureProp *textures;
  int num_objects;
  int num_textures;
} Scene;

/* Some structure macros: reflectance, patch, object, etc */
#define obj_reflect(obj) ((obj)->surface->shade.p)
#define poly_reflect(ply) ((ply)->Mfather->Ofather->surface->shade.p)
#define poly_emit(ply) ((ply)->Mfather->Ofather->surface->rad.E)
#define poly_patch(p) ((p)->Pfather)
#define poly_object(p) ((p)->Mfather->Ofather)

/**********************************************************************/
/* Function prototypes */
/**********************************************************************/
extern void print_PolyList();
extern void print_Polygon();
extern void print_Vertex();
extern void print_VertexList();
extern void print_Colour();
extern void print_Spectra();
extern void print_ShadeType();
extern void print_Radiance();
extern void print_LightProp();
extern void print_TextureMap();
extern void print_Object();
extern void print_Scene();
extern char *ProgName;

#endif /* STRUCT_H */
