/*LINTLIBRARY*/

/*  @(#)sunview.c 1.2 92/01/05
 *
 *  SunView dependent graphics routines used by popi.
 *  written by Rich Burridge - Sun Microsystems.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include <suntool/tty.h>

Canvas canvas ;
Cursor busy_cursor, main_cursor ;
Frame cframe, tframe ;
Icon popi_icon ;
Panel panel ;
Panel_item slider ;
Pixrect *memarea ;                       /* Offscreen image area. */
Pixrect *pr ;
Pixwin *cpw ;
Tty ttysw ;

short busy_cursor_array[] = {
#include <images/hglass.cursor>
} ;
mpr_static(busy_cursor_pr, 16, 16, 1, busy_cursor_array) ;

unsigned short icon_image[] = {
#include "popi.icon"
} ;
mpr_static(icon_pr, 64, 64, 1, icon_image) ;

extern int errno ;
extern char *sys_errlist[] ;

static void canvas_proc(), canvas_repaint(), destroy_proc() ;

char cmsname[MAXLINE] ;
char ttybuf[MAXLINE] ;    /* Input from user in ttysw window. */
unsigned char *mptr ;     /* Pointer to scanline data. */

int iscolor ;             /* Set if this is a color screen. */
int opercent = -1 ;       /* Previous display percent done value. */
int tbufptr ;             /* Current pointer into ttybuf. */

/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c)     - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(l,y,w,c)    - to draw an image scanline triple.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype, pos) - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)   - load new colormap.
 */


void
disp_init(argc, argv)        /* Called from main at the start. */
int argc ;
char *argv[] ;
{
  char title[MAXLINE] ;      /* Used to constructure window title lines. */
  int cht ;                  /* Height of the popi term window. */
  int cx ;                   /* X position of the popi term window. */
  int cy ;                   /* Y position of the popi term window. */
  int ttyfd ;                /* File descriptor for tty subwindow. */

  STRCPY(ttybuf, "") ;       /* Zeroise tty input buffer. */
  tbufptr = 0 ;              /* Reset tty buffer pointer. */

  if (getenv("WINDOW_PARENT") == NULL)
    {
      FPRINTF(stderr,"%s: Not a native SunView window\n", ProgName) ;
      exit(1) ;
    }

  popi_icon = icon_create(ICON_IMAGE, &icon_pr, 0) ;     /* Create icon. */
  SPRINTF(title, "%s commands", ProgName) ;
  tframe = window_create((Window) 0,      FRAME,    /* Create tty frame. */
                        FRAME_ICON,       popi_icon,
                        WIN_ROWS,         10,
                        WIN_COLUMNS,      80,
                        FRAME_LABEL,      title,
                        FRAME_ARGS,       argc, argv,
                        0) ;
  ttysw = window_create(tframe,          TERM,     /* Create tty window. */
                        TTY_ARGV,        TTY_ARGV_DO_NOT_FORK,
                        0) ;

  cx = (int) window_get(tframe, WIN_X) ;
  cy = (int) window_get(tframe, WIN_Y) ;
  cht = (int) window_get(tframe, WIN_HEIGHT) ;
  SPRINTF(title, "%s image canvas", ProgName) ;
  cframe = window_create((Window) 0,       FRAME,  /* Create canvas frame. */
                         WIN_X,            cx,
                         WIN_Y,            cy + cht + 10,
                         FRAME_LABEL,      title,
                         FRAME_SHOW_LABEL, TRUE,
                         0) ;
  panel = window_create(cframe,   PANEL,    /* Create panel for slider. */
                        WIN_SHOW, FALSE,
                        0) ;
  slider = panel_create_item(panel, PANEL_SLIDER,
                             PANEL_LABEL_STRING, "% done",
                             PANEL_MIN_VALUE,    0,
                             PANEL_MAX_VALUE,    100,
                             0) ;
  canvas = window_create(cframe,             CANVAS,     /* Create canvas. */
                         WIN_WIDTH,          Xsize,
                         WIN_HEIGHT,         Ysize,
                         CANVAS_REPAINT_PROC, canvas_repaint,
                         CANVAS_RETAINED,    FALSE,
                         WIN_EVENT_PROC,     canvas_proc,
                         0) ;
  cpw = canvas_pixwin(canvas) ;
  scr_depth = cpw->pw_pixrect->pr_depth;
  window_fit(cframe) ;

  memarea = mem_create(Xsize, Ysize, scr_depth) ;
  (void) pr_rop(memarea, 0, 0, Xsize, Ysize, PIX_CLR, memarea, 0, 0) ;

  (void) signal(SIGHUP, SIG_IGN) ;
  ttyfd = (int) window_get(ttysw, TTY_TTY_FD) ;
  (void) dup2(ttyfd, 0) ;
  (void) dup2(ttyfd, 1) ;
  (void) dup2(ttyfd, 2) ;

  (void) window_set(canvas, WIN_INPUT_DESIGNEE,
                     (int) window_get(ttysw, WIN_DEVICE_NUMBER), 0) ;
 
  main_cursor = window_get(canvas, WIN_CURSOR) ;
  busy_cursor = cursor_create(CURSOR_IMAGE, &busy_cursor_pr, 0) ;

  iscolor = (scr_depth > 1) ? 1 : 0 ;
  if (dtype == IS_MONO) iscolor = 0 ;
  if (!iscolor) dtype = IS_MONO ;
  pr = mem_create(Xsize, 1, scr_depth) ;
  SPRINTF(cmsname, "popi%10ld", getpid()) ;

  (void) notify_interpose_destroy_func(cframe, destroy_proc) ;
  (void) notify_interpose_destroy_func(tframe, destroy_proc) ;
  (void) window_set(tframe, WIN_SHOW, TRUE, 0) ;
  (void) window_set(cframe, WIN_SHOW, TRUE, 0) ;
  (void) notify_dispatch() ;      /* Make the window appear. */
  (void) notify_do_dispatch() ;
  init_dither() ;    /* Initialise dither arrays and variables. */
}


void
disp_finish()    /* Called from main prior to exit - null routine. */
{}



/*ARGSUSED*/
void
disp_imgstart(width, height, ncolors)  /* Called prior to drawing an image. */
int width, height, ncolors ;
{
  (void) window_set(cframe, WIN_SHOW, TRUE, 0) ;
  (void) pw_writebackground(cpw, 0, 0, Xsize, Ysize, PIX_CLR) ;
  (void) window_set(panel, WIN_SHOW, FALSE, 0) ;
  (void) notify_dispatch() ;
}


void
disp_imgend()                /* Called after drawing an image. */
{}


void
disp_putline(lines, y, width, ncolors)    /* Draw an image scanline. */
unsigned char **lines ;
int y, width, ncolors ;
{
  int color, i, j, off, x ;
  unsigned char *line, tmp ;

  mptr = (unsigned char *) ((struct mpr_data *) pr->pr_data)->md_image ;
  if ((dtype == IS_GRAY || dtype == IS_MONO) && ncolors == 3)
    line = ntsc_luma(lines, y, width) ;
  else
    line = &lines[0][y*width] ;

  if (scr_depth == 32)
    {
      off = y * width ;
      if (colors == 1)
        for (i = j = 0; i < width; i++)
          {
            tmp       = lines[0][off+i] ;
            mptr[j++] = 0 ;                          /* X */
            mptr[j++] = tmp ;                        /* B */
            mptr[j++] = tmp ;                        /* G */
            mptr[j++] = tmp ;                        /* R */
          }
        else
          for (i = j = 0; i < width; i++)
            {
              mptr[j++] = 0 ;                        /* X */
              for (color = 2; color >= 0; color--)
                  mptr[j++] = lines[color][off+i] ;  /* G, B, R */
            }  
    }
  else if (dtype == IS_COLOR || dtype == IS_GRAY)
    {
      for (x = 0; x < width; x++) mptr[x] = line[x] ;
    }
  else
    {
      halftone(line, y, width) ;
    }
  (void) pr_rop(memarea, 0, y, width, 1, PIX_SRC, pr, 0, 0) ;
  (void) pw_rop(cpw, 0, y, width, 1, PIX_SRC, pr, 0, 0) ;
}


disp_getchar()                /* Get next user typed character. */
{
  int c ;

  if (!tbufptr)                 /* If buffer empty, get next line. */
    if (gets(ttybuf) == NULL) {	/* fake up 'q' for ^D... */
      ttybuf[0] = 'q';
      ttybuf[1] = 0;
    }
  c = ttybuf[tbufptr++] ;  
  if (c == '\0')                       /* Is it end of input buffer? */
    {
      c = '\n' ;
      tbufptr = 0 ;                    /* Reset to force another read. */
      set_cursor(BUSY_CUR) ;           /* We will now get busy! */
    }
  return c ;
}


disp_prompt()        /* Display popi prompt and clear input line. */
{
  char *prompt = "\r-> " ;

  set_cursor(NORMAL_CUR) ;
  PRINTF(prompt) ;
  FFLUSH(stdout) ;
  return(sizeof prompt - 1) ;
}


void
disp_error(errtype, pos)    /* Display error message. */
int errtype ;
int pos ;
{
  int i ;

  if (errtype & ERR_PARSE)
    {
      for (i = 1; i < pos; ++i) PUTC('-', stderr) ;
      PUTC('^', stderr) ;
      PUTC('\n', stderr) ;
    }

  FPRINTF(stderr, "%s\n", ErrBuf) ;

/* We assume errno hasn't been reset by the preceding output */

  if (errtype & ERR_SYS)
    FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]) ;
  FFLUSH(stderr) ;
}


void
disp_percentdone(percent)
int percent ;
{
  if (percent == opercent) return ;   /* Same as last time? */
  panel_set_value(slider, percent) ;
  if (percent == 0)   (void) window_set(panel, WIN_SHOW, TRUE,  0) ;
  if (percent == 100) (void) window_set(panel, WIN_SHOW, FALSE, 0) ;
  opercent = percent ;
  (void) notify_dispatch() ;
}


void
disp_resize(width, height)    /* Resize popi image window. */
int width, height ;
{
  Rect *rect ;

  rect           = (Rect *) LINT_CAST(window_get(cframe, FRAME_OPEN_RECT)) ;
  rect->r_height = height ;
  rect->r_width  = width ;
  (void) window_set(cframe, FRAME_OPEN_RECT, rect, 0) ;
  (void) window_set(canvas, WIN_WIDTH, width, 0) ;
  (void) window_set(canvas, WIN_HEIGHT, height, 0) ;

  (void) pr_destroy(pr) ;
  (void) pr_destroy(memarea) ;
  pr      = mem_create(width, 1, scr_depth) ;
  memarea = mem_create(width, height, scr_depth) ;
  (void) pr_rop(memarea, 0, 0, width, height, PIX_CLR, memarea, 0, 0) ;
}


void
disp_colormap(n, red, green, blue)      /* Load new colormap. */
int n ;
unsigned char *red, *green, *blue ;
{
  SPRINTF(cmsname, "popi%10ld", getpid()) ;
  (void) pw_setcmsname(cpw, cmsname) ;
  (void) pw_putcolormap(cpw, 0, n, red, green, blue) ;
}


/*ARGSUSED*/
static void
canvas_proc(window, event)
Window window ;
Event *event ;
{
  window_default_event_proc(canvas, event, (char *) 0) ;
}


/*ARGSUSED*/
static void
canvas_repaint(canvas, pw, repaint_area)
Canvas canvas ;
Pixwin *pw ;
Rectlist *repaint_area ;
{
  (void) pw_rop(cpw, 0, 0, Xsize, Ysize, PIX_SRC, memarea, 0, 0) ;
}


/*ARGSUSED*/
static void
destroy_proc(client, status)
Notify_client client ;
Destroy_status status ;
{
  exit(0) ;
}


set_cursor(type)
enum cur_type type ;
{
  switch (type)
    {
        case BUSY_CUR  :
            (void) window_set(canvas, WIN_CURSOR, busy_cursor, 0) ;
            break ;
        case NORMAL_CUR:
            (void) window_set(canvas, WIN_CURSOR, main_cursor, 0) ;
    }
}
