/*LINTLIBRARY*/

/*  @(#)x11.c 1.3 92/01/05
 *
 *  X11 dependent graphics routines used by popi.
 *  written by Rich Burridge - Sun Microsystems.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#ifdef HASPOLL
#include <poll.h>
#endif /*HASPOLL*/
 
#define  POPI_BORDER_WIDTH  2

#define  FRAME_MASK  (ExposureMask)

unsigned short icon_image[] = {
#include "popi.icon"
} ;

#define  POPI_INPUT  0      /* Types of input selected on. */
#define  KEY_INPUT   1

#ifdef NOSELECT
#ifdef HASPOLL
struct pollfd pfd[2] ;
unsigned long npfd ;         /* Number of file descriptors to monitor. */
#endif /*HASPOLL*/
#else NOSELECT
#ifdef NO_43SELECT
int fullmask ;               /* Full mask of file descriptors to check on. */
int readmask ;               /* Readmask used in select call. */
#else
fd_set fullmask ;            /* Full mask of file descriptors to check on. */
fd_set readmask ;            /* Readmask used in select call. */
#endif /*NO_43SELECT*/
#endif /*NOSELECT*/

Atom protocol_atom, kill_atom ;
Colormap cmap = NULL ;
Cursor busy_cursor, main_cursor ;
Display *dpy ;
GC gc, pix_gc ;
Pixmap memarea, popi_icon, load_icon() ;
Visual *visual ;
Window frame, frame_icon, root ;
XColor ccol ;
XEvent event ;
XGCValues gc_val ;
XSizeHints size ;
XWMHints wm_hints ;

unsigned long gc_mask ;
int screen ;
int xfd ;                 /* Server connection file descriptor. */
unsigned long backgnd, foregnd ;
unsigned long palette[CMAPLEN] ;
int colsused = 0 ;

int iscolor ;                  /* Set if this is a color screen. */
int ix = 0 ;                   /* Initial X position of the icon. */
int iy = 0 ;                   /* Initial Y position of the icon. */

extern char geometry[] ;       /* X11 geometry information. */
extern char x11_display[] ;    /* X11 display information. */
extern int iconic ;            /* Set if the window is in an iconic state. */
extern unsigned char *mptr ;   /* Pointer to scanline data. */

/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c)     - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(l,y,w,c)    - to draw an image scanline triple.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype, pos) - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)   - load new colormap.
 */


void
disp_init(argc, argv)        /* Called from main at the start. */
int argc ;
char *argv[] ;
{
  unsigned int h, w ;       /* Window dimensions. */
  int flags ;
  int x, y ;                /* Window position. */
  XSetWindowAttributes winattrs ;

  if ((dpy = XOpenDisplay(x11_display)) == NULL)
    {
      FPRINTF(stderr,"%s: Couldn't open display %s\n", ProgName,
              (getenv ("DISPLAY") ? getenv("DISPLAY") : x11_display)) ;
      exit(1) ;
    } 

  screen = DefaultScreen(dpy) ;
  root   = RootWindow(dpy, screen) ;
  xfd    = ConnectionNumber(dpy) ;

/* Setup the file descriptors for X server connection and standard input. */

#ifdef NOSELECT
#ifdef HASPOLL
  pfd[0].fd     = xfd ;
  pfd[1].fd     = 0 ;
  pfd[0].events = pfd[1].events = POLLIN ;
  npfd          = 2L ;
#endif /*HASPOLL*/
#else
#ifdef NO_43SELECT
  fullmask = 1 ;
  fullmask |= (1 << xfd) ;
#else
  FD_ZERO(&fullmask) ;
  FD_SET(0, &fullmask) ;
  FD_SET(xfd, &fullmask) ;
#endif /*NO_43SELECT*/
#endif /*NOSELECT*/

  if (!geometry)
    STRCPY(geometry, XGetDefault(dpy, ProgName, "Geometry")) ;

  scr_depth = DefaultDepth(dpy, screen) ;

  init_dither() ;            /* Initialise dither arrays and variables. */

  iscolor = (scr_depth > 2) ;
  if (dtype == IS_MONO) iscolor = 0 ;
  if (!iscolor) dtype = IS_MONO ;

  foregnd   = BlackPixel(dpy, screen) ;
  backgnd   = WhitePixel(dpy, screen) ;
  popi_icon = load_icon(icon_image) ;

  rev_video = (scr_depth == 1 && BlackPixel(dpy, screen) == 0) ;
  size.flags = PMinSize | PMaxSize | PPosition | PSize ;
  size.x = 0 ;
  size.y = 0 ;
  size.max_width  = size.min_width  = size.width  = Xsize ;
  size.max_height = size.min_height = size.height = Ysize ;

  if (strlen(geometry))
    {
      flags = XParseGeometry(geometry, &x, &y, &w, &h) ;
      if (XValue & flags)
        {
          if (XNegative & flags)
            x = DisplayWidth(dpy, screen) + x - size.width ;
            size.flags |= USPosition ;
            size.x = x ;
        }
      if (YValue & flags)
        {
          if (YNegative & flags)
            y = DisplayHeight(dpy, screen) + y - size.height ;
            size.flags |= USPosition ;
            size.y = y ;
        }
    }    

  winattrs.background_pixel = backgnd ;
  winattrs.border_pixel     = backgnd ;
  winattrs.backing_store    = NotUseful ;
  winattrs.bit_gravity      = NorthWestGravity ;
  winattrs.save_under       = False ;
  winattrs.event_mask       = FRAME_MASK ;

  frame = XCreateWindow(dpy, root,
                        size.x, size.y, size.width, size.height,
                        POPI_BORDER_WIDTH, CopyFromParent,
                        InputOutput, CopyFromParent,
                        CWBackPixel   | CWBackingStore | CWBitGravity |
                        CWBorderPixel | CWEventMask    | CWSaveUnder,
                        &winattrs) ;

  memarea = XCreatePixmap(dpy, frame, size.width, size.height, scr_depth) ;

  protocol_atom = XInternAtom(dpy, "WM_PROTOCOLS", False) ;
  kill_atom = XInternAtom(dpy, "WM_DELETE_WINDOW", False) ;

#ifndef X11R3
  XSetWMProtocols(dpy, frame, &kill_atom, 1) ;
#endif /*!X11R3*/

  XSetStandardProperties(dpy, frame, "popi", NULL, popi_icon,
                         argv, argc, &size) ;

  wm_hints.icon_x      = ix ;
  wm_hints.icon_y      = iy ;
  wm_hints.input       = True ;
  wm_hints.icon_pixmap = popi_icon ;
  wm_hints.flags       = IconPositionHint | InputHint | IconPixmapHint ;
  if (iconic)
    {
      wm_hints.initial_state = IconicState ;
      wm_hints.flags |= StateHint ;
    }
  XSetWMHints(dpy, frame, &wm_hints) ;

  gc_mask = GCFunction | GCForeground | GCBackground | GCGraphicsExposures ;
  gc_val.foreground = foregnd ;
  gc_val.background = backgnd ;
  gc_val.function   = GXcopy ;
  gc_val.graphics_exposures = False ;
  gc = XCreateGC(dpy, root, gc_mask, &gc_val) ;

  pix_gc = DefaultGC(dpy, screen) ;

  main_cursor = XCreateFontCursor(dpy, XC_top_left_arrow) ;
  busy_cursor = XCreateFontCursor(dpy, XC_watch) ;

  gc_val.foreground = backgnd ;
  gc_val.background = foregnd ;
  XChangeGC(dpy, gc, GCBackground | GCForeground, &gc_val) ;
  XFillRectangle(dpy, memarea, gc, 0, 0,
                 (unsigned int) size.width, (unsigned int) size.height) ;

  XSelectInput(dpy, frame, FRAME_MASK) ;
  XMapWindow(dpy, frame) ;
  XSync(dpy, 0) ;
}


void
disp_finish()      /* Called from main prior to exit - null routine. */
{
}


/*ARGSUSED*/
void
disp_imgstart(width, height, ncolors)  /* Called prior to drawing an image. */
int width, height, ncolors ;
{
  PRINTF("Displaying image.\n") ;
  XClearWindow(dpy, frame) ;
}


void
disp_imgend()      /* Called after drawing an image - null routine. */
{
}


void
disp_putline(lines, y, width, ncolors)    /* Draw an image scanline triple. */
pixel_t **lines ;
int y, width, ncolors ;
{
  XImage *image ;
  int i, len ;
  unsigned char *line ;

  len = (scr_depth == 1) ? ((width / 8) + 1) : width ;
  mptr = (unsigned char *) Emalloc((unsigned int) len) ;
  if ((dtype == IS_GRAY || dtype == IS_MONO) && ncolors == 3)
    line = ntsc_luma(lines, y, width) ;
  else
    line = &lines[0][y*width] ;

  if (dtype == IS_COLOR || dtype == IS_GRAY)
    {
      for (i = 0; i < len; i++) mptr[i] = palette[line[i]] ;
    }
  else
    {
      halftone(line, y, width) ;
    }

  image = XCreateImage(dpy, DefaultVisual(dpy, screen), scr_depth,
                       (scr_depth == 1) ? XYPixmap : ZPixmap,
                       0, (char *) mptr, width, 1, 8, len) ;
  XPutImage(dpy, memarea, pix_gc, image, 0, 0, 0, y,
            (unsigned) image->width, (unsigned) image->height) ;
  XDestroyImage(image) ;
  XCopyArea(dpy, memarea, frame, gc, 0, y, width, 1, 0, y) ;
}


disp_getchar()       /* Get next user typed character. */
{
  char c ;
  int fd ;
  struct timeval *tval = NULL ;

  XSync(dpy, 0) ;
  for (;;)
    {
#ifdef NOSELECT
#ifdef HASPOLL
      fd = -1 ;
      POLL(pfd, npfd, -1) ;
           if (pfd[0].revents == POLLIN) fd = POPI_INPUT ;
      else if (pfd[1].revents == POLLIN) fd = KEY_INPUT ;
#else
      FPRINTF(stderr, "This situation in not being properly handled yet.\n") ;
#endif /*HASPOLL*/
#else
      readmask = fullmask ;
#ifdef NO_43SELECT
      SELECT(32, &readmask, 0, 0, tval) ;
           if (readmask && (1 << xfd)) fd = POPI_INPUT ;
      else if (readmask && (1 << 0))   fd = KEY_INPUT ;
#else
      SELECT(FD_SETSIZE, &readmask, (fd_set *) 0, (fd_set *) 0, tval) ;
           if (FD_ISSET(xfd, &readmask)) fd = POPI_INPUT ;
      else if (FD_ISSET(0,   &readmask)) fd = KEY_INPUT ;
#endif /*NO_43SELECT*/
#endif /*NOSELECT*/

      switch (fd)
        {     
          case POPI_INPUT : check_X_events() ;
                            break ;
          case KEY_INPUT  : READ(0, &c, 1) ;
                            if (c == RETURN || c == LINEFEED)
                              set_cursor(BUSY_CUR) ;
                            return(c) ;
        }
    }
}


void
disp_percentdone(percent)
int percent ;
{
  static int lastpercent = 100 ;

  if (percent == 100)
    {
      PRINTF("\r    \n") ;
      FFLUSH(stdout) ;
      return ;
    }
  if (percent != lastpercent && percent % 5 == 0)
    {
      PRINTF("\r%2d%% ", percent) ;
      FFLUSH(stdout) ;
      lastpercent = percent ;
    }
}


void
disp_resize(width, height)              /* Resize popi image window. */
int width, height ;
{
  XSizeHints hints ;
  long supplied ;

  XResizeWindow(dpy, frame, width, height) ;
#ifdef X11R3
  XGetNormalHints(dpy, frame, &hints) ;
#else
  XGetWMNormalHints(dpy, frame, &hints, &supplied) ;
#endif /*X11R3*/
  hints.flags      = PMinSize | PMaxSize ;
  hints.max_width  = hints.min_width  = width ;
  hints.max_height = hints.min_height = height ;
#ifdef X11R3
  XSetNormalHints(dpy, frame, &hints) ;
#else
  XSetWMNormalHints(dpy, frame, &hints) ;
#endif /*X11R3*/

  XFreePixmap(dpy, memarea) ;
  memarea = XCreatePixmap(dpy, frame, width, height, scr_depth) ;
  gc_val.foreground = backgnd ;
  gc_val.background = foregnd ;
  XChangeGC(dpy, gc, GCBackground | GCForeground, &gc_val) ;
  XFillRectangle(dpy, memarea, gc, 0, 0,
                 (unsigned int) width, (unsigned int) height) ;
}


void
disp_colormap(cmaplen, red, green, blue)      /* Load new colormap. */
int cmaplen ;
unsigned char *red, *green, *blue ;
{
  int i, newmap ;
  Visual *visual ;
  XSetWindowAttributes winattrs ;
 
  if (DisplayCells(dpy, screen) <= 2) return ;
  visual = DefaultVisual(dpy, screen) ;
  if (colsused) XFreeColors(dpy, cmap, palette, colsused, 0) ;
  if (cmap != NULL) XFreeColormap(dpy, cmap) ;

  newmap = 0 ;
  cmap = DefaultColormap(dpy, screen) ;

/*  Attempt to use the default colormap. If we can't allocate all the colors
 *  we need, then attempt to create a private colormap.
 */

  ccol.flags = DoRed | DoGreen | DoBlue ;
  iscolor = 1 ;
  for (i = 0; i < CMAPLEN; i++)
    {
      ccol.red   = red[i]   << 8 ;
      ccol.green = green[i] << 8 ;
      ccol.blue  = blue[i]  << 8 ;
      if (!XAllocColor(dpy, cmap, &ccol))
        if ((visual->class == StaticColor) ||
            (visual->class == StaticGray))
          {
            FPRINTF(stderr, "popi: XAllocColor failed on a static visual\n") ;
            iscolor = 0 ;
            return ;
          }
        else
          { 

/*  We can't allocate the colors shareable so free all the colors we had
 *  allocated and create a private colormap.
 */

            XFreeColors(dpy, cmap, palette, i, 0) ;
            newmap = 1 ;
            break ;
          }
        palette[i] = ccol.pixel ;
    }

  if (newmap)
    {
      if ((visual->class == PseudoColor) || (visual->class == GrayScale))
        cmap = XCreateColormap(dpy, root, visual, AllocNone) ;
      else
        cmap = XCreateColormap(dpy, root, visual, AllocAll) ;
 
      if (!XAllocColorCells(dpy, cmap, False, NULL, 0, palette, cmaplen))
        {
          FPRINTF(stderr, "popi: failed to allocate private colormap.\n") ;
          iscolor = 0 ;
          return ;
        } 
 
      ccol.flags = DoRed | DoGreen | DoBlue ;
      for (i = 0; i < cmaplen; i++)
        {
          ccol.pixel = palette[i] ;
          ccol.red   = red[i]   << 8 ;
          ccol.green = green[i] << 8 ;
          ccol.blue  = blue[i]  << 8 ;
          XStoreColor(dpy, cmap, &ccol) ;
        } 
    }
 
  winattrs.colormap = cmap ;
  XChangeWindowAttributes(dpy, frame, CWColormap, &winattrs) ;
}


disp_prompt()        /* Display popi prompt and clear input line. */
{
  static char prompt[] = "-> " ;

  set_cursor(NORMAL_CUR) ;
  PRINTF(prompt) ;
  FFLUSH(stdout) ;
  return(sizeof(prompt-1)) ;
}


void
disp_error(errtype, pos)    /* Display error message. */
int errtype ;
int pos ;
{
  extern int errno ;
  extern char *sys_errlist[] ;

  if (errtype & ERR_PARSE)
    {
      int i ;

      for (i = 1; i < pos; ++i) PUTC('-', stderr) ;
      PUTC('^', stderr) ;
      PUTC('\n', stderr) ;
    }

  FPRINTF(stderr, "%s\n", ErrBuf) ;

/* We assume errno hasn't been reset by the preceding output. */

  if (errtype & ERR_SYS)
    FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]) ;
  FFLUSH(stderr) ;
}


check_X_events()      /* Check and process pending X events. */
{
  XClientMessageEvent *ev ;

  XSync(dpy, 0) ;
  while (XPending(dpy) && (XPeekEvent(dpy, &event), (event.type == Expose)))
    {
      XNextEvent(dpy, &event) ;

      switch (event.type)
        {
          case ClientMessage    : /* Catch ICCCM kill from WM. */

                                  ev = (XClientMessageEvent *) &event ;
                                  if (ev->message_type == protocol_atom &&
                                      ev->data.l[0] == kill_atom)
                                    exit(0) ;
                                  break ;

          case Expose           : process_expose((XExposeEvent *) &event) ;
        }
    }
}


Pixmap
load_icon(sbuf)
unsigned short sbuf[] ;
{
  char cbuf[512] ;
  int i ;
  GC igc ;
  Pixmap pixmap ;
  XImage *image ;

  for (i = 0; i < 256; i++)
    {
      cbuf[i*2  ] = (sbuf[i] >> 8) & 0xFF ;
      cbuf[i*2+1] =  sbuf[i]       & 0xFF ;
    }
  pixmap = XCreatePixmap(dpy, root, 64, 64, 1) ;
  gc_mask = GCFunction | GCForeground | GCBackground | GCGraphicsExposures ;
  gc_val.foreground         = foregnd ;
  gc_val.background         = backgnd ;
  gc_val.function           = GXcopy ;
  gc_val.graphics_exposures = False ;
  igc   = XCreateGC(dpy, pixmap, gc_mask, &gc_val) ;
  image = XCreateImage(dpy, DefaultVisual(dpy, screen), 1, XYPixmap,
                       0, cbuf, 64, 64, BitmapPad(dpy), 0) ;
  XPutImage(dpy, pixmap, igc, image, 0, 0, 0, 0, 64, 64) ;
  return(pixmap) ;
}


process_expose(event)
XExposeEvent *event ;
{
  int doframe = 0 ;

  do
    {
      if (event->count == 0)
        if (event->window == frame) doframe++ ;
    }    
  while (XCheckMaskEvent(dpy, ExposureMask, (XEvent *) event)) ;

  if (doframe)
    XCopyArea(dpy, memarea, frame, gc, 0, 0, Xsize, Ysize, 0, 0) ;
}


set_cursor(type)
enum cur_type type ;
{
  switch (type)
    {
      case BUSY_CUR   : XDefineCursor(dpy, frame, busy_cursor) ;
                        break ;
      case NORMAL_CUR : XDefineCursor(dpy, frame, main_cursor) ;
    }
}
