
/*  @(#)sparcgen.c 1.3 93/06/01
 *
 *  Popi sparc runtime assembler generation routines.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#ifdef SOLARIS2
#include <sys/exechdr.h>
#else
#include <sys/exec.h>
#endif /*SOLARIS2*/

#include <unistd.h>
#include <string.h>
#include "popi.h"
#include "expr.h"

#define DUMPFILE "popi.out"

typedef unsigned long Inst ;
typedef int (*intfun) P((int, int)) ;

static void gen       P((Tree *)) ;
static void binop     P((Tree *, int, int, int)) ;
static void binconst  P((Tree *, int, int, int)) ;
static void dodump    P((Inst *, int)) ;
static void clearregs P((void)) ;
static void freereg   P((int)) ;

static int findreg    P((void)) ;

extern intfun mul     P((void)) ;
extern intfun Pdiv    P((void)) ;
extern intfun mod     P((void)) ;

static Inst *code ;
static ccset, boolean, last = 18, regs[6], spp, sppmax, result ;

#define  zero       0
#define  tmp        1
#define  base       24        /* first arg */
#define  offset     25        /* etc. */
#define  NREGVAR    5
#define  REGVARBASE 26
#define  rx         16
#define  ry         17
#define  sp         14
#define  o0         8        /* first arg out */
#define  o1         9
#define  o2         10
#define  o3         11

#define  OP(rd, op, rs, i, r_or_d)  \
                ((2 << 30) | (rd << 25) | (op << 19) | (rs << 14) | \
                 (i << 13) | (r_or_d & 017777))
#define  MOP(rd, op, rs, i, r_or_d) \
                ((3 << 30) | (rd << 25) | (op << 19) | (rs << 14) | \
                 (i << 13) | (r_or_d & 017777))
#define  SETHI(rd, op, imm)  ((rd << 25) | (op << 22) | (imm & 017777777))

#define  _ADD(a, b, c)      *code++ = OP(c, 000, a, 0, b)
#define  _ADDcc(a, b, c)    *code++ = OP(c, 020, a, 0, b)
#define  _SUB(a, b, c)      *code++ = OP(c, 004, a, 0, b)
#define  _SUBcc(a, b, c)    *code++ = OP(c, 024, a, 0, b)
#define  _XOR(a, b, c)      *code++ = OP(c, 003, a, 0, b)
#define  _XORcc(a, b, c)    *code++ = OP(c, 023, a, 0, b)
#define  _OR(a, b, c)       *code++ = OP(c, 002, a, 0, b)
#define  _ORcc(a, b, c)     *code++ = OP(c, 022, a, 0, b)
#define  _AND(a, b, c)      *code++ = OP(c, 001, a, 0, b)
#define  _ANDcc(a, b, c)    *code++ = OP(c, 021, a, 0, b)
#define  _NOR(a, b, c)      *code++ = OP(c, 006, a, 0, b)
#define  _XORI(a, b, c)     *code++ = OP(c, 003, a, 1, b)
#define  _XORIcc(a, b, c)   *code++ = OP(c, 023, a, 1, b)
#define  _ORI(a, b, c)      *code++ = OP(c, 002, a, 1, b)
#define  _ORIcc(a, b, c)    *code++ = OP(c, 022, a, 1, b)
#define  _ADDI(a, b, c)     *code++ = OP(c, 000, a, 1, b)
#define  _ADDIcc(a, b, c)   *code++ = OP(c, 020, a, 1, b)
#define  _ANDI(a, b, c)     *code++ = OP(c, 001, a, 1, b)
#define  _ANDIcc(a, b, c)   *code++ = OP(c, 021, a, 1, b)
#define  _LBU(a, b, c)      *code++ = MOP(a, 001, b, 0, c)
#define  _LBIU(a, b, c)     *code++ = MOP(a, 001, b, 1, c)
#define  _SB(a, b, c)       *code++ = MOP(a, 005, b, 0, c)
#define  _SBI(a, b, c)      *code++ = MOP(a, 005, b, 1, c)
#define  _LW(a, b, c)       *code++ = MOP(a, 000, b, 0, c)
#define  _LWI(a, b, c)      *code++ = MOP(a, 000, b, 1, c)
#define  _SW(a, b, c)       *code++ = MOP(a, 004, b, 0, c)
#define  _SWI(a, b, c)      *code++ = MOP(a, 004, b, 1, c)
#define  _SETHI(a, b)       *code++ = SETHI(a, 04, b >> 10)
#define  _SLL(a, b, c)      *code++ = OP(c, 045, a, 0, b)
#define  _SRL(a, b, c)      *code++ = OP(c, 046, a, 0, b)
#define  _SRA(a, b, c)      *code++ = OP(c, 047, a, 0, b)
#define  _SLLI(a, b, c)     *code++ = OP(c, 045, a, 1, b)
#define  _SRLI(a, b, c)     *code++ = OP(c, 046, a, 1, b)
#define  _SRAI(a, b, c)     *code++ = OP(c, 047, a, 1, b)

#define  _CMP(a, b)    *code++ = OP(zero,024,a,0,b)  /* _SUBcc(a,b,zero) ; */
#define  _CMPI(a, b)   *code++ = OP(zero,024,a,1,b)  /* _SUBIcc(a,b,zero) ; */

#define  _B(cond, disp)    SETHI(cond, 02, disp)

#define  ALWAYS            010
#define  EQ                001
#define  NE                011
#define  GE                013
#define  LE                002
#define  GT                012
#define  LT                003
#define  _JAL(a)           *code++ = OP(15, 070, a, 0, zero)
#define  _SAVE(a)          *code++ = OP(sp, 074, sp, 1, a)
#define  _RET              *code++ = 0x81c7e008    /* jmpl      %o7+8,%g0  */
#define  _RESTORE          *code++ = 0x81e80000    /* restore   %g0,%g0,%g0  */
#define  _NOP              *code++ = SETHI(zero, 04, 0)

#define  _PUSH(a) { \
                    _SW(a, sp, spp) ; \
                    spp += 4 ; \
                    if (spp > sppmax) \
                      sppmax = spp ; \
                  }

#define  _POP(a) { \
                   spp -= 4 ; \
                   _LWI(a, sp, spp) ; \
                 }

/* Calculate possible cost to generate code tree. */

int
cost(t)
Tree *t ;
{
  int cst = 0, i ;

  switch(t->t)
    {
      case T_Pang   : cst += 7 ;
                      break ;
      case T_Prad   : cst += 12 ;
                      break ;
      case T_Mul    : cst += 5 ;
                      break ;
      case T_Div    :
      case T_Mod    : cst += 7 ;
                      break ;
      case T_Sin    :
      case T_Cos    :
      case T_Tan    : cst += 4 ;
                      break ;
      case T_Rand   : cst += 5 ;
                      break ;
      case T_Coord  : cst += 3 ;
                      break ;
      case T_Polar  : cst += 9 ;
                      break ;
      case T_Assign :
      case T_Var    : cst += 2 ;    /* Average register/memory variables. */
                      break ;
      default       : cst++ ;
    }
  for (i = 0; i < t->siz; i++)
    cst += cost(t->kids[i]) ;
  return(cst) ;
}


static void
loadimm(i, j)
int i, j ;
{
  if (j > 0x1777 || j < 0)
    {
      _SETHI(i, j) ;
      _ORIcc(i, (j & 01777), i) ;
    }
  else _ORIcc(zero, j, i) ;
}


static void
readvar(var, r)
int var, r ;
{
  struct var *v = findVar(var) ;
 
  if (v->sc == Reg)
    {
      _ADDcc(zero, v->v, r) ;
      ccset = 1 ;
    }
  else
    {
      loadimm(tmp, (int) &v->v) ;
      _LW(r, tmp, zero) ;
      ccset = 0 ;
    }
}
 

static void
setvar(var, r)
int var, r ;
{
  struct var *v = findVar(var) ;
 
  if (v->sc == Reg) _ADDcc(zero, r, v->v) ;
  else
    {
      loadimm(tmp, (int) &v->v) ;
      _SW(r, zero, tmp) ;
    }
}
 

extern void
allocRegs(v, n)
struct var *v ;
int n ;
{
  int i;
 
  for (i = 0; i < n; i++)
    {
      if (i < 4)
        {
          v[i].sc = Reg ;
          v[i].v = i+26 ;
        }
      else v[i].sc = Mem ;
    }
}

 
static int sl_ptor, sl_mul, sl_div, sl_mod, sl_rad, sl_ang ;
 

static void
varfns(t)
Tree *t ;
{
  int i ;
 
  for (i = 0; i < nbuiltin; i++)
    if (builtins[i].t == t->t)
      {
        char buf[16] ;

        SPRINTF(buf, ".%s", builtins[i].n) ;
        if (builtins[i].func != NULL) builtins[i].var = allocName(buf) ;
        break ;
      }

  switch (t->t)
    {
      case T_Mul   :
      case T_Sqr   : sl_mul = allocName(".mul") ;
                     break ;
      case T_Div   : sl_div = allocName(".Pdiv") ;
                     break ;
      case T_Sin   :
      case T_Cos   :
      case T_Tan   :
      case T_Mod   : sl_mod = allocName(".mod") ;
                     break ;
      case T_Prad  : sl_rad = allocName(".dist") ;
                     break ;
      case T_Pang  : sl_ang = allocName(".angle") ;
                     break ;
      case T_Polar : sl_ptor = allocName(".ptor") ;
                     break ;
    }
  for (i = 0; i < t->siz; i++)
    if (t->kids[i]) varfns(t->kids[i]) ;
}


static void
jvar(r)
int r ;
{
  struct var *v = findVar(r) ;

  if (v->sc == Reg) _JAL(v->v) ;
  else
    {
      readvar(r, tmp) ;
      _JAL(tmp) ;
    }
}


void
compile(t)
Tree *t ;
{
  char dumpstr[MAXLINE] ;
  int color, i, ino, tvar ;
  Inst prog[2048], *remem1, *remem2 ;
  void (*progp)() = (void (*)()) prog ;

  code = prog ;
  if (t->t != T_Top) run_error(ERR_TOP) ;
  clearregs() ;
  sl_ang = sl_ptor = sl_rad = sl_div = sl_mul = sl_mod = -1 ;
  for (i = 0; i < nbuiltin; i++) builtins[i].var = -1 ;
  varfns(t) ;
  allocVars() ;
  last = findreg() ;
  code = prog ;
  _SAVE(0) ;                /* Value is or'ed in later. */
  for (i = 0; i < nbuiltin; i++)
    if (builtins[i].var != -1 && builtins[i].func != NULL)
      {
        loadimm(last, (int) builtins[i].func) ;
        setvar(builtins[i].var, last) ;
      }

  if (sl_div != -1)
    {
      loadimm(last, (int) Pdiv()) ;
      setvar(sl_div, last) ;
    }
  if (sl_mul != -1)
    {
      loadimm(last, (int) mul()) ;
      setvar(sl_mul, last) ;
    }
  if (sl_mod != -1)
    {
      loadimm(last, (int) mod()) ;
      setvar(sl_mod, last) ;
    }
  if (sl_rad != -1)
    {
      loadimm(last, (int) dist) ;
      setvar(sl_rad, last) ;
    }
  if (sl_ang != -1)
    {
      loadimm(last, (int) angle) ;
      setvar(sl_ang, last) ;
    }
  if (sl_ptor != -1)
    {
      loadimm(last, (int) polar) ;
      setvar(sl_ptor, last) ;
    }
  ccset = boolean = 0 ;
  gen(t->kids[0]) ;
  loadimm(ry, Ysize-1) ;
  remem1 = code ;
  ccset = boolean = 0 ;
  gen(t->kids[1]) ;
  loadimm(rx, Xsize-1) ;
  remem2 = code ;
  ccset = boolean = 0 ;
  gen(t->kids[2]) ;
  ccset = boolean = 0 ;
  gen(t->kids[3]) ;
  _ADDI(offset, -1, offset) ;
  _CMP(rx, zero) ;
  *code = _B(NE, (remem2 - code)) ;
  code++ ;
  _ADDI(rx, -1, rx) ;
  ccset = boolean = 0 ;
  gen(t->kids[4]) ;
  _CMP(ry, zero) ;
  *code = _B(NE, (remem1 - code)) ;
  code++ ;
  _ADDI(ry, -1, ry) ;
  _RET ;
  _RESTORE ;
  *prog |= (-(sppmax + 80)) & 017777 ;
  dodump(prog, (code - prog) * sizeof (Inst)) ;
  ino = getImageNo("new") ;
  for (color = 0; color < colors; color++)
    progp(Images[ino].pix[color], (Xsize * Ysize) - 1) ;

  if (Debug)
    {
      SPRINTF(dumpstr, "echo ',-1?' | adb %s", DUMPFILE) ;
      system(dumpstr) ;
    }

  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


static void
gen(t)
Tree *t ;
{
  Inst *remem ;
  int i, j, saved, b1, b2, c1, c2 ;

  for (j = 0; j < nbuiltin; j++)
    {
      if (builtins[j].t == t->t && builtins[j].var != -1)
        {
          if (t->siz == 0)
            {
              jvar(builtins[j].var) ;
              _NOP ;
            }
          else
            {
              int i, args[4] ;
              static int oreg[] = { o0, o1, o2, o3 } ;
 
              for (i = 0; i < t->siz; i++)
                {
                  gen(t->kids[i]) ;
                  args[i] = result ;
                  if (result != rx && result != ry && (last = findreg()) == -1)
                    {
                      last = result ;
                      args[i] = -1 ;
                      _PUSH(last) ;
                    }
                }
              for (i = t->siz-1; i >= 0; i--)
                {
                  if (i == 0) jvar(builtins[j].var) ;
                  if (args[i] == -1)
                    {
                       _POP(oreg[i]) ;
                    }
                  else
                    {
                      _OR(zero, args[i], oreg[i]) ;
                      if (args[i] != rx && args[i] != ry) freereg(args[i]) ;
                    }
                }
            }
          _ADDcc(zero, o0, last) ;
          result = last ;
          boolean = 0 ;
          ccset = 1 ;
          return ;
        }
    }

  switch (t->t)
    {
      case T_Inum     : if (t->i == 0)
                          {
                            result = zero ;
                            ccset = 0 ;
                            boolean = 1 ;
                          }
                        else
                          {
                            loadimm(result = last, t->i) ;
                            boolean = !(t->i & ~1) ;
                            ccset = 1 ;
                          }
                        break ;
      case T_Var      : readvar(t->i, result = last) ;
                        boolean = 0 ;
                        break ;
      case T_Xcoord   : result = rx ;
                        boolean = 0 ;
                        ccset = 0 ;
                        break ;
      case T_Ycoord   : result = ry ;
                        boolean = 0 ;
                        ccset = 0 ;
                        break ;
      case T_Pang     : j = sl_ang ;
                        goto docode ;
      case T_Prad     : j = sl_rad ;
      docode          : _ADD(zero, rx, o0) ;
                        _ADD(zero, ry, o1) ;
                        loadimm(o2, Xsize / 2) ;
                        jvar(j) ;
                        loadimm(o3, Ysize / 2) ;
                        _ADDcc(zero, o0, last) ;
                        result = last ;
                        boolean = 0 ;
                        break ;
      case T_Assign   : gen(t->kids[0]) ;
                        setvar(t->i, result) ;
                        boolean = 0 ;
                        ccset = 0 ;
                        break ;
      case T_Bang     : gen(t->kids[0]) ;
                        if (!ccset) _CMP(result, zero) ;
                        *code++ = _B(EQ, 3) ;
                        _ADDIcc(zero, 1, last) ;
                        _ADDIcc(zero, 0, last) ;
                        result = last ;
                        boolean = 1 ;
                        ccset = 1 ;
                        break ;
      case T_Not      : gen(t->kids[0]) ;
                        _XORIcc(result, -1, last) ;
                        result = last ;
                        boolean = 0 ;
                        ccset = 1 ;
                        break ;
      case T_Neg      : gen(t->kids[0]) ;
                        _SUBcc(zero, result, last) ;
                        result = last ;
                        boolean = 0 ;
                        ccset = 1 ;
                        break ;
      case T_Abs      : gen(t->kids[0]) ;
                        _CMP(result, zero) ;
                        *code++ = _B(GE, 3) ;
                        _ADDcc(result, zero, last) ;
                        _SUBcc(zero, result, last) ;
                        result = last ;
                        boolean = 0 ;
                        ccset = 1 ;
                        break ;
      case T_Sqr      : gen(t->kids[0]) ;
                        _ADD(zero, result, o0) ;
                        jvar(sl_mul) ;
                        _ADD(zero, result, o1) ;
                        _ADDcc(zero, o0, last) ;
                        ccset = 1 ;
                        boolean = 0 ;
                        break ;
      case T_Sin      :
      case T_Cos      :
      case T_Tan      : gen(t->kids[0]) ;
                        loadimm(o1, MAX_ANG) ;
                        jvar(sl_mod) ;
                        _ADD(zero, result, o0) ;
                        _ADDcc(zero, o0, last) ;
                        _SLLI(last, 2, last) ;
                        loadimm(tmp, (int) ((t->t == T_Sin) ? sin_cache :
                                            (t->t == T_Cos) ? cos_cache :
                                                              tan_cache)) ;
                        _LW(last, tmp, last) ;
                        result = last ;
                        boolean = 0 ;
                        ccset = 0 ;
                        break ;
      case T_Cond     : gen(t->kids[0]) ;                   /* test */
                        if (!ccset) _CMP(result, zero) ;
                        remem = code ;
                        *code++ = _B(EQ, 0) ;
                        _NOP ;
                        gen(t->kids[1]) ;                /* iftrue */
                        b1 = boolean ;
                        c1 = ccset ;
                        if (result != last) _ADD(result, zero, last) ;
                        saved = last ;
                        *remem |= code - remem + 1 ;
                        remem = code - 1 ;
                        *code = code[-1] ;
                        code[-1] = _B(ALWAYS, 0) ;       /* fill delay slot */
                        code++ ;
                        gen(t->kids[2]) ;                /* iffalse */
                        b2 = boolean ;
                        c2 = ccset ;
                        if (result != saved) _ADD(result, zero, saved) ;
                        last = result = saved ;
                        boolean = (b1 & b2) ;
                        ccset = (c1 & c2) ;
                        *remem |= code - remem ;
                        break ;
      case T_Comma    : gen(t->kids[0]) ;
                        gen(t->kids[1]) ;
                        break ;
      case T_Land     :
      case T_Lor      : gen(t->kids[0]) ;
                        b1 = boolean ;
                             if (result != last) _ADDcc(result, zero, last) ;
                        else if (!ccset)         _CMP(result, zero) ;
                        saved = last ;
                        remem = code ;
                        if (t->t == T_Land) *code++ = _B(EQ, 0) ;
                        else                *code++ = _B(NE, 0) ;
                        _NOP ;
                        gen(t->kids[1]) ;
                        b2 = boolean ;
                        if (result != saved)
                          {
                            _ADDcc(result, zero, saved) ;
                            ccset = 1 ;
                          }
                        result = saved ;
                        *remem |= code - remem ;
                        if (!(b1 & b2))       /* "normalize" expr to {0, 1} */
                          {
                            if (!ccset) _CMP(result, 0) ;
                            *code++ = _B(NE, 3) ;
                            _ADDIcc(zero, 0, result) ;
                            _ADDIcc(zero, 1, result) ;
                            ccset = 1 ;
                          }
                        boolean = 1 ;
                        break ;
      case T_Mapcoord : gen(t->kids[0]) ;
                        saved = result ;
                        if (result != rx && result != ry &&
                            (last = findreg()) == -1)
                          run_error(ERR_NOREGS) ;
                        if (t->siz == 2)
                          {
                            gen(t->kids[1]) ;
                            j = last ;
                          }
                        else j = offset ;
                        if (t->i == 0) _SB(saved, base, j) ;
                        else
                          {
                            loadimm(tmp, (int) Images[t->i].pix[0]) ;
                            _SB(saved, tmp, j) ;
                          }
                        result = last ;
                        ccset = 0 ;
                        boolean = 0 ;
                        break ;
      case T_Coord    : if (t->i > nimages || t->i < 0) run_error(ERR_NOBUF) ;
                        FPRINTF(stderr, "gen: T_Coord - image %d\n", t->i) ;
                        if (t->siz == 1)
                          {
                            gen(t->kids[0]) ;
                            j = last ;
                          }
                        else j = offset ;
                        loadimm(tmp, (int) Images[t->i].pix[0]) ;
                        _LBU(last, tmp, j) ;
                        result = last ;
                        ccset = 0 ;
                        boolean = 0 ;
                        break ;
      case T_Nop      : break ;
      default         : gen(t->kids[0]) ;
                        saved = result ;
                        if (((Tree *) t->kids[1])->t == T_Inum &&
                            (i = ((Tree *) t->kids[1])->i) < 4096 && i > -4096)
                          {
                            binconst(t, result, i, last) ;
                          }
                        else
                          {
                            if (saved != rx && saved != ry &&
                                (last = findreg()) == -1)
                              {
                                last = saved ;
                                saved = -1 ;
                                _PUSH(last) ;
                              }
                            gen(t->kids[1]) ;
                            if (saved == -1)
                              {
                                _POP(tmp) ;
                                binop(t, tmp, result, last) ;
                              }
                            else
                              {
                                binop(t, saved, result, last) ;
                                if (saved != rx && saved != ry)
                                  freereg(saved) ;
                              }
                          }
                        result = last ;
    }
}


static void
binop(t, s1, s2, d)
Tree *t ;
int s1, s2, d ;
{
  boolean = 0 ;
  ccset = 1 ;
  switch (t->t)
    {
      default       : FPRINTF(stderr, "binop: operator %d\n", t->t) ;
                      run_error(ERR_NOOP) ;
      case T_Add    : _ADDcc(s1, s2, d) ;
                      break ;
      case T_Sub    : _SUBcc(s1, s2, d) ;
                      break ;
      case T_Polar  : _ADD(zero, s1, o0) ;
                      jvar(sl_ptor) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Mul    : _ADD(zero, s1, o0) ;
                      jvar(sl_mul) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Div    : _ADD(zero, s1, o0) ;
                      jvar(sl_div) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Mod    : _ADD(zero, s1, o0) ;
                      jvar(sl_mod) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Gt     : _CMP(s1, s2) ;
                      *code++ = _B(GT, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Lt     : _CMP(s1, s2) ;
                      *code++ = _B(LT, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ge     : _CMP(s1,s2) ;
                      *code++ = _B(GE, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Le     : _CMP(s1, s2) ;
                      *code++ = _B(LE, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Eq     : _CMP(s1, s2) ;
                      *code++ = _B(EQ, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ne     : _CMP(s1, s2) ;
                      *code++ = _B(NE, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Xor    : _XORcc(s1, s2, d) ;
                      break ;
      case T_And    : _ANDcc(s1, s2, d) ;
                      break ;
      case T_Or     : _ORcc(s1, s2, d) ;
                      break ;
      case T_Lshift : _SLL(s1, s2, d) ;
                      ccset = 0 ;
                      break ;
      case T_Rshift : _SRA(s1, s2, d) ;
                      ccset = 0 ;
    }
}


static void
binconst(t, s1, s2, d)
Tree *t ;
int s1, s2, d ;
{
  boolean = 0 ;
  ccset = 1 ;
  switch (t->t)
    {
      default       : run_error(ERR_NOCONST) ;
      case T_Add    : _ADDIcc(s1, s2, d) ;
                      break ;
      case T_Sub    : _ADDIcc(s1, -s2, d) ;
                      break ;
      case T_Mul    : {
                        int shift, tv ;
 
                        if (s2 < 0) _SUB(zero, s1, tmp) ;
                        else        _ADD(zero, s1, tmp) ;
                        _ADD(zero, s2 & 1 ? s1 : 0, d) ;
                        s2 &= ~1 ;
                        for (tv = shift = 0; s2; s2 >>= 1, shift++)
                          {
                            if (s2 & 1)
                              {
                                if (shift)
                                  {
                                    _SLLI(tmp, (shift - tv), tmp) ;
                                    tv = shift ;
                                  }
                                _ADDcc(tmp, d, d);
                              }        
                          }
                      }
                      break ;

      case T_Div    :
      case T_Mod    : loadimm(o1, s2) ;
                      jvar(((t->t == T_Mul) ? sl_mul :
                            (t->t == T_Mod  ? sl_mod : sl_div))) ;
                      _ADD(zero, s1, o0) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Gt     : _CMPI(s1, s2) ;
                      *code++ = _B(GT, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Lt     : _CMPI(s1, s2) ;
                      *code++ = _B(LT, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ge     : _CMPI(s1, s2) ;
                      *code++ = _B(GE, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Le     : _CMPI(s1, s2) ;
                      *code++ = _B(LE, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Eq     : _CMPI(s1, s2) ;
                      *code++ = _B(EQ, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ne     : _CMPI(s1, s2) ;
                      *code++ = _B(NE, 3) ;
                      _ADDIcc(zero, 1, d) ;
                      _ADDIcc(zero, 0, d) ;
                      boolean = 1 ;
                      break ;
      case T_Xor    : _XORIcc(s1, s2, d) ;
                      break ;
      case T_And    : _ANDIcc(s1, s2, d) ;
                      break ;
      case T_Or     : _ORIcc(s1, s2, d) ;
                      break ;
      case T_Lshift : _SLLI(s1, s2, d) ;
                      ccset = 0 ;
                      break ;
      case T_Rshift : _SRAI(s1, s2, d) ;
                      ccset = 0 ;
    }
}


static void
dodump(prog, size)
Inst *prog ;
int size ;
{
  if (Debug)
    {
      int fd = creat(DUMPFILE, 0644) ;
      struct exec exec ;

      exec.a_machtype = M_SPARC ;
      exec.a_magic    = OMAGIC ;
      exec.a_text     = size ;
      exec.a_data     = 0 ;
      exec.a_bss      = 0 ;
      exec.a_syms     = 0 ;
      exec.a_entry    = 0 ;
      exec.a_trsize   = 0 ;
      exec.a_drsize   = 0 ;
      if (fd < 0)
        {
          perror(DUMPFILE) ;
          exit(1) ;
        }
      WRITE(fd, &exec, sizeof(exec)) ;
      WRITE(fd, prog, size) ;
      CLOSE(fd) ;
    }
}


static void
clearregs()
{
  int i ;

  for (i = 0 ; i < 6 ; i++) regs[i] = 0 ;
}


static void
freereg(i)
int i ;
{
  regs[i-18] = 0 ;
}


static int
findreg()
{
  int i ;

  for (i = 0 ; i < 6 ; i++)
    if (regs[i] == 0)
      break ;
  if (i == 6) return(-1) ;
  regs[i] = 1 ;
  return(i+18) ;
}
