/*LINTLIBRARY*/

/*  @(#)apollo.c 1.3 92/04/16
 *
 *  Popi device driver for an Apollo display.
 *  Written by Tim Lambert - University of New South Wales.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include "graphics.h"
#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/pad.h>
#include <apollo/error.h>

/* geometry, and x11_display are used for x11 options - we don't use them */
char x11_display[MAXLINE] ;       /* X11 display information. */
char geometry[MAXLINE] ;          /* X11 geometry information. */

/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c,l)   - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(l,y,w,c)    - to draw an image scanline triple.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype)      - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)   - load new colormap
 */

status_$t            status;
gpr_$bitmap_desc_t   display_bitmap;
gpr_$offset_t        display_bitmap_size;
gpr_$rgb_plane_t     hi_plane;

#define  RES  8
int dither[RES][RES] = {        /* dither matrix */
  {   0, 128,  32, 160,   8, 136,  40, 168, },
  { 192,  64, 224,  96, 200,  72, 232, 104, },
  {  48, 176,  16, 144,  56, 184,  24, 152, },
  { 240, 112, 208,  80, 248, 120, 216,  88, },
  {  12, 140,  44, 172,   4, 132,  36, 164, },
  { 204,  76, 236, 108, 196,  68, 228, 100, },
  {  60, 188,  28, 156,  52, 180,  20, 148, },
  { 252, 124, 220,  92, 244, 116, 212,  84, },
} ;
void check(char *messagex)
{
   if (status.all)
   {   error_$print (status);
       printf("Error occurred while %s.\n", messagex);
   }
}

/*ARGSUSED*/
void
disp_init(argc,argv)           /* called from main at the atart. */
int argc;
char *argv[];
{
static short int         unit = 1;
static short int         disp_len = sizeof(gpr_$disp_char_t);
       short int         disp_len_returned;
       short int         unobscured;
gpr_$disp_char_t     display_characteristics;
static gpr_$display_mode_t mode = gpr_$direct;
        stream_$id_t         graphics_str;
        pad_$window_desc_t   window;
        gpr_$color_vector_t  colour_map;
        short int            i;
        int                  intense;
        gpr_$rgb_plane_t     plane;

   gpr_$inq_disp_characteristics(mode, unit, disp_len,
                                 &display_characteristics,
                                  &disp_len_returned, &status);
   check("in disp_init after inquiring");

   display_bitmap_size.x_size = Xsize;
   display_bitmap_size.y_size = Ysize;
   hi_plane                   = display_characteristics.n_planes - 1;

   window.top   =  0;     window.left   =  0;
   window.width =  Xsize;   window.height =  Ysize;
   pad_$create_window((char *)NULL,0,pad_$transcript,unit,window,&graphics_str,&status);
   pad_$set_full_window(graphics_str,(short)1,&window,&status);
   pad_$set_border(graphics_str,(short)1,false,&status);
   pad_$set_auto_close(graphics_str,(short)1,true,&status);
   gpr_$init(mode, graphics_str, display_bitmap_size,
             hi_plane, &display_bitmap, &status);
   check("in disp_init after initializing");
   gpr_$set_auto_refresh(true,&status);
   gpr_$set_obscured_opt(gpr_$pop_if_obs,&status);
   gpr_$set_cursor_active(true,&status);
   pad_$def_pfk(graphics_str,"M3  ","=",1,&status); /* M3 reports current cursor position */
   pad_$def_pfk(graphics_str,"M3U ","",0,&status); /* just in case M3U does something */
   for(plane=0;plane<=hi_plane;plane++)
	gpr_$set_raster_op(plane,gpr_$rop_not_dst,&status);  /* so that horiz_line below uses XOR */
   if(hi_plane==7){ /* 8 plane colour display */
       /* first 16 colours are used by DM, so we'll use remaining 240,
          mapping intensities in the range 0..255 to entries 16..255 */
       gpr_$inq_color_map( 0, 256, colour_map, &status );
       for(i=16;i<=255;i++){
           intense = (255*(i-16)/240);
           colour_map[i] = (intense << 16) + (intense << 8) + intense;
       }
       gpr_$acquire_display(&status);
       gpr_$set_color_map( 0, 256, colour_map, &status );
       gpr_$release_display(&status);
   }
}


void
disp_finish()                  /* called from main prior to exit. */
{
    gpr_$terminate(false,&status);
}


/* Called prior to drawing an image. */

void
disp_imgstart(width, height, ncolors, len)
int width, height, ncolors, len ;
{
}


void
disp_imgend()                  /* called after drawing an image. */
{
}


void
disp_putline(lines, y, width, ncolors)  /* called to draw image scanline y. */
pixel_t **lines;
int y, width, ncolors;
{
  linteger pixel_array[2000];
  pixel_t *line;
  static gpr_$window_t                  scanline = {{0,0},{1,1}};
  int x;

  line = ntsc_luma(lines, y, width);
  for (x=0;x<width;x++){
    if (hi_plane==7){
        pixel_array[x] = (gpr_$pixel_value_t) line[x]*240/256 +16;
    } else {
        pixel_array[x] = !(line[x] >= dither[y % RES][x % RES]);
    }
  }
  scanline.window_size.y_size = (short)1;
  scanline.window_size.x_size = (short)width;
  scanline.window_base.y_coord = (short)y;
  gpr_$set_obscured_opt(gpr_$pop_if_obs,&status);
  gpr_$acquire_display(&status); check("acquiring display");
  gpr_$write_pixels((gpr_$pixel_value_t *)pixel_array,scanline,&status); check("write pixels");
  gpr_$release_display(&status); 
}


disp_getchar()                 /* get next user typed character. */
{
  return(getchar());
}


disp_prompt()                  /* display popi prompt. */
{
  PRINTF("-> ");
  return 3;
}


void
disp_error(errtype, pos)            /* display error message. */
int	errtype,
	pos;
{
    extern int  errno;
    extern char *sys_errlist[];

    if (errtype & ERR_PARSE)
    {
        int     i;
 
        for (i=1; i < pos; ++i)
            PUTC('-', stderr);
        PUTC('^', stderr);
        PUTC('\n', stderr);
    }
 
    FPRINTF(stderr, "%s\n", ErrBuf);
    /* we assume errno hasn't been reset by the preceding output */
    if (errtype & ERR_SYS)
        FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]);
}

void horiz_line(percent)
int percent;
/* draw a horizontal line percent % of the way down */
{
#define MAX_WINDOWS 50
    gpr_$window_t vis_list[MAX_WINDOWS];
    int i;
    short int slots_total;
    short int x,y;

    y = Ysize * percent / 100;
    x = Xsize;

    gpr_$set_obscured_opt(gpr_$ok_if_obs,&status);
    gpr_$acquire_display(&status); check("acquiring display");
    gpr_$inq_vis_list (MAX_WINDOWS, &slots_total, vis_list, &status); check("inq vis list");
    for(i=0; i<slots_total; i++){
      gpr_$set_clip_window (vis_list[i], &status) ; check("set clip window");
      gpr_$move(0,y,&status);
      gpr_$line(x,y,&status);
    }
    gpr_$release_display(&status);
}



void
disp_percentdone(percent)                
int	percent;                             
{                                        
    static int	lastpercent = 100;
                                         
    if (percent != lastpercent){
        horiz_line(percent);
	if(!(percent == 0 || percent == 100))
	    horiz_line(lastpercent);
	lastpercent = percent;                  
    }                                    
}                                        


void
disp_resize(width, height)    /* Resize popi image window. */
int width, height ;
{
}


void
disp_colormap(n, red, green, blue)      /* Load new colormap. */
int n ;
unsigned char *red, *green, *blue ;
{
}
