/* file: perfpoly - do performance timing on code processing polygons */

/* Author: Fred Fisher
 * 
 * This program facilitates deciding which algorithm is fastest for a given
 * processing task. A number of random polygons are generated, then each
 * algorithm is called to process the set of polygons. Alternatively,
 * a polygon may be read from a file, then each algorithm is called.
 *
 * Customizing macro's and typedef's are in perf.h
 *
 * To add a new test:
 *	1) add the routine name to the 'PROG_OPTIONS' macro
 *	2) put the routine in this file or another to be resolved at link time.
 *	   Each routine should return a compatible value to be checked with
 *	   the CHECK_RETURN macro.
 *
 * you can control the amount of output with the verbosity option, try:
 *      perf 0 < file           ;number of vertices tested
 *      perf 1 < file           ;number of vertices, sorted results
 *      perf 2 < file           ;long summary after all vertex sets
 *      perf 3 < file           ;summary after one vertex set
 *      perf 4 < file           ;onPolygon #, long results of one test
 *      perf 5 < file           ;
 *      perf 6 < file           ;show polygon data
 *      perf 7 < file           ;trace algorithm
 *      perf 9 < file           ;the most
 *
 * arg2 is the number of times to loop when timing each algorithm:
 * 	perf xx 100000
 * will loop 100,000 times when timing each algorithm
 *
 * arg3 ... argN are for other options:
 *	z 	means use testing using 3D coordinate polygons (else 2D)
 *
 * if the polygon is being read from a file, and the file contains 999 for 
 * the first X coord, then program options are read from the file and 
 * random polygons are generated. See code.
 *
 * specify the following in 'PROG_OPTIONS'
 * 	<routine name>
 *		this is the actual "C" routine name.
 *	<number of times to test routine with a given polygon> 
 *		this should be > 1 / (HZ * approx. single test time in seconds)
 */
#define	PROG_OPTIONS( macName )						\
	macName(isConvexSchorn, NUM_TEST);				\
	macName(isConvexSchorn2, NUM_TEST);				\
	macName(isConvexSchorn3, NUM_TEST);				\
	macName(isConvexSchorn4, NUM_TEST);				\
	macName(isConvexSchorn5, NUM_TEST);				\
	macName(isConvexSchorn6, NUM_TEST);				\
	macName(isConvexFred,  NUM_TEST);				\
	macName(isConvexSloan, NUM_TEST);				\
	macName(isConvexHeckbert, NUM_TEST);				\
	macName(isConvexBoucher, NUM_TEST);				\

#if 0
#endif

#define	MAX_TEST_ROUTINE	50	/* max routines that may be specified */
#define	MAX_POLY_VERT		20000	/* max vertices that may be generated */
#define	MAX_TIME_VERT		10	/* max verts to keep in summary table */
#define	NUM_TEST		100000
#undef	NUM_TEST
#define	NUM_TEST		100

#include "stdio.h"
#include <math.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/times.h>

#define X       0
#define Y       1
#define Z       2

#define GIMax(x, y)             ( (x) >= (y) ? (x) :  (y))
#define GIMin(x, y)             ( (x) <= (y) ? (x) :  (y))

/*****************************************************************************/
int	verbose    = 0;		/* verbosity for debugging */

static	int	numPolygon = 10;/* num of diff polygons to try for each nvert */
static	int	minVert	   = 3;	/* min number of polygon vertices to generate */
static	int	maxVert	   = 10;/* max number of polygon vertices to generate */
static	float	bboxMin[3], bboxMax[3];	/* to save bounding box */

float	epsilon = 123.45;	/* epsilon used by some routines */

double	drand48();

/*****************************************************************************/
long
convertStarStopToUSec( start, stop )
long		       start, stop;
/* make this a routine so you can test the timer function and make sure HZ
 * is defined right on your machine. Use verbosity option 't' to test time.
 */
{
    long uSec = 1000000.0 *(float)(stop - start) / (float)(HZ);
    return( uSec );
}

/*****************************************************************************/
testTimer()
{
    long        timestart, timestop;
    struct  tms timebuf;
    float       uSec;

    printf("Start timer test...\n");
    timestart = times( &timebuf );
    while ( 1 ) {
	timestop = times( &timebuf ); 
	if ( convertStarStopToUSec(timestart, timestop) / 1000000.0 > 10.0) 
	   break;
    }
    timestop = times( &timebuf );
    uSec = convertStarStopToUSec( timestart, timestop );
    printf("\7\7\7End test in %g uSec, %g seconds\n", uSec, uSec / 1000000.0 );
}

/*****************************************************************************/
sortByFloatWithData( ndata, pfloat, pint, pchar )
int	ndata;		/* number of items to sort */
float	*pfloat;	/* ptr to array of floats */
int	*pint;		/* ptr to corresponding array of integers */
char	**pchar;	/* ptr to corresponding array of ptrs */
/* sort using the 'pfloat' array. Other data is optional */
{
    int		ii, home, theint;
    float	thefloat;
    char	*thepchar;

    for ( ii = 1;  ii < ndata;  ii++ ) {
	/* move data at location 'ii' to the proper spot */
	thefloat = pfloat[ii];
	if ( pint  ) theint = pint[ii];
	if ( pchar ) thepchar = pchar[ii];
	home = ii - 1;
	do {
	    if ( thefloat >= pfloat[home] ) break;
	    /* shift array data up one location */
	    pfloat[home+1] = pfloat[home];
	    if ( pint  ) pint[home+1]  = pint[home];
	    if ( pchar ) pchar[home+1] = pchar[home];
	} while ( --home >= 0 );
	pfloat[++home] = thefloat;
	if ( pint  ) pint[  home] = theint;
	if ( pchar ) pchar[ home] = thepchar;
    }
}


/*****************************************************************************/
readPolygon2D(stream, nvert, pVert )
FILE 	*stream;
int	*nvert;		/* return number of vertices read here */
float	pVert[][2];	/* return vertices here */
/* This routine reads in a stream of 2D polygon coords from the input stream */
{
    int		status, go = 1;
    float	dx, dy;

    if ( verbose >= 6 ) {
	printf("Read polygon data:\n");
    }
    *nvert = 0;
    while ( go ) {
	status = fscanf(stream,"%g%g",&(pVert[*nvert][X]), &(pVert[*nvert][Y]));
	switch ( status ) {
	    case EOF :
		go = 0;
		break;
	    case 2 : {
		if ( verbose >= 6 ) {
		    printf("%2d) %8g %8g\n", *nvert, 
					pVert[*nvert][X], pVert[*nvert][Y]);
		}
		if ( !(*nvert) ) {
		    /* init to compute bounding box */
		    bboxMin[X] = bboxMax[X] = pVert[*nvert][X];
		    bboxMin[Y] = bboxMax[Y] = pVert[*nvert][Y];
		} else {
		    /* adjust bounding box */
		    bboxMin[X] = GIMin( bboxMin[X], pVert[*nvert][X] );
		    bboxMax[X] = GIMax( bboxMax[X], pVert[*nvert][X] );
		    bboxMin[Y] = GIMin( bboxMin[Y], pVert[*nvert][Y] );
		    bboxMax[Y] = GIMax( bboxMax[Y], pVert[*nvert][Y] );
		}
		(*nvert)++;
		break;
	    }
	    default : {
    		fprintf( stderr, "\7\7\7???BAD POLYGON DATA\n");
		usage();
		break;
	    }
	}
    }
    dx = bboxMax[X] - bboxMin[X];
    dy = bboxMax[Y] - bboxMin[Y];
    epsilon = 1.0;
    if ( dx ) epsilon = GIMin( epsilon, dx );
    if ( dy ) epsilon = GIMin( epsilon, dy );
    epsilon *= 1e-5;
}

#include "perf.h"

/*****************************************************************************/
printPoly( nvert, pVert )
int        nvert;
Number	   pVert[][2];
{
    int	ii;

    for ( ii = 0;  ii < nvert;  ii++ ) {
	printf("%2d) %8g %8g\n", ii, pVert[ii][X], pVert[ii][Y]);
    }
}

/*****************************************************************************/
printResultHeader( nvert )
{
    if ( !nvert ) 
        printf( "uSecs/Test is the time taken for one polygon test\n");
    else
	printf(
	"uSecs/Test is the ave of the average time for a %d vertex polygon\n",
		nvert );

    printf("uSecs/Test  totalSec onPolygon  numTest nameTest\n");
    printf("---------- --------- --------- -------- --------\n");
}

/*****************************************************************************/
static	char	*progName = 0;
usage()
{
    fprintf(stderr, "Usage: %s [arg1 [arg2]]\n", progName );
    fprintf(stderr, "arg1 is either: 0-9 to specify verbosity level\n");
    fprintf(stderr, "                t   for timer test only\n");
    fprintf(stderr, "arg2 is the number of times to call test routine\n");
    fprintf(stderr, "     default value for arg2 is diff for each test\n");
    fprintf(stderr, "arg3 ... argn are for other flags:\n");
    fprintf(stderr, "		     z	 for using 3d polygons\n");
}

/*****************************************************************************/
int
main(argc,argv)
int argc;  char *argv[];
{
    int		doRandom = 0, ii, jj, onVert, onPolygon, onRoutine, onTest, 
		timeIndex, status[MAX_TEST_ROUTINE], 
		useTableNumTest = 1, fixedNumTest, useNumTest, use2d = 1;
    char	*pname[MAX_TEST_ROUTINE];
    long	timestart, timestop;
    struct  tms	timebuf;
    float	uSec, saveTime[MAX_TEST_ROUTINE];
    int		nvertex;	/* number of vertices in current polygon */
    Number	pgon2D[MAX_POLY_VERT][2];	/* current 2D polygon */
    Number	pgon3D[MAX_POLY_VERT][3];	/* current 3D polygon */
    float	uSecPerTest[MAX_TEST_ROUTINE][MAX_TIME_VERT];

    progName = argv[0];
    if ( argc > 1 ) {
        verbose = argv[1][0] - '0';
	if ( argv[1][0] == 't' ) { testTimer(); verbose = 0; exit(0); }
	if ( verbose < 0  ||  verbose > 9 ) { usage(); exit(0); }
    }
    if ( argc > 2 ) {
	useTableNumTest = 0;			/* ignore table value */
	fixedNumTest = atoi( argv[2] );
	if ( fixedNumTest <= 0 ) { usage(); exit(0); }
    }
    for ( ii = 3;  ii < argc;  ii++ ) {
	if ( argv[ii][0] == 'z' ) use2d = 0;
    }
    srand( 12345 );
    for ( ii = 0;  ii < MAX_TEST_ROUTINE;  ii++ ) {
	for ( jj = 0;  jj < MAX_TIME_VERT;  jj++ ) {
	    uSecPerTest[ ii ][ jj ] = 0.0;
	}
    }

    readPolygon2D( stdin, &nvertex, pgon2D );
    if ( pgon2D[0][X] > 998.0 ) {
	/* generate random data */
	doRandom   = 1;
				   	numPolygon   = pgon2D[0][Y];
	minVert         = pgon2D[1][X]; maxVert      = pgon2D[1][Y];
	useTableNumTest = pgon2D[2][X];	fixedNumTest = pgon2D[2][Y];
	if ( minVert < 0 ) {
	    minVert = 0;
	}
    } else {
 	/* use polygon data from file for test */
	minVert    = maxVert = nvertex;
	numPolygon = 1;
        for ( ii = 0; ii < nvertex; ii++ ) {
 	    /* copy 2D points from file over to 3D array */
	    pgon3D[ii][X] = pgon2D[ii][X];
	    pgon3D[ii][Y] = pgon2D[ii][Y];
	    pgon3D[ii][Z] = 0.0;
        }
    }

    if ( verbose ) {
        printf("Test %d polygons for each vertex count in the range [%d,%d]\n",
		numPolygon, minVert, maxVert );
	printf("epsilon = %g\n", epsilon );
    }

    /***** loop for each vertex size *****/
    for ( onVert = minVert;  onVert <= maxVert;  onVert++ ) {
	if ( verbose  &&  (maxVert - minVert) )
	    printf("********** Test for %d vertices\n", onVert );
	if ( onVert >= MAX_TIME_VERT ) {
	    timeIndex = MAX_TIME_VERT - 1;
	} else {
	    timeIndex = onVert;
	}
	
	if ( verbose >= 4 ) {
	    printResultHeader( 0 );
	}

	/***** test for each polygon of the current vertex size *****/
	for ( onPolygon = 0; onPolygon < numPolygon; onPolygon++ ) {
	    if ( verbose >= 4 ) printf("onPolygon %2d\n", onPolygon );

	    if ( doRandom ) {
		/* make an arbitrary polygon fitting 0-1 range in x and y */
		for ( ii = 0; ii < onVert; ii++ ) {
		    pgon2D[ii][X] = pgon3D[ii][X] = drand48();
		    pgon2D[ii][Y] = pgon3D[ii][Y] = drand48();
		                    pgon3D[ii][Z] = drand48();
		}
	    }


/*****************************************************************************/
	    /* define macro to run through all tests. It's true that this
	     * makes it hard to debug the code, but the type of code here 
	     * doesn't need much more than visual inspection, it makes it 
	     * easier to add new tests, AND it's working now.
	     */
#define	DO_TEST(rname, numtest)						\
	    if ( useTableNumTest ) useNumTest = numtest;		\
	    else useNumTest = fixedNumTest;				\
	    if ( use2d ) {						\
	        timestart = times( &timebuf );				\
	        for ( onTest = 0;  onTest < useNumTest;  onTest++ ) {	\
		    TEST_SETUP_2D;					\
		    status[onRoutine] = rname( onVert, pgon2D );	\
	        }							\
	        timestop = times( &timebuf );				\
									\
	    } else {							\
	        timestart = times( &timebuf );				\
	        for ( onTest = 0;  onTest < useNumTest;  onTest++ ) {	\
		    TEST_SETUP_3D;					\
		    status[onRoutine] = rname( onVert, pgon3D );	\
	        }							\
	        timestop = times( &timebuf );				\
	    }								\
	    uSec = convertStarStopToUSec(timestart, timestop);		\
	    uSecPerTest[ onRoutine ][timeIndex] += uSec / useNumTest;	\
	    if ( verbose >= 4 ) {					\
	        /* print results of this one test */			\
	        printf("%10g %9g %9d %8d ", uSec/useNumTest,		\
			uSec / 1000000.0, onPolygon, useNumTest, "rname" );\
	    }								\
	    FIX_RETURN;							\
	    if ( verbose >= 4 ) {					\
		PRINT_ONE_TEST_RESULT("rname");				\
	    }								\
	    CHECK_RETURN("rname");					\
	    onRoutine++;
/***** end macro ***** end macro ***** end macro ***** end macro *************/

	    onRoutine 	= 0;
	    PROG_OPTIONS( DO_TEST );
	}	/***** end of loop for this polygon *****/

/*****************************************************************************/
#define	PRINT_RESULTS(rname, numtest)					\
	if ( useTableNumTest ) useNumTest = numtest;			\
	else useNumTest = fixedNumTest;					\
	useNumTest *= numPolygon;					\
	uSec = uSecPerTest[ onRoutine ][timeIndex] / (float)(numPolygon); \
	if ( actuallyPrint ) {						\
	    printf("%10g %9g  aveOfAve %8d ", uSec,			\
	        uSecPerTest[ onRoutine ][timeIndex] / 1000000.0, useNumTest);\
	    PRINT_ONE_TEST_RESULT("rname");				\
	}								\
	pname[onRoutine] = "rname";					\
	saveTime[onRoutine] = uSec;					\
	onRoutine++;
/***** end macro ***** end macro ***** end macro ***** end macro *************/

	if ( verbose >= 3 ) {
	    int actuallyPrint = 1;
	    printf(
"***** results for %d vertices, ave of times for %d polygons\n", 
							onVert, numPolygon );
	    printResultHeader( onVert );
	    onRoutine = 0;
	    PROG_OPTIONS( PRINT_RESULTS );
	}

    }		/***** end of loop for this vertex count *****/

/*****************************************************************************/
#define	PRINT_ORDER(rname, numtest)					\
	printf("%d) %9g uSecs ", onRoutine, saveTime[onRoutine] );	\
	if ( onRoutine  &&  saveTime[0] ) {				\
	    printf( "(%6.2f%% or %4.2f times slower) ", 		\
		100.0 * saveTime[onRoutine] / saveTime[0] - 100.0,	\
		saveTime[onRoutine] / saveTime[0] );			\
	}								\
	printf("for %s\n", pname[onRoutine]);				\
	onRoutine++;
/***** end macro ***** end macro ***** end macro ***** end macro *************/

    if ( verbose  &&  (maxVert - minVert) ) {
	printf("\nSummary of test results ordered by number of vertices:\n");
    }
    for ( onVert = minVert;  onVert <= maxVert && verbose;  onVert++ ) {
	int actuallyPrint = 0;
	printf("\n********** Summary for %d vertices\n", onVert );
	if ( onVert >= MAX_TIME_VERT ) {
	    timeIndex = MAX_TIME_VERT - 1;
	} else {
	    timeIndex = onVert;
	}

	if ( verbose >= 2 ) {
	    actuallyPrint = 1;
	    printResultHeader( onVert );
	}
	/* fill 'saveTime[] and pname[] arrays, maybe print also */
	onRoutine   = 0;
	PROG_OPTIONS( PRINT_RESULTS );

  	printf("Summary ordered by speed\n");
	sortByFloatWithData( onRoutine, saveTime, 0, pname );
	onRoutine = 0;
	PROG_OPTIONS( PRINT_ORDER );
    }

    if ( status[0] == ReturnConvex  ||  status[0] == ReturnDegenerateConvex)
	return(0);
    return(-1);
}
