/*
 * convex.c: test convexity of polygon on stdin
 *
 * Paul Heckbert	27 Oct 1992
 */

#include <stdio.h>
#include "perf.h"

/* change the following two lines if your data types differ */
#define FORMAT "%F%F"

typedef enum {POS, NEG, ZERO} Sign;

/*
 * convexp: read polygon on stream fp, consisting of ascii (x,y) coords,
 * returning 1 if polygon convex and simple, 0 if not.
 *
 * A variation on Peter Schorn's algorithm (comp.graphics, 15 Oct 92).
 * Concise, but not super-optimized implementation.
 */

isConvexHeckbert(fp)
FILE *fp;
{
    int nturn;
    Sign anglesign, projsign;
    Number ax, ay, bx, by, cx, cy, px, py, qx, qy, rx, ry;

    if (fscanf(fp, FORMAT, &cx, &cy) != 2) return ReturnConvex;
    do {
	if (fscanf(fp, FORMAT, &ax, &ay) != 2) return ReturnConvex;
	px = ax-cx; py = ay-cy;
    } while (px==0 && py==0);
    rx = px; ry = py;
    anglesign = ZERO;	/* sign of angle at current vertex */
    projsign = ZERO;	/* sign of projection of current edge */
    nturn = 0;		/* number of changes in sign of projection */
    signchange2(&projsign, px, py);

    while (fscanf(fp, FORMAT, &bx, &by) == 2) {
	qx = bx-ax; qy = by-ay;
	/*
	 * invariants: if we call (bx,by)=vert[i], then (ax,ay)=vert[i-1],
	 * (qx,qy)=vert[i]-vert[i-1], (px,py)=vert[i-1]-vert[i-2],
	 * (cx,cy)=vert[0], (rx,ry)=vert[1]-vert[0]
	 * where we've eliminated repeated vertices from input
	 */
	if (qx!=0 || qy!=0) {
	    if (signchange(&anglesign, px*qy-py*qx)) return ReturnNonConvex;
	    if (signchange2(&projsign, qx, qy) && ++nturn==3) 
						     return ReturnNonConvex;
	    ax = bx; ay = by;
	    px = qx; py = qy;
	}
    }

    qx = cx-ax; qy = cy-ay;
    if (qx!=0 || qy!=0) {
	if (signchange(&anglesign, px*qy-py*qx)) return ReturnNonConvex;
	if (signchange2(&projsign, qx, qy) && ++nturn==3) 
						 return ReturnNonConvex;
	if (signchange(&anglesign, qx*ry-qy*rx)) return ReturnNonConvex;
    }
    else
	if (signchange(&anglesign, px*ry-py*rx)) return ReturnNonConvex;

    return ReturnConvex;
}

/*
 * signchange: test if val has a different sign than *sign, updating *sign.
 * Signs of zero (val==0) are ignored.
 */

int signchange(sign, val)
Sign *sign;
Number val;
{
    switch (*sign) {
	case POS: if (val<0) {*sign = NEG; return 1;} break;
	case NEG: if (val>0) {*sign = POS; return 1;} break;
	case ZERO:
	    if (val>0) *sign = POS;
	    else if (val<0) *sign = NEG;
	    break;
    }
    return 0;
}

/*
 * signchange2: test if the pair (a,b) has a lexicographically different sign
 * than *sign, updating *sign.
 * (a,b) is considered positive if a>0 || a==0 && b>0
 * (a,b) is considered negative if a<0 || a==0 && b<0
 */

int signchange2(sign, a, b)
Sign *sign;
Number a, b;
{
    return a!=0 ? signchange(sign, a) : signchange(sign, b);
}
