/*
  $Header: tools2.c,v 1.9 93/08/30 11:01:42 hmallat Exp $
  */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "misc.h"
#include "list.h"
#include "draw.h"
#include "X_misc.h"
#include "convert.h"
#include "tools1.h"
#include "tools2.h"
#include "undo.h"
#include "fill.h"
#include "error.h"

#define FIRST_NA first_x = first_y = -1;
#define PREV_NA prev_x = prev_y = -1;

/* FUNCTION:       point(int x, int y)
   DESCRIPTION:    Draws a point to the current picture on the position where
                   the mouse button is pressed.
   PARAMETERS:
   int x           Mouse x coordinate
   int y           Mouse y coordinate
   RETURNS:        Nothing.

   Written by Hannu Mallat, last modifications 14th July 1993.
*/
void point(int x, int y, int kind)
{
  extern PLOT xor_data, put_data;
  extern int b_w, b_h, x_off, y_off, show_edges;

  if(kind == PRESS)
    {
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      memorize(cur_pic, x-x_off, y-y_off, x+b_w-x_off, y+b_h-y_off, NULL,
	       "point");
      (*put_data)(x, y);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(x-x_off, y-y_off, x+b_w-x_off, y+b_h-y_off, TRUE);
      touch(cur_pic);
      PREV_NA;
    }
}

/* FUNCTION:       draw(int x, int y, int kind)
   DESCRIPTION:    The freehand draw. Draws a line from the last point where 
                   the mouse was, button down, to the current one.
   PARAMETERS:
   int x           Mouse x coordinate
   int y           Mouse y coordinate
   int kind        What kind of an event?
   RETURNS:        Nothing.

   Written by Hannu Mallat, last modifications 14th July 1993.
*/
void draw(int x, int y, int kind)
{
  extern PLOT xor_data, put_data;
  extern int prev_x, prev_y, cancelled, left, show_edges, b_w, b_h;

  switch(kind)
    {
    case PRESS:
      all_busy();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      memorize(cur_pic, 0, 0, cur_pic->width-1, cur_pic->height-1, NULL,
	       "freehand draw");
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      cancelled = FALSE;
      left = FALSE;
      linate(x, y, x, y, b_w, b_h, put_data, TRUE);
      touch(cur_pic);
      all_busy_nomore();
      prev_x = x;
      prev_y = y;
      break;
    case MOVE_W:
      if(cancelled == FALSE && prev_x != -1 && prev_y != -1 && 
	 (prev_x != x || prev_y != y))
	{
	  if(left == TRUE)
	    {
	      left = FALSE;
	      prev_x = x;
	      prev_y = y;
	    }
	  linate(prev_x, prev_y, x, y, b_w, b_h, put_data, TRUE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    case LEAVE:
      left = TRUE;
      break;
    case MOVE_WO:
    case CANCEL:
      cancelled = TRUE;
      left = FALSE;
      PREV_NA;
      break;
    }
}

/* FUNCTION:       airbrush(int x, int y, int kind)
   DESCRIPTION:    The airbrush.
   PARAMETERS:
   int x           Mouse x coordinate
   int y           Mouse y coordinate
   int kind        What kind of an event?
   RETURNS:        Nothing.

   Written by Hannu Mallat, last modifications 14th July 1993.
*/
void airbrush(int x, int y, int kind)
{
  extern PLOT xor_data, put_data;
  extern int ab_max_r, ab_flow, cancelled, left, show_edges, b_w, b_h, 
  x_off, y_off;
  int n, x_offset, y_offset, r, xmin = x, xmax = x, ymin = y, ymax = y;
  double angle;
  
  switch(kind)
    {
    case PRESS:
      all_busy();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      memorize(cur_pic, 0, 0, cur_pic->width-1, cur_pic->height-1, NULL,
	       "airbrush");
      cancelled = FALSE;
      left = FALSE;
	for(n = 0; n<ab_flow; n++)
	  {
	    angle = rnd_double()*(double)PI*2.0;
	    r = rnd_int() % ab_max_r;
	    x_offset = cos(angle)*r;
	    y_offset = sin(angle)*r;
	    (*put_data)(x+x_offset, y+y_offset);
	    if (x+x_offset<xmin) xmin = x+x_offset;
	    if (x+x_offset>xmax) xmax = x+x_offset;
	    if (y+y_offset<ymin) ymin = y+y_offset;
	    if (y+y_offset>ymax) ymax = y+y_offset;
	  }
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(xmin-x_off, ymin-y_off, 
	       xmax+b_w-x_off, ymax+b_h-y_off, TRUE); 
      touch(cur_pic);
      all_busy_nomore();
      break;
    case MOVE_W:
      if(left == TRUE)
	{
	  left = FALSE;
	  cancelled = FALSE;
	}
      if(cancelled == FALSE && (prev_x != x || prev_y != y))
	{
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  for(n = 0; n<ab_flow; n++)
	    {
	      angle = rnd_double()*(double)PI*2.0;
	      r = rnd_int() % ab_max_r;
	      x_offset = cos(angle)*r;
	      y_offset = sin(angle)*r;
	      line(x+x_offset, y+y_offset, x+x_offset, y+y_offset, put_data);
	      if (x+x_offset<xmin) xmin = x+x_offset;
	      if (x+x_offset>xmax) xmax = x+x_offset;
	      if (y+y_offset<ymin) ymin = y+y_offset;
	      if (y+y_offset>ymax) ymax = y+y_offset;
	    }
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(xmin-x_off, ymin-y_off, 
		   xmax+b_w-x_off, ymax+b_h-y_off, TRUE); 
	}
      break;
    case LEAVE:
      left = TRUE;
    case CANCEL:
      cancelled = TRUE;
      break;
    }
  PREV_NA;
}

/* FUNCTION:       rect(int x, int y, int kind)
   DESCRIPTION:    Interactive rectangle drawing with a rubberband
   PARAMETERS:
   int x           Mouse x coordinate
   int y           Mouse y coordinate
   int kind        What kind of an event?
   RETURNS:        Nothing.

   Written by Hannu Mallat, last modifications 14th July 1993.
*/
void rubber_rect(int x, int y, int kind)
{
  extern PLOT xor_data, put_data, fill_data, rubber_data;
  extern struct picture *cur_block, *fill_pattern;
  extern int prev_x, prev_y, cancelled, show_edges, first_x, first_y, b_w, b_h,
  x_off, y_off, filled, perimeter;
  struct picture *help_block = cur_block;

  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  boxate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, FALSE);
	  if(show_edges == TRUE) {plot_mask_edge(xor_data);update_edge();}
	}
      clear_undo();
      cancelled = TRUE;
      FIRST_NA;
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  first_x = x;
	  first_y = y;
	  prev_x = x;
	  prev_y = y;
	  boxate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	}
      else
	{
	  boxate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, FALSE);
	  all_busy();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  memorize(cur_pic, 
		   MIN(first_x, x)-x_off, MIN(first_y, y)-y_off,
		   MAX(first_x, x)+b_w-x_off, MAX(first_y, y)+b_h-y_off, NULL,
		   "rectangle");
	  if(filled == TRUE)
	    {
	      cur_block = fill_pattern;
	      fill_box(first_x, first_y, x, y, fill_data);
	      cur_block = help_block;
	      if(perimeter == TRUE)
		box(first_x, first_y, x, y, put_data);
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      blockade(MIN(first_x, x)-x_off, MIN(first_y, y)-y_off,
		       MAX(first_x, x)+b_w-x_off, MAX(first_y, y)+b_h-y_off, 
		       TRUE);
	    }
	  else if(filled == FALSE)
	    {
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      boxate(first_x, first_y, x, y, b_w, b_h, put_data, TRUE);
	    }
	  touch(cur_pic);
	  all_busy_nomore();
	  FIRST_NA;
	  PREV_NA;
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      cancelled = FALSE;
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  boxate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, FALSE);
	  boxate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       boxate(prev_x, prev_y, x, y, b_w, b_h, *put_pixel, all)
   DESCRIPTION:    To shorten and clean up code
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications th July 1993.
*/   
void boxate(int prev_x, int prev_y, int x, int y, int b_h, int b_w,
	    void (*put_pixel)(int x, int y), int all)
{
  extern PLOT xor_data;
  extern int show_edges, x_off, y_off;
  int x_offset = x_off, y_offset = y_off;
  
  if(b_h == 1 && b_w == 1)
    x_offset = y_offset = 0;
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  box(prev_x, prev_y, x, y, put_pixel);
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  blockade(MIN(prev_x, x)-x_offset, MIN(prev_y, y)-y_offset,
	   MIN(prev_x, x)+b_w-x_offset, MAX(prev_y, y)+b_h-y_offset, all);
  blockade(MAX(prev_x, x)-x_offset, MIN(prev_y, y)-y_offset,
	   MAX(prev_x, x)+b_w-x_offset, MAX(prev_y, y)+b_h-y_offset, all);
  blockade(MIN(prev_x, x)-x_offset, MIN(prev_y, y)-y_offset,
	   MAX(prev_x, x)+b_w-x_offset, MIN(prev_y, y)+b_h-y_offset, all);
  blockade(MIN(prev_x, x)-x_offset, MAX(prev_y, y)-y_offset,
	   MAX(prev_x, x)+b_w-x_offset, MAX(prev_y, y)+b_h-y_offset, all);
}

/* FUNCTION:       rubber_line(int x, int y, int kind, int which);
   DESCRIPTION:    Line or polyline
   PARAMETERS:
   x, y            coords
   kind            event
   which           LINE or POLYLINE
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 16th July 1993.
*/   
void rubber_line(int x, int y, int kind, int which)
{
  extern PLOT xor_data, put_data, rubber_data;
  extern int prev_x, prev_y, cancelled;
  extern int first_x, first_y, show_edges, b_w, b_h, x_off, y_off;

  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  linate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, FALSE);
	}
      clear_undo();
      cancelled = TRUE;
      FIRST_NA; 
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  prev_x = x;
	  prev_y = y;
	  first_x = x;
	  first_y = y;
	  if(which == POLYLINE)
	    {
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      memorize(cur_pic, 
		       0, 0, cur_pic->width-1, cur_pic->height-1, NULL,
		       "polyline");
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	    }
	  linate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	}
      else
	{
	  linate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, FALSE);
	  all_busy();
	  if(which == LINE)
	    {
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      memorize(cur_pic, 
		       MIN(first_x, x)-x_off, MIN(first_y, y)-y_off,
		       MAX(first_x, x)+b_w-x_off, MAX(first_y, y)+b_h-y_off, 
		       NULL, "line");
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	    }
	  linate(first_x, first_y, x, y, b_w, b_h, put_data, TRUE);
	  touch(cur_pic);
	  all_busy_nomore();
	  if(which == LINE)
	    {
	      FIRST_NA;
	      PREV_NA;
	    }
	  else
	    {
	      first_x = x;
	      first_y = y;
	      prev_x = x;
	      prev_y = y;
	      linate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	    }
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  linate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, FALSE);
	  linate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       linate(prev_x, prev_y, x, y, b_w, b_h, *put_pixel)
   DESCRIPTION:    To shorten and clean up the code
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications 15th July 1993.
*/   
void linate(int prev_x, int prev_y, int x, int y, int b_w, int b_h,
	    void(*put_pixel)(int x, int y), int all)
{
  extern PLOT xor_data;
  extern struct plist *pixel_p;
  extern int counter, limit, show_edges, x_off, y_off;
  struct plist *p_p;
  int x_offset = x_off, y_offset = y_off;
  
  if(b_h == 1 && b_w == 1)
    x_offset = y_offset = 0;

  pixel_p = NULL;
  limit = 1;
  counter = 1;

  if(show_edges == TRUE) plot_mask_edge(xor_data);
  line(prev_x, prev_y, x, y, put_pixel);
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  line(prev_x, prev_y, x, y, list_pixel);

  add_coords(x, y);

  p_p = pixel_p;
  
  while(p_p != NULL)
    {
      x = p_p->x_c;
      y = p_p->y_c;
      blockade(x-x_offset-skip_value/2, y-y_offset-skip_value/2, 
	       x+b_w-x_offset+skip_value/2, y+b_h-y_offset+skip_value/2, all);
      p_p = p_p->next;
    }
  if(pixel_p != NULL)
    del_coords();
}

/* FUNCTION:       rubber_circle(int x, int y, int kind)
   DESCRIPTION:    Draw a rubberbanding circle
   PARAMETERS:
   x, y            Coords
   kind            Event
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 16th July 1993.
*/   
void rubber_circle(int x, int y, int kind)
{
  extern PLOT xor_data, put_data, fill_data, rubber_data;
  extern int prev_x, prev_y, cancelled, first_x, first_y, b_w, b_h, 
  x_off, y_off,
  filled, perimeter, show_edges;
  extern struct picture *cur_block, *fill_pattern;
  struct picture *help_block = cur_block;
  int r;

  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  circulate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, 
		    FALSE);
	}
      clear_undo();
      cancelled = TRUE;
      FIRST_NA;
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  prev_x = x;
	  prev_y = y;
	  first_x = x;
	  first_y = y;
	  circulate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	}
      else
	{
	  circulate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, 
		    FALSE);
	  all_busy();
	  r = sqrt(pow(first_x-x, 2)+pow(first_y-y, 2));
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  memorize(cur_pic, 
		   (first_x-r)-x_off, (first_y-r)-y_off,
		   (first_x+r)+b_w-x_off, (first_y+r)+b_h-y_off, NULL,
		   "circle");
	  if(filled == TRUE)
	    {
	      cur_block = fill_pattern;
	      fill_circle(first_x, first_y, r, fill_data);
	      cur_block = help_block;
	      if(perimeter == TRUE)
		circle(first_x, first_y, r, put_data);
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      blockade((first_x-r)-x_off, (first_y-r)-y_off,
		       (first_x+r)+b_w-x_off, (first_y+r)+b_h-y_off, TRUE);
	      
	    }
	  else if(filled == FALSE)
	    {
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      circulate(first_x, first_y, x, y, b_w, b_h, put_data, TRUE);
	    }
	  touch(cur_pic);
	  all_busy_nomore();
	  FIRST_NA;
	  PREV_NA;
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  circulate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, 
		    FALSE);
	  circulate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       circulate(prev_x, prev_y, x, y, b_w, b_h, *put_pixel, all)
   DESCRIPTION:    To shorten and clean up the code
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications 16th July 1993.
*/   
void circulate(int prev_x, int prev_y, int x, int y, int b_w, int b_h,
	    void(*put_pixel)(int x, int y), int all)
{
  extern PLOT xor_data;
  extern struct plist *pixel_p;
  extern int counter, limit, show_edges, x_off, y_off;
  struct plist *p_p;
  int r;
  int x_offset = x_off, y_offset = y_off;
  
  if(b_h == 1 && b_w == 1)
    x_offset = y_offset = 0;

  pixel_p = NULL;
  limit = 8;
  counter = 8;  
  r = sqrt(pow(prev_x-x, 2)+pow(prev_y-y, 2));

  if(show_edges == TRUE) plot_mask_edge(xor_data);
  circle(prev_x, prev_y, r, put_pixel);
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  circle(prev_x, prev_y, r, list_pixel);

  p_p = pixel_p;

  while(p_p != NULL)
    {
      x = p_p->x_c;
      y = p_p->y_c;
      blockade(x-x_offset-skip_value/1, 
	       y-y_offset-skip_value/1, 
	       x+b_w-x_offset+skip_value/1, 
	       y+b_h-y_offset+skip_value/1, all);
      p_p = p_p->next;
    }
  if(pixel_p != NULL)
    del_coords();
}

/* FUNCTION:       rubber_ellipse(int x, int y, int kind)
   DESCRIPTION:    Draw a rubberbanding ellipse
   PARAMETERS:
   x, y            Coords
   kind            Event
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 16th July 1993.
*/   
void rubber_ellipse(int x, int y, int kind)
{
  extern PLOT xor_data, put_data, fill_data, rubber_data;
  extern int prev_x, prev_y, cancelled, first_x, first_y, b_w, b_h, 
  x_off, y_off, filled, perimeter, show_edges;
  extern struct picture *cur_block, *fill_pattern;
  struct picture *help_block = cur_block;

  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  ellipsate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, 
		    FALSE);
	}
      clear_undo();
      cancelled = TRUE;
      FIRST_NA;
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  prev_x = x;
	  prev_y = y;
	  first_x = x;
	  first_y = y;
	  ellipsate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	}
      else
	{
	  ellipsate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, 
		    FALSE);
	  all_busy();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  memorize(cur_pic, 
		   (first_x-ABS(first_x-x))-x_off, 
		   (first_y-ABS(first_y-y))-y_off,
		   (first_x+ABS(first_x-x))+b_w-x_off, 
		   (first_y+ABS(first_y-y))+b_h-y_off, NULL, "ellipse");
	  if(filled == TRUE)
	    {
	      cur_block = fill_pattern;
	      fill_ellipse(first_x, first_y, 
			   ABS(first_x-x), ABS(first_y-y), fill_data);
	      cur_block = help_block;
	      if(perimeter == TRUE)
		ellipse(first_x, first_y, ABS(first_x-x), ABS(first_y-y), 
			put_data);
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      blockade((first_x-ABS(first_x-x))-x_off, 
		       (first_y-ABS(first_y-y))-y_off,
		       (first_x+ABS(first_x-x))+b_w-x_off, 
		       (first_y+ABS(first_y-y))+b_h-y_off, TRUE); 
	    }
	  else if(filled == FALSE)
	    {
	      if(show_edges == TRUE) plot_mask_edge(xor_data);
	      ellipsate(first_x, first_y, x, y, b_w, b_h, put_data, TRUE);
	    }
	  touch(cur_pic);
	  all_busy_nomore();
	  FIRST_NA;
	  PREV_NA;
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  ellipsate(first_x, first_y, prev_x, prev_y, 1, 1, rubber_data, 
		    FALSE);
	  ellipsate(first_x, first_y, x, y, 1, 1, rubber_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       ellipsate(prev_x, prev_y, x, y, b_w, b_h, *put_pixel, all)
   DESCRIPTION:    To shorten and clean up code
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications 16th July 1993.
*/   
void ellipsate(int prev_x, int prev_y, int x, int y, int b_w, int b_h,
	       void(*put_pixel)(int x, int y), int all)
{
  extern PLOT xor_data;
  extern struct plist *pixel_p;
  extern int counter, limit, show_edges, x_off, y_off;
  struct plist *p_p;
  int x_offset = x_off, y_offset = y_off;
  
  if(b_h == 1 && b_w == 1)
    x_offset = y_offset = 0;

  pixel_p = NULL;
  limit = 4;
  counter = 4;  

  if(ABS(prev_x-x == 0) || ABS(prev_y-y) == 0)
    {
      limit = 1;
      counter = 1;
    }

  if(show_edges == TRUE) plot_mask_edge(xor_data);
  ellipse(prev_x, prev_y, ABS(prev_x-x), ABS(prev_y-y), put_pixel);
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  ellipse(prev_x, prev_y, ABS(prev_x-x), ABS(prev_y-y), list_pixel);

  p_p = pixel_p;

  while(p_p != NULL)
    {
      x = p_p->x_c;
      y = p_p->y_c;
      blockade(x-x_offset-skip_value/1, 
	       y-y_offset-skip_value/1, 
	       x+b_w-x_offset+skip_value/1, 
	       y+b_h-y_offset+skip_value/1, all);
      p_p = p_p->next;
    }
  if(pixel_p != NULL)
    del_coords();
}

/* FUNCTION:       color_ext(int x, int y, int button);
   DESCRIPTION:    Extract a colour from a picture and set it as primary
                   or secondary colour.
   PARAMETERS:
   x, y            Mouse coordinates
   button          Mouse button
   RETURNS:        

   Written by Hannu Mallat, last modifications 14th July 1993.
   Fixed by Petri Kuittinen, last modifications 14th July 1993.
*/   
void color_ext(int x, int y, int kind, int button)
{
  extern struct view *cur_view;
  extern int cancelled;

  switch(kind)
    {
    case CANCEL:
      cancelled = TRUE;
      break;
    case PRESS:
      cancelled = FALSE;
      /* fallthru */
    case MOVE_W:
      if(button == LEFT)
	{
	  cur_view->color1 = get_color(x, y);
	}
      else if (button == CENTER)
	{
	  cur_view->color2 = get_color(x, y);
	}
      create_brush(cur_block_type, cur_block->width);
      show_colors(cur_view);
      break;
    }
}

/* FUNCTION:       flood(int x, int y, int kind)
   DESCRIPTION:    flood fill an area
   PARAMETERS:
   x, y            mouse coords
   kind            event
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 23th August 1993.
*/   
void flood(int x, int y, int kind)
{
  extern PLOT xor_data, fill_data;
  extern struct picture *cur_pic, *cur_block, *fill_pattern;
  extern int show_edges, fill_tolerance;
  struct picture *help_block = cur_block; 

  switch(kind)
    {
    case PRESS:
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      all_busy();
      memorize(cur_pic, 0, 0, cur_pic->width-1, cur_pic->height-1, NULL,
	       "flood fill");
      cur_block = fill_pattern;
      flood_fill(x, y, fill_tolerance, fill_data);
      cur_block = help_block;
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1,
	       TRUE);
      touch(cur_pic);
      all_busy_nomore();
      break;
    }
}

/* FUNCTION:       select_rect(int x, int y, int kind)
   DESCRIPTION:    Select a rectangle from the image
   PARAMETERS:
   x, y            mouse coords
   kind            what event?
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void select_rect(int x, int y, int kind, int button)
{
  extern PLOT xor_data;
  extern int prev_x, prev_y, cancelled;
  extern int first_x, first_y, show_edges;
  static int save_edges;

  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  boxate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  show_edges = save_edges;
	  if(show_edges == FALSE) plot_mask_edge(xor_data);
	  update_edge();
	}
      clear_undo();
      cancelled = TRUE;
      FIRST_NA;
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  if(show_edges == FALSE) {plot_mask_edge(xor_data); update_edge();}
	  first_x = x;
	  first_y = y;
	  prev_x = x;
	  prev_y = y;
	  save_edges = show_edges;
	  show_edges = FALSE;
	  boxate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	}
      else
	{
	  boxate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  show_edges = save_edges;
	  all_busy();
	  plot_mask_edge(xor_data);
	  if(button == CENTER) update_edge();
	  memorize(cur_pic, first_x, first_y, x, y, NULL, "select rectangle");
	  if(button == LEFT)
	    fill_box(first_x, first_y, x, y, set_mask);
	  else
	    fill_box(first_x, first_y, x, y, clear_mask);
	  if(cur_pic->n_edges == 0)
	    {
	      cur_pic->min_x = MIN(first_x, x);
	      cur_pic->min_y = MIN(first_y, y);
	      cur_pic->max_x = MAX(first_x, x);
	      cur_pic->max_y = MAX(first_y, y);
	    }
	  else
	    {
	      cur_pic->min_x = MIN(cur_pic->min_x, MIN(first_x, x));
	      cur_pic->min_y = MIN(cur_pic->min_y, MIN(first_y, y));
	      cur_pic->max_x = MAX(cur_pic->max_x, MAX(first_x, x));
	      cur_pic->max_y = MAX(cur_pic->max_y, MAX(first_y, y));
	    }
	  minimize_mask_edge();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  update_edge();
	  all_busy_nomore();
	  update_menu();
	  FIRST_NA;
	  PREV_NA;
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      cancelled = FALSE;
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  boxate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  boxate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       select_ellipse(int x, int y, int kind)
   DESCRIPTION:    Select an ellipse from the image
   PARAMETERS:
   x, y            mouse coords
   kind            what event?
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 5th August 1993.
*/   
void select_ellipse(int x, int y, int kind, int button)
{
  extern PLOT xor_data;
  extern int prev_x, prev_y, cancelled;
  extern int first_x, first_y, show_edges;
  static int save_edges;

  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  ellipsate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  show_edges = save_edges;
	  if(show_edges == FALSE) plot_mask_edge(xor_data);
	  update_edge();
	}
      clear_undo();
      cancelled = TRUE;
      FIRST_NA;
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  if(show_edges == FALSE) {plot_mask_edge(xor_data); update_edge();}
	  prev_x = x;
	  prev_y = y;
	  first_x = x;
	  first_y = y;
	  save_edges = show_edges;
	  show_edges = FALSE;
	  ellipsate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	}
      else
	{
	  ellipsate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  show_edges = save_edges;
	  all_busy();
	  plot_mask_edge(xor_data);
	  if(button == CENTER) update_edge();
	  memorize(cur_pic, first_x-ABS(first_x-x), first_y-ABS(first_y-y), 
		   first_x+ABS(first_x-x), first_y+ABS(first_y-y), NULL, 
		   "select ellipse");
	  if(button == LEFT)
	    fill_ellipse(first_x, first_y, ABS(first_x-x), ABS(first_y-y), 
			 set_mask);
	  else
	    fill_ellipse(first_x, first_y, ABS(first_x-x), ABS(first_y-y), 
			 clear_mask);
	  if(cur_pic->n_edges == 0)
	    {
	      cur_pic->min_x = first_x-ABS(first_x-x);
	      cur_pic->min_y = first_y-ABS(first_y-y);
	      cur_pic->max_x = first_x+ABS(first_x-x);
	      cur_pic->max_y = first_y+ABS(first_y-y);
	    }
	  else
	    {
	      cur_pic->min_x = MIN(cur_pic->min_x, first_x-ABS(first_x-x));
	      cur_pic->min_y = MIN(cur_pic->min_y, first_y-ABS(first_y-y));
	      cur_pic->max_x = MAX(cur_pic->max_x, first_x+ABS(first_x-x));
	      cur_pic->max_y = MAX(cur_pic->max_y, first_y+ABS(first_y-y));
	    }
	  minimize_mask_edge();
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  update_edge();
	  all_busy_nomore();
	  update_menu();
	  FIRST_NA;
	  PREV_NA;
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  ellipsate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  ellipsate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       select_polygon(int x, int y, int kind)
   DESCRIPTION:    Select a polygon from the image
   PARAMETERS:
   x, y            mouse coords
   kind            what event?
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 4th August 1993.
*/   
void select_poly(int x, int y, int kind, int button)
{
  extern PLOT xor_data;
  extern int prev_x, prev_y, cancelled;
  extern int first_x, first_y, show_edges;
  static int save_edges;
  static int primus_x, primus_y, xmin, ymin, xmax, ymax;
  static struct polygon *poly;
 
  switch(kind)
    {
    case CANCEL:
      if(first_x != -1 && first_y != -1)
	{
	  linate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  all_busy();
	  plot_mask_edge(xor_data);
	  add_polygon_edge(poly, first_x, first_y, primus_x, primus_y);
	  if(button == CENTER) update_edge();
	  memorize(cur_pic, xmin, ymin, xmax, ymax, NULL, "select polygon");
	  if(button == LEFT)
	    fill_polygon(poly, set_mask);
	  else
	    fill_polygon(poly, clear_mask);
	  delete_polygon(poly);
	  if(cur_pic->n_edges == 0)
	    {
	      cur_pic->min_x = xmin;
	      cur_pic->min_y = ymin;
	      cur_pic->max_x = xmax;
	      cur_pic->max_y = ymax;
	    }
	  else
	    {
	      cur_pic->min_x = MIN(cur_pic->min_x, xmin);
	      cur_pic->min_y = MIN(cur_pic->min_y, ymin);
	      cur_pic->max_x = MAX(cur_pic->max_x, xmax);
	      cur_pic->max_y = MAX(cur_pic->max_y, ymax);
	    }
	  minimize_mask_edge();
	  show_edges = save_edges;
	  if(show_edges == TRUE) plot_mask_edge(xor_data); 
	  update_edge();
	  all_busy_nomore();
	}
      update_menu();
      cancelled = TRUE;
      FIRST_NA;
      PREV_NA;
      break;
    case PRESS:
      cancelled = FALSE;
      if(first_x == -1 || first_y == -1)
	{
	  if(show_edges == FALSE) {plot_mask_edge(xor_data); update_edge();}
	  poly = create_polygon();
	  if(poly == NULL)
	    panic("Cannot allocate memory! (create_polygon())", TRUE);
	  prev_x = first_x = primus_x = xmin = xmax = x;
	  prev_y = first_y = primus_y = ymin = ymax = y;
	  save_edges = show_edges;
	  show_edges = FALSE;
	  linate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	}
      else
	{
	  linate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  plot_mask_edge(xor_data);
	  line(first_x, first_y, x, y, add_mask_edge);
	  add_polygon_edge(poly, first_x, first_y, x, y);
	  plot_mask_edge(xor_data);
	  update_edge();
	  first_x = x;
	  first_y = y;
	  prev_x = x;
	  prev_y = y;
	  if(xmin>x) xmin = x; else if(xmax<x) xmax = x;
	  if(ymin>y) ymin = y; else if(ymax<y) ymax = y;
	  linate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	}
      break;
    case MOVE_W:
    case MOVE_WO:
      if(first_x != -1 && first_y != -1 && (prev_x != x || prev_y != y))
	{
	  linate(first_x, first_y, prev_x, prev_y, 1, 1, xor_data, FALSE);
	  linate(first_x, first_y, x, y, 1, 1, xor_data, FALSE);
	  prev_x = x;
	  prev_y = y;
	}
      break;
    }
}

/* FUNCTION:       select_lasso(int x, int y, int kind)
   DESCRIPTION:    Select an area from the image with lasso
   PARAMETERS:
   x, y            mouse coords
   kind            what event?
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 4th August 1993.
*/   
void select_lasso(int x, int y, int kind, int button)
{
  extern PLOT xor_data;
  extern int prev_x, prev_y, first_x, first_y, left, show_edges;
  static int save_edges, xmin, ymin, xmax, ymax;
  static struct polygon *poly;

  switch(kind)
    {
    case PRESS:
      if(show_edges == FALSE) {plot_mask_edge(xor_data); update_edge();}
      poly = create_polygon();
      if(poly == NULL)
	panic("Cannot allocate memory! (create_polygon())", TRUE);
      prev_x = first_x = xmin = xmax = x;
      prev_y = first_y = ymin = ymax = y;
      save_edges = show_edges;
      show_edges = FALSE;
      left = FALSE;
      plot_mask_edge(xor_data);
      line(prev_x, prev_y, x, y, add_mask_edge);
      add_polygon_edge(poly, prev_x, prev_y, x, y);
      plot_mask_edge(xor_data);
      update_edge();
      break;
    case MOVE_W:
      if(prev_x != -1 && prev_y != -1 && 
	 (prev_x != x || prev_y != y))
	{
	  plot_mask_edge(xor_data);
	  line(prev_x, prev_y, x, y, add_mask_edge);
	  add_polygon_edge(poly, prev_x, prev_y, x, y);
	  plot_mask_edge(xor_data);
	  blockade(prev_x, prev_y, x, y, FALSE);
	  prev_x = x;
	  prev_y = y;
	  if(xmin>x) xmin = x; else if(xmax<x) xmax = x;
	  if(ymin>y) ymin = y; else if(ymax<y) ymax = y;
	  left = FALSE;
	}
      break;
    case LEAVE:
      if(prev_x != -1 && prev_y != -1)
	{
	  left = TRUE;
	  if(ABS(x-cur_pic->width)>ABS(x)) x = 0;
	  else x = cur_pic->width-1;
	  if(ABS(y-cur_pic->height)>ABS(y)) y = 0;
	  else y = cur_pic->height-1;
	  plot_mask_edge(xor_data);
	  line(prev_x, prev_y, x, y, add_mask_edge);
	  add_polygon_edge(poly, prev_x, prev_y, x, y);
	  plot_mask_edge(xor_data);
	  blockade(prev_x, prev_y, x, y, FALSE);
	  prev_x = x;
	  prev_y = y;    
	  if(xmin>x) xmin = x; else if(xmax<x) xmax = x;
	  if(ymin>y) ymin = y; else if(ymax<y) ymax = y;
	}
      break;
    case MOVE_WO:
      left = FALSE;
      FIRST_NA;
      PREV_NA;
      break;
    case RELEASE:
      if(prev_x != -1 && prev_y != -1)
	{
	  all_busy();
	  if(xmin>x) xmin = x; else if(xmax<x) xmax = x;
	  if(ymin>y) ymin = y; else if(ymax<y) ymax = y;
	  plot_mask_edge(xor_data);
	  add_polygon_edge(poly, prev_x, prev_y, first_x, first_y);
	  if(button == CENTER) update_edge();
	  memorize(cur_pic, xmin, ymin, xmax, ymax, NULL, "select lasso");
	  if(button == LEFT)
	    fill_polygon(poly, set_mask);
	  else
	    fill_polygon(poly, clear_mask);
	  delete_polygon(poly);
	  if(cur_pic->n_edges == 0)
	    {
	      cur_pic->min_x = xmin;
	      cur_pic->min_y = ymin;
	      cur_pic->max_x = xmax;
	      cur_pic->max_y = ymax;
	    }
	  else
	    {
	      cur_pic->min_x = MIN(cur_pic->min_x, xmin);
	      cur_pic->min_y = MIN(cur_pic->min_y, ymin);
	      cur_pic->max_x = MAX(cur_pic->max_x, xmax);
	      cur_pic->max_y = MAX(cur_pic->max_y, ymax);
	    }
	  minimize_mask_edge();
	  show_edges = save_edges;
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  update_edge();
	  all_busy_nomore();
	  update_menu();
	  FIRST_NA;
	  PREV_NA;
	}
      break;
    }
}

/* FUNCTION:       select_n_colors(int x, y, kind, button)
   DESCRIPTION:    select a range of colors
   PARAMETERS:
   x, y            mouse coords
   kind            event
   button          mouse button
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 28th August 1993.
*/   
void select_n_colors(int x, int y, int kind, int button)
{
  extern PLOT xor_data;
  extern int select_tolerance, show_edges;

  switch(kind)
    {
    case PRESS:
      all_busy();
      if(show_edges == TRUE) {plot_mask_edge(xor_data); update_edge();}
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, 
	       "select color range");
      cur_pic->min_x = 0;
      cur_pic->min_y = 0;
      cur_pic->max_x = cur_pic->width-1;
      cur_pic->max_y = cur_pic->height-1;
      if(button == LEFT)
	fill_by_color(get_color(x, y), select_tolerance, set_mask);
      else
	fill_by_color(get_color(x, y), select_tolerance, clear_mask);
      minimize_mask_edge();
      if(show_edges == TRUE) {plot_mask_edge(xor_data); update_edge();}
      all_busy_nomore();
      update_menu();
      break;
    }
}
