/* FILE:   draw.c
 
   Routines for basic geometric shapes and filling areas are here.

   Written by Petri Kuittinen, last modifications 23th July 1993.
*/

#include "misc.h"
#include "draw.h"


/* FUNCTION:     void line (int x1, int y1, int x2, int y2,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  draw line from (x1, y1) to (x2, y2) using Bresenham's
   algorithm. This funtion only calculates the coordinates and 
   calls another funtion to do the actual pixel drawing.
   PARAMETRES:
   int x1
   int y1
   int x2
   int y2
   void (*put_pixel) (x,y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 25th June 1993
*/
void line (int x1, int y1, int x2, int y2, void (*put_pixel) (int x, int y))
{
  /* d = delta, x and y are current coordinates, ax and ay are the
     absolute offset between x and y coordinates, sx and sy are signs
     of the offsets */
  register int d, x, y, ax, ay, sx, sy; 

  sx = 1; ax = (x2-x1)<<1;
  if (ax<0) {ax = -ax; sx = -sx;}
  sy = 1; ay = (y2-y1)<<1;
  if (ay<0) {ay = -ay; sy = -sy;}
  x = x1; y = y1;

  if (ax>ay)
    {		
      /* x offset higher */
      d = ay-(ax>>1);
      for (;;)
	{
	  (*put_pixel) (x, y);
	  if (x==x2) return;
	  if (d>=0) {y += sy; d -= ax;}
	  x += sx; d += ay;
	}
    }
    else
      {			
	/* y offset higher */
	d = ax-(ay>>1);
	for (;;)
	  {
	    (*put_pixel) (x, y);
	    if (y==y2) return;
	    if (d>=0) {x += sx; d -= ay;}
	    y += sy; d += ax;
	  }
      }
}

/* FUNCTION:     void hline (int x1, int x2, int y,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  draw horizontal line from (x1, y) to (x2, y). This
   funtion only calculates the coordinates and calls another funtion
   to do the actual pixel drawing.
   PARAMETRES:
   int x1
   int x2
   int y
   void (*put_pixel) (x, y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 2nd July 1993
*/
void hline (int x1, int x2, int y, void (*put_pixel)(int x, int y))
{
  if (x2<x1) {SWAP (x1,x2);}
  do
    (*put_pixel) (x1++,y);
  while (x1<=x2);
}


/* FUNCTION:     void vline (int x, int y1, int y2,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  draw vertical line from (x, y1) to (x, y2). This
   funtion only calculates the coordinates and calls another funtion
   to do the actual pixel drawing.
   PARAMETRES:
   int x
   int y1
   int y2
   void (*put_pixel) (x, y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 2nd July 1993
*/
void vline (int x, int y1, int y2, void (*put_pixel)(int x, int y))
{
  if (y2<y1) {SWAP (y1,y2);}
  do
    (*put_pixel) (x, y1++);
  while (y1<=y2);
}


/* FUNCTION:     void box (int x1, int y1, int x2, y2,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  draw the egdes of rectangular box. Points (x1, y1) and
   (x2, y2) are the opposite vertexes of the box.
   PARAMETRES:
   int x1
   int y1
   int x2
   int y2
   void (*put_pixel) (x, y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 12nd July 1993
*/
void box (int x1, int y1, int x2, int y2, void (*put_pixel)(int x, int y))
{
  if (y2<y1) { SWAP (y1, y2);}
  hline(x1, x2, y1, put_pixel);
  if (y1==y2) return;
  hline(x1, x2, y2, put_pixel);
  if (ABS(y2-y1)==1) return;
  vline(x1, y1+1, y2-1, put_pixel);
  vline(x2, y1+1, y2-1, put_pixel);
}


/* FUNCTION:     void fill_box (int x1, int y1, int x2, y2,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Fill a box shaped area. Points (x1, y1) and
   (x2, y2) are the opposite vertexes of the box.
   PARAMETRES:
   int x1
   int y1
   int x2
   int y2
   void (*put_pixel) (x, y) pointer to pixel drawing/pattern fill plot routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 15th July 1993
*/
void fill_box (int x1, int y1, int x2, int y2,
	       void (*put_pixel) (int x, int y))
{
  if (y2<y1) { SWAP (y1, y2);}
  if (x2<x1) { SWAP (x1, x2);}
  do
    hline (x1, x2, y1, put_pixel);
  while (y1++<y2);
}


/* FUNCTION:     void circle (int xc, int yc, int r,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Draw a circle, whose centre is at (x, y) and radius is r.
   Calls external routine to do the actual plotting. Uses second order
   difference algorithm presented in Foley's and van Dam's book; Computer
   Graphics. The algorithm calculates only one octant of circle and uses
   symmetry to define the rest of points.
   PARAMETRES:
   int xc        coordinates of circle centre
   int yc
   int r         radius of circle (>=0)
   void (*put_pixel) (x, y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 19th July 1993
   Now this routine should no longer plot the same pixel twice (checks added
   to prevent this).
*/
void circle (int xc, int yc, int r, void (*put_pixel) (int x, int y))
{
  register int x, y, d, de, dse;
  
  if (r<=0)
    {
      put_pixel(xc, yc);
      return;
    }

  x = 0;
  y = r;
  d = 1-r;
  de = 3;
  dse = -2*r+5;
  
  put_pixel (xc+x, yc+y);
  put_pixel (xc+x, yc-y);
  put_pixel (xc+y, yc+x);
  put_pixel (xc-y, yc+x);

  while (y>x)
    {
      if (d<0)
	{
	  d += de;
	  de += 2;
	  dse += 2;
	  x++;
	}
      else
	{
	  d += dse;
	  de += 2;
	  dse += 4;
	  x++;
	  y--;
	}
      if (x<=y)
	{
	  /* 8-way symmetry */
	  put_pixel (xc+x, yc+y);
	  put_pixel (xc+x, yc-y);
	  put_pixel (xc-x, yc-y);
	  put_pixel (xc-x, yc+y);
	  if (x!=y)
	    {
	      put_pixel (xc+y, yc+x);
	      put_pixel (xc-y, yc+x);
	      put_pixel (xc+y, yc-x);
	      put_pixel (xc-y, yc-x);
	    }
	}
    }
}

/* hline filling macros used in circle filling routine */
#define CIRCLE_FILL1 hline (xc-x, xc+x, yc+y, put_pixel); \
  hline (xc-x, xc+x, yc-y, put_pixel);
#define CIRCLE_FILL2 hline (xc-y, xc+y, yc+x, put_pixel); \
if (x) { hline (xc-y, xc+y, yc-x, put_pixel);}

/* FUNCTION:     void fill_circle (int xc, int yc, int r,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Fill the area of a circle, whose centre is at (x, y) and
   radius is r. Uses the same algorithm as above routine.
   PARAMETRES:
   int xc        coordinates of circle centre
   int yc
   int r         radius of circle (>=0)
   void (*put_pixel) (x, y) pointer to pixel drawing/pattern fill plot routine
   RETURNS:      nothing
   
   
   Written by Petri Kuittinen, last modifications 19th July 1993
   Now this routine should no longer fill the same hline twice.
*/
void fill_circle (int xc, int yc, int r, void (*put_pixel) (int x, int y))
{
  register int x, y, d, de, dse;
  
  if (r<=0)
    {
      put_pixel (xc, yc);
      return;
    }

  x = 0;
  y = r;
  d = 1-r;
  de = 3;
  dse = -2*r+5;

  while (y>x)
    {
      if (d<0)
	{
	  d += de;
	  de += 2;
	  dse += 2;
	  CIRCLE_FILL2;
	  x++;
	}
      else
	{
	  d += dse;
	  de += 2;
	  dse += 4;
	  CIRCLE_FILL1;
	  CIRCLE_FILL2;
	  x++;
	  y--;
	}
    }
  if (x==y)
    {
      CIRCLE_FILL2;
    }
}


/* FUNCTION:     void ellipse (int xc, int yc, int rx, int ry,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Draw an ellipse, whose centre is at (x, y) and radiuses are
   rx and ry, using an algorithm derived from Foley's and van Dam's book;
   Computer Graphics. The algorithm calculates two octants of ellipse and uses
   symmetry to define the rest of points.

   Kuittinen).
   PARAMETRES:
   int xc        coordinates of circle centre
   int yc
   int rx        x-radius of circle (>=0)
   int ry        y-radius of circle (>=0)
   void (*put_pixel) (x, y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 23th July 1993
*/
void ellipse (int xc, int yc, int rx, int ry, void (*put_pixel) (int x, int y))
{
  register int x, y, d, de, dse, rx2, ry2, re, k1, k2;
  
  if (rx==0)
    {
      vline (xc, yc-ry, yc+ry, put_pixel);
      return;
    }
  if (ry==0)
    {
      hline (xc-rx, xc+rx, yc, put_pixel);
      return;
    }
  
  x = 0;
  y = ry;

  rx2 = rx*rx;
  ry2 = ry*ry;

  /* First octant */

  d = ry2-rx2*ry+rx2/4;
  de = 3*ry2;
  dse = de+rx2*(-2*ry+2);
  
  k1 = 2*ry2;
  k2 = k1+2*rx2;
  re = rx2/2+ry2;

  put_pixel (xc+x, yc+y);
  put_pixel (xc+x, yc-y);

  while (rx2*y-ry2*x>re)
    {
      if (d<0)
	{
	  d += de;
	  de += k1;
	  dse += k1;
	  x++;
	}
      else
	{
	  d += dse;
	  de += k1;
	  dse += k2;
	  x++;
	  y--;
	}
      put_pixel (xc+x, yc+y);
      put_pixel (xc-x, yc+y);
      if (y!=0)
	{
	  put_pixel (xc+x, yc-y);
	  put_pixel (xc-x, yc-y);
	}
    }

  /* Second octant */

  d = ry2*(x*x+x-rx2)+ry2/4+rx2*(y*y-2*y+1);
  dse = rx2*(-2*y+3);
  de = dse+ry2*(2*x+2);
  k1 = 2*rx2;
  k2 = k1+2*ry2;

  while (y>0)
    {
      if (d<0)
	{
	  d += de;
	  de += k2;
	  dse += k1;
	  x++;
	  y--;
	}
      else
	{
	  d += dse;
	  de += k1;
	  dse += k1;
	  y--;
	}
      put_pixel (xc+x, yc+y);
      put_pixel (xc-x, yc+y);
      if (y!=0)
	{
	  put_pixel (xc+x, yc-y);
	  put_pixel (xc-x, yc-y);
	}
    }
}

/* A hline filling macro used in ellipse filling routine */
#define ELLIPSE_FILL hline (xc-x, xc+x, yc+y, put_pixel); \
  if (y!=0) hline (xc-x, xc+x, yc-y, put_pixel);

/* FUNCTION:     void fill_ellipse (int xc, int yc, int rx, int ry,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Fill the area of an ellipse, whose centre is at (x, y) and
   radiuses are rx and ry, using the same algorithm as above.

   PARAMETRES:
   int xc        coordinates of circle centre
   int yc
   int rx        x-radius of circle (>=0)
   int ry        y-radius of circle (>=0)
   void (*put_pixel) (x, y) pointer to pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 23th July 1993
*/
void fill_ellipse (int xc, int yc, int rx, int ry,
		   void (*put_pixel) (int x, int y))
{
  register int x, y, d, de, dse, rx2, ry2, re, k1, k2;
  
  if (rx==0)
    {
      vline (xc, yc-ry, yc+ry, put_pixel);
      return;
    }
  if (ry==0)
    {
      hline (xc-rx, xc+rx, yc, put_pixel);
      return;
    }
  
  x = 0;
  y = ry;

  rx2 = rx*rx;
  ry2 = ry*ry;

  /* First octant */

  d = ry2-rx2*ry+rx2/4;
  de = 3*ry2;
  dse = de+rx2*(-2*ry+2);
  
  k1 = 2*ry2;
  k2 = k1+2*rx2;
  re = rx2/2+ry2;

  while (rx2*y-ry2*x>re)
    {
      if (d<0)
	{
	  d += de;
	  de += k1;
	  dse += k1;
	  x++;
	}
      else
	{
	  d += dse;
	  de += k1;
	  dse += k2;
	  ELLIPSE_FILL;
	  x++;
	  y--;
	}
    }

  /* Second octant */

  d = ry2*(x*x+x-rx2)+ry2/4+rx2*(y*y-2*y+1);
  dse = rx2*(-2*y+3);
  de = dse+ry2*(2*x+2);
  k1 = 2*rx2;
  k2 = k1+2*ry2;

  while (y>0)
    {
      if (d<0)
	{
	  d += de;
	  de += k2;
	  dse += k1;
	  ELLIPSE_FILL;
	  x++;
	  y--;
	}
      else
	{
	  d += dse;
	  de += k1;
	  dse += k1;
	  ELLIPSE_FILL;
	  y--;
	}
    }
  ELLIPSE_FILL;
}
