/*
 *      File:	List.c
 *    Module:	libUmsc
 *      Date:	07/21/92
 *    Author:	Daniel B. Waylonis (danw@msc.edu)
 *   Purpose:	Linked list library routines
 *
 *   Copyright @ 1992 by Research Equipment Inc. (dba)
 *   Minnesota Supercomputer Center, Inc.
 *
 * RESTRICTED RIGHTS LEGEND
 *   Use, duplication, or disclosure of this software and its documentation
 *   by the Government is subject to restrictions as set forth in subdivision
 *   { (b) (3) (ii) } of the Rights in Technical Data and Computer Software
 *   clause at 52.227-7013.
 *
 * $Log: List.c,v $
 * Revision 1.3  1993/01/20  20:45:52  danw
 * Changed internals of UmscListDump to use void* instead of int*.
 *
 * Revision 1.2  1992/12/02  17:33:28  danw
 * Changed the incorrect declaration of UmscListIsIntitialized from 'int'
 * to 'char'.
 *
 * Revision 1.1  1992/12/01  19:31:09  danw
 * Initial revision
 *
 * 
 */
static char* RCSId = "$Header: /g/src/prod/MSC/Xtools/msc/lib/Umsc/RCS/List.c,v 1.3 1993/01/20 20:45:52 danw Exp danw $";

#include <stdio.h>

#include <Umsc/ListP.h>
#include <Umsc/Misc.h>

/*-------------------------------------------------------------------------*/
/* local prototypes */
static void	UnlinkNode(UmscListRec*, UmscNode);

/*-------------------------------------------------------------------------*/
/*
 * default comparison routine
 */
static int DefaultComparison(void	*a,
			     void	*b)
{
    return (a != b);
}

/*-------------------------------------------------------------------------*/
/*
 * default free routine
 */
static void DefaultFree(void *a)
{
    if (a)
	free(a);
}

/*-------------------------------------------------------------------------*/
/*
 * Return the current object (NULL if current is not set)
 */
void* UmscListGetCurrent(UmscList theList)
{
    if (theList->current)
	return(theList->current->obj);
    else
	return(NULL);
}

/*-------------------------------------------------------------------------*/
/*
 * Return True if the list is initialized
 */
char UmscListIsInitialized(UmscList theList)
{
    return(theList->init);
}

/*-------------------------------------------------------------------------*/
/*
 * Return the next object (NULL if current is not set) and set
 * current to that object
 */
void* UmscListGetNext(UmscList theList)
{
    if (theList->current) {
	theList->current = theList->current->next;
	return(UmscListGetCurrent(theList));
    }
    else
	return(NULL);
}

/*-------------------------------------------------------------------------*/
/*
 * Return the previous object (NULL if current is not set) and set
 * current to that object
 */
void* UmscListGetPrev(UmscList theList)
{
    if (theList->current) {
	theList->current = theList->current->prev;
	return(UmscListGetCurrent(theList));
    }
    else
	return(NULL);
}

/*-------------------------------------------------------------------------*/
/*
 * Return the number of objects in the list
 */
int UmscListGetCount(UmscList theList)
{
    return(theList->cnt);
}

/*-------------------------------------------------------------------------*/
/*
 * Set the current object to the previously saved object
 */
void* UmscListRestoreCurrent(UmscList theList)
{
    theList->current = theList->saved;
    return(UmscListGetCurrent(theList));
}

/*-------------------------------------------------------------------------*/
/* Save the current object pointer
 */
void UmscListSaveCurrent(UmscList theList)
{
    theList->saved = theList->current;
}

/*-------------------------------------------------------------------------*/
/*
 * Set the current pointer to an integer position [1, theList->cnt]
 * Special positions:
 *	UmscFIRST = first position
 *	UmscLAST = last position
 *
 * Note:  For speed, a check is done to see if the item is closer to the
 *	  front or back of the list and then processed accordingly
 */
void *UmscListSetCurrent(UmscList	theList,
			 int		pos)
{
    int		i;
    UmscNode	node;

    /* check if pos is a special position */
    if ((pos == UmscFIRST) ||
	(pos == 0))
	theList->current = theList->first;
    else if ((pos == UmscLAST) ||
	     (pos == theList->cnt))
	theList->current = theList->last;
    else if ((pos <= 0) ||
	     (pos > theList->cnt)) {
	fprintf(stderr, "UmscListSetCurrent:  %d not in range [1, %d]\n",
		pos, theList->cnt);
	return(NULL);
    }
    else {

	/* search from first to last */
	if (pos < theList->cnt / 2) {
	    node = theList->first;

	    for (i = 1; i < pos; i++)
		node = node->next;
	}

	/* search from last to first */
	else {
	    node = theList->last;

	    for (i = theList->cnt; i > pos; i--)
		node = node->prev;
	}

	theList->current = node;
    }

    return(UmscListGetCurrent(theList));
}

/*-------------------------------------------------------------------------*/
/*
 * Initialize the list structure
 * if cmpRtn or freeRtn are NULL, default ones will be used
 *
 */
UmscSTATUS UmscListInit(UmscList	*theList,
			int		(*cmpRtn)(),
			void		(*freeRtn)())
{
    UmscList	tempList;
    /*
     * Check if list is valid
     */
    if (!theList) {
	fprintf(stderr, "UmscListInit:  Null list pointer\n");
	return(UmscFAILURE);
    }

    if (! *theList) {
	tempList = UmscNew(UmscListRec);

	if (! tempList) {
	    fprintf(stderr, "UmscListInit:  Out of memory -- cannot init\n");
	    return(UmscFAILURE);
	}

	tempList->cnt = 0;
	tempList->first = NULL;
	tempList->current = NULL;
	tempList->last = NULL;
	tempList->saved = NULL;
	tempList->init = True;

	/* use defaults if none given */
	tempList->cmp  = (cmpRtn) ? cmpRtn: DefaultComparison;
	tempList->free = (freeRtn) ? freeRtn : DefaultFree;

	/* set the list */
	*theList = tempList;
    }
    else {
	fprintf(stderr, "UmscListInit:  List already initialized\n");
	return(UmscWARNING);

    }
    
    return(UmscSUCCESS);
}

/*-------------------------------------------------------------------------*/
/*
 * Alternate way of creating a UmscList
 * if cmpRtn or freeRtn are NULL, default ones will be used
 *
 */
UmscList UmscListNew(int	(*cmpRtn)(),
		     void	(*freeRtn)())
{
    UmscList	tempList = UmscNew(UmscListRec);

    if(! tempList) {
	fprintf(stderr, "UmscListNew:  Out of memory -- cannot init\n");
	return(NULL);
    }

    tempList->cnt = 0;
    tempList->first = NULL;
    tempList->current = NULL;
    tempList->last = NULL;
    tempList->saved = NULL;
    tempList->init = True;

    /* use defaults if none given */
    tempList->cmp = (cmpRtn) ? cmpRtn : DefaultComparison;
    tempList->free = (freeRtn) ? freeRtn : DefaultFree;

    return(tempList);
}

/*-------------------------------------------------------------------------*/
/*
 * Insert theObj in a newNode.  Put the node before/after the current pointer,
 * if current is NULL, put at end of list.
 * Adjust the theList->first and theList->last pointers
 * Set current to be new node. 
 *
 * return UmscSUCCESS if no err, UmscFAILURE otherwise
 */
UmscSTATUS UmscListInsert(UmscList	theList,
			  void		*theObj,
			  int		direction)
{
    UmscNode	newNode;
    UmscNode	*first = &theList->first;
    UmscNode    *last = &theList->last;
    UmscNode	*current = &theList->current;

    if (!theList) {
	fprintf(stderr, "UmscListInsert:  List is NULL\n");
	return(UmscFAILURE);
    }

    newNode = UmscNew(UmscNodeRec);
    if (! newNode) {
	fprintf(stderr,
		"UmscListInsert:  Unable to allocate memory for node\n");
	return(UmscFAILURE);
    }
    
    newNode->obj = theObj;
    newNode->prev = NULL;
    newNode->next = NULL;
    
    if (! *current)
	*current = *last;

    /*
     * check direction -- if invalid, warn and add as next
     */

    if ((direction != UmscNEXT) &&
	(direction != UmscPREV)) {
	fprintf(stderr,
		"UmscListInsert:  Unknown add direction (%d) -- using NEXT\n",
		direction);
	direction = UmscNEXT;
    }

    /*
     * add node to the right place
     */
 
    if (! *current) {
	if (direction == UmscNEXT)
	    *current = *last;
	else
	    *current = *first;
    }

    if (*current) {

	if (direction == UmscNEXT) {
	    if ((*current)->next)
		(*current)->next->prev = newNode;
	
	    newNode->prev = *current;
	    newNode->next = (*current)->next;
	    (*current)->next = newNode;
	} /* add to next pointer */

	else if (direction == UmscPREV) {
	    if ((*current)->prev)
		(*current)->prev->next = newNode;

	    newNode->next = *current;
	    newNode->prev = (*current)->prev;
	    (*current)->prev = newNode;

	} /* add to previous pointer */

    } /* if there is a current pointer */
    
    if ((! *first) ||
	((! newNode->prev) &&
	 (direction == UmscPREV)))
	*first = newNode;
    
    if ((! *last) ||
	((! newNode->next) &&
	 (direction == UmscNEXT)))
	*last = newNode;
    
    theList->current = newNode; 
    theList->cnt++;

    /* normal exit */
    return(UmscSUCCESS);
}

/*-------------------------------------------------------------------------*/
/*
 * Unlink a node from the list and return its pointer
 */
static void UnlinkNode(UmscList	theList,
		       UmscNode	node)
{
    if (! node)
	return;

    /*
     * update the links
     */
    if (node->prev)
	node->prev->next = node->next;
    
    if (node->next)
	node->next->prev = node->prev;
    
    if (theList->first == node)
	theList->first = node->next;
    
    if (theList->last == node)
	theList->last = node->prev;
}

/*-------------------------------------------------------------------------*/
/*
 * Move current to first or last position
 */
void UmscListMove(UmscList	theList,
		  int		position)
{
    UmscNode	newCurrent;

    /*
     * Check if list is valid
     */
    if (!theList) {
	fprintf(stderr, "UmscListMove:  List is NULL\n");
	return;
    }

    if (theList->current) {

	if ((position == UmscFIRST) ||
	    (position == UmscLAST)) {
	    UnlinkNode(theList, theList->current);

	    /*
	     * move to first position
	     * update first pointer (if necessary)
	     */
	    if (position == UmscFIRST) {
		theList->current->prev = NULL;

		if (theList->current != theList->first) {
		    theList->current->next = theList->first;
		    theList->current->next->prev = theList->current;
		    theList->first = theList->current;
		}
		else if (theList->first != theList->last)
		    theList->current->next = theList->first->next;

	    }
	    /*
	     * move to last position
	     * update last pointer (if necessary)
	     */
	    else {
		theList->current->next = NULL;

		if (theList->current != theList->last) {
		    theList->current->prev = theList->last;
		    theList->current->prev->next = theList->current;
		    theList->last = theList->current;
		}
		else if (theList->first != theList->last)
		    theList->current->prev = theList->last->prev;
	    }

	    /*
	     * check and update first and last pointers
	     */
	    if (! theList->first)
		theList->first = theList->current;

	    if (! theList->last)
		theList->last = theList->current;
				     
	} /* if valid position */
	else
	    fprintf(stderr, "UmscListMove:  Invalid position (%d)\n",
		    position);

    } /* if the current object is defined */
    else
	fprintf(stderr, "UmscListMove:  Current is NULL\n");
		
} /* UmscListMove */

/*-------------------------------------------------------------------------*/
/*
 * Remove the object (if any) pointed to by the current pointer
 * Set to next or prev pointer as specified by dir
 */
void UmscListRemove(UmscList	theList,
		    int		direction)
{
    UmscNode	newCurrent;

    /*
     * Check if list is valid
     */
    if (!theList) {
	fprintf(stderr, "UmscListRemove:  List is NULL\n");
	return;
    }

    if (theList->current) {

	if ((direction == UmscNEXT) ||
	    (direction == UmscPREV)) {
	    UnlinkNode(theList, theList->current);
	    newCurrent = ((direction == UmscNEXT) ? theList->current->next :
			  theList->current->prev);

	    /* free the memory */
	    theList->free(theList->current->obj);
	    free(theList->current);
	    theList->current = newCurrent;
	    theList->cnt--;
	}
	else
	    fprintf(stderr, "UmscListRemove:  Unknown direction (%d)\n",
		    direction);
    }
    else
	fprintf(stderr, "UmscListRemove:  Current object is NULL\n");
}

/*-------------------------------------------------------------------------*/
/*
 * Free the space used by the list
 */
void UmscListFree(UmscList	theList,
		  int		freeObjects)
{
    UmscNode	next;
    UmscNode	current;
    
    /*
     * Check if list is valid
     */
    if (!theList) {
	fprintf(stderr, "UmscListFree:  List is NULL\n");
	return;
    }

    theList->current = theList->first;
    current = theList->current;
    
    while (current) {
	next = current->next;

	if (freeObjects)
	    theList->free(current->obj);

	free(current);
	current = next;
    }
    
    theList->cnt = 0;
    theList->first = NULL;
    theList->current = NULL;
    theList->last = NULL;
}

/*-------------------------------------------------------------------------*/
/*
 * Try to find the requested object.  Return object if found, NULL otherwise
 * Set the current pointer to the object
 */
void *UmscListFind(UmscList	theList,
		   void		*searchObj)
{
    void	*theObj = NULL;

    /*
     * Check if list is valid
     */
    if (!theList) {
	fprintf(stderr, "UmscListFind:  List is NULL\n");
	return (theObj);
    }
    
    /*
     * start at beginning of list
     */
    theList->current = theList->first;
    theObj = UmscListGetCurrent(theList);

    /*
     * use the user defined comparison routine and check until found or list
     * is exhausted
     */
    while ((theObj) &&
	   (theList->cmp(theObj, searchObj))) {
	UmscListGetNext(theList);
	theObj = UmscListGetCurrent(theList);
    }
    
    return (theObj);
    
} /* UmscListFind */

/*-------------------------------------------------------------------------*/
/*
 * Temporarily set the search routine to be 'searchRtn'
 * Call UmscFind.  Reset the search routine.
 */
void* UmscListFindWithFn(UmscList	theList,
			 int		(*searchRtn)(),
			 void		*searchObj)
{
    void	*theObj = NULL;
    int		(*oldSearch)();

    /* Check if list is valid */
    if (!theList) {
	fprintf(stderr, "UmscListFind:  List is NULL\n");
	return (theObj);
    }

    /* save old search routine */
    oldSearch = theList->cmp;

    /* set to new, if not null */
    if (searchRtn)
	theList->cmp = searchRtn;
    else
	theList->cmp = DefaultComparison;
 
    /* find it */
    theObj = UmscListFind(theList, searchObj);

    /* restore the search */
    theList->cmp = oldSearch;

    /* return the object, if any */
    return (theObj);
    
}

/*-------------------------------------------------------------------------*/
/*
 * make a new data structure that points to a list of the values in the
 * list (NULL terminated)
 */
void** UmscListDump(UmscList theList)
{
    UmscNode	node;
    void**	dump;
    void**	dumpPtr;
    int		i;

    /* check if list is valid */
    if ((! theList) ||
	(! theList->init)) {
	fprintf(stderr, "UmscListDump:  List (%#x) not initialized\n",
		theList);
	return(NULL);
    }

    /* allocate space for a new list */
    dump = (void**)malloc(sizeof(void*) * theList->cnt + 1);
    dumpPtr = dump;

    if (dump == NULL) {
	fprintf(stderr,
		"UmscListDump:  Unable to allocate memory for dump\n");
	return(NULL);
    }

    node = theList->first;
    i = 0;

    /* loop through the list */
    while (node) {
	*dumpPtr++ = node->obj;
	node = node->next;
    }

    /* normal exit */
    *dumpPtr = NULL;
    return(dump);
}

/*-------------------------------------------------------------------------*/
/*
 * Unlink the object (if any) pointed to by the current pointer
 * Set to next or prev pointer as specified by dir
 */
void UmscListUnlink(UmscList	theList,
		    int		direction)
{
    UmscNode	newCurrent;

    /*
     * Check if list is valid
     */
    if (!theList) {
	fprintf(stderr, "UmscListUnlink:  List is NULL\n");
	return;
    }

    if (theList->current) {

	if ((direction == UmscNEXT) ||
	    (direction == UmscPREV)) {
	    UnlinkNode(theList, theList->current);
	    newCurrent = ((direction == UmscNEXT) ? theList->current->next :
			  theList->current->prev);

	    /* free the memory */
	    free(theList->current);
	    theList->current = newCurrent;
	    theList->cnt--;
	}
	else
	    fprintf(stderr, "UmscListUnlink:  Unknown direction (%d)\n",
		    direction);
    }
    else
	fprintf(stderr, "UmscListUnlink:  Current object is NULL\n");
}

/*-------------------------------------------------------------------------*/
