/* combimg -- perform a pixelwise combination of a set of 2D images     */
/*                                                                      */
/* operations: average (avg)                                            */
/*             minimum (min)                                            */
/*             maximum (max)                                            */
/*             median  (med)                                            */
/*             rank or order statistic (os) nnn (rank 1 == max)         */
/*             ordinary linear combination (olc) coefficients a1 ... aN */
/*             rank-ordered linear comb. (rlc) coefficients a1 ... aN   */

/* version 1.0     26 Oct 1992 */

/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */


#include <math.h>		
#include <stdio.h>
#include <strings.h>
#include <values.h>
#include <sys/types.h>
#include "rasterio.h"
#include "combimg.h"


main( argc, argv ) 
   unsigned int argc;    /* count of arguments parsed by operating system    */
   char *argv[];         /* pointers to arguments parsed by operating system */
   {
   FILE *fptr;
   char filename[80];
   char compname[80];
   char **filelist;
   char *outfile;
   char *slash;
   byte **image;
   float *coefflist;
   float *c;
   float f;
   struct rasterfile H; /* rasterfile headers */
   byte *cmap;          /* colormaps */
   byte *I,*O;          /* image pointers */
   byte *u,*v,*w,*z;   	/* image pointers */
   int X,Y,N;           /* image dimensions */
   byte *pix;
   int i,j,k;           /* indices */
   int r,s;
   int nimgs;
   int ncoeffs;
   int n = argc;
   int mode;
   int rank;
   int done;

   char Index[IXLGTH+2];
   char Suffix[SXLGTH+1];
   char Format[FTLGTH];



   if (argc < 7)
      {
      fprintf(stderr,
         "usage: combimg -i filename_file | filename1 ... filenameN\n\
               -o output_filename  \n\
               -c avg | min | max | med | os nnn | olc[|rlc] a1 ... aN\n");
      exit(0);
      }

   filelist = (char **)NULL;
   mode = 0;
   rank = 0;

   /* get arguments */
   while ( --n )
      {
      if ( argv[n][0] == '-' )          /* then this is a switch */
         {
         if ( argv[n][1] == 'i' )       /* filenames follow */
            {
            if ( (n+2 >= argc) || (argv[n+2][0] == '-') ) /* then filenames */
               {                                          /* are in a file. */
               if ( !(fptr = fopen( argv[n+1], "r" )) )
                  {
                  fprintf(stderr,"Unable to open %s\n",argv[n+1]);
                  exit( 0 );
                  }
               nimgs = -1;
               while ( !feof(fptr) )
                  {
                  fscanf( fptr,"%s",filename );
                  ++nimgs;
                  }
               if ( nimgs < 2 )
                  {
                  fprintf(stderr,"There must be at least two filenames\n");
                  exit( 0 );
                  }
               else
                  {
                  fprintf(stderr,"%d filenames are specified.\n",nimgs);
                  }
               if ( !(filelist = 
                     (char **)calloc( nimgs, sizeof(char *) + NAMESIZE )) )
                  {
                  fprintf(stderr,"error allocating filelist\n");
                  exit( 0 );
                  }
               rewind(fptr);
               for ( i=0; i<nimgs; ++i)
                  {
                  *(filelist+i) = 
                   (char *)(filelist) + nimgs*sizeof(char *) + i*NAMESIZE;
                  fscanf( fptr, "%s",*(filelist+i) );
                  /*fprintf(stderr,"name[%d]: %s\n",i,*(filelist+i));*/
                  }
               fclose(fptr);
               }
            else /* filenames are on command line */
               {
               nimgs = 0;
               while ( (n+1+nimgs<argc) && (argv[n+1+nimgs][0] != '-') ) 
                 ++nimgs;
               if ( !(filelist = 
                     (char **)calloc( nimgs, sizeof(char *) )) )
                  {
                  fprintf(stderr,"error allocating filelist\n");
                  exit( 0 );
                  }
               for ( i=0; i<nimgs; ++i)
                  {
                  *(filelist+i) = argv[n+1+i];
                  /*fprintf(stderr,"name[%d]: %s\n",i,*(filelist+i));*/
                  }
               }
            }
         else if ( argv[n][1] == 'o' )       /* output filename follows */
            {
            outfile = argv[n+1];
            }
         else if ( argv[n][1] == 'c' )       /* combination mode follows */
            {
            if ( !strcasecmp( argv[n+1], "avg" ) )
               {
               mode = MAVG;
               }
            else if ( !strcasecmp( argv[n+1], "min" ) )
               {
               mode = MMIN;
               }
            else if ( !strcasecmp( argv[n+1], "max" ) )
               {
               mode = MMAX;
               }
            else if ( !strcasecmp( argv[n+1], "med" ) )
               {
               mode = MMED;
               }
            else if ( !strcasecmp( argv[n+1], "os" ) )
               {
               mode = MOS;
               rank = atoi( argv[n+2] );
               }
            else if ( !strcasecmp( argv[n+1], "olc" ) )
               {
               mode = MOLC;
               ncoeffs = 0;
               while ( (n+2+ncoeffs<argc) && (argv[n+2+ncoeffs][0] != '-') ) 
                 ++ncoeffs;
               if ( !(coefflist = 
                     (float *)calloc( 1, ncoeffs*(sizeof(float)) )) )
                  {
                  fprintf(stderr,"error allocating coefficient list\n");
                  exit( 0 );
                  }
               for ( i=0; i<ncoeffs; ++i)
                  {
                  *(coefflist+i) = atof(argv[n+2+i]);
                  /*fprintf(stderr,"coeff[%d]: %f\n",i,*(coefflist+i));*/
                  }
               }
            else if ( !strcasecmp( argv[n+1], "rlc" ) )
               {
               mode = MRLC;
               ncoeffs = 0;
               while ( (n+2+ncoeffs<argc) && (argv[n+2+ncoeffs][0] != '-') ) 
                 ++ncoeffs;
               if ( !(coefflist = 
                     (float *)calloc( 1, ncoeffs*(sizeof(float)) )) )
                  {
                  fprintf(stderr,"error allocating coefficient list\n");
                  exit( 0 );
                  }
               for ( i=0; i<ncoeffs; ++i)
                  {
                  *(coefflist+i) = atof(argv[n+2+i]);
                  /*fprintf(stderr,"coeff[%d]: %f\n",i,*(coefflist+i));*/
                  }
               }
            }
         }
      }

   if ( !filelist || !outfile || !mode )
      {
      fprintf(stderr,"-i, -o, and -c must be specified\n");
      exit( 0 );
      }

   if ( ((mode == MOLC) || (mode == MRLC)) && (nimgs != ncoeffs) )
      {
      fprintf(stderr,"Number of filenames must equal number of coefficients\n");
      exit( 0 );
      }

   if ( mode == MMED )
      {
      rank = nimgs/2+1;
      }
   else if ( mode == MOS )
      {
      if ( (rank < 0) || (rank > nimgs) )
         {
         fprintf(stderr,"Rank must be between 1 and %d\n",nimgs);
         exit( 0 );
         }
      else if ( rank == 1 )
         {
         mode == MMAX;
         }
      else if ( rank == nimgs )
         {
         mode == MMIN;
         }
      }

      

   if ( !(image = (byte **)calloc( nimgs, sizeof(byte *) )) )
      {
      fprintf(stderr,"error allocating image pointer list\n");
      exit( 0 );
      }

   /* try to open the first file */
   fptr = OpenFile( *filelist, "", "r" );
   if (  ReadRasterFile( fptr, &H, &cmap, &I, 0, 0, 0, 0, ALLOC )  ) exit( 0 );
   fclose( fptr );

   /* get size info */
   X = H.ras_width;
   Y = H.ras_height;
   N = X*Y;

   /* allocate image buffers */

   *image = I;
   for ( i=1; i<nimgs; ++i)
      {
      if ( !(*(image+i) = (byte *)calloc( N, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating image %d\n", i);
         exit( 0 );
         }
      }
   if ( !(O = (byte *)calloc( N, sizeof(byte)) ) )
      {
      fprintf(stderr,"error allocating output image \n");
      exit( 0 );
      }

   if ( !( (mode == MAVG)  || (mode == MOLC) ) )
      if ( !(pix = (byte *)calloc( nimgs, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating pixel sort array \n");
         exit( 0 );
         }



   /* get the images */
   for ( i=0; i<nimgs; ++i)
      {
      fptr = OpenFile( *(filelist+i), "", "r" );
      if (  ReadRasterFile( fptr, &H, &cmap, image+i, 0, 0, 0, 0, 0 )  ) 
         exit( 0 );
      fclose(fptr);
      /* remap the image */
      ExtractLuminance( &H, cmap, *(image+i), 0, 0, 0, 0 );
      }

   /* do the various combinations */
   if ( mode == MAVG )  /* straight average */
      {
      z=O;
      for ( j=0; j<N; ++j )
         {
         r = 0;
         for ( i=0; i<nimgs; i++ )
            r += *(*(image+i)+j);
         *(z++) = r / nimgs;
         }
      }
   else if ( mode == MOLC )  /* ordinary linear combination */
      {
      z=O;
      for ( j=0; j<N; ++j )
         {
         f = 0;
         c = coefflist;
         for ( i=0; i<nimgs; i++ )
            f += *(c++) * (float)(*(*(image+i)+j));
         *(z++) = (byte)(f > WHITE) 
                ? WHITE : ( (f < BLACK) ? BLACK : f );
         }
      }
   else if ( mode == MMAX )  /* maximum */
      {
      z=O;
      for ( j=0; j<N; ++j )
         {
         r = BLACK;
         for ( i=0; i<nimgs; i++ )
            {
            s = *(*(image+i)+j);
            if ( s > r )
               r = s;
            }
         *(z++) = r;
         }
      }
   else if ( mode == MMIN )  /* minimum */
      {
      z=O;
      for ( j=0; j<N; ++j )
         {
         r = WHITE;
         for ( i=0; i<nimgs; i++ )
            {
            s = *(*(image+i)+j);
            if ( s < r )
               r = s;
            }
         *(z++) = r;
         }
      }
   else if ( (mode == MMED) || (mode == MOS) || (mode == MRLC) )
      {    /* median, order stat, or rank-ordered linear combination */
      z=O;
      for ( j=0; j<N; ++j ) /* process loop */
         {
         w = pix;
         for ( i=0; i<nimgs; i++ )  /* get pixels */
            {
            *(w++) = *(*(image+i)+j);
            }

         for ( i=1; i<nimgs; i++ )  /* sort loop */
            {
            v = pix+i;
            u = --v;
            if ( *u > *v )  /* order loop */
               {
                r = *u;
               *u = *v;
               *v =  r;
               j=i-1;
               while ( j )  /* bubble loop */
                  {
                  v = pix+j;
                  u = --v;
                  if ( *u <= *v ) /* are ok */
                     {
                     break;       /* go to next order */
                     }
                  else            /* swap them and */
                     {
                      r = *u;
                     *u = *v;
                     *v =  r;
                     }
                  --j;            /* check next bubble */
                  }  /* bubble loop */
               }  /* order loop */
            }  /* sort loop */

         if ( mode != MRLC ) /* (mode == MMED) || (mode == MOS) */
            {
            *(z++) = *(pix + nimgs - rank);
            }
         else /* mode == MRLC */
            {
            f = 0;
            u = pix;
            c = coefflist + ncoeffs;
            for ( i=0; i<nimgs; i++ )
               {
               f += *(--c) * (float)*(u++);
               }
            *(z++) = (byte)(f > WHITE) 
                   ? WHITE : ( (f < BLACK) ? BLACK : f );
            }

         }  /* process loop */
      }  /* mode == MMED, MOS, or MRLC */

   /* make the output file */
   fptr = OpenFile( outfile, "", "w" );
   WriteRasterFile( fptr, &H, cmap, O, 0, 0, 0, 0 ); 
   fclose( fptr );               
   }

