/* wmesa.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2 beta
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Windows driver by: Mark E. Peterson (markp@ic.mankato.mn.us)
 */

/*
$Id: wmesa.c,v 1.5 1995/07/24 18:56:00 brianp Exp $

$Log: wmesa.c,v $
 * Revision 1.5  1995/07/24  18:56:00  brianp
 * added dd_finish()
 *
 * Revision 1.4  1995/06/21  15:39:26  brianp
 * removed #include "wmesaP.h"
 *
 * Revision 1.3  1995/06/21  15:38:10  brianp
 * added struct_wmesa_context, don't need wmesaP.h anymore
 *
 * Revision 1.2  1995/06/09  18:53:39  brianp
 * removed compute_mult_and_shift()
 * dd_read/write_color_span/pixels() take GLubyte arrays instead of GLints
 *
 * Revision 1.1  1995/06/09  18:52:29  brianp
 * Initial revision
 *
 */


#include <stdlib.h>
#include <windows.h>
#include "gl\wmesa.h"
#include "bresenhm.h"
#include "context.h"
#include "dd.h"
#include "quikpoly.h"
#include "xform.h"
#include "wing.h"


/* Bit's used for dest: */
#define FRONT_PIXMAP	1
#define BACK_PIXMAP	2
#define BACK_XIMAGE	4

struct wmesa_context
{
  struct gl_context *gl_ctx;	/* the main library context */
  HWND Window;
  HDC Compat_DC;                /* Display context for double buffering. */
  HBITMAP Old_Compat_BM,Compat_BM;            /* Bitmap for double buffering */
  GLuint width, height,ScanWidth;
  GLboolean db_flag;	/* double buffered? */
  GLboolean rgb_flag;	/* RGB mode? */
  GLuint depth;		/* bits per pixel (1, 8, 24, etc) */
  unsigned long pixel;	/* current color index or RGBA pixel value */
  unsigned long clearpixel; /* pixel for clearing the color buffers */
  GLint rmult, gmult, bmult, amult;	/* Multiplier */
	GLint rshift, gshift, bshift, ashift;	/* Bit shifts */
  char *ScreenMem; // WinG memory
  BITMAPINFO *IndexFormat;
  HPALETTE hPal; // Current Palette
};


#ifdef NDEBUG
  #define assert(ignore)	((void) 0)
#else
  void Mesa_Assert(void *Cond,void *File,unsigned Line)
  {
    char Msg[512];
    sprintf(Msg,"%s %s %d",Cond,File,Line);
    MessageBox(NULL,Msg,"Assertion failed.",MB_OK);
    exit(1);
  }
  #define assert(e)	if (!e) Mesa_Assert(#e,__FILE__,__LINE__);
#endif
#define DD_GETDC ((Current->db_flag) ? Current->Compat_DC : GetDC(Current->Window))
#define DD_RELEASEDC if (!Current->db_flag) ReleaseDC(Current->Window,DC)
static WMesaContext Current = NULL;
#ifdef __SYMANTEC_BUGS
  struct dd_function_table DD;
  #include "vb.h"
  struct vertex_buffer VB;
#endif
#define FLIP(Y)  (Current->height-(Y)-1)
/* Finish all pending operations and synchronize. */
void dd_finish(void)
{
}
void dd_flush(void)
{
}
/* Return characteristics of the output buffer. */
void dd_buffer_info( GLuint *width, GLuint *height,GLuint *depth)
{
  int New_Size;
  RECT CR;
  GetClientRect(Current->Window,&CR);
  *width=CR.right;
  *height=CR.bottom;
  *depth = Current->depth;
  New_Size=((*width)!=Current->width) || ((*height)!=Current->height);
  if (New_Size)
  {
    Current->width=*width;
    Current->ScanWidth=Current->width;
    if ((Current->ScanWidth%sizeof(long))!=0)
      Current->ScanWidth+=(sizeof(long)-(Current->ScanWidth%sizeof(long)));
    Current->height=*height;
    if (Current->db_flag)
    {
      if (Current->rgb_flag==GL_TRUE)
        Current->Compat_BM=CreateCompatibleBitmap(Current->Compat_DC,Current->width,Current->height);
      else
      {
        Current->IndexFormat->bmiHeader.biWidth=Current->width;
        if (Current->IndexFormat->bmiHeader.biHeight<0)
          Current->IndexFormat->bmiHeader.biHeight=-Current->height;
        else
          Current->IndexFormat->bmiHeader.biHeight=Current->height;
        Current->Compat_BM=WinGCreateBitmap(Current->Compat_DC,Current->IndexFormat,&((void *) Current->ScreenMem));
      }
      DeleteObject(SelectObject(Current->Compat_DC,Current->Compat_BM));
    }
  }
}
/*
 * Specify which buffer to read from.
 * Input:  mode - as passed to glReadBuffer
 * Return:  new mode or GL_FALSE if error.
 */
GLenum dd_read_buffer(GLenum mode)
{
  GLenum m;
  switch (mode)
  {
    case GL_FRONT:
    case GL_FRONT_LEFT:
    case GL_LEFT:
      m = mode;
	    //Current->Readable = Current->Window;
    break;
    case GL_BACK:
    case GL_BACK_LEFT:
	    m = mode;
      //if (Current->backpixmap)
	    //  Current->readable = Current->backpixmap;
      //else if (Current->backimage)
	    //  Current->readable = None;
    break;
    case GL_AUX0:
    case GL_FRONT_RIGHT:
    case GL_RIGHT:
    case GL_BACK_RIGHT:
    default:
	    return GL_FALSE;
  }
  return m;
}
/*
 * Specify which buffer(s) to draw into.
 * Input:  mode - as passed to glDrawBuffer
 * Return:  new output mode or GL_FALSE if error.
 */
GLenum dd_draw_buffer(GLenum mode)
{
  GLenum m;
  switch (mode)
  {
    case GL_NONE:
      m = GL_NONE;
      //Current->dest = 0;
    break;
    case GL_FRONT:
    case GL_FRONT_LEFT:
	    m = mode;
	    //Current->dest = FRONT_PIXMAP;
	  break;
    case GL_BACK:
    case GL_BACK_LEFT:
	    m = mode;
      //Current->dest = 0;
    break;
    case GL_LEFT:
    case GL_FRONT_AND_BACK:
	    m = mode;
      //Current->dest = FRONT_PIXMAP;
    break;
    case GL_RIGHT:
    case GL_FRONT_RIGHT:
    case GL_BACK_RIGHT:
    case GL_AUX0:
    default:
	   /* Don't change drawbuffer, return error signal */
	   return GL_FALSE;
  }
  return m;
}
/*
 * Set the color index used to clear the color buffer.
 * This implements glClearIndex().
 */
void dd_clear_index(GLuint index)
{
  Current->clearpixel = index;
}
/*
 * Set the color used to clear the color buffer.
 * This implements glClearColor().
 */
void dd_clear_color(const GLfloat color[4])
{
  Current->clearpixel=RGB(color[0],color[1],color[2]);
}
/*
 * Clear the specified region of the color buffer using the clear color
 * or index as specified by one of the two functions above.
 */
void dd_clear(GLboolean all,GLint x, GLint y, GLint width, GLint height )
{
  if (all)
  {
    x=y=0;
    width=Current->width;
    height=Current->height;
  }
  if (Current->rgb_flag==GL_TRUE)
  {
    HDC DC=DD_GETDC;
    HPEN Pen=CreatePen(PS_SOLID,1,Current->clearpixel);
    HBRUSH Brush=CreateSolidBrush(Current->clearpixel);
    HPEN Old_Pen=SelectObject(DC,Pen);
    HBRUSH Old_Brush=SelectObject(DC,Brush);
    Rectangle(DC,x,y,x+width,y+height);
    SelectObject(DC,Old_Pen);
    SelectObject(DC,Old_Brush);
    DeleteObject(Pen);
    DeleteObject(Brush);
    DD_RELEASEDC;
  }
  else
  {
    int i;
    char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
    for (i=0; i<height; i++)
    {
      memset(Mem,Current->clearpixel,width);
      Mem+=width;
    }
  }
}
/*
 * Set the pixel logic operation.  Return GL_TRUE if the device driver
 * can perform the operation, otherwise return GL_FALSE.  If GL_FALSE
 * is returned, the logic op will be done in software by Mesa.
 */
GLboolean dd_logicop( GLenum op )
{
  return GL_FALSE;
}
/**********************************************************************/
/*****                Simple rendering functions                  *****/
/**********************************************************************/
/* Set the current color index. */
void dd_index(GLuint index)
{
  Current->pixel=index;
}
/* Set the index mode bitplane mask. */
void dd_index_mask(GLuint mask)
{
  assert(0);
}
/* Set the current RGBA color. */
void dd_color( const GLfloat color[4] )
{
  Current->pixel=RGB(color[0]*Current->rmult,
                     color[1]*Current->gmult,
                     color[2]*Current->bmult);
}
/* Set the RGBA drawing mask. */
void dd_color_mask( GLboolean rmask, GLboolean gmask,GLboolean bmask, GLboolean amask)
{
  assert(0);
}
/* Write a pixel using current index or color. */
void dd_draw_pixel( GLint x, GLint y )
{
  if (Current->rgb_flag==GL_TRUE)
  {
    HDC DC=DD_GETDC;
    y=FLIP(y);
    SetPixel(DC,x,y,Current->pixel);
    DD_RELEASEDC;
  }
  else
    Current->ScreenMem[(y*Current->ScanWidth)+x]=Current->pixel;
}
#define BRESENHAM_PLOT(x,y) dd_draw_pixel(x,y);
/* Draw a line using the current index or color. */
void dd_draw_line( GLint x0, GLint y0, GLint x1, GLint y1 )
{
  if (Current->rgb_flag==GL_TRUE)
  {
    HDC DC=DD_GETDC;
    HPEN Pen=CreatePen(PS_SOLID,1,Current->pixel);
    HPEN Old_Pen=SelectObject(DC,Pen);
    y0=FLIP(y0);
    y1=FLIP(y1);
    MoveToEx(DC,x0,y0,NULL);
    LineTo(DC,x1,y1);
    SelectObject(DC,Old_Pen);
    DeleteObject(Pen);
    DD_RELEASEDC;
  }
  else
  {
    if (y0==y1)
    {
      if (x1<x0)
      {
        GLint t=x1;
        x1=x0;
        x0=t;
      }
      memset(Current->ScreenMem+y0*Current->ScanWidth+x0,Current->pixel,x1-x0+1);
    }
    else
      BRESENHAM(x0,y0,x1,y1);
  }
}
/* Draw a convex polygon using the current index or color. */
void dd_draw_polygon( GLuint n, GLint x[], GLint y[] )
{
  if (Current->rgb_flag==GL_TRUE)
  {
    POINT *Pts=(POINT *) malloc(n*sizeof(POINT));
    int i;
    HDC DC=DD_GETDC;
    HPEN Pen=CreatePen(PS_SOLID,1,Current->pixel);
    HBRUSH Brush=CreateSolidBrush(Current->pixel);
    HPEN Old_Pen=SelectObject(DC,Pen);
    HBRUSH Old_Brush=SelectObject(DC,Brush);
    for (i=0; i<n; i++)
    {
      Pts[i].x=x[i];
      Pts[i].y=FLIP(y[i]);
    }
    Polygon(DC,Pts,n);
    SelectObject(DC,Old_Pen);
    SelectObject(DC,Old_Brush);
    DeleteObject(Pen);
    DeleteObject(Brush);
    DD_RELEASEDC;
    free(Pts);
  }
  else
    gl_quick_polygon(n,x,y);
}
/**********************************************************************/
/*****             Span-based pixel drawing and reading           *****/
/**********************************************************************/
/* Write a horizontal span of color-index pixels with a boolean mask. */
void dd_write_index_span( GLuint n, GLint x, GLint y,const GLuint index[], const GLubyte mask[] )
{
  int i;
  char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    if (mask[i])
      Mem[i]=index[i];
}
/*
 * Write a horizontal span of pixels with a boolean mask.  The current
 * color index is used for all pixels.
 */
void dd_write_monoindex_span(GLuint n,GLint x,GLint y,const GLubyte mask[])
{
  int i;
  char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    if (mask[i])
      Mem[i]=Current->pixel;
}
/* Read a horizontal span of color-index pixels. */
void dd_read_index_span( GLuint n, GLint x, GLint y, GLuint index[])
{
  int i;
  char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    index[i]=Mem[i];
}
/* Write a horizontal span of color pixels with a boolean mask. */
void dd_write_color_span( GLuint n, GLint x, GLint y,
			  const GLubyte red[], const GLubyte green[],
			  const GLubyte blue[], const GLubyte alpha[],
			  const GLubyte mask[] )
{
  if (Current->rgb_flag==GL_TRUE)
  {
    int i;
    HDC DC=DD_GETDC;
    y=FLIP(y);
    for (i=0; i<n; i++)
      if (mask[i])
        SetPixel(DC,x+i,y,RGB(red[i],green[i],blue[i]));
    DD_RELEASEDC;
  }
  else
  {
    int i;
    char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
    for (i=0; i<n; i++)
      if (mask[i])
        Mem[i]=GetNearestPaletteIndex(Current->hPal,RGB(red[i],green[i],blue[i]));
  }
}
/*
 * Write a horizontal span of pixels with a boolean mask.  The current color
 * is used for all pixels.
 */
void dd_write_monocolor_span( GLuint n, GLint x, GLint y,const GLubyte mask[])
{
  int i;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  y=FLIP(y);
  for (i=0; i<n; i++)
    if (mask[i])
      SetPixel(DC,x+i,y,Current->pixel);
  DD_RELEASEDC;
}
/* Read an array of color pixels. */
void dd_read_color_pixels( GLuint n, const GLint x[], const GLint y[],
				  GLubyte red[], GLubyte green[],
				  GLubyte blue[], GLubyte alpha[] )
{
  int i;
  COLORREF Color;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  for (i=0; i<n; i++)
  {
    Color=GetPixel(DC,x[i],FLIP(y[i]));
    red[i]=GetRValue(Color);
    green[i]=GetGValue(Color);
    blue[i]=GetBValue(Color);
    alpha[i]=255;
  }
  DD_RELEASEDC;
  memset(alpha,0,n*sizeof(GLint));
}
/* Read a horizontal span of color pixels. */
void dd_read_color_span( GLuint n, GLint x, GLint y,
		         GLubyte red[], GLubyte green[],
			 GLubyte blue[], GLubyte alpha[] )
{
  int i;
  COLORREF Color;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  y=FLIP(y);
  for (i=0; i<n; i++)
  {
    Color=GetPixel(DC,x+i,y);
    red[i]=GetRValue(Color);
    green[i]=GetGValue(Color);
    blue[i]=GetBValue(Color);
    alpha[i]=255;
  }
  DD_RELEASEDC;
  memset(alpha,0,n*sizeof(GLint));
}
/**********************************************************************/
/*****            Array-based pixel drawing and reading           *****/
/**********************************************************************/
/* Write an array of pixels with a boolean mask. */
void dd_write_index_pixels( GLuint n, const GLint x[], const GLint y[],
				  const GLuint index[], const GLubyte mask[] )
{
  int i;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
  {
    if (mask[i])
    {
      dd_index(index[i]);
      dd_draw_pixel(x[i],y[i]);
    }
  }
}
/*
 * Write an array of pixels with a boolean mask.  The current color
 * index is used for all pixels.
 */
void dd_write_monoindex_pixels( GLuint n,
				       const GLint x[], const GLint y[],
				       const GLubyte mask[] )
{
  int i;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
  {
    if (mask[i])
      dd_draw_pixel(x[i],y[i]);
  }
}
/* Read an array of color index pixels. */
void dd_read_index_pixels( GLuint n,
				  const GLint x[], const GLint y[],
				  GLuint indx[] )
{
  int i;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    indx[i]=*(Current->ScreenMem+y[i]*Current->ScanWidth+x[i]);
}
/* Write an array of pixels with a boolean mask. */
void dd_write_color_pixels( GLuint n, const GLint x[], const GLint y[],
				   const GLubyte r[], const GLubyte g[],
				   const GLubyte b[], const GLubyte a[],
				   const GLubyte mask[] )
{
  int i;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  for (i=0; i<n; i++)
    if (mask[i])
      SetPixel(DC,x[i],FLIP(y[i]),RGB(r[i],g[i],b[i]));
  DD_RELEASEDC;
}
/*
 * Write an array of pixels with a boolean mask.  The current color
 * is used for all pixels.
 */
void dd_write_monocolor_pixels( GLuint n,
				       const GLint x[], const GLint y[],
    		       const GLubyte mask[] )
{
  int i;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  for (i=0; i<n; i++)
    if (mask[i])
      SetPixel(DC,x[i],FLIP(y[i]),Current->pixel);
  DD_RELEASEDC;
}
#define PAL_SIZE 256
void GetPalette(HPALETTE Pal,RGBQUAD *aRGB)
{
	int i;
	HDC hdc;
	struct
	{
		WORD Version;
		WORD NumberOfEntries;
		PALETTEENTRY aEntries[PAL_SIZE];
	} Palette =
	{
		0x300,
		PAL_SIZE
	};
	hdc=GetDC(NULL);
	if (Pal!=NULL)
    GetPaletteEntries(Pal,0,PAL_SIZE,Palette.aEntries);
  else
    GetSystemPaletteEntries(hdc,0,PAL_SIZE,Palette.aEntries);
	if (GetSystemPaletteUse(hdc) == SYSPAL_NOSTATIC)
	{
		for(i = 0; i <PAL_SIZE; i++)
			Palette.aEntries[i].peFlags = PC_RESERVED;
		Palette.aEntries[255].peRed = 255;
		Palette.aEntries[255].peGreen = 255;
		Palette.aEntries[255].peBlue = 255;
		Palette.aEntries[255].peFlags = 0;
		Palette.aEntries[0].peRed = 0;
		Palette.aEntries[0].peGreen = 0;
		Palette.aEntries[0].peBlue = 0;
		Palette.aEntries[0].peFlags = 0;
	}
	else
	{
		int nStaticColors;
		int nUsableColors;
		nStaticColors = GetDeviceCaps(hdc, NUMCOLORS)/2;
		for (i=0; i<nStaticColors; i++)
			Palette.aEntries[i].peFlags = 0;
		nUsableColors = PAL_SIZE-nStaticColors;
		for (; i<nUsableColors; i++)
			Palette.aEntries[i].peFlags = PC_RESERVED;
		for (; i<PAL_SIZE-nStaticColors; i++)
			Palette.aEntries[i].peFlags = PC_RESERVED;
		for (i=PAL_SIZE-nStaticColors; i<PAL_SIZE; i++)
			Palette.aEntries[i].peFlags = 0;
	}
	ReleaseDC(NULL,hdc);
  for (i=0; i<PAL_SIZE; i++)
  {
    aRGB[i].rgbRed=Palette.aEntries[i].peRed;
    aRGB[i].rgbGreen=Palette.aEntries[i].peGreen;
    aRGB[i].rgbBlue=Palette.aEntries[i].peBlue;
    aRGB[i].rgbReserved=Palette.aEntries[i].peFlags;
  }
}
WMesaContext WMesaCreateContext(HWND hWnd,HPALETTE Pal,GLboolean rgb_flag,GLboolean db_flag)
{
  BITMAPINFO *Rec;
  HDC DC;
  RECT CR;
  WMesaContext c;
  DD.draw_pixel=dd_draw_pixel;
  DD.draw_line=dd_draw_line;
  DD.draw_polygon=dd_draw_polygon;
  DD.write_color_span=dd_write_color_span;
  DD.write_monocolor_span=dd_write_monocolor_span;
  DD.write_color_pixels=dd_write_color_pixels;
  DD.write_monocolor_pixels=dd_write_monocolor_pixels;
  DD.write_index_span=dd_write_index_span;
  DD.write_monoindex_span=dd_write_monoindex_span;
  DD.write_index_pixels=dd_write_index_pixels;
  DD.write_monoindex_pixels=dd_write_monoindex_pixels;
  DD.read_index_span=dd_read_index_span;
  DD.read_color_span=dd_read_color_span;
  DD.read_index_pixels=dd_read_index_pixels;
  DD.read_color_pixels=dd_read_color_pixels;
  c = (struct wmesa_context *) calloc(1,sizeof(struct wmesa_context));
  if (!c)
    return NULL;
  /* allocate a new Mesa context */
  c->gl_ctx = gl_new_context(NULL);
  c->Window=hWnd;
  if (rgb_flag==GL_FALSE)
  {
    /* COLOR-INDEXED WINDOW:
     * Even if the visual is TrueColor or DirectColor we treat it as
     * being color indexed.  This is weird but might be useful to someone.
     */
    c->rgb_flag = GL_FALSE;
    c->gl_ctx->RGBAflag = GL_FALSE;
    c->pixel = 1;
    db_flag=GL_TRUE; // WinG requires double buffering
    //c->gl_ctx->BufferDepth = windepth;
  }
  else
  {
    c->rgb_flag = GL_TRUE;
    c->gl_ctx->RGBAflag = GL_TRUE;
    c->pixel = 0;
    c->gl_ctx->RedScale   = c->rmult = 255;
    c->gl_ctx->GreenScale = c->gmult = 255;
    c->gl_ctx->BlueScale  = c->bmult = 255;
    c->gl_ctx->AlphaScale = c->amult = 255;
    c->rshift=0;
    c->gshift=0;
    c->bshift=0;
    c->ashift=0;
  }
  GetClientRect(c->Window,&CR);
  c->width=CR.right;
  c->height=CR.bottom;
  if (db_flag)
  {
    c->db_flag = 1;
    c->gl_ctx->Color.DrawBuffer = GL_BACK;
    c->gl_ctx->Pixel.ReadBuffer = GL_BACK;
    /* Double buffered */
    if (c->rgb_flag==GL_TRUE)
    {
      DC=GetDC(c->Window);
      c->Compat_DC=CreateCompatibleDC(DC);
      c->Compat_BM=CreateCompatibleBitmap(DC,c->width,c->height);
      ReleaseDC(c->Window,DC);
      c->Old_Compat_BM=SelectObject(c->Compat_DC,c->Compat_BM);
    }
    else
    {
      c->Compat_DC=WinGCreateDC();
      Rec=(BITMAPINFO *) malloc(sizeof(BITMAPINFO)+(PAL_SIZE-1)*sizeof(RGBQUAD));
      c->hPal=Pal;
      GetPalette(Pal,Rec->bmiColors);
      WinGRecommendDIBFormat(Rec);
      Rec->bmiHeader.biWidth=c->width;
      Rec->bmiHeader.biHeight*=c->height;
      Rec->bmiHeader.biClrUsed=PAL_SIZE;
      if (Rec->bmiHeader.biPlanes!=1 || Rec->bmiHeader.biBitCount!=8)
      {
        MessageBox(NULL,"Error.","This code presumes a 256 color, single plane, WinG Device.\n",MB_OK);
        exit(1);
      }
      c->Compat_BM=WinGCreateBitmap(c->Compat_DC,Rec,&((void *) c->ScreenMem));
      c->Old_Compat_BM=SelectObject(c->Compat_DC,c->Compat_BM);
      WinGSetDIBColorTable(c->Compat_DC,0,PAL_SIZE,Rec->bmiColors);
      c->IndexFormat=Rec;
      c->ScanWidth=c->width;
      if ((c->ScanWidth%sizeof(long))!=0)
        c->ScanWidth+=(sizeof(long)-(c->ScanWidth%sizeof(long)));
    }
  }
  else
  {
    /* Single Buffered */
    c->db_flag = 0;
    c->gl_ctx->Color.DrawBuffer = GL_FRONT;
    c->gl_ctx->Pixel.ReadBuffer = GL_FRONT;
  }
  c->gl_ctx->DBflag = db_flag;
  return c;
}
void WMesaDestroyContext( WMesaContext c )
{
   gl_destroy_context( c->gl_ctx );
   if (c->db_flag)
   {
     SelectObject(c->Compat_DC,c->Old_Compat_BM);
     DeleteDC(c->Compat_DC);
     DeleteObject(c->Compat_BM);
   }
   free( (void *) c );
}
void WMesaMakeCurrent( WMesaContext c )
{
   gl_set_context( c->gl_ctx );
   Current = c;
   if (Current->gl_ctx->Viewport.Width==0)
      /* initialize viewport to window size */
      gl_viewport( 0, 0, Current->width, Current->height );
}
void WMesaSwapBuffers( void )
{
  // *** Perhaps the DC should be saved in WMesaContext?
  HDC DC;
  if (Current->db_flag)
  {
    DC=GetDC(Current->Window);
    if (Current->rgb_flag)
      BitBlt(DC,0,0,Current->width,Current->height,Current->Compat_DC,0,0,SRCCOPY);
    else
      WinGBitBlt(DC,0,0,Current->width,Current->height,Current->Compat_DC,0,0);
    ReleaseDC(Current->Window,DC);
  }
}
void WMesaPaletteChange(HPALETTE Pal)
{
  if (Current && Current->rgb_flag==GL_FALSE)
  {
    Current->hPal=Pal;
    GetPalette(Pal,Current->IndexFormat->bmiColors);
    WinGSetDIBColorTable(Current->Compat_DC,0,PAL_SIZE,Current->IndexFormat->bmiColors);
  }
}

