/* enable.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: enable.c,v 1.12 1995/06/21 15:10:47 brianp Exp $

$Log: enable.c,v $
 * Revision 1.12  1995/06/21  15:10:47  brianp
 * allocate stencil buffer when stencil test is enabled
 *
 * Revision 1.11  1995/06/20  22:11:18  brianp
 * fixed depth buffer allocation bug in glEnable(), per Asif Khan
 *
 * Revision 1.10  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.9  1995/05/12  19:26:43  brianp
 * replaced CC.Mode!=0 with INSIDE_BEGIN_END
 *
 * Revision 1.8  1995/03/27  20:31:26  brianp
 * new Texture.Enabled scheme
 *
 * Revision 1.7  1995/03/24  20:03:21  brianp
 * added gl_update_pixel_logic call to GL_BLEND case
 *
 * Revision 1.6  1995/03/24  17:00:22  brianp
 * added gl_update_pixel_logic
 *
 * Revision 1.5  1995/03/09  21:41:39  brianp
 * call gl_udpate_rasterflags for GL_CULL_FACE
 *
 * Revision 1.4  1995/03/09  19:07:39  brianp
 * removed gl_disable, fixed color_material bug
 *
 * Revision 1.3  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.2  1995/02/24  15:25:46  brianp
 * added error checks to gl_enable and gl_disable
 *
 * Revision 1.1  1995/02/24  14:21:54  brianp
 * Initial revision
 *
 */


#include <string.h>
#include "context.h"
#include "depth.h"
#include "dither.h"
#include "draw.h"
#include "list.h"
#include "macros.h"
#include "stencil.h"




/*
 * Perform glEnable and glDisable calls.
 */
void gl_enable( GLenum cap, GLboolean state )
{
   GLuint p;

   if (INSIDE_BEGIN_END) {
      if (state) {
	 gl_error( GL_INVALID_OPERATION, "glEnable" );
      }
      else {
	 gl_error( GL_INVALID_OPERATION, "glDisable" );
      }
      return;
   }

   switch (cap) {
      case GL_ALPHA_TEST:
         CC.Color.AlphaEnabled = state;
	 gl_update_rasterflags();
	 break;
      case GL_AUTO_NORMAL:
	 CC.Eval.AutoNormal = state;
	 break;
      case GL_BLEND:
         CC.Color.BlendEnabled = state;
	 gl_update_pixel_logic();
	 gl_update_rasterflags();
	 break;
      case GL_CLIP_PLANE0:
      case GL_CLIP_PLANE1:
      case GL_CLIP_PLANE2:
      case GL_CLIP_PLANE3:
      case GL_CLIP_PLANE4:
      case GL_CLIP_PLANE5:
	 CC.Transform.ClipEnabled[cap-GL_CLIP_PLANE0] = state;
	 /* Check if any clip planes enabled */
         CC.Transform.AnyClip = GL_FALSE;
         for (p=0;p<MAX_CLIP_PLANES;p++) {
            if (CC.Transform.ClipEnabled[p]) {
               CC.Transform.AnyClip = GL_TRUE;
               break;
            }
         }
	 break;
      case GL_COLOR_MATERIAL:
	 CC.Light.ColorMaterialEnabled = state;
	 if (state) {
	    gl_color( CC.Current.Color );
	 }
	 break;
      case GL_CULL_FACE:
         CC.Polygon.CullFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_DEPTH_TEST:
	 CC.Depth.Test = state;
	 gl_update_rasterflags();
	 if (state && !CC.DepthBuffer) {
	    /* need to allocate a depth buffer now */
	    gl_alloc_depth_buffer();
	 }
         break;
      case GL_DITHER:
	 CC.Color.DitherFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_FOG:
	 CC.Fog.Enabled = state;
	 gl_update_rasterflags();
	 break;
      case GL_LIGHT0:
      case GL_LIGHT1:
      case GL_LIGHT2:
      case GL_LIGHT3:
      case GL_LIGHT4:
      case GL_LIGHT5:
      case GL_LIGHT6:
      case GL_LIGHT7:
         CC.Light.Light[cap-GL_LIGHT0].Enabled = state;
         break;
      case GL_LIGHTING:
         CC.Light.Enabled = state;
         break;
      case GL_LINE_SMOOTH:
	 CC.Line.SmoothFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_LINE_STIPPLE:
	 CC.Line.StippleFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_LOGIC_OP:
	 CC.Color.LogicOpEnabled = state;
	 gl_update_pixel_logic();
	 gl_update_rasterflags();
	 break;
      case GL_MAP1_COLOR_4:
	 CC.Eval.Map1Color4 = state;
	 break;
      case GL_MAP1_INDEX:
	 CC.Eval.Map1Index = state;
	 break;
      case GL_MAP1_NORMAL:
	 CC.Eval.Map1Normal = state;
	 break;
      case GL_MAP1_TEXTURE_COORD_1:
	 CC.Eval.Map1TextureCoord1 = state;
	 break;
      case GL_MAP1_TEXTURE_COORD_2:
	 CC.Eval.Map1TextureCoord2 = state;
	 break;
      case GL_MAP1_TEXTURE_COORD_3:
	 CC.Eval.Map1TextureCoord3 = state;
	 break;
      case GL_MAP1_TEXTURE_COORD_4:
	 CC.Eval.Map1TextureCoord4 = state;
	 break;
      case GL_MAP1_VERTEX_3:
	 CC.Eval.Map1Vertex3 = state;
	 break;
      case GL_MAP1_VERTEX_4:
	 CC.Eval.Map1Vertex4 = state;
	 break;
      case GL_MAP2_COLOR_4:
	 CC.Eval.Map2Color4 = state;
	 break;
      case GL_MAP2_INDEX:
	 CC.Eval.Map2Index = state;
	 break;
      case GL_MAP2_NORMAL:
	 CC.Eval.Map2Normal = state;
	 break;
      case GL_MAP2_TEXTURE_COORD_1: 
	 CC.Eval.Map2TextureCoord1 = state;
	 break;
      case GL_MAP2_TEXTURE_COORD_2:
	 CC.Eval.Map2TextureCoord2 = state;
	 break;
      case GL_MAP2_TEXTURE_COORD_3:
	 CC.Eval.Map2TextureCoord3 = state;
	 break;
      case GL_MAP2_TEXTURE_COORD_4:
	 CC.Eval.Map2TextureCoord4 = state;
	 break;
      case GL_MAP2_VERTEX_3:
	 CC.Eval.Map2Vertex3 = state;
	 break;
      case GL_MAP2_VERTEX_4:
	 CC.Eval.Map2Vertex4 = state;
	 break;
      case GL_NORMALIZE:
	 CC.Transform.Normalize = state;
	 break;
      case GL_POINT_SMOOTH:
	 CC.Point.SmoothFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_POLYGON_SMOOTH:
	 CC.Polygon.SmoothFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_POLYGON_STIPPLE:
	 CC.Polygon.StippleFlag = state;
	 gl_update_rasterflags();
	 break;
      case GL_SCISSOR_TEST:
	 CC.Scissor.Enabled = state;
	 gl_update_rasterflags();
	 break;
      case GL_STENCIL_TEST:
	 CC.Stencil.Enabled = state;
	 gl_update_rasterflags();
	 if (state && !CC.StencilBuffer) {
	    /* need to allocate a stencil buffer now */
	    gl_alloc_stencil_buffer();
	 }
	 break;
      case GL_TEXTURE_1D:
	 if (state) {
	    CC.Texture.Enabled |= 1;
	 }
	 else {
	    CC.Texture.Enabled &= 2;
	 }
	 gl_update_rasterflags();
	 break;
      case GL_TEXTURE_2D:
	 if (state) {
	    CC.Texture.Enabled |= 2;
	 }
	 else {
	    CC.Texture.Enabled &= 1;
	 }
	 gl_update_rasterflags();
	 break;
      case GL_TEXTURE_GEN_Q:
	 CC.Texture.TexGenEnabledQ = state;
	 break;
      case GL_TEXTURE_GEN_R:
	 CC.Texture.TexGenEnabledR = state;
	 break;
      case GL_TEXTURE_GEN_S:
	 CC.Texture.TexGenEnabledS = state;
	 break;
      case GL_TEXTURE_GEN_T:
	 CC.Texture.TexGenEnabledT = state;
	 break;
      default:
	 if (state) {
	    gl_error( GL_INVALID_ENUM, "glEnable" );
	 }
	 else {
	    gl_error( GL_INVALID_ENUM, "glDisable" );
	 }
         break;
   }
   CC.PointFunc = NULL;
   CC.LineFunc = NULL;
   CC.PolygonFunc = NULL;
}




void glEnable( GLenum cap )
{
   if (CC.CompileFlag) {
      gl_save_enable( cap );
   }
   if (CC.ExecuteFlag) {
      gl_enable( cap, GL_TRUE );
   }
}



void glDisable( GLenum cap )
{
   if (CC.CompileFlag) {
      gl_save_disable( cap );
   }
   if (CC.ExecuteFlag) {
      gl_enable( cap, GL_FALSE );
   }
}



GLboolean glIsEnabled( GLenum cap )
{
   switch (cap) {
      case GL_ALPHA_TEST:
         return CC.Color.AlphaEnabled;
      case GL_AUTO_NORMAL:
	 return CC.Eval.AutoNormal;
      case GL_BLEND:
         return CC.Color.BlendEnabled;
      case GL_CLIP_PLANE0:
      case GL_CLIP_PLANE1:
      case GL_CLIP_PLANE2:
      case GL_CLIP_PLANE3:
      case GL_CLIP_PLANE4:
      case GL_CLIP_PLANE5:
	 return CC.Transform.ClipEnabled[cap-GL_CLIP_PLANE0];
      case GL_COLOR_MATERIAL:
	 return CC.Light.ColorMaterialEnabled;
      case GL_CULL_FACE:
         return CC.Polygon.CullFlag;
      case GL_DEPTH_TEST:
         return CC.Depth.Test;
      case GL_DITHER:
	 return CC.Color.DitherFlag;
      case GL_FOG:
	 return CC.Fog.Enabled;
      case GL_LIGHTING:
         return CC.Light.Enabled;
      case GL_LIGHT0:
      case GL_LIGHT1:
      case GL_LIGHT2:
      case GL_LIGHT3:
      case GL_LIGHT4:
      case GL_LIGHT5:
      case GL_LIGHT6:
      case GL_LIGHT7:
         return CC.Light.Light[cap-GL_LIGHT0].Enabled;
      case GL_LINE_SMOOTH:
	 return CC.Line.SmoothFlag;
      case GL_LINE_STIPPLE:
	 return CC.Line.StippleFlag;
      case GL_LOGIC_OP:
	 return CC.Color.LogicOpEnabled;
      case GL_MAP1_COLOR_4:
	 return CC.Eval.Map1Color4;
      case GL_MAP1_INDEX:
	 return CC.Eval.Map1Index;
      case GL_MAP1_NORMAL:
	 return CC.Eval.Map1Normal;
      case GL_MAP1_TEXTURE_COORD_1:
	 return CC.Eval.Map1TextureCoord1;
      case GL_MAP1_TEXTURE_COORD_2:
	 return CC.Eval.Map1TextureCoord2;
      case GL_MAP1_TEXTURE_COORD_3:
	 return CC.Eval.Map1TextureCoord3;
      case GL_MAP1_TEXTURE_COORD_4:
	 return CC.Eval.Map1TextureCoord4;
      case GL_MAP1_VERTEX_3:
	 return CC.Eval.Map1Vertex3;
      case GL_MAP1_VERTEX_4:
	 return CC.Eval.Map1Vertex4;
      case GL_MAP2_COLOR_4:
	 return CC.Eval.Map2Color4;
      case GL_MAP2_INDEX:
	 return CC.Eval.Map2Index;
      case GL_MAP2_NORMAL:
	 return CC.Eval.Map2Normal;
      case GL_MAP2_TEXTURE_COORD_1: 
	 return CC.Eval.Map2TextureCoord1;
      case GL_MAP2_TEXTURE_COORD_2:
	 return CC.Eval.Map2TextureCoord2;
      case GL_MAP2_TEXTURE_COORD_3:
	 return CC.Eval.Map2TextureCoord3;
      case GL_MAP2_TEXTURE_COORD_4:
	 return CC.Eval.Map2TextureCoord4;
      case GL_MAP2_VERTEX_3:
	 return CC.Eval.Map2Vertex3;
      case GL_MAP2_VERTEX_4:
	 return CC.Eval.Map2Vertex4;
      case GL_NORMALIZE:
	 return CC.Transform.Normalize;
      case GL_POINT_SMOOTH:
	 return CC.Point.SmoothFlag;
      case GL_POLYGON_SMOOTH:
	 return CC.Polygon.SmoothFlag;
      case GL_POLYGON_STIPPLE:
	 return CC.Polygon.StippleFlag;
      case GL_SCISSOR_TEST:
	 return CC.Scissor.Enabled;
      case GL_STENCIL_TEST:
	 return CC.Stencil.Enabled;
      case GL_TEXTURE_1D:
	 return (CC.Texture.Enabled & 1) ? GL_TRUE : GL_FALSE;
      case GL_TEXTURE_2D:
	 return (CC.Texture.Enabled & 2) ? GL_TRUE : GL_FALSE;
      case GL_TEXTURE_GEN_Q:
	 return CC.Texture.TexGenEnabledQ;
      case GL_TEXTURE_GEN_R:
	 return CC.Texture.TexGenEnabledR;
      case GL_TEXTURE_GEN_S:
	 return CC.Texture.TexGenEnabledS;
      case GL_TEXTURE_GEN_T:
	 return CC.Texture.TexGenEnabledT;
      default:
	 gl_error( GL_INVALID_ENUM, "glIsEnabled" );
	 return GL_FALSE;
   }
}

