/* dd.h */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: dd.h,v 1.13 1995/07/24 18:55:20 brianp Exp $

$Log: dd.h,v $
 * Revision 1.13  1995/07/24  18:55:20  brianp
 * added dd_finish()
 *
 * Revision 1.12  1995/06/12  15:38:23  brianp
 * changed color arrays to GLubyte
 *
 * Revision 1.11  1995/05/22  20:59:34  brianp
 * Release 1.2
 *
 * Revision 1.10  1995/04/12  15:37:05  brianp
 * removed dd_draw_pixel(), dd_draw_line(), and dd_draw_polygon()
 *
 * Revision 1.9  1995/04/11  14:04:57  brianp
 * introduced DD struct of function pointers
 *
 * Revision 1.8  1995/03/30  21:07:49  brianp
 * updated to use pointers to CC.write_* functions
 *
 * Revision 1.7  1995/03/22  21:36:53  brianp
 * removed mode from dd_buffer_info()
 *
 * Revision 1.6  1995/03/08  15:10:02  brianp
 * added support for dd_logicop
 * added dd_clear_index and dd_clear_color
 *
 * Revision 1.5  1995/03/07  19:01:39  brianp
 * added dd_read_index_pixels() and dd_read_color_pixels()
 *
 * Revision 1.4  1995/03/07  14:20:41  brianp
 * updated for new XSetForeground/GC scheme
 *
 * Revision 1.3  1995/03/04  19:25:29  brianp
 * 1.1 beta revision
 *
 * Revision 1.2  1995/02/27  22:48:36  brianp
 * modified for PB
 *
 * Revision 1.1  1995/02/24  14:20:25  brianp
 * Initial revision
 *
 */


/*
 * Device driver function prototypes.
 *
 * Each of these functions must be implemented in the system-specific
 * interface (XMesa, AMesa, etc).
 */


#ifndef DD_INCLUDED
#define DD_INCLUDED


#include "GL/gl.h"



/*
 * The following structure contains function pointers which must be
 * initialized by the device driver.
 *
 * These functions are called by Mesa to read and write pixels to and
 * from the frame buffer.  The reason to use function pointers is that
 * the device driver might want to use different implementations of
 * these functions depending on the "mode" of the frame buffer.  For
 * example, in the X/Mesa driver, our frame buffer can either be a Window,
 * Pixmap, or XImage.  Furthermore, we have special cases for certain
 * visual types such as 24-bit TrueColor and 8-bit PseudoColor.  We
 * just point these functions at the appropriate device driver and off
 * we go!  It's also faster than using if/thens or switches inside each
 * function.
 *
 * It's up to the device driver to initialize the pointers whenever the
 * dd_draw_buffer(), dd_read_buffer() or "context switch" functions are
 * called.
 *
 * Look at xmesa.c to see how these function pointers are initialized
 * depending on the "flavor" of output buffer.
 * 
 * Here's a quick description of each function pointer:
 *
 * draw_point - render a single pixel
 * draw_line - render a simple line segment
 * draw_polygon - render a simple polygon w/out depth buffering, etc.
 *
 * write_color_span - write a horizontal run of RGBA pixels
 * write_monocolor_span - write a horizontal run of mono-RGBA pixels
 * write_color_pixels - write a random array of RGBA pixels
 * write_monocolor_pixels - write a random array of mono-RGBA pixels
 *
 * write_index_span - write a horizontal run of CI pixels
 * write_nonoindex_span - write a horizontal run of mono-CI pixels
 * write_index_pixels - write a random array of CI pixels
 * write_monoindex_pixels - write a random array of mono-CI pixels
 *
 * read_index_span - read a horizontal run of color index pixels
 * read_color_span - read a horizontal run of RGBA pixels
 * read_index_pixels - read a random array of CI pixels
 * read_color_pixels - read a random array of RGBA pixels
 *
 * NOTES:
 *   RGBA = red/green/blue/alpha
 *   CI = color index (color mapped mode)
 *   mono = all pixels have the same color or index
 *
 * IN ALL CASES:
 *      X coordinates start at 0 at the left and increase to the right
 *      Y coordinates start at 0 at the bottom and increase upward
 */



struct dd_function_table {
   /*
    * Functions for simple points, line segments and polygons:
    */
   void (*draw_pixel)( GLint x, GLint y );
   void (*draw_line)( GLint x0, GLint y0, GLint x1, GLint y1 );
   void (*draw_polygon)( GLuint n, GLint x[], GLint y[] );

   /*
    * Functions for writing pixels to the frame buffer:
    */
   void (*write_color_span)( GLuint n, GLint x, GLint y,
			     const GLubyte red[], const GLubyte green[],
			     const GLubyte blue[], const GLubyte alpha[],
			     const GLubyte mask[] );
   void (*write_monocolor_span)( GLuint n, GLint x, GLint y,
				 const GLubyte mask[] );
   void (*write_color_pixels)( GLuint n, const GLint x[], const GLint y[],
			       const GLubyte red[], const GLubyte green[],
			       const GLubyte blue[], const GLubyte alpha[],
			       const GLubyte mask[] );
   void (*write_monocolor_pixels)( GLuint n, const GLint x[], const GLint y[],
				   const GLubyte mask[] );
   void (*write_index_span)( GLuint n, GLint x, GLint y, const GLuint index[],
			     const GLubyte mask[] );
   void (*write_monoindex_span)( GLuint n, GLint x, GLint y,
				 const GLubyte mask[] );
   void (*write_index_pixels)( GLuint n, const GLint x[], const GLint y[],
			       const GLuint index[], const GLubyte mask[] );
   void (*write_monoindex_pixels)( GLuint n, const GLint x[], const GLint y[],
				   const GLubyte mask[] );

   /*
    * Functions to read pixels from frame buffer:
    */
   void (*read_index_span)( GLuint n, GLint x, GLint y, GLuint index[] );
   void (*read_color_span)( GLuint n, GLint x, GLint y,
			    GLubyte red[], GLubyte green[],
			    GLubyte blue[], GLubyte alpha[] );
   void (*read_index_pixels)( GLuint n, const GLint x[], const GLint y[],
			      GLuint indx[] );
   void (*read_color_pixels)( GLuint n, const GLint x[], const GLint y[],
			      GLubyte red[], GLubyte green[],
			      GLubyte blue[], GLubyte alpha[] );

};



extern struct dd_function_table DD;




/**********************************************************************/
/*****                      Miscellaneous                         *****/
/**********************************************************************/



/* Implements glFinish() */
extern void dd_finish( void );


/* Implements glFlush() */
extern void dd_flush( void );


/* Return characteristics of the output buffer. */
extern void dd_buffer_info( GLuint *width, GLuint *height, GLuint *depth );


/*
 * Specify which buffer to read from.
 * Input:  mode - as passed to glReadBuffer
 * Return:  new mode or GL_FALSE if error.
 */
extern GLenum dd_read_buffer( GLenum mode );


/*
 * Specify which buffer(s) to draw into.
 * Input:  mode - as passed to glDrawBuffer
 * Return:  new output mode or GL_FALSE if error.
 */
extern GLenum dd_draw_buffer( GLenum mode );



/*
 * Set the color index used to clear the color buffer.
 * This implements glClearIndex().
 */
extern void dd_clear_index( GLuint index );



/*
 * Set the color used to clear the color buffer.
 * This implements glClearColor().
 */
extern void dd_clear_color( const GLfloat color[4] );



/*
 * Clear the color buffer using the clear color or index as specified
 * by one of the two functions above.
 */
extern void dd_clear( GLboolean all,
		      GLint x, GLint y, GLint width, GLint height );



/*
 * Set the pixel logic operation.  Return GL_TRUE if the device driver
 * can perform the operation, otherwise return GL_FALSE.  If GL_FALSE
 * is returned, the logic op will be done in software by Mesa.
 */
extern GLboolean dd_logicop( GLenum op );



/**********************************************************************/
/*****                Simple rendering functions                  *****/
/**********************************************************************/


/* Set the current color index. */
extern void dd_index( GLuint index );


/* Set the index mode bitplane mask. */
extern void dd_index_mask( GLuint mask );


/* Set the current RGBA color. */
extern void dd_color( const GLfloat color[4] );


/* Set the RGBA drawing mask. */
extern void dd_color_mask( GLboolean rmask, GLboolean gmask,
			   GLboolean bmask, GLboolean amask );




/**********************************************************************/
/*****              Color buffer reading functions                *****/
/**********************************************************************/

/**** OBSOLETE!!! ****/


/* Read a horizontal span of color-index pixels. */
extern void dd_read_index_span( GLuint n, GLint x, GLint y, GLuint index[] );


/* Read a horizontal span of color pixels. */
extern void dd_read_color_span( GLuint n, GLint x, GLint y,
			        GLubyte red[], GLubyte green[],
			        GLubyte blue[], GLubyte alpha[] );


/* Read an array of color index pixels. */
extern void dd_read_index_pixels( GLuint n,
				  const GLint x[], const GLint y[],
				  GLuint indx[] );


/* Read an array of color pixels. */
extern void dd_read_color_pixels( GLuint n, const GLint x[], const GLint y[],
				  GLubyte red[], GLubyte green[],
				  GLubyte blue[], GLubyte alpha[] );



#endif

