/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoObject/RCS/Pioneer4200Driver.c,v 0.27 92/09/28 12:39:53 drapeau Exp $ */
/* $Log:	Pioneer4200Driver.c,v $
 * Revision 0.27  92/09/28  12:39:53  drapeau
 * PlayFromTo() was modified to reflect the new semantics of the VideoObject
 * library.
 * 
 * Revision 0.26  92/07/30  15:28:41  drapeau
 * Several changes:
 * * Removed non-used functions.
 * * Re-formatted all function declarations to conform to ANSI function
 *   prototype standards.
 * * Replaced hard-coded references to 30 frame-per-second frame rates with
 *   new definition "FrameRate".  All math is based on this definition now,
 *   allowing the driver to be used in places where the frame rate is not
 *   30 fps.
 * * Improved diagnostic messages.  Diagnostics now report the serial port
 *   being used for the command when possible.
 * 
 * Revision 0.25  92/01/03  16:59:08  drapeau
 * Modified Pioneer4200PlayFromTo() to fix an error in playback.  Before,
 * the function always uses Pioneer4200PlayatSpeedDir, which sends a
 * different opcode to the Pioneer4200 than Pioneer4200Play.  The
 * opcode for PlayAtSpeedDir always mutes the audio, even if the
 * speed is the normal 30 frames per second.  The function now checks for
 * the device speed; if the speed is 30 fps, then the normal Pioneer4200Play
 * is used; otherwise, Pioneer4200PlayAtSpeedDir is used.
 * 
 * Revision 0.24  91/09/30  17:06:27  lim
 * Implemented Pioneer4200Ping.
 * 
 * Revision 0.23  91/08/24  17:50:02  lim
 * Implemented PrintDiagnostics.
 * 
 * Revision 0.22  91/08/24  13:38:06  lim
 * 1. Updated to use status codes in new PlayerStatus.h
 * 2. Clear Marker() removed as part of video object.
 * 
 * Revision 0.21  91/08/20  16:29:04  lim
 * Parameters for SetVideo : FeatureOff - Video mute
 * 			     FeatureOn - Video on
 * 
 * Revision 0.20  91/08/17  20:40:28  lim
 * 1. Pause After Search is now implemented.
 * 
 * Revision 0.19  91/08/09  10:05:06  lim
 * Fixed error regarding ErrorDecode().
 * 
 * 
 * Revision 0.18  91/08/08  17:16:38  lim
 *  1. PlayAtSpeedDir() - try to input device-dependent speed rather than an arbitrary speed.
 *    ie use CalcSpeed before calling PlayAtSpeedDir()
 * 2. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *     normal play.
 * 3. QueryMedium() - one parameter added, 'result'. Used to return non-integer results.
 * 
 * Revision 0.17  91/08/07  13:06:19  lim
 * 1. Made as part of VideoLib.a. Removed references to vEdit application.
 * 2. Added instance pointer, "theObject" to all public functions. 
 * 3. ErrorDecode() is now a private function. It calls DisplayError() 
 * (in 'videoObjects.c') to display the error message. 
 * 
 * Revision 0.16  91/08/02  12:51:38  lim
 * 1. SetDefaults function changed to take in 4 parameters"
 * audio, addressingMode, addressDisplayOnOff, and displayMode
 * 2. xv_set is no longer called within driver code, but from
 * vEdit application.
 * 3. Pause is removed.
 * 
 * Revision 0.15  91/07/29  22:25:55  lim
 * Updated all player statuses to conform with specs.
 * 
 * Revision 0.14  91/07/27  22:27:45  lim
 * Changed speed to int from double.
 * 
 * Revision 0.13  91/07/23  15:51:55  lim
 * 1. Set slider maximum to 120, minimum to -120. This is done because
 * all devices will have to set their own slider max and min to match the
 * max number of frames that they can play per second.
 * 2. CalcSpeed and PlayAtSpeedDir has been changed to match change in
 * slider values. It used to be that the slider value was twice the max
 * number of frames so that 1.5 and 2.5 f/s speeds can be accommodated.
 * 
 * Revision 0.12  91/07/19  13:54:18  lim
 * Fixed error : Eject is implemented with 2 "RJ"
 * Changed Pioneer4200PlayFromTo :
 * 1. Speed field is added.
 * 2. NULL in from field means resume play, ie no search.
 * 3. PlayAtSpeedDir used rather than Play.
 * 4. Speed field changed from float to double.
 * 
 * Revision 0.11  91/07/17  22:49:03  lim
 * Now SetDefaults also sets the Player in 'ready' state.
 * 
 * Revision 0.10  91/07/12  16:16:10  lim
 * Initial revision implementing VideoObject.
 *  */

#include "Pioneer4200Driver.h"

static char pioneer4200Rcsid[] = "$Header: /Source/Media/collab/VideoObject/RCS/Pioneer4200Driver.c,v 0.27 92/09/28 12:39:53 drapeau Exp $";

static char	diagMsg[64];


/* Sets up default values for video options */
int
  Pioneer4200SetDefaults(VideoObject*	theObject,
			 int		audio,
			 int		addressingMode,
			 int		addressDisplayOnOff,
			 int		displayMode)
{  
  int status;
  
  status = Pioneer4200QueryStatus(theObject);
  Pioneer4200SetAudio(theObject, audio);			    /* set audio */
  if ((status != PlayerDoorOpen) && (status != PlayerPark))
    Pioneer4200SetAddMode(theObject, addressingMode);
  Pioneer4200SetAddressDisplay(theObject, addressDisplayOnOff,
			       displayMode);			    /* set address display */
  Pioneer4200Play(theObject);					    /* puts player in 'ready' state */
  Pioneer4200Still(theObject);
  return status;
}								    /* end function Pioneer4200SetDefaults */


int
  Pioneer4200SendCode(VideoObject*	theObject,
		      char*		code)
{
  char	in[20];
  char	out[20];
  char	errorMsg[50];
  int	nr;
  int	err;
  int	num;
  
  strcpy(in,code);
  strcat(in,"\015");
  write(theObject->DevConfig->fd,in,strlen(in));
  sprintf(diagMsg, "%s :\tSending %s.\n",
	  theObject->DevConfig->serialPort,
	  code);
  PrintDiagnostics(diagMsg);
  nr = read(theObject->DevConfig->fd,out,20);
  out[nr-2] = '\0';
  sprintf(diagMsg, "%s :\tReading %s.\n",
	  theObject->DevConfig->serialPort,
	  out);
  PrintDiagnostics(diagMsg);
  if (out[0] == 'R')
    return PlayerOk;
  else if (out[0] == 'E') 
  {
    sscanf(&out[1],"%d",&err);
    switch (err) 
    {
     case 0:
      err = Pioneer4200CommunicationError;
     case 4:
      err = Pioneer4200FeatureNotAvailable;
     case 6:
      err = Pioneer4200MissingArgument;
     case 11:
      err = Pioneer4200NoDisc;
     case 12:
      err = Pioneer4200SearchError;
     case 13:
      err = Pioneer4200DefocussingError;
     case 15:
      err = Pioneer4200PictureStop;
     case 99:
      err = Pioneer4200PanicError;
     default:
      err = PlayerUnknownError;
    }								    /* end of switch (err) */
    Pioneer4200ErrorDecode(theObject, -err, errorMsg);
    return err;
  }								    /* end of else if (out[0] == 'E') */
  else if (isdigit(out[0])) 
  {
    sscanf(out,"%d",&num);
    return num;
  }
  else if (out[0] == 'P') 
  {
    sscanf(&out[1],"%d",&num);
    if (strlen(&out[1]) == 2) 
    {
      switch (num) 
      {
       case 0:
	return PlayerDoorOpen;
       case 1:
	return PlayerPark;
       case 4:
	return PlayerForwardPlay;
       case 5:
	return PlayerStill;
       case 6:
	return PlayerPause;
       case 9:
	return PlayerMultiSpeedPlay;
       default:
	return PlayerUnknownReturnCode;
      }		 						    /* end of switch (num) */
    }                                                               /* end of if (strlen(&out[1]) == 2) */
    else 
      return num;
  }								    /* end of else if (out[0] == 'P') */
  else if (out[0] == 'C') 
  {
    sscanf(&out[2],"%d",&num);
    return num;
  }
  else 
    return PlayerUnknownReturnCode;
}								    /* end function Pioneer4200SendCode */



/*  Player functions  */

int
  Pioneer4200Stop(VideoObject* theObject)
{
  return(Pioneer4200SendCode(theObject, "PA"));			    /* Implemented as a pause for maintaining 'ready' state */
}								    /* end function Pioneer4200Stop */


int 
  Pioneer4200Eject(VideoObject* theObject)
{
  Pioneer4200SendCode(theObject, "RJ");
  return(Pioneer4200SendCode(theObject, "RJ"));
}								    /* end function Pioneer4200Eject */



int
  Pioneer4200Start(VideoObject* theObject)
{
  return(Pioneer4200SendCode(theObject, "SA"));
}								    /* end function Pioneer4200Start */


int
  Pioneer4200Pause(VideoObject* theObject)
{
  return(Pioneer4200SendCode(theObject, "PA"));
}								    /* end function Pioneer4200Pause */



int
  Pioneer4200Play(VideoObject* theObject)
{
  return(Pioneer4200SendCode(theObject, "PL"));
}								    /* end function Pioneer4200Play */


int
  Pioneer4200PlayFromTo(VideoObject*	theObject,
			int		from,
			int		to,
			int		speed)
{
  char	fr[20];
  int	status;
  int	deviceSpeed;
  
  if ((from == to) || ((from > 0) && (to == 0)))		    /* Case a & b: search to "from" and still (non-blocking... */
  {								    /* ...search doesn't apply here since search time is quick) */
    return(Pioneer4200Search(theObject, from));			    /* So both types of searches fall into the same search code */
  }
  if ((from == 0) && (to > 0))					    /* Case c: play from current position until address... */
  {								    /* ..."to" has been reached. */
    Pioneer4200SetStop(theObject, to);				    /* This code makes the player return control ...*/
    deviceSpeed = Pioneer4200CalcSpeed(theObject, speed, 0);	    /* ... to the program once play has been started, ... */
    if (deviceSpeed == FrameRate)				    /* ... instead of waiting until play has ended ...*/
      return(Pioneer4200Play(theObject));
    else
      return(Pioneer4200PlayAtSpeedDir(theObject, deviceSpeed, Forward));
  }
  if ((from > 0) && (to > 0) && (from < to))			    /* Case d: play from "from" to "to" */
  {
    Pioneer4200Search(theObject, from);
    Pioneer4200SetStop(theObject, to);
    deviceSpeed = Pioneer4200CalcSpeed(theObject, speed, 0);
    if (deviceSpeed == FrameRate)
      return(Pioneer4200Play(theObject));
    else
      return(Pioneer4200PlayAtSpeedDir(theObject, deviceSpeed, Forward));
  }
  return(PlayerReturnError);					    /* If this code is reached, incorrect from & to were requested */
}								    /* end function Pioneer4200PlayFromTo */


int
  Pioneer4200Still(VideoObject* theObject)
{
  return(Pioneer4200SendCode(theObject, "ST"));    
}								    /* end function Pioneer4200Still */



int
  Pioneer4200Step(VideoObject*		theObject,
		  enum Direction	dir)
{
  int i;
  
  for (i=0; i < StepRate; ++i)
    if (dir == Forward)
      Pioneer4200SendCode(theObject, "SF");
    else
      Pioneer4200SendCode(theObject, "SR");
  if (dir == Forward)
    return(Pioneer4200SendCode(theObject, "SF"));
  else
    return(Pioneer4200SendCode(theObject, "SR"));
}								    /* end function Pioneer4200Step */



int
  Pioneer4200FastForward(VideoObject* theObject)
{
  int i;
  int nm1;
  
  nm1 = ScanRate - 1;
  for (i=0; i<nm1; ++i)
    Pioneer4200SendCode(theObject, "NF");
  return(Pioneer4200SendCode(theObject, "NF"));
}								    /* end function Pioneer4200FastForward */



int
  Pioneer4200Reverse(VideoObject* theObject)
{
  int i; 
  int nm1;
  
  nm1 = ScanRate - 1;
  for (i=0; i<nm1; ++i)
    Pioneer4200SendCode(theObject, "NR");
  return(Pioneer4200SendCode(theObject, "NR"));
}								    /* end function Pioneer4200Reverse */


int
  Pioneer4200PlayAtSpeedDir(VideoObject*	theObject,
			    int			framesPerSecond,
			    enum Direction	direction)
{
  char		fr[16];
  int		speed;
  static int	lastSpeed = 0;
  
  speed = framesPerSecond;
  
  if (speed < (FrameRate / 6))
    speed = FrameRate / 10;
  else if (speed < (FrameRate / 4))
    speed = FrameRate / 6;
  else if (speed < (FrameRate / 3))
    speed = FrameRate / 4;
  else if (speed < (FrameRate / 2))
    speed = FrameRate / 3;
  else if (speed < FrameRate)
    speed = FrameRate / 2;
  else if (speed < (FrameRate * 2))
    speed = FrameRate;
  else if (speed < (FrameRate * 3))
    speed = FrameRate * 2;
  else if (speed < (FrameRate * 4))
    speed = FrameRate * 3;
  else
    speed = FrameRate * 4;
  
  speed = speed * 2;						    /* Pioneer4200 multiplies speed values by 2 (i.e., 60 = 1x) */
  if (speed != lastSpeed)					    /* Is it really necessary to send a new command? */
  {								    /* Yes, prepare the command and send it to the player */
    lastSpeed = speed;						    /* Update last speed played */
    sprintf(diagMsg, "%s :\tSpeed is %d frames per second.\n",
	    theObject->DevConfig->serialPort,
	    speed);
    PrintDiagnostics(diagMsg);
    if (direction == Forward)
      sprintf(fr,"%dSPMF",speed);
    else
      sprintf(fr,"%dSPMR",speed);
    
    return(Pioneer4200SendCode(theObject, fr));
  }
  else
    return(PlayerOk);
}								    /* end function Pioneer4200PlayAtSpeedDir */



/* Returns number of frames per second. */
int
  Pioneer4200CalcSpeed(VideoObject*	theObject,
		       int		value,
		       int		playMode)		    /* unused */
{
  int speed;
  
  if (value < 0)
    value = -value;
  if (value < (FrameRate / 6))					    /* 1/10th speed */
    speed = FrameRate / 10;
  else if (value < (FrameRate / 4))				    /* 1/6th speed */
    speed = FrameRate / 6;
  else if (value < (FrameRate / 3))				    /* 1/4th speed */
    speed = FrameRate / 4;
  else if (value < (FrameRate / 2))				    /* 1/3rd speed */
    speed = FrameRate / 3;
  else if (value < FrameRate)					    /* 1/2 speed */
    speed = FrameRate / 2;
  else if (value < (FrameRate * 2))				    /* 1x speed */
    speed = FrameRate;
  else if (value < (FrameRate * 3))				    /* 2x speed */
    speed = FrameRate * 2;
  else if (value < (FrameRate * 4))				    /* 3x speed */
    speed = FrameRate * 3;
  else
    speed = FrameRate * 4;					    /* 4x speed */
  
  return (speed);
}								    /* end function Pioneer4200CalcSpeed */


int 
  Pioneer4200Power(VideoObject*	theObject,
		   int		value)
{
  /* NOT IMPLEMENTED */
  return;
}								    /* end function Pioneer4200Power */


int
  Pioneer4200Search(VideoObject*	theObject,
		    int			to)
{
  char fr[16];
  
  sprintf(fr,"%dSE",to);
  return(Pioneer4200SendCode(theObject, fr));
}								    /* end function Pioneer4200Search */


int
  Pioneer4200SetStop(VideoObject*	theObject,
		     int		stopM)
{
  char fr[16];
  
  sprintf(fr,"%dSM",stopM);
  return(Pioneer4200SendCode(theObject, fr));
}								    /* end function Pioneer4200SetStop */


int
  Pioneer4200SetAddressDisplay(VideoObject*	theObject,
			       int		on,
			       int		mode)
{
  switch (mode)
  {
   case DisplayFrame:
    Pioneer4200SendCode(theObject, "1RA"); 
    break;
   case DisplayChapter:
    Pioneer4200SendCode(theObject, "2RA"); 
    break;
   case DisplayFrameAndChapter:    
    Pioneer4200SendCode(theObject, "3RA"); 
    break;
  }
  
  if (on == FeatureOn) 
    return(Pioneer4200SendCode(theObject, "1DS"));
  else 
    return(Pioneer4200SendCode(theObject, "0DS"));
}								    /* end function Pioneer4200SetAddressDisplay */



int
  Pioneer4200SetAudio(VideoObject*	theObject,
		      int		ac)
{
  switch (ac)
  {
   case PlayerAudioMute:
    return(Pioneer4200SendCode(theObject, "0AD"));
    break;
   case PlayerAudioLeft:
    return(Pioneer4200SendCode(theObject, "1AD"));
    break;
   case PlayerAudioRight:
    return(Pioneer4200SendCode(theObject, "2AD"));
    break;
   case PlayerAudioStereo:
    return(Pioneer4200SendCode(theObject, "3AD"));
    break;
  }
  return(PlayerOk);
}								    /* end function Pioneer4200SetAudio */


int
  Pioneer4200SetVideo(VideoObject*	theObject,
		      int		vc)
{
  if (vc == FeatureOn)
    return(Pioneer4200SendCode(theObject, "1VD"));
  else
    return(Pioneer4200SendCode(theObject, "0VD"));
}								    /* end function Pioneer4200SetVideo */


int
  Pioneer4200SetAddMode(VideoObject*	theObject,
			int		mode)
{
  switch (mode) 
  {
   case PlayerFrameMode:
    return(Pioneer4200SendCode(theObject, "FR"));
   case PlayerTimeMode:
    return(Pioneer4200SendCode(theObject, "TM"));
   case PlayerChapterMode:
    return(Pioneer4200SendCode(theObject, "CH"));
   default:
    return PlayerUnknownCommand;
  }
}								    /* end function Pioneer4200SetAddMode */


int
  Pioneer4200QueryFrame(VideoObject* theObject)
{  
  return(Pioneer4200SendCode(theObject, "?F"));
}								    /* end function Pioneer4200QueryFrame */



int
  Pioneer4200QueryChapter(VideoObject* theObject)
{  
  return(Pioneer4200SendCode(theObject, "?C"));
}								    /* end function Pioneer4200QueryChapter */


int
  Pioneer4200QueryAudio(VideoObject* theObject)
{  
  /* NOT IMPLEMENTED */
  return PlayerOk;
}								    /* end function Pioneer4200QueryAudio */


int
  Pioneer4200QueryVideo(VideoObject* theObject)
{  
  /* NOT IMPLEMENTED */
  return PlayerOk;
}								    /* end function Pioneer4200QueryVideo */



/* Queries disc status */
int
  Pioneer4200QueryMedium(VideoObject*	theObject,
			 char*		result)			    /* unused */
{  
  return(Pioneer4200SendCode(theObject, "?D"));
}								    /* end function Pioneer4200QueryMedium */


int
  Pioneer4200QueryStatus(VideoObject* theObject)
{  
  return(Pioneer4200SendCode(theObject, "?P"));
}								    /* end function Pioneer4200QueryStatus */


int 
  Pioneer4200Ping(VideoObject* theObject)
{
  int nr;
  char out[5];
  
  write(theObject->DevConfig->fd, "?D\r", 3);			    /* Query disk status command */
  sprintf(diagMsg, "%s :\tSent ping.\n",
	  theObject->DevConfig->serialPort);
  PrintDiagnostics(diagMsg);
  nr = read(theObject->DevConfig->fd, out, 1);
  if (nr)
  {
    sprintf(diagMsg, "%s :\tSuccessful ping.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  
  else 
  {
    sprintf(diagMsg, "%s :\tPing unsuccessful.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  
  return nr;
}								    /* end function Pioneer4200Ping */


void 
  Pioneer4200ErrorDecode(VideoObject*	theObject,
			 int		error,
			 char*		errorMsg)
{
  switch (error) 
  {
   case PlayerCantInitDevice:
    strcpy(errorMsg,"Cannot initialize device properly");
    break;
   case PlayerInvalidBaudRate:
    strcpy(errorMsg,"Baud rate is invalid");
    break;
   case Pioneer4200CommunicationError:
    strcpy(errorMsg,"There has been a communication error");
    break;
   case Pioneer4200FeatureNotAvailable:
    strcpy(errorMsg,"Feature/command not available in stop mode");
    break;
   case Pioneer4200MissingArgument:
    strcpy(errorMsg,"An argument is missing");
    break;
   case Pioneer4200NoDisc:
    strcpy(errorMsg,"Please insert a videodisc first");
    break;
   case Pioneer4200SearchError:
    strcpy(errorMsg,"Search error");
    break;
   case Pioneer4200DefocussingError:
    strcpy(errorMsg,"Defocussing error");
    break;
   case Pioneer4200PictureStop:
    strcpy(errorMsg,"Picture Stop");
    break;
   case Pioneer4200PanicError:
    strcpy(errorMsg,"CATASTROPHIC ERROR");
    break;
   case PlayerUnknownError:
    strcpy(errorMsg,"Cannot understand error code");
    break;
   case PlayerUnknownReturnCode:
    strcpy(errorMsg,"Cannot understand return code");
    break;
   case PlayerUnknownCommand:
    strcpy(errorMsg,"An unknown command has been issued");
   default:
    strcpy(errorMsg, "Unknown error code");
    break;
  }
  DisplayError(errorMsg, " ");
}								    /* end function Pioneer4200ErrorDecode */
