/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/DTR/RCS/dtr.c,v 1.12 92/05/29 12:42:01 drapeau Exp $ */
/* $Log:	dtr.c,v $
 * Revision 1.12  92/05/29  12:42:01  drapeau
 * Modified code to track new name of the MAEstro "Selection" structure;
 * it is now named "MAESelection".
 * 
 * Revision 1.11  92/03/02  12:27:17  drapeau
 * Modified calls to SetCurrentSelection() to pass an extra parameter:
 * a pointer to a MAEstro Selection structure.  This was done to
 * accommodate calls to SetCurrentSelection from the Receiver
 * method SetSelection().  Before these changes, SetSelection() (and
 * SetCurrentSelection()) did not handle requests for partial selections.
 * 
 * Revision 1.10  92/01/09  12:41:10  drapeau
 * Slight modifications to the code to make it ANSI-compliant.
 * 
 * Revision 1.0  92/01/07  15:07:42  drapeau
 * *Modified main(), so that in now sets default locations for the DTR
 *  main and edit list panels.
 * *Modified QuitDtr() so that it now checks for unsaved audio files
 *  and unsaved edit lists.
 * * Removed EditingHandler() since it is no longer needed.  The text
 *  field that used to show the name of the current document has been
 *  removed.
 * *Removed EditModeSettingNotifyHandler(), since the Edit Mode choice
 *  has been replaced with a menu item.
 * *Removed EditListPopUpDoneButtonHandler() since it is no longer needed.
 * *Removed NewEditListTextFieldHandler() since it is no longer needed.
 * Also, Made a number of cosmetic changes to make code easier to read and
 * to conform to programming specifications.
 * 
 * Revision 0.34  91/10/09  17:21:42  derek
 * Save Edit List bug fixed.
 * 
 * Revision 0.33  91/09/30  16:13:07  derek
 * DTR will now disconnect from the Port Manager when it quits.  Also,
 * a number of printf statements are removed.
 * 
 * Revision 0.32  91/09/24  16:49:43  derek
 * Zoom level's max value is changed from 50 to 4.
 * 
 * Revision 0.31  91/09/18  22:47:18  derek
 * The following things are done:
 * 1.	The Makefile is changed corresponding to the changes in collab/.
 * 2.	Copyright messages included.
 * 3.	Some minor bugs fixed.
 * 
 * Revision 0.30  91/08/28  13:02:59  derek
 * The following changes are made:
 * 1.	I ahve written an if-statement checking if NewSender() returns
 * 	NULL.  If so, DTR will notice_prompt the user that the
 * 	PortManager is absent.
 * 2.	"Illegal Zoom value" bug in ZoomLevelTextField is fixed.
 * 
 * Revision 0.29  91/08/27  18:06:01  derek
 * The SizeToFit bug is fixed.
 * 
 * Revision 0.28  91/08/27  17:22:40  derek
 * The zooming bug is fixed in this version.
 * 
 * Revision 0.27  91/08/21  17:47:00  derek
 * The big canvas "mis-placed" bug is fixed.
 * 
 * Revision 0.26  91/08/21  11:34:11  derek
 * The following changes are made:
 * 1.	Now the duration and size of the recorded sound will be displayed
 * 	during recording.
 * 2.	I have changed GetSelection() corresponding to the request of Tek joo
 * 3.	Info Panel is added to the application.
 * 4.	Fixed SizeToFitHandler() so that when no file or buffer is currently
 * 	loaded, it would not do anything (except giving a warning
 * 	notice_prompt).
 * 5.	Inplemented the `Close' Menu option in the document menu.
 * 6.	Fixed the bug in which after ClearAll and I press PreviewEdit,
 * 	the edit wont be played.
 * 7.	I have made the changes corresponding to the change in OpenPanel's
 * 	name.  (from OpenPanel to Browse).
 * 8.	Incorporated BrowseCheck to check command line arg.
 * 9.	Changed most EditingStatusMessages to NoticePrompts.
 * 10.	SoundFileSaveAsPopUp and EditListSaveAsPopUp are removed 
 * 	from the application.
 * 
 * Revision 0.25  91/08/16  18:10:31  derek
 * 
 * The following things are done:
 * 1.	I have fixed an openpanel bug in my code in which I 
 * 	made the app return the wrong values to OpenHandler and
 * 	the SaveHandler.
 * 2.	The flashing color of the play button has been changed from
 * 	Red to Green.
 * 3.	Fixed a quantization bug: Buffer.play.end, when converted
 * 	from endingTimeInSec, should not exceed Buffer_hdr_data_size - 1.
 * 
 * Revision 0.24  91/08/14  18:16:42  derek
 * The following things are done:
 * 1.	I changed the name of the "save as" and "save" menu options
 * 	to be more specific.  ie. to "save sound file" and "save edit list".
 * 2.	I removed the preview panel popup and replaced it with a preview
 * 	edit button.
 * 3.	I changed the message displayed by UpdateSpaceOnTmp() to consist
 * 	only of the time remaining and the amount of space left.  There is
 * 	no more percentage indicating the capacity of /usr/tmp.
 * 
 * Revision 0.23  91/08/14  16:13:59  derek
 * Fixed a few saving/appending bugs.
 * 
 * Revision 0.22  91/08/13  20:38:23  derek
 * The buttons (play, pause, record) will now flash after they are pressed.
 * This only applies to times when audio files (not edit-lists) are 
 * played.
 * 
 * Revision 0.21  91/08/08  21:44:31  derek
 * Fixed a number of bugs.
 * 
 * Revision 0.20  91/08/08  11:02:04  derek
 * This is a cleaner version.  I have removed lots of printf/fprintf 
 * statements from it, and have also cleaned up the code using xsaber.
 * This version should run substantially faster.
 * 
 * Revision 0.19  91/08/07  16:24:04  derek
 * The Edit list part of DTR is done.  OpenPanel is also incorporated.
 * 
 * Revision 0.18  91/08/06  12:41:09  derek
 * Edit list panel is done.  Still need to link it to the network code.
 * 
 * Revision 0.17  91/07/30  11:45:31  derek
 * I have fixed the tmp file conflict bug.
 * 
 * Revision 0.16  91/07/27  11:45:47  derek
 * I have added audio output options to the application.  User can choose
 * either headphone or speaker as output device.
 * 
 * Revision 0.15  91/07/26  13:18:17  derek
 * Some saving bugs fixed.
 * 
 * Revision 0.14  91/07/23  21:21:31  derek
 * This version is not ready for release.  Disk space editing is half-done:
 * the application can play an infinite sound and the canvases can handle
 * infinite sound files.  The app is pretty bug free too, I think.  The
 * weakness is that it cannot record sound infinitely.  
 * 
 * Revision 0.13  91/06/26  15:55:02  derek
 * I have reformatted the code to conform coding specs.
 * 
 * Revision 0.12  91/06/20  19:55:19  derek
 * The network part should be working.  Also fixed numerous minor parts
 * involving the canvas and the display.
 * 
 * Revision 0.11  91/06/02  10:27:53  derek
 * Changes made in the save-as function
 * 
 * Revision 0.10  1991/04/25  01:46:22  derek
 * This version is checked in on 4/24/91
 * */
static char rcsid[] = "$Header: /Source/Media/collab/DTR/RCS/dtr.c,v 1.12 92/05/29 12:42:01 drapeau Exp $";

#include "dtr.h"              
#include "dtr_ui.h"
#include "getopt.h"

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;

dtr_mainWindow_objects*			dtr_mainWindow;
dtr_globalWaveCanvasPopUp_objects*	dtr_globalWaveCanvasPopUp;
dtr_editListPanelPopUp_objects*		dtr_editListPanelPopUp;
dtr_infoPopUp_objects*			dtr_infoPopUp;

extern Scrollbar			WaveCanvasScrollbar;

void
  main(argc, argv)
int		argc;
char		**argv;
{
  Rect	tempRect;

  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);		    /*  Initialize XView.                               */
  INSTANCE = xv_unique_key();
  strcpy(prog, argv[0]);

  dtr_mainWindow =						    /*  Initialize user interface components.           */
    dtr_mainWindow_objects_initialize(NULL, NULL);
  dtr_globalWaveCanvasPopUp =
    dtr_globalWaveCanvasPopUp_objects_initialize(NULL, dtr_mainWindow->mainWindow);
  dtr_editListPanelPopUp =
    dtr_editListPanelPopUp_objects_initialize(NULL, dtr_mainWindow->mainWindow);
  dtr_infoPopUp =
    dtr_infoPopUp_objects_initialize(NULL, dtr_mainWindow->mainWindow);
  
  frame_get_rect(dtr_mainWindow->mainWindow,&tempRect);	    /* Get the size of the top-level window */
  tempRect.r_top = 275;
  tempRect.r_left = 30;
  frame_set_rect(dtr_mainWindow->mainWindow,&tempRect);	    /* Set the position of the top-level window */

  frame_get_rect(dtr_editListPanelPopUp->editListPanelPopUp,
		 &tempRect);
  tempRect.r_top = 275;
  tempRect.r_left = 640;
  frame_set_rect(dtr_editListPanelPopUp->editListPanelPopUp,
		 &tempRect);

  frame_get_rect(dtr_globalWaveCanvasPopUp->globalWaveCanvasPopUp,
		 &tempRect);
  tempRect.r_top = 60;
  tempRect.r_left = 200;
  frame_set_rect(dtr_globalWaveCanvasPopUp->globalWaveCanvasPopUp,
		 &tempRect);

  frame_get_rect(dtr_infoPopUp->infoPopUp,&tempRect);
  tempRect.r_top = 250;
  tempRect.r_left = 340;
  frame_set_rect(dtr_infoPopUp->infoPopUp,&tempRect);

  InitOpenPanel();						    /*  Initialize the open panel stuff.                */
  InitEditList();
  InitWaveCanvasScrollbar();					    /*  Initialize the scrollbar for wave canvas.       */
  InitScaleCanvasScrollbar();
  InitWaveCanvas();
  InitGlobalWaveCanvas();
  InitVUMeterCanvas();
  InitWaveEditVariables();
  
  (void) notify_set_signal_func((Notify_client)dtr_mainWindow,	    /*  Set up to catch SIGPOLL asynchronously.         */
				(Notify_func)SigpollAsyncHandler, 
				SIGPOLL, NOTIFY_ASYNC); 
  
  (void) notify_set_event_func(SIGPOLL,				    /*  Set a synchronous event handler for SIGPOLL...  */
			       (Notify_func)SigpollSyncHandler,	    /*  ...to schedule.                                 */
			       NOTIFY_SAFE);
  
  InitAudio();							    /*  Initialize audio parameters.                    */
  InitAudioControl();						    /*  Initialize audio control device.                */
  InitBuffer();							    /*  Initialize the sound buffer.                    */
  InitRecord();							    /*  Initialize the recording tmp file name.         */
  AudioUpdatePanel(TRUE);					    /*  Update audio control (by calling...             */
								    /*  ...AudioReadState().                            */
  InitDisplay();						    /*  Set Initial display.                            */
  InitNetwork(argc, argv);					    /*  Init networking stuff, set up connection...     */
								    /*  ...with PortManager.                            */
/************
  if (strcmp(currentSoundFile, "Untitled") != 0)
  {
  UpdateHeader(FALSE);
    
    FileReady = ReadSoundFile(FALSE);
    SoundBufferReady = FALSE;				
    SameSoundFile = TRUE;
    if (FileReady) 
    {
      RepaintWaveCanvas();
      RepaintGlobalWaveCanvas();
    }
  }
************/

  notify_interpose_destroy_func(dtr_mainWindow->mainWindow,
				DestroyMainWindow);

  xv_main_loop(dtr_mainWindow->mainWindow);		    /*  Turn control over to XView.                     */
  exit(0);
}



/*
 * Menu handler for `DocumentMenu (Open)'.
 */
Menu_item
  OpenFileHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    Browse(NULL, BrowseOpen, IS_SOUNDFILE, NULL, NULL);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `DocumentMenu (Save Sound File)'.
 */
Menu_item
  SaveSoundFileMenuOption(item, op)
Menu_item	item;
Menu_generate	op;
{
  char soundFileName[80];

  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    if ((strcmp(currentSoundFile, "Untitled") == 0) || IsTmpSoundFile(currentSoundFile))
    {
      Browse(NULL, BrowseSave, IS_SOUNDFILE, NULL, NULL);
    }
    else
    {
      Browse(currentSoundFile, BrowseCheckSave, IS_SOUNDFILE, NULL, NULL);
    }
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `DocumentMenu (Save Sound File As)'.
 */
Menu_item
SaveSoundFileAsMenuOption(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    Browse(NULL, BrowseSave, IS_SOUNDFILE, NULL, NULL);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `DocumentMenu (Info)'.
 */
Menu_item
  InfoHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    xv_set(dtr_infoPopUp->infoPopUp, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(dtr_infoPopUp->infoPopUp, XV_SHOW, TRUE, NULL);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 *  Menu handler for `DocumentMenu (Close)'.
 *  This function closes the current document.  (Same as ClearAll.)
 */
Menu_item
CloseHandler(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  switch (op) {
   case MENU_DISPLAY:
    break;

   case MENU_DISPLAY_DONE:
    break;

   case MENU_NOTIFY:
    ClearAllSound();
    break;

   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `DocumentMenu (quit)'.
 */
Menu_item
  QuitDtr(item, op)
Menu_item	item;
Menu_generate	op;
{
  int    reply;

  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (unsavedChangesExist == TRUE)
    {
      reply = notice_prompt(dtr_editListPanelPopUp->editListUpperControlPanel, NULL,
			    NOTICE_MESSAGE_STRINGS,
			    "Unsaved changes exist in your current edit list.",
			    "Quit Anyway?",
			    NULL,
			    NOTICE_BUTTON, "Yes", 'Y',
			    NOTICE_BUTTON, "Cancel", 'C',
			    NULL);
      if (reply == 'C')
	return item;
    }
    if (BufferSaved == FALSE)
    {
      reply = notice_prompt(dtr_mainWindow->menuControlPanel, NULL,
			    NOTICE_MESSAGE_STRINGS,
			    "Buffer not saved yet.",
			    "Quit Anyway?",
			    NULL,
			    NOTICE_BUTTON, "Yes", 'Y',
			    NOTICE_BUTTON, "Cancel", 'C',
			    NULL);
      if (reply == 'C')
	return item;
    }
    AudioFlushClose();
    RemoveSoundFileInTmp();
    SenderDisconnectFromPortMgr(sender, &(receiver->receivePort));
    exit(0);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}								    /* end function QuitDtr */


/*
 * Menu handler for `EditMenu (Cut)'.
 */
Menu_item
  CutHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
   case MENU_DISPLAY_DONE:
   case MENU_NOTIFY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (WaveEditMode == FALSE)
    {
      break;
    }
    WaveCutHandler();
    break;
  }
  return item;
}


/*
 * Menu handler for `EditMenu (Copy)'.
 */
Menu_item
  CopyHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  extern	BOOL	AboutToCopy;
  
  switch (op) 
  {
   case MENU_DISPLAY:
   case MENU_DISPLAY_DONE:
   case MENU_NOTIFY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (WaveEditMode == FALSE)
    {
      break;
    }
    AboutToCopy = TRUE;
    break;
    
  }
  return item;
}


/*
 * Menu handler for `EditMenu (Undo)'.
 */
Menu_item
  UndoHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  unsigned char  *tempptr;
  unsigned        tempu;
  int             tempi;
  extern   BOOL   CanUndo;
  
  switch (op) 
  {
   case MENU_DISPLAY:
   case MENU_NOTIFY_DONE:
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (WaveEditMode == FALSE)
    {
      break;
    }
    if (!CanUndo)
      break;
    if (Buffer.olddata) 
    {
      ClearAllSound();
      
      tempptr = Buffer.data;
      Buffer.data = Buffer.olddata;
      Buffer.olddata = tempptr;
      
      tempu = Buffer.alloc_size;
      Buffer.alloc_size = Buffer.old_alloc_size;
      Buffer.old_alloc_size = tempu;
      
      tempi = Buffer.hdr.data_size;
      Buffer.hdr.data_size = Buffer.old_hdr_data_size;
      Buffer.old_hdr_data_size = tempi;
      
      SoundBufferReady = TRUE;
      Buffer.display.start = Buffer.play.start = 0;
      Buffer.display.end = Buffer.play.end = 
	Buffer.hdr.data_size - 1;
      FileUpdate();
      UpdateMessageDisplay();
      Zoom = 1.0;
      UpdateZoomLevelDisplay();
      
      oldLeftFloatMarker = 0.0;					    /*  Reset Markers                                   */
      oldRightFloatMarker = (double) Buffer.hdr.data_size - 1.0;
      WaveCanvasRightMarkerSet = FALSE;
      oldLeftFloatGbWaveMarker = 0.0;
      oldRightFloatGbWaveMarker = 
	(double) Buffer.hdr.data_size - 1.0;
      GlobalWaveCanvasRightMarkerSet = FALSE;
      RepaintWaveCanvas();
      RepaintGlobalWaveCanvas();
      
      SameSoundFile = TRUE;					    /*  Reset Same Sound File flag.                     */
    }
    else 
    {
	AlertByNoticePrompt(dtr_mainWindow->menuControlPanel,
			    "Cannot undo any further.");
    }
    break;
  }
  return item;
}								    /* end function UndoHandler */


/*
 * Menu handler for `EditMenu (Paste)'.
 */
Menu_item
  PasteHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  extern	BOOL	AboutToPaste;
  
  switch (op) 
  {
   case MENU_DISPLAY:
   case MENU_DISPLAY_DONE:
   case MENU_NOTIFY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (WaveEditMode == FALSE)
    {
      break;
    }
    AboutToPaste = TRUE;
    break;
  }
  return item;
}


/*
 * Menu handler for `EditMenu (Clear All)'.
 */
Menu_item
  ClearAllHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
   case MENU_DISPLAY_DONE:
   case MENU_NOTIFY_DONE:
    break;
    
   case MENU_NOTIFY:
    ClearAllSound();
    break;
    
  }
  return item;
}


/*
 * Menu handler for `OptionsMenu (Reset Markers)'.
 */
Menu_item
  ResetMarkersHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  if (op == MENU_NOTIFY) 
  {
    if (WaveCanvasRightMarkerSet)
      ResetWaveCanvasMarkers();
    if (GlobalWaveCanvasRightMarkerSet)
      ResetGlobalWaveCanvasMarkers();
  }
  return item;
}


/*
 * Menu handler for `OptionsMenu (Size To Fit)'.
 */
Menu_item
  SizeToFitHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  double             newZoom;
  double             save;
  
  save = WaveCanvasDataSamplingInterval;
  
  switch (op) 
  {
   case MENU_DISPLAY:
   case MENU_DISPLAY_DONE:
   case MENU_NOTIFY_DONE:
    break;
    
   case MENU_NOTIFY:
    
    if ((FileReady == FALSE) && (SoundBufferReady == FALSE))	    /*  If no sound has been loaded yet, return...      */
    {								    /*  ...doing anything.                              */
      AlertByNoticePrompt(dtr_mainWindow->mainWindow,
			  "There is no sound to fit.");
      return item;
    }
  
    if (WaveCanvasRightMarkerSet)				    /*  Adjust the zoom level.                          */
    {
      WaveCanvasDataSamplingInterval = 
	(oldRightFloatMarker - oldLeftFloatMarker) / WaveWindowWidth;
    }
    else 
    {
      WaveCanvasDataSamplingInterval = Buffer.hdr.data_size /	    /*  If no selection is specified, we still want...  */
	(double) WaveWindowWidth;				    /*  ...to resize the whole canvas.                  */
    }
    
    newZoom = DefaultWaveCanvasDataSamplingInterval / 
      WaveCanvasDataSamplingInterval;
    
    if (newZoom > (double) MAX_ZOOM) 
    {
      AlertByNoticePrompt(dtr_mainWindow->mainWindow,
			  "Selection is too small to resize.");
      WaveCanvasDataSamplingInterval = save;
      return item;
    }
    else Zoom = newZoom;
    
    UpdateZoomLevelDisplay();
    WaveCanvasDataSamplingInterval = DefaultWaveCanvasDataSamplingInterval /
      Zoom;
/******
    FrameStartingSecond = (oldLeftFloatMarker /
			   DefaultWaveCanvasDataSamplingInterval) /
			     (WaveWindowWidth / 2)) * 5.0;
********/
    FrameStartingSecond = (oldLeftFloatMarker /
			   DEVICE_SAMPLE_RATE);
    RepaintWaveCanvas();
    
    if (WaveCanvasRightMarkerSet) 
    {
      xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 
	     irint(((oldLeftFloatMarker - (FrameStartingSecond * DEVICE_SAMPLE_RATE))/ 
		    WaveCanvasDataSamplingInterval)
		   / 10.0), NULL);
    }
    else 
    {
      xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    }
    UpdateCanvasFrame();
    break;
  }
  return item;
}								    /* end function SizeToFitHandler */


/*
 * Menu handler for `OptionsMenu (Global Wave Canvas)'.
 */
Menu_item
  GlobalWaveCanvasHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    xv_set(dtr_globalWaveCanvasPopUp->globalWaveCanvasPopUp, 
	   FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(dtr_globalWaveCanvasPopUp->globalWaveCanvasPopUp, 
	   XV_SHOW, TRUE, NULL);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `optionsMenu (Enable Sound Editing)'.
 */
Menu_item
  ToggleSoundEditing(item, op)
Menu_item	item;
Menu_generate	op;
{
  dtr_mainWindow_objects * ip = (dtr_mainWindow_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  
  switch (op)
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (WaveEditMode == TRUE)
    {
      WaveEditMode = FALSE;
      xv_set(item, MENU_STRING, "Enable Sound Editing",NULL);
    }
    else
    {
      WaveEditMode = TRUE;
      xv_set(item, MENU_STRING, "Disable Sound Editing",NULL);
    }
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `audioOutputOptionsMenu (icons/speaker.icon)'.
 */
Menu_item
  SetSpeakerOutput(item, op)
Menu_item	item;
Menu_generate	op;
{
  static int	playPort;
  
  switch (op)
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    playPort = AUDIO_SPEAKER;
    audio_set_play_port(Audioctl_fd, &playPort);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `audioOutputOptionsMenu (icons/headphone.icon)'.
 */
Menu_item
  SetHeadphoneOutput(item, op)
Menu_item	item;
Menu_generate	op;
{
  static int	playPort;
  
  switch (op)
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    playPort = AUDIO_HEADPHONE;
    audio_set_play_port(Audioctl_fd, &playPort);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}



/*
 * Notify callback function for `openEditListButton'.
 */
void
ShowEditListPanelHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  xv_set(dtr_editListPanelPopUp->editListPanelPopUp,
	 FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set(dtr_editListPanelPopUp->editListPanelPopUp,
	 XV_SHOW, TRUE, NULL);
}




/*
 * Menu handler for `EditListDocumentMenu (Open)'.
 */
Menu_item
OpenEditListHandler(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  switch (op) {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    Browse(NULL, BrowseOpen, IS_EDITLIST, documentFirstLine, documentFileType);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `EditListDocumentMenu (Save Edit List)'.
 */
Menu_item
  SaveEditListHandler(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op)
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    if (strcmp(absoluteEditListName, "Untitled") == 0)		    /* No filename specified yet */
    {
      Browse(NULL, BrowseSave, IS_EDITLIST, documentFirstLine, documentFileType);
    }
    else
    {
      Browse(absoluteEditListName, BrowseCheckSave, IS_EDITLIST,
	     documentFirstLine, documentFileType);
    }
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}



/*
 * Menu handler for `EditListDocumentMenu (Save Edit List As)'.
 */
Menu_item
EditListSaveAsHandler(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  switch (op) {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    Browse(NULL, BrowseSave, IS_EDITLIST, documentFirstLine, documentFileType);
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


/*
 * Menu handler for `EditListDocumentMenu (New)'.
 */
Menu_item
NewEditListHandler(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  switch (op) {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    NewEditList();
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}



/*
 * Notify callback function for `CanvasRewind2Button'.
 */
void
CanvasRewind2Handler(item, event)
	Panel_item	item;
	Event		*event;
{
  if (FrameStartingSecond == 0.0)
  {
    return;
  }
  else if (FrameStartingSecond >= 5.0)
  {
    FrameStartingSecond -= 10.0;
    if (FrameStartingSecond < 0.0)
      FrameStartingSecond = 0.0;
    RepaintWaveCanvas();
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    UpdateCanvasFrame();
  }
  else 
  {
    FrameStartingSecond = 0.0;
    RepaintWaveCanvas();
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    UpdateCanvasFrame();
  }    
}


/*
 * Notify callback function for `CanvasRewind1Button'.
 */
void
  CanvasRewind1Handler(item, event)
Panel_item	item;
Event		*event;
{
  if (FrameStartingSecond == 0.0)
  {
    return;
  }
  else  if (FrameStartingSecond >= 5.0)
  {
    FrameStartingSecond -= 5.0;
    RepaintWaveCanvas();
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    UpdateCanvasFrame();
  }
  else
  {
    FrameStartingSecond = 0.0;
    RepaintWaveCanvas();
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    UpdateCanvasFrame();
  }
}


/*
 * Notify callback function for `CanvasFastForward1Button'.
 */
void
  CanvasFastForward1Handler(item, event)
Panel_item	item;
Event		*event;
{
  double		duration;
	
  duration = Buffer.hdr.data_size / ((double) DEVICE_SAMPLE_RATE);
  if (duration > FrameStartingSecond + 5.0)
  {
    FrameStartingSecond += 5.0;
    RepaintWaveCanvas();
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    UpdateCanvasFrame();
  }
}


/*
 * Notify callback function for `CanvasFastForward2Button'.
 */
void
  CanvasFastForward2Handler(item, event)
Panel_item	item;
Event		*event;
{
  double   duration;
	
  duration = Buffer.hdr.data_size / ((double) DEVICE_SAMPLE_RATE);
  if (duration > FrameStartingSecond + 10.0)
  {
    FrameStartingSecond += 10.0;
    if (duration < FrameStartingSecond)
      FrameStartingSecond -= 5.0;
    RepaintWaveCanvas();
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);
    UpdateCanvasFrame();
  }
}



/*
 * Event callback function for `WaveCanvas'.
 */
Notify_value
  WaveCanvasEventHandler(win, event, arg, type)
Xv_window	win;
Event		*event;
Notify_arg	arg;
Notify_event_type type;
{
  static 	int 	sawLeftDown = 0;
  static 	int 	sawMiddleDown = 0;
  static 	int	lastWinX;
  int                   GbWaveWinX;
  int                   overallWaveWinX;
  extern        BOOL    AboutToCopy;
  extern	BOOL	AboutToPaste;
  extern	BOOL	CutPortionReady;
  
  if (ActiveFlag)						    /*  If playing or recording, return w/o doing...    */
    return notify_next_event_func(win, (Notify_event) event,	    /*  ...anything.                                    */
				  arg, type);
  if (AboutToCopy)						    /*  Check if the mouse click is meant to be a ...   */
  {								    /*  ...copy-click.                                  */
    if (WaveCanvasRightMarkerSet && (event_id(event) == MS_RIGHT ||
				     event_id(event) == MS_MIDDLE ||
				     event_id(event) == MS_LEFT))
      WaveCopyHandler(irint(event_x(event) * 
			    WaveCanvasDataSamplingInterval));
    return notify_next_event_func(win, (Notify_event) event, 
				  arg, type);
  }
  if (AboutToPaste)						    /*  Check if the mouse click is meant to be a ...   */
  {								    /*  ...paste-click.                                 */
    if (CutPortionReady && (event_id(event) == MS_RIGHT ||
			    event_id(event) == MS_MIDDLE ||
			    event_id(event) == MS_LEFT))
      WavePasteHandler(irint(event_x(event) * 
			     WaveCanvasDataSamplingInterval));
    return notify_next_event_func(win, (Notify_event) event, 
				  arg, type);
  }
  overallWaveWinX = irint(event_x(event) +
			  ((FrameStartingSecond / 10.0) * WaveWindowWidth
			   * DefaultWaveCanvasDataSamplingInterval /
			   WaveCanvasDataSamplingInterval));
  
  if (GlobalWaveCanvasDataSamplingInterval != 0.0)
    GbWaveWinX = irint(overallWaveWinX * 
		       WaveCanvasDataSamplingInterval/
		       GlobalWaveCanvasDataSamplingInterval);
  else GbWaveWinX = 0;
  if (event_id(event) == MS_LEFT)
  {
    if (event_is_down(event)) 
    {
      sawLeftDown = 1;
      leftButtonHoldPoint = overallWaveWinX;
      GbWaveLeftButtonHoldPoint = GbWaveWinX;
      WaveCanvasMarkerHandler(overallWaveWinX, &oldLeftFloatMarker);
      WaveCanvasMarkerHandler(overallWaveWinX, &oldRightFloatMarker);
      
      GlobalWaveCanvasMarkerHandler(GbWaveWinX, 
				    &oldLeftFloatGbWaveMarker);
      GlobalWaveCanvasMarkerHandler(GbWaveWinX, 
				    &oldRightFloatGbWaveMarker);
    }
    else if (event_is_up(event)) 
    {
      sawLeftDown = 0;
    }
  }								    /* end if (event_id(event) == MS_LEFT) */
  else if (event_id(event) == MS_MIDDLE)
  {
    if (event_is_down(event))
    {
      sawMiddleDown = 1;
      if (leftButtonHoldPoint == NOTREADY)
      {
	leftButtonHoldPoint = overallWaveWinX;
	GbWaveLeftButtonHoldPoint = GbWaveWinX;
	WaveCanvasMarkerHandler(overallWaveWinX, &oldLeftFloatMarker);
	WaveCanvasMarkerHandler(overallWaveWinX, &oldRightFloatMarker);
	GlobalWaveCanvasMarkerHandler(GbWaveWinX, 
				      &oldLeftFloatGbWaveMarker);
	GlobalWaveCanvasMarkerHandler(GbWaveWinX, 
				      &oldRightFloatGbWaveMarker);
      }								    /* end if (leftButtonHoldPoint == NOTREADY)  */
      else 
      {
	if (overallWaveWinX < leftButtonHoldPoint)
	{
	  /***
	    if (lastWinX > leftButtonHoldPoint)  {
	    WaveCanvasMarkerHandler(leftButtonHoldPoint, 
	    &oldRightFloatMarker);
	    GlobalWaveCanvasMarkerHandler(GbWaveLeftButtonHoldPoint, 
	    &oldRightFloatGbWaveMarker);
	    }
	    **********/
	  WaveCanvasBackwardMarkerHandler(overallWaveWinX, 
					  &oldLeftFloatMarker);
	  GlobalWaveCanvasBackwardMarkerHandler(GbWaveWinX, 
						&oldLeftFloatGbWaveMarker);
	}							    /* end if (overallWaveWinX < leftButtonHoldPoint) */
	else 
	{
	  if (lastWinX < leftButtonHoldPoint)  
	  {
	    WaveCanvasBackwardMarkerHandler(leftButtonHoldPoint, 
					    &oldLeftFloatMarker);
	    GlobalWaveCanvasBackwardMarkerHandler(GbWaveLeftButtonHoldPoint, 
						  &oldLeftFloatGbWaveMarker);
	  }
	  WaveCanvasMarkerHandler(overallWaveWinX, &oldRightFloatMarker);
	  GlobalWaveCanvasMarkerHandler(GbWaveWinX, 
					&oldRightFloatGbWaveMarker);
	}							    /* end else (overallWaveWinX < leftButtonHoldPoint) */
      }								    /* end else (leftButtonHoldPoint == NOTREADY) */
    }								    /* end if (event_is_down(event)) */
    else if (event_is_up(event)) 
    {
      sawMiddleDown = 0;
    }
  }								    /* end else if (event_id(event) == MS_MIDDLE) */
  else if (event_id(event) == LOC_DRAG && (sawLeftDown || sawMiddleDown)) 
  {
    if (overallWaveWinX < leftButtonHoldPoint)
    {
      if (lastWinX > leftButtonHoldPoint)  
      {
	WaveCanvasMarkerHandler(leftButtonHoldPoint, 
				&oldRightFloatMarker);
	GlobalWaveCanvasMarkerHandler(GbWaveLeftButtonHoldPoint, 
				      &oldRightFloatGbWaveMarker);
      }
      WaveCanvasBackwardMarkerHandler(overallWaveWinX, &oldLeftFloatMarker);
      GlobalWaveCanvasBackwardMarkerHandler(GbWaveWinX, 
					    &oldLeftFloatGbWaveMarker);
    }								    /* end if (overallWaveWinX < leftButtonHoldPoint) */
    else
    {
      if (lastWinX < leftButtonHoldPoint) 
      {
	WaveCanvasBackwardMarkerHandler(leftButtonHoldPoint,
					&oldLeftFloatMarker);
	GlobalWaveCanvasBackwardMarkerHandler(GbWaveLeftButtonHoldPoint, 
					      &oldLeftFloatGbWaveMarker);
      }
      WaveCanvasMarkerHandler(overallWaveWinX, &oldRightFloatMarker);
      GlobalWaveCanvasMarkerHandler(GbWaveWinX, 
				    &oldRightFloatGbWaveMarker);
    }								    /* end else (overallWaveWinX < leftButtonHoldPoint) */
  }
  lastWinX = overallWaveWinX;
  UpdateSelection();						    /* Load new selection info into the Edit List panel */
  return notify_next_event_func(win, (Notify_event) event, arg, type);
}								    /* end function WaveCanvasEventHandler */


/*
 * Repaint callback function for `WaveCanvas'.
 */
void
  WaveCanvasRepaintHandler(canvas, paint_window, display, xid, rects)
Canvas		canvas;
Xv_window	paint_window;
Display		*display;
Window		xid;
Xv_xrectlist	*rects;
{
  int  			pt;
  int  			i;
  int  			from, to;
  extern   GC 		gcWave;
  extern   Scrollbar    WaveCanvasScrollbar;
  int      		heightOffset;
  double   		floatpt;
  int      		leftWinX, rightWinX;
  double                scaleMarkerInterval;
  double                x;
  static unsigned char  soundData[80000];
  int                   canvasDataSize;
  
  XClearWindow(display, xid);
  
  WaveCanvasDataSamplingInterval = DefaultWaveCanvasDataSamplingInterval /
    Zoom;
  UpdateCanvasesAndScrollbarParameters();
  if (Buffer.hdr.data_size < (FrameStartingSecond + 10) * DEVICE_SAMPLE_RATE)
  {
    canvasDataSize = Buffer.hdr.data_size -			    /*  find out how much data to display over one...   */
      irint(FrameStartingSecond * DEVICE_SAMPLE_RATE);		    /*  ...window.                                      */
  }
  else
  {
    canvasDataSize = 80000;
  }
  
  if (FileReady)
  {
    soundfd = open(Buffer.filename, O_RDONLY);
    lseek(soundfd, irint(FrameStartingSecond * DEVICE_SAMPLE_RATE), L_SET);
    read(soundfd, (char *)soundData, canvasDataSize);
    close(soundfd);
  }
  
  heightOffset = (int) (WaveCanvasMidHeight - 15);
  WaveCanvasPenColor("White");
  to = heightOffset;
  
  if (FileReady)						    /*  Canvas data size is defined to be the diff...   */
  {								    /*  between last byte of the data and the closest.. */
    floatpt = 0.0;						    /*  ...multiple of 80000.                           */
    for(i=0, pt=0 ; pt < canvasDataSize && i < WaveCanvasWidth;
	i++)
    {
      from = to;
      to   = ((audio_u2c(soundData[pt]) * WaveScopeWidth) >> 7)
	+ heightOffset;
      
      XDrawLine(display, xid, gcWave, i-1, from, i, to);
      
      floatpt = floatpt + WaveCanvasDataSamplingInterval;
      pt = irint(floatpt);
    }
  }    
  else if (SoundBufferReady == TRUE)
  {
    floatpt = (double) FrameStartingSecond * DEVICE_SAMPLE_RATE;
    
    for(i=0, pt= irint(floatpt) ;
	pt < canvasDataSize + (FrameStartingSecond * DEVICE_SAMPLE_RATE)
	&& i <= WaveCanvasWidth;
	i++)
    {
      from = to;
      to   = ((audio_u2c(Buffer.data[pt]) * WaveScopeWidth) >> 7)
	+ heightOffset;
      
      XDrawLine(display, xid, gcWave, i-1, from, i, to);
      
      floatpt = floatpt + WaveCanvasDataSamplingInterval;
      pt = irint(floatpt);
    }
  }
  
  scaleMarkerInterval = WaveWindowWidth / 10.0 * Zoom;		    /*  Draw the horizontal time axis.                  */
  WaveCanvasPenColor("Red");
  
  XDrawLine(display, xid, gcWave, 0, WaveCanvasMidHeight-15, 
	    WaveCanvasWidth, WaveCanvasMidHeight-15);
  for(x = 0.0 ; x < (double) WaveCanvasWidth ; x += scaleMarkerInterval)
    XDrawLine(display, xid, gcWave, 
	      irint(x), WaveCanvasMidHeight-15+5,
	      irint(x), WaveCanvasMidHeight-15-5);
  
  if (WaveCanvasRightMarkerSet)					    /*  Highlignt marked area.                          */
  {
    rightWinX = irint(oldRightFloatMarker/WaveCanvasDataSamplingInterval);
    leftWinX  = irint(oldLeftFloatMarker/WaveCanvasDataSamplingInterval);
    ReverseWaveCanvasArea(leftWinX, rightWinX);
  }
}								    /* end function WaveCanvasRepaintHandler */


/*
 * Notify callback function for `PauseButton'.
 */
void
  PauseButtonHandler(item, event)
Panel_item	item;
Event		*event;
{
  Pause();
}


/*
 * Notify callback function for `StopButton'.
 */
void
  StopButtonHandler(item, event)
Panel_item	item;
Event		*event;
{
  if (ActiveFlag & PLAY)
    StopPlay();
  else
    if (ActiveFlag & RECORD)
      StopRecord();
  CancelButtonGlow();
}


/*
 * Notify callback function for `PlayButton'.
 */
void
  PlayButtonHandler(item, event)
Panel_item	item;
Event		*event;
{
  dtr_mainWindow_objects	*ip = 
    (dtr_mainWindow_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  
  if (ActiveFlag & PLAY)					    /* Don't do anything if already playing a sound */
    return;
  if (WaitFlag & PLAY) 
  {
    AlertByNoticePrompt(dtr_mainWindow->mainWindow,
			"Device is not ready.  Play Stopped");
    StopPlay();
    return;
  }
  
  if (WaveEditMode == FALSE)
  {
    FileReady = ReadSoundFile(FALSE);
    SoundBufferReady = FALSE;
  }
  else 
  {
    SoundBufferReady = ReadSoundFile(TRUE);
    FileReady = FALSE;
  }
  
  if((SoundBufferReady == TRUE) || (FileReady == TRUE)) 
  {
    if (!SameSoundFile)  
    {
      RepaintWaveCanvas(); 
      RepaintGlobalWaveCanvas();
    }
    Play(ip);
    SameSoundFile = TRUE;
  }
}								    /* end function PlayButtonHandler */



/*
 * Repaint callback function for `ScaleCanvas'.
 */
void
  ScaleCanvasRepaintHandler(canvas, paint_window, display, xid, rects)
Canvas		canvas;
Xv_window	paint_window;
Display		*display;
Window		xid;
Xv_xrectlist	*rects;
{
  double		scaleMarkerInterval;
  int			canvasWidth;
  float		        time, timeInterval;
  extern	GC	gcScale;
  double                x;
  extern      int       ScaleAxisHeight;
  extern      int       ScaleTimeHeight;
  int                   width;
  int                   i;
  
  XClearWindow(display, xid);
  
  canvasWidth = irint(Zoom *WaveWindowWidth);
  if (canvasWidth < WaveWindowWidth)
    canvasWidth = WaveWindowWidth;
  ScaleCanvasPenColor("black");
  timeInterval = 1.0;
  
  scaleMarkerInterval = WaveWindowWidth / 10.0 * Zoom;
  
  for(i = 1 ; scaleMarkerInterval > 200.0; )
  {
    if (i == 1)
    {
      timeInterval /= 2.0;
      scaleMarkerInterval /= 2.0;
      i = 5;
    }
    else
    {
      timeInterval /= 5.0;
      scaleMarkerInterval /= 5.0;
      i = 1;
    }
  }
  
  XDrawLine(display, xid, gcScale, 0, ScaleAxisHeight, 
	    canvasWidth, ScaleAxisHeight);
  
  x = 0.0;
  XDrawLine(display, xid, gcScale, 
	    irint(x), ScaleAxisHeight+5, irint(x), ScaleAxisHeight-5);
  
  for(x = scaleMarkerInterval, time = FrameStartingSecond + timeInterval; 
      x < (double) canvasWidth ; 
      x += scaleMarkerInterval, time += timeInterval)
  {
    ScaleCanvasPenColor("black");	
    XDrawLine(display, xid, gcScale, 
	      irint(x), ScaleAxisHeight+5, irint(x), ScaleAxisHeight-5);
    sprintf(msg, "%.2f s", time);
    width = strlen(msg) * 5.0;
    ScaleCanvasPenColor("black");
    XDrawString(display, xid, gcScale, 
		irint(x-(width/2.0)), ScaleTimeHeight, msg, strlen(msg));
  }
  
}


/*
 * Notify callback function for `RecordButton'.
 */
void
  RecordButtonHandler(item, event)
Panel_item	item;
Event		*event;
{
  dtr_mainWindow_objects	*ip = 
    (dtr_mainWindow_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  
  if ((ActiveFlag & PLAY) || (WaitFlag & PLAY))			    /*  Record stops in-progress Play.                  */
    StopPlay();
  
  if (WaitFlag & RECORD)
  {
    StopRecord();						    /*  Never got the device.                           */
    return;
  }
  else if (ActiveFlag & RECORD)
  {								    /*  Stop button pressed.                            */
    StopRecord();
    if (Audio_state.record.error)
    {
      AlertByNoticePrompt(dtr_mainWindow->mainWindow,
			  "Overflow detected during Record.");
    }
    return;
  }
  
  switch (AudioOpen(RECORD))
  {
   case 0:							    /*  Open succeeded.                                 */
    break;
    
   case 1:							    /*  Open returned EBUSY.                            */
    AlertByNoticePrompt(dtr_mainWindow->mainWindow,
			"Audio device currently in use.");
    return;
    
   case -1:							    /*  Open error.                                     */
   default:
    AlertByNoticePrompt(dtr_mainWindow->menuControlPanel,
			"Error opening audio device.");
    return;
  }
  
  strcpy(Buffer.filename,"\0");					    /*  Clear buffer filename to avoid filename...      */
  sprintf(currentSoundFile,"Untitled");				    /*  ...conflict.                                    */
  UpdateHeader(FALSE);
  ClearMessageDisplay();
  ClearWaveCanvas();
  ClearGlobalWaveCanvas();
  StartRecord();
}								    /* end function RecordButtonHandler */


/*
 * Notify callback function for `PlayGain'.
 */
void
  PlayGainSlider(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  double  gain;
  
  gain = ScaleGain(value);
  Audio_state.play.gain = ~0;
  (void) audio_set_play_gain(Audioctl_fd, &gain);
  if (item == dtr_mainWindow->playGain)				    /* Make sure the other volume level slider shows the same... */
    xv_set(dtr_editListPanelPopUp->editListVolumeSlider,	    /* ...value as this one does */
	   PANEL_VALUE, value, NULL);
  else
    xv_set(dtr_mainWindow->playGain,
	   PANEL_VALUE, value, NULL);
  return;
}


/*
 * Notify callback function for `RecordGain'.
 */
void
  RecordGainSlider(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  double gain;
  
  gain = ScaleGain(value);					    /*  Let SIGPOLL handler adjust displayed value.     */
  Audio_state.record.gain = ~0;
  (void) audio_set_record_gain(Audioctl_fd, &gain);
}


/*
 * Repaint callback function for `VUMeterCanvas'.
 */
void
  VUMeterCanvasRepaintHandler(canvas, paint_window, rects)
Canvas		canvas;
Xv_window	paint_window;
Rectlist	*rects;
{
  ClearVUMeterCanvas();
}


/*
 * Notify callback function for `ZoomSlider'.
 */
void
  ZoomSliderHandler(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  extern Xv_Window paintWinWave;
  extern Display   *dpyWave;
  extern Window    xid;
  extern int       ScrollbarPixelsPerUnit;
  
  if (strcmp(currentSoundFile, "Untitled") != 0)
  {
    return;
  }
  xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);	    /*  Reset scrollbar first, this fixes the weird.... */
								    /*  ...canvas bug.                                  */
  Zoom = value/100.0;
  sprintf(msg, "%.2lf", Zoom);					    /*  Update the zoom-level text field.               */
  xv_set(dtr_mainWindow->zoomLevelTextField, 
	 PANEL_VALUE, msg, NULL);
  FrameStartingSecond = (irint(oldLeftFloatMarker /
			       DefaultWaveCanvasDataSamplingInterval) /
			 (WaveWindowWidth / 2)) * 5.0;
  RepaintWaveCanvas();
  CenterWaveCanvasHighlightedSegment();
  UpdateCanvasFrame();
}


/*
 * Event callback function for `ZoomLevelTextField'.
 */
Panel_setting
  ZoomLevelTextFieldHandler(item, event)
Panel_item	item;
Event		*event;
{
  char     *value = (char *) xv_get(item, PANEL_VALUE);
  double   newZoom;
  
  if (strcmp(currentSoundFile, "Untitled") ==0)
  {
    ClearWaveCanvas();
    return panel_text_notify(item, event);
  }  
  newZoom = atof(value);
  if (newZoom <= 1.0 || newZoom > (double) MAX_ZOOM)
  {
    AlertByNoticePrompt(dtr_mainWindow->mainWindow,
			"Error:  Illegal zoom value.");
  }
  else
  {
    xv_set(WaveCanvasScrollbar, SCROLLBAR_VIEW_START, 0, NULL);	    /*  Reset scrollbar first, this fixes the weird.... */
								    /*  ...canvas bug.                                  */
    Zoom = newZoom;
    UpdateZoomLevelDisplay();
    RepaintWaveCanvas();
    CenterWaveCanvasHighlightedSegment();
    UpdateCanvasFrame();
  }
  return panel_text_notify(item, event);
}


/*
 * Event callback function for `globalWaveCanvasPopUp'.
 */
Notify_value
  GlobalWaveCanvasEventHandler(win, event, arg, type)
Xv_window	win;
Event		*event;
Notify_arg	arg;
Notify_event_type type;
{
  extern   	int     GbWaveLeftButtonHoldPoint;
  static 	int 	sawLeftDown = 0;
  static 	int 	sawMiddleDown = 0;
  static 	int	lastWinX;
  int                   WaveCanvasWinX;
  extern        BOOL    AboutToCopy;
  extern	BOOL	AboutToPaste;
  extern	BOOL	CutPortionReady;
  
  if (ActiveFlag)						    /*  If playing or recording, return without doing.. */
    return notify_next_event_func(win, (Notify_event) event,	    /*  ...anything.                                    */
				  arg, type);
  
  if (AboutToCopy)						    /*  Check if the mouse click is meant to be a...    */
  {								    /*  ...copy-click.                                  */
    if (WaveCanvasRightMarkerSet && (event_id(event) == MS_RIGHT ||
				     event_id(event) == MS_MIDDLE ||
				     event_id(event) == MS_LEFT)) 
      WaveCopyHandler(irint(event_x(event) * 
			    GlobalWaveCanvasDataSamplingInterval));
    return notify_next_event_func(win, (Notify_event) event, 
				  arg, type);
  }
  if (AboutToPaste)						    /*  Check if the mouse click is meant to be a...    */
  {								    /*  ...paste-click.                                 */
    if (CutPortionReady && (event_id(event) == MS_RIGHT ||
			    event_id(event) == MS_MIDDLE ||
			    event_id(event) == MS_LEFT)) 
      WavePasteHandler(irint(event_x(event) * 
			     GlobalWaveCanvasDataSamplingInterval));
    return notify_next_event_func(win, (Notify_event) event, 
				  arg, type);
  }
  WaveCanvasWinX = irint(event_x(event) *			    /*  Calculate the other wave canvas's window x...   */
			 GlobalWaveCanvasDataSamplingInterval /	    /*  ...value.                                       */
			 WaveCanvasDataSamplingInterval);
  
  if (event_id(event) == MS_LEFT)
  {
    if (event_is_down(event))
    {
      sawLeftDown = 1;
      GbWaveLeftButtonHoldPoint = event_x(event);
      leftButtonHoldPoint = WaveCanvasWinX;
      WaveCanvasMarkerHandler(WaveCanvasWinX, &oldLeftFloatMarker);
      WaveCanvasMarkerHandler(WaveCanvasWinX, &oldRightFloatMarker);
      GlobalWaveCanvasMarkerHandler(event_x(event),
				    &oldLeftFloatGbWaveMarker);
      GlobalWaveCanvasMarkerHandler(event_x(event),
				    &oldRightFloatGbWaveMarker);
    }
    else if (event_is_up(event))
    {
      sawLeftDown = 0;
    }
  }
  else if (event_id(event) == MS_MIDDLE)
  {
    if (event_is_down(event))
    {
      sawMiddleDown = 1;
      if (GbWaveLeftButtonHoldPoint == NOTREADY)
      {
	GbWaveLeftButtonHoldPoint = event_x(event);
	leftButtonHoldPoint = WaveCanvasWinX;
	WaveCanvasMarkerHandler(WaveCanvasWinX, &oldLeftFloatMarker);
	WaveCanvasMarkerHandler(WaveCanvasWinX, &oldRightFloatMarker);
	GlobalWaveCanvasMarkerHandler(event_x(event),
				      &oldLeftFloatGbWaveMarker);
	GlobalWaveCanvasMarkerHandler(event_x(event),
				      &oldRightFloatGbWaveMarker);
      } 
      else
      {
	if (event_x(event) < GbWaveLeftButtonHoldPoint) {
	  /********
	    if (lastWinX > GbWaveLeftButtonHoldPoint) {
	    WaveCanvasMarkerHandler(leftButtonHoldPoint,
	    &oldRightFloatMarker);
	    GlobalWaveCanvasMarkerHandler(GbWaveLeftButtonHoldPoint,
	    &oldRightFloatGbWaveMarker);
	    }
	    ********/
	  WaveCanvasBackwardMarkerHandler(WaveCanvasWinX, 
					  &oldLeftFloatMarker);
	  
	  GlobalWaveCanvasBackwardMarkerHandler(event_x(event),
						&oldLeftFloatGbWaveMarker);
	}
	else
	{
	  if (lastWinX < GbWaveLeftButtonHoldPoint)
	  {
	    WaveCanvasBackwardMarkerHandler(
					    leftButtonHoldPoint, 
					    &oldLeftFloatMarker);
	    GlobalWaveCanvasBackwardMarkerHandler(
						  GbWaveLeftButtonHoldPoint, 
						  &oldLeftFloatGbWaveMarker);
	  }
	  WaveCanvasMarkerHandler(WaveCanvasWinX, 
				  &oldRightFloatMarker);
	  GlobalWaveCanvasMarkerHandler(event_x(event),
					&oldRightFloatGbWaveMarker);
	}	
      }
      
    }
    else if (event_is_up(event))
    {
      sawMiddleDown = 0;
    }
  } 
  else if (event_id(event) == LOC_DRAG && (sawLeftDown || sawMiddleDown))
  {
    if (event_x(event) < GbWaveLeftButtonHoldPoint)
    {
      if (lastWinX > GbWaveLeftButtonHoldPoint)
      {
	WaveCanvasMarkerHandler(leftButtonHoldPoint, 
				&oldRightFloatMarker);
	GlobalWaveCanvasMarkerHandler(GbWaveLeftButtonHoldPoint, 
				      &oldRightFloatGbWaveMarker);
      }
      WaveCanvasBackwardMarkerHandler(WaveCanvasWinX, 
				      &oldLeftFloatMarker);
      GlobalWaveCanvasBackwardMarkerHandler(event_x(event),
					    &oldLeftFloatGbWaveMarker);
    }
    else
    {
      if (lastWinX < GbWaveLeftButtonHoldPoint)
      {
	WaveCanvasBackwardMarkerHandler(leftButtonHoldPoint, 
					&oldLeftFloatMarker);
	GlobalWaveCanvasBackwardMarkerHandler(GbWaveLeftButtonHoldPoint, 
					      &oldLeftFloatGbWaveMarker);
      }
      WaveCanvasMarkerHandler(WaveCanvasWinX, &oldRightFloatMarker);
      GlobalWaveCanvasMarkerHandler(event_x(event),
				    &oldRightFloatGbWaveMarker);
    }
  } 
  
  lastWinX = event_x(event);
  UpdateSelection();
  return notify_next_event_func(win, (Notify_event) event, arg, type);
}								    /* end function GlobalWaveCanvasEventHandler */

/*
 * Repaint callback function for `GlobalWaveCanvas'.
 */
void
  GlobalWaveCanvasRepaintHandler(canvas, paint_window, display, xid, rects)
Canvas		canvas;
Xv_window	paint_window;
Display		*display;
Window		xid;
Rectlist	*rects;
{
  int  		pt;
  int  		dataEnd;
  int  		i;
  int  		from, to;
  extern   GC 	gcGbWave;
  int      	heightOffset;
  double   	floatpt;
  int      	leftWinX, rightWinX;
  unsigned char data;
  
    
  XClearWindow(display, xid);
  
  GlobalWaveCanvasDataSamplingInterval = ((double)Buffer.hdr.data_size)/
    (double)GlobalWaveWindowWidth;
  
  heightOffset = GlobalWaveCanvasMidHeight;
  GlobalWaveCanvasPenColor("White");
  to = heightOffset;
  floatpt = 0.0;
  dataEnd = irint((double)Buffer.hdr.data_size - 
		  GlobalWaveCanvasDataSamplingInterval);
  
  if (FileReady)
  {
    soundfd = open(Buffer.filename, O_RDONLY);
    lseek(soundfd, Buffer.hdr_size, L_SET);
    
    for(i=0, pt=0; pt < dataEnd ; i++)
    {
      from = to;
      read(soundfd, &data, 1);
      to   = ((audio_u2c(data) * GlobalWaveScopeWidth) >> 7)
	+ heightOffset;
      XDrawLine(display, xid, gcGbWave, i-1, from, i, to);
      floatpt += GlobalWaveCanvasDataSamplingInterval;
      lseek(soundfd, irint(floatpt) - pt, L_INCR);
      pt = irint(floatpt);
    }
    close(soundfd);
  }
  else if (SoundBufferReady == TRUE)
  {
    for(i=0, pt=0 ; pt < dataEnd ; i++)
    {
      from = to;
      to   = ((audio_u2c(Buffer.data[pt]) * GlobalWaveScopeWidth) >> 7)
	+ heightOffset;
    
      XDrawLine(display, xid, gcGbWave, i-1, from, i, to);
    
      floatpt = floatpt + GlobalWaveCanvasDataSamplingInterval;
      pt = irint(floatpt);
    }
  }
  
  XDrawLine(display, xid, gcGbWave, 0, GlobalWaveCanvasMidHeight,
	    GlobalWaveCanvasWidth-1, GlobalWaveCanvasMidHeight);
  
  if (GlobalWaveCanvasRightMarkerSet)				    /*  HighLight marked area.                          */
  {
    rightWinX = irint(oldRightFloatGbWaveMarker/
		      GlobalWaveCanvasDataSamplingInterval);
    leftWinX  = irint(oldLeftFloatGbWaveMarker/
		      GlobalWaveCanvasDataSamplingInterval);
    ReverseGlobalWaveCanvasArea(leftWinX, rightWinX);
  }
  DisplayCanvasFrame();
}								    /* end function GlobalWaveCanvasRepaintHandler */


/*
 * Event callback function for `GlobalWaveCanvasDoneButton'.
 */
void
  GlobalWaveCanvasDoneHandler(item, event)
Panel_item	item;
Event		*event;
{
  xv_set(dtr_globalWaveCanvasPopUp->globalWaveCanvasPopUp, 
	 FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(dtr_globalWaveCanvasPopUp->globalWaveCanvasPopUp, 
	 XV_SHOW, FALSE, NULL);
  panel_default_handle_event(item, event);
}


/*
 * Notify callback function for `EditListScrollList'.
 */
int
DumpEntry(item, string, client_data, op, event)
     Panel_item	item;
     char		*string;
     Xv_opaque	client_data;
     Panel_list_op	op;
     Event		*event;
{
  switch(op)
  {
   case PANEL_LIST_OP_DESELECT:
    DeselectList();
    break;
    
   case PANEL_LIST_OP_SELECT:
    if (ActiveFlag & PLAY)
      StopPlay();
    if (ActiveFlag & RECORD)
      StopRecord();
    
    currentSelection = atoi(string) - 1;
    SetCurrentSelection(currentSelection, (MAESelection*)NULL);
    xv_set(dtr_editListPanelPopUp->deleteButton,
	   PANEL_INACTIVE, FALSE, NULL);			    /*  De-activate the Delete and Modify buttons.      */
    xv_set(dtr_editListPanelPopUp->modifyButton,
	   PANEL_INACTIVE, FALSE, NULL);
    xv_set(dtr_editListPanelPopUp->deleteAllButton,
	   PANEL_INACTIVE, FALSE, NULL);
    break;
    
   case PANEL_LIST_OP_VALIDATE:
   case PANEL_LIST_OP_DELETE:
    break;
  }
  return XV_OK;
}								    /* end function DumpEntry */


/*
 * Notify callback function for `LabelTextField'.
 */
Panel_setting
LabelTextFieldHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  return panel_text_notify(item, event);
}



void UpdateSelection()
{
  double	startingTimeInSec;
  double	endingTimeInSec;
  int		min;
  double	sec;
  char		proposedPath[MAXPATHLEN];
  
  if ((strcmp(currentSoundFile, "Untitled") != 0) || currentSoundFile == NULL)
  {
    realpath(currentSoundFile, proposedPath);
    xv_set(dtr_editListPanelPopUp->soundFileTextField,
	   PANEL_VALUE, proposedPath,
	   NULL);
  }
  else
  {
    xv_set(dtr_editListPanelPopUp->soundFileTextField,
	   PANEL_VALUE, "", NULL);
  }
  startingTimeInSec = Buffer.play.start / (double) DEVICE_SAMPLE_RATE;
  min = irint(floor(startingTimeInSec / 60.0));
  sec = startingTimeInSec - (60.0 * min);
  sprintf(msg, "%d", min);
  xv_set(dtr_editListPanelPopUp->startingTimeMinuteTextField,
	 PANEL_VALUE, msg, NULL);
  sprintf(msg, "%.2lf", sec);
  xv_set(dtr_editListPanelPopUp->startingTimeSecondTextField,
	 PANEL_VALUE, msg, NULL);
  endingTimeInSec   = Buffer.play.end   / (double) DEVICE_SAMPLE_RATE;
  min = irint(floor(endingTimeInSec / 60.0));
  sec = endingTimeInSec - (60.0 * min);
  sprintf(msg, "%d", min);
  xv_set(dtr_editListPanelPopUp->endingTimeMinuteTextField,
	 PANEL_VALUE, msg, NULL);
  sprintf(msg, "%.4lf", sec);
  xv_set(dtr_editListPanelPopUp->endingTimeSecondTextField,
	 PANEL_VALUE, msg, NULL);
}								    /* end function UpdateSelection */



/*
 * Notify callback function for `SetEditSelectionButton'.
 */
void
SetEditSelectionButtonHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  if (ActiveFlag & PLAY)
    StopPlay();
  if (ActiveFlag & RECORD)
    StopRecord();
  SetCurrentSelection(currentSelection, (MAESelection*)NULL);
}


/*
 * Notify callback function for `AddButton'.
 */
void
AddButtonHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  AddSelection();
}



/*
 * Notify callback function for `ModifyButton'.
 */
void
ModifyButtonHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  ModifySelection();
}


/*
 * Notify callback function for `DeleteButton'.
 */
void
DeleteButtonHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  DeleteSelection();
}


/*
 * Notify callback function for `DeleteAllButton'.
 */
void
DeleteAllButtonHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  DeleteAllSelections();
}


/*
 * Notify callback function for `PreviewEditButton'.
 */
void
PreviewEditButtonHandler(item, event)
	Panel_item	item;
	Event		*event;
{
  if (ActiveFlag & PLAY)
    return;
  if (ActiveFlag & RECORD)
    return;
  if (WaitFlag & PLAY)
  {
    AlertByNoticePrompt(dtr_editListPanelPopUp->editListUpperControlPanel,
			"Device is not ready.  Play Stopped");
    StopPlay();
    return;
  }
  if (!currentSelectionReadyToPlay)
    SetCurrentSelection(currentSelection, (MAESelection*)NULL);
  if ((FileReady == TRUE) || (SoundBufferReady == TRUE))
  {
    Play(dtr_mainWindow);
    SameSoundFile = TRUE;
  }
}								    /* end function PreviewEditButtonHandler */
