#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "pixlib.h"

char Usage[] =
"%s [--] height width [-rle] [-dump] [-tiles vt ht] [-title 'name']\n\
    [-xoff x] [-yoff y] [-vers v] [-alpha a] [-addr offset]\n\
    [-tilefile 'file'] [-format f]\n";

char *Example[] = {
    "\tMklffhdr writes a file consisting solely of a lucasfilm image file\n",
    "\theader and tile pointer table. The output header will be long enough\n",
    "\tto contain the header and table, rounded up to a multiple of\n",
    "\tLFF_BLOCKSIZE (1024) bytes. The height and width to put in the header\n",
    "\tmust be specified, the remaining parameters change the default header\n",
    "\tparameters and are:\n",
    "\t\t-rle  : Storage will be run-length encoded\n",
    "\t\t-dump : Storage will be dumped (default)\n",
    "\t\t-tiles: Tile size as specified (default 1 1)\n",
    "\t\t-title: Title as specified (default '')\n",
    "\t\t-xoff : X offset as specified (default 0)\n",
    "\t\t-yoff : Y offset as specified (default 0)\n",
    "\t\t-vers : Version as specified (default 0)\n",
    "\t\t-alpha: Alpha mode as specified (default 0)\n",
    "\t\t-addr : First tile pointer as specified (default end of header)\n",
    "\t\t-tilefile: Input file of tile pointers (& lengths) to be used\n",
    "\t\t-format: Format as specified (default FORMAT_RGB)\n",
    "\n\tExample: mklffhdr 2048 2048 -dump -title 'barstool' > barhdr\n",
    NullPtr(char)
};

main (argc, argv)
int	argc;
char   *argv[];
{
    int tileheight, tilewidth, ntiles, tilesize, needed, temp,
	i,
	height, width, numflag,
	rleflag, dumpflag, addrflag, tableflag,
	vtiles, htiles, xoff, yoff, version, alpha, address, format;
    char *title, *tilefile, *malloc();
    TILEPTR *tile_table;
    LFF_HDR hdr;

    rleflag = dumpflag = addrflag = tableflag = FALSE;
    numflag = 0;

    height = width = -1;
    vtiles = htiles = 1;
    version = LFF_VERSION;
    xoff = yoff = alpha = 0;
    title = "";
    format = FORMAT_RGB;

    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i],"--"))
	    userhelp(argv[0],Usage,Example);
	else if (!strcmp(argv[i],"-rle"))
	    rleflag = TRUE;
	else if (!strcmp(argv[i],"-dump"))
	    dumpflag = TRUE;
	else if (!strcmp(argv[i],"-tiles")) {
	    musthaveint(argv[++i],&vtiles,"Bad vertical tile #");
	    musthaveint(argv[++i],&htiles,"Bad horizontal tile #");
	} else if (!strcmp(argv[i],"-title"))
	    title = argv[i];
	else if (!strcmp(argv[i],"-xflag"))
	    musthaveint(argv[++i],&xoff,"Bad x offset");
	else if (!strcmp(argv[i],"-yflag"))
	    musthaveint(argv[++i],&yoff,"Bad y offset");
	else if (!strcmp(argv[i],"-vers"))
	    musthaveint(argv[++i],&version,"Bad version");
	else if (!strcmp(argv[i],"-alpha"))
	    musthaveint(argv[++i],&alpha,"Bad alpha");
	else if (!strcmp(argv[i],"-addr")) {
	    musthaveint(argv[++i],&address,"Bad address");
	    addrflag = TRUE;
	} else if (!strcmp(argv[i],"-tilefile")) {
	    tilefile = argv[++i];
	    tableflag = TRUE;
	} else if (!strcmp(argv[i],"-format")) {
	    musthaveint(argv[++i],&format,"Bad format");
	} else {
	    int parm;

	    if (!sscanf(argv[i],"%d",&parm))
		gr_bomb("Bad numeric height or width (%s)\n",argv[i]);
	    if (numflag == 0)
		height = parm, numflag++;
	    else if (numflag == 1)
		width = parm, numflag++;
	    else
		gr_error("Excess numeric parameter (%s), ignored\n",argv[i]);
	}
    }

    /* Verify some of the input parameters */
    if (height < 0 || width < 0)
	gr_bomb("Invalid or unspecified height (%d) or width (%d)\n",
	    height,width);
    if (rleflag && dumpflag)
	gr_bomb("Output cannot be both encoded and dumped\n");
    if (vtiles < 1 || htiles < 1)
	gr_bomb("Number of tiles must both be positive (vert = %d, horiz = %d)\n",
	    vtiles,htiles);
    if (strlen(title) > 245) {
	gr_error("Warning: title is too long, and will be chopped to 245 chars\n");
	title[245] = '\0';
    }

    tileheight = height / vtiles;
    tilewidth = width / htiles;
    if (tileheight * vtiles != height)
	gr_error("Warning: %d tiles does not go evenly into image height %d\n",
	    vtiles,height);
    if (tilewidth * htiles != width)
	gr_error("Warning: %d tiles does not go evenly into image width %d\n",
	    htiles,width);
    if (tileheight < 0 || tilewidth < 0)
	gr_error("Warning: negative tile height (%d) or width (%d)\n",
	    tileheight,tilewidth);

    hdr.hdr_magic = MAGIC_LFF;
    hdr.hdr_version = version;
    strcpy(hdr.hdr_label,title);
    hdr.hdr_labelptr = 0;
    hdr.hdr_height = height;
    hdr.hdr_width = width;
    hdr.hdr_tileheight = tileheight;
    hdr.hdr_tilewidth = tilewidth;
    hdr.hdr_format = format;
    hdr.hdr_storage = rleflag ? STORAGE_RLE : STORAGE_DUMP;
    hdr.hdr_blocking = LFF_BLOCKSIZE;
    hdr.hdr_alphamode = alpha;
    hdr.hdr_xoffset = xoff;
    hdr.hdr_yoffset = yoff;
    hdr.hdr_map_ptr = 0;	/* No color map */
    hdr.hdr_map_name[0] = '\0'; /* Same */

    /* Set up the tile pointer table. Figure the number of tiles we need,
	then round up to a multiple of the image blocksize (including
	the image header).
    */
    ntiles = vtiles * htiles;
    needed = ntiles * sizeof(TILEPTR);
    temp = needed + sizeof(LFF_HDR);
    if (temp % LFF_BLOCKSIZE) {
	temp += LFF_BLOCKSIZE;
	temp -= temp % LFF_BLOCKSIZE;
    }

    tile_table = (TILEPTR *)malloc(temp - sizeof(LFF_HDR));
    tilesize = tileheight * tilewidth * pixel_length(format);
    if (tableflag) {
	FILE *fp = fopen(tilefile,"r");
	char buf[80];
	int i;

	if (fp == NullPtr(FILE)) {
	    gr_error("Error opening tile file %s, no header generated\n",tilefile);
	    exit(-1);
	}

	/* Read each line and encode into table */
	for (i = 0; i < ntiles; i++) {
	    if (!fgets(buf,80,fp)) {
		gr_error("Warning: premature EOF in tile table file '%s', line %d\n",
		    tilefile,i+1);
		break;
	    }
	    sscanf(buf,"%ld %ld",
		&tile_table[i].tileptr,&tile_table[i].tilelength);
	}
    } else {
	int i;

	for (i = 0; i < ntiles; i++) {
	    tile_table[i].tileptr = temp + tilesize * i;
	    tile_table[i].tilelength = 0;
	}
    }

    /* If -addr was specified, it overrides input file or calculated offsets */
    if (addrflag) {
	tile_table[0].tileptr = address;
	tile_table[0].tilelength = 0;
    }

    hdr_to_vax(&hdr);
    tile_table_to_vax(tile_table,vtiles * htiles);

    if (fwrite(&hdr,sizeof(hdr),1,stdout) != 1 ||
	fwrite(tile_table,temp-sizeof(hdr),1,stdout) != 1) {
	gr_error("Failure writing header & tile table\n");
	exit(-1);
    } else
	gr_error("Wrote %d bytes of header and tile table\n",temp);
    exit(0);
}

musthaveint(strptr,iptr,message)
char *strptr;
int *iptr;
char *message;
{
    if (!sscanf(strptr,"%d",iptr))
	gr_error("%s (%s)\n",message,strptr);
}


