#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "pixlib.h"

char Usage[] =
"%s [--] input-file output-file\n";

char *Example[] = {
    "\tlfffilter filters an image by replacing each pixel with the median",
    "\tof itself and its 8 neighbors\n",
    NullPtr(char)
};

main (argc, argv)
int	argc;
char   *argv[];
{
    int  i, getpixfun(), status;
    char *lffname = NULL, *outname = NULL;
    RGBPIXEL *image, *filtered;
    LFF_HDR hdr;

    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i],"--"))
	    userhelp(argv[0],Usage,Example);
	else {
	    if (lffname == NULL)
		lffname = argv[i];
	    else if (outname == NULL)
		outname = argv[i];
	    else {
		fprintf(stderr,"Unknown switch %s\n",argv[i]);
		userhelp(argv[0],Usage,Example);
	    }
	}
    }

    if (lffname == NULL || outname == NULL)
	userhelp(argv[0], Usage, Example);

    /*	Open the image */
    status = open_lff(lffname,&hdr);
    if (status != LFF_OK)
	gr_bomb("Can't open '%s', open_lff returns: %s\n",lffname,pixerrmess(status));

    /*	Allocate memory for the image in core */
    i = (hdr.hdr_height + 2) * (hdr.hdr_width + 2) * sizeof(RGBPIXEL);
    image = (RGBPIXEL *)malloc(i);
    if (image == (RGBPIXEL *)NULL)
	gr_bomb("Failed to allocate input image buffer\n");

    /*	Allocate memory for the filtered image */
    filtered = (RGBPIXEL *)malloc(hdr.hdr_height * hdr.hdr_width * sizeof(RGBPIXEL));
    if (filtered == NULL)
	gr_bomb("Failed to allocate output image buffer\n");

    printf("\nReading input file %s\n",lffname);

    /*	Scan each tile in order and save in the incore buffer */
    {
	int j, vtiles,htiles;

	vtiles = hdr.hdr_height / hdr.hdr_tileheight;
	htiles = hdr.hdr_width / hdr.hdr_tilewidth;
	for (i = 0; i < vtiles; i++) {
	    int yoff = i * hdr.hdr_tileheight;

	    for (j = 0; j < htiles; j++) {
		int y, xoff = j * hdr.hdr_tilewidth;

		if (lff_seek_tile(i,j) != LFF_OK)
		    gr_bomb("Error reading tile %d, row %d, column %d\n",
			i*htiles+j,i,j);
		for (y = yoff; y < yoff+hdr.hdr_tileheight; y++) {
		    int newy;
		    /*
			This is a little unconventional:
			The image is imbedded in an array extended 1 pixel
			    on each side, so we need to offset
			    appropriately
		    */
		    RGBPIXEL *ptr = &image[(hdr.hdr_width+2)*(y+1) + xoff + 1];

		    status = lff_read_line(ptr,&newy);
		    if (status != LFF_OK)
			gr_bomb("lff_read_line(y=%d) returns: %s\n",
			    newy,pixerrmess(status));
		}
	    }
	}
    }

    /*	Filter the image */
    printf("Filtering image\n");
    filter_image(image, filtered, hdr.hdr_height, hdr.hdr_width);

    /*	Write the filtered image */
    printf("Writing output image\n");
    pixinit(filtered, hdr.hdr_height, hdr.hdr_width);
    status = write_lff(
	outname,
	hdr.hdr_height,
	hdr.hdr_width,
	hdr.hdr_height / hdr.hdr_tileheight,
	hdr.hdr_width  / hdr.hdr_tilewidth,
	hdr.hdr_format,
	hdr.hdr_storage,
	hdr.hdr_xoffset,
	hdr.hdr_yoffset,
	hdr.hdr_version,
	hdr.hdr_alphamode,
	getpixfun);
    if (status != LFF_OK)
	gr_bomb("Error writing output file %s: %s\n", outname, pixerrmess(status));
}

filter_image(image, filtered, height, width)
RGBPIXEL *image, *filtered;
int height, width;
{
    RGBPIXEL **temp, **in, **out;
    int row, col;

    /*	Allocate pointers to the input and output image scanlines */
    temp = (RGBPIXEL **)malloc((height + 2) * sizeof(RGBPIXEL *));
    in = temp + 1;
    out = (RGBPIXEL **)malloc(height * sizeof(RGBPIXEL *));

    if (temp == NULL || out == NULL)
	gr_bomb("Failed to allocate scanline pointers\n");

    /*	Set up pointers */
    in[-1] = &image[1];
    in[0] = &image[width + 2 + 1];
    out[0] = filtered;

    for (row = 1; row < height; row++) {
	in[row] = in[row - 1] + width + 2;
	out[row] = out[row - 1] + width;
    }

    in[height] = in[height - 1] + width + 2;

    /*	Replicate top and bottom scanlines */
    for (col = 0; col < width; col++) {
	in[-1][col] = in[0][col];
	in[height][col] = in[height - 1][col];
    }
    /*	Replicate left and right */
    for (row = 0; row < height; row++) {
	in[row][-1] = in[row][0];
	in[row][width] = in[row][width-1];
    }
    /*	Replicate corners diagonally */
    in[-1][-1]	      = in[0][0];
    in[-1][width]     = in[0][width-1];
    in[height][-1]    = in[height - 1][0];
    in[height][width] = in[height - 1][width - 1];

    for (row = 0; row < height; row++)
	for (col = 0; col < width; col++) {
	    register RGBPIXEL p;
	    int red[9], grn[9], blu[9];

	    /* Previous row */
	    p = in[row-1][col-1];
	    red[0] = GETRED(p); grn[0] = GETGRN(p); blu[0] = GETBLU(p);
	    p = in[row-1][col  ];
	    red[1] = GETRED(p); grn[1] = GETGRN(p); blu[1] = GETBLU(p);
	    p = in[row-1][col+1];
	    red[2] = GETRED(p); grn[2] = GETGRN(p); blu[2] = GETBLU(p);

	    /* This row */
	    p = in[row	][col-1];
	    red[3] = GETRED(p); grn[3] = GETGRN(p); blu[3] = GETBLU(p);
	    p = in[row	][col  ];
	    red[4] = GETRED(p); grn[4] = GETGRN(p); blu[4] = GETBLU(p);
	    p = in[row	][col+1];
	    red[5] = GETRED(p); grn[5] = GETGRN(p); blu[5] = GETBLU(p);

	    /* Next row */
	    p = in[row+1][col-1];
	    red[6] = GETRED(p); grn[6] = GETGRN(p); blu[6] = GETBLU(p);
	    p = in[row+1][col  ];
	    red[7] = GETRED(p); grn[7] = GETGRN(p); blu[7] = GETBLU(p);
	    p = in[row+1][col+1];
	    red[8] = GETRED(p); grn[8] = GETGRN(p); blu[8] = GETBLU(p);

	    out[row][col] =
		RGB8(order(5, 9, red-1),order(5, 9, grn-1),order(5, 9, blu-1));
	}
}

/*  Find k'th order statistic in array a[1..n] */
order(k, n, a)
int k, n;
register int *a;
{
    register int i, cut, base, size;
    int below[10],
	above[10],
	below_idx,  /* # of elements below cut */
	above_idx,  /* # of elements above cut */
	mid_count;  /* # of elements = cut */

    base = 1;	    /* Starting index of array */
    size = n;	    /* Length of array */

    while (size > 1) {
	cut = a[size / 2];

	/* Split array into upper and lower halves */
	below_idx = 1;	/* 1 + # of elements < cut */
	above_idx = 1;	/* 1 + # of elements > cut */
	mid_count = 0;	/* # of elements = cut */

	for (i = 1; i <= size; i++)
	    if (a[i] < cut)
		below[below_idx++] = a[i];
	    else if (a[i] > cut)
		above[above_idx++] = a[i];
	    else
		mid_count++;

	/*
	   Choose halve containg k'th order statistic and copy back into a
	    for next pass
	*/
	if (k < base + below_idx - 1) { /* stat is in below[] array */
	    /* base is unchanged */
	    size = below_idx - 1;
	    for (i = 1; i <= size; i++)
		a[i] = below[i];
	} else if (k < base + below_idx - 1 + mid_count)
	    return cut;
	else {
	    base = base + below_idx - 1 + mid_count;
	    size = above_idx - 1;
	    for (i = 1; i <= size; i++)
		a[i] = above[i];
	}
    }
    return a[1];
}



















static RGBPIXEL *Image;
static int Height, Width;

pixinit(image,height,width)
RGBPIXEL *image;
int height, width;
{
    Image = image;
    Height = height;
    Width = width;
}

/*
    Function called by write_lff, which returns the next line from the
	input lucasfilm file.
*/
int getpixfun(row,col,length,pixels)
int row, col, length;
RGBPIXEL pixels[];
{
    RGBPIXEL *pixsrc;

    pixsrc = &Image[row * Width + col];

    bcopy( (char *)pixsrc,(char *)pixels,length * sizeof(RGBPIXEL) );
    return length;
}

