/*
  Routines to write a .lff image onto a file, one scan line at a time.

  int write_lff(char *name,
		int   height,width,format,vtiles,htiles,
		      storage,xoff,yoff,version,alphamode,
		int (*pixfun)())

    The specified .lff file 'name' is  created.  It  will  contain  an
	image  of  size  height*width,	 and   additional   parameters
	specified as follows:
	    vtiles,htiles - number of tiles in vertical and horizontal
		directions. Must divide height and width integrally.
	    format  -  pixels  stored,	either	FORMAT_R,  FORMAT_RGB,
		FORMAT_RGBA, or FORMAT_BINARY (1 bit/pixel).
	    storage  -	encoding  method,   either   STORAGE_DUMP   or
		STORAGE_RLE (RLE is illegal for BINARY images).
	    xoff, yoff - offset into the image to restore the  picture
		at
	    version - version number, usually LFF_VERSION
	    alphamode - 0 = matted to black, 1 = unassociated
    In addition a pointer to a function which will return  the	pixels
	in the image must be passed as pixfun. This function  will  be
	called as follows:

	    pixfun(row,column,length,pixels)
	    int row, column, length;
	    RGBPIXEL pixels[];

    The function should put length pixels from the image  starting  at
	the specified row and column  (origin  in  upper-left  corner)
	into the pixels array. It should return the actual  number  of
	pixels put into the array. If this is not equal to the	passed
	length, a fatal error probably will have occured.  There  will
	be some provision for error recovery later.

    This convention is extended for binary images  as  follows:  if  a
	binary image is being created, pixfun() will  ask  for	length
	pixels as usual, but what is put into the array  must  be  the
	bit-mapped pixels of the row themselves (LSB to the left),  in
	the same format as they are stored in the image.

    If the image is successfully written, 0 (LFF_OK) will be returned;
	otherwise, one of the following error codes:

	    LFF_BADFILE     Can't open specified file
	    LFF_BADSIZE     Bad height or width (<= 0)
	    LFF_BADTILE     Bad number of vertical or horizontal tiles
	    LFF_BADFORMAT   Unknown pixel format
	    LFF_BADSTORAGE  Unknown storage code
	    LFF_READ_ERR    pixfun() failed at some point
	    LFF_WRITE_ERR   Write to file failed at some point

  int write_lff_fd(FD	 fd,
		   off_t offset,
		   int	 height,width,format,vtiles,htiles,
			 storage,xoff,yoff,version,alphamode,
		   int (*pixfun)())

    Like write_lff, but fd must be an already opened  file  descriptor
	onto which to write the image.	Offset	is  a  'virtual  image
	base' within the file; the image is written starting  at  that
	offset.
*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include "pixlib.h"
#include "lffbufio.h"

/* Globals */
int write_lff();
int write_lff_fd();
int pixel_length(); /* Return length of a specified pixel format */

/* Private functions */
static int checkparms();    /* Check header parameters */
static int write_lff_hdr(); /* Actually write header and file */
static int write_dump();    /* Routine to dump image */
static int pack_pixels();   /* Pack pixels into a character buffer */
static int write_rle();     /* Routine to run-length encode image */

/* These are all support functions for write_rle() */
static int runlen();
static int rle_encode();
static RPIXEL *do_run();
static void put_dump();
static void put_run();
/*static*/ void add_packet();
static void flush_block();

static FD Lff_out = BAD_FD;
static TILEPTR *Tile_table = NULL;  /* Tile position table */

int write_lff(name,height,width,vtiles,htiles,format,
	      storage,xoff,yoff,version,alphamode,pixfun)
char *name;
int height, width, vtiles, htiles, format, storage,
    xoff, yoff, version, alphamode;
int (*pixfun)();
{
    int status, tileheight, tilewidth;
    off_t pixel_offset;

    if (name == NullPtr(char))
	return LFF_BADFILE;

    status = checkparms(height, width, vtiles, htiles, format, storage,
			&tileheight, &tilewidth, &pixel_offset);
    if (status != LFF_OK)
	return status;

    Lff_out = _open_lff(name, LFF_WRITE, (off_t)0);
    if (Lff_out == BAD_FD)
	return LFF_BADFILE;

    return
	write_lff_hdr(name, pixel_offset,
		      height, width, vtiles, htiles,
		      tileheight, tilewidth, format,
		      storage, xoff, yoff, version, alphamode, pixfun);
}

int write_lff_fd(fd, offset, height, width, vtiles, htiles, format,
		 storage, xoff, yoff, version, alphamode, pixfun)
FD fd;
off_t offset;
int height, width, vtiles, htiles, format, storage,
    xoff, yoff, version, alphamode;
int (*pixfun)();
{
    int status, tileheight, tilewidth;
    off_t pixel_offset;

    status = checkparms(height, width, vtiles, htiles, format, storage,
			&tileheight, &tilewidth, &pixel_offset);
    if (status != LFF_OK)
	return status;

    Lff_out = _open_lff_fd(fd, LFF_WRITE, offset);
    if (Lff_out == BAD_FD)
	return LFF_BADFILE;

    return
	write_lff_hdr((char *)NULL, pixel_offset,
		      height, width, vtiles, htiles,
		      tileheight, tilewidth, format,
		      storage, xoff, yoff, version, alphamode, pixfun);
}

/*
    Once the lff file has been opened, actually write the header and
    image
*/
static int write_lff_hdr(name, pixel_offset,
			 height, width, vtiles, htiles,
			 tileheight, tilewidth, format,
			 storage, xoff, yoff, version, alphamode, pixfun)
char *name;
off_t pixel_offset; /* Where pixels begin after tile table */
int height, width, tileheight, tilewidth, format, storage, xoff, yoff,
    version, alphamode;
int (*pixfun)();
{
    LFF_HDR hdr;
    int pixsize, status;

    /* If we're making a BINARY  image,  figure  out  the  hdr_tilepad
	size; this is the number of bytes/scanline, and is the	number
	of bits (pixels)/scanline rounded up to a multiple of 8.
    */
    if (format == FORMAT_BINARY) {
	hdr.hdr_tilepad = tilewidth;
	if ((tilewidth % 8) != 0)
	    hdr.hdr_tilepad += 8 - (tilewidth % 8);
	hdr.hdr_tilepad /= 8;
    }

    /* Create the header block and dump it */
    hdr.hdr_magic   = MAGIC_LFF;
    hdr.hdr_version = version;
    if (name != NULL)
	strcpy(hdr.hdr_label,name);
    else
	hdr.hdr_label[0] = '\0';
    hdr.hdr_height  = height;
    hdr.hdr_tileheight = tileheight;
    hdr.hdr_width   = width;
    hdr.hdr_tilewidth = tilewidth;
    hdr.hdr_format  = format;
    hdr.hdr_storage = storage;
    hdr.hdr_blocking= LFF_BLOCKSIZE;
    hdr.hdr_alphamode = alphamode;
    hdr.hdr_xoffset = xoff;
    hdr.hdr_yoffset = yoff;
    hdr.hdr_map_ptr = 0;	/* No color map */
    hdr.hdr_map_name[0] = '\0'; /* Same */

    /* Seek to pixel area */
    _seek_lff(Lff_out, (off_t)pixel_offset, L_SET);
    pixsize = pixel_length(format);

    /* Write pixels */
    if (storage == STORAGE_DUMP)
	status = write_dump(&hdr,pixsize,pixfun,vtiles,htiles);
    else
	status = write_rle(&hdr,pixsize,pixfun,vtiles,htiles);

    if (status != LFF_OK) {
	_close_lff(Lff_out);
	goto done;
    }

    /* Now seek to start and write header, which should have tile pointer
	table filled in
    */
    _seek_lff(Lff_out, (off_t)0, L_SET);

    hdr_to_vax(&hdr);
    tile_table_to_vax(Tile_table,vtiles * htiles);

    if (_write_lff(Lff_out, &hdr, sizeof(hdr)) == -1 ||
	_write_lff(Lff_out, Tile_table, pixel_offset - sizeof(hdr)) == -1) {
	_close_lff(Lff_out);
	status = LFF_WRITE_ERR;
	goto done;
    } else
	_close_lff(Lff_out);
done:
    free(Tile_table);
    Tile_table = NULL;
    return status;
}


static int checkparms(height, width, vtiles, htiles, format, storage,
		      tileheight, tilewidth, pixel_offset)
int height, width, vtiles, htiles, format, storage;
int *tileheight, *tilewidth;
off_t *pixel_offset;
{
    int ntiles, needed;

    if (height <= 0 || width <= 0)
	return LFF_BADSIZE;

    *tileheight = height / vtiles;
    *tilewidth = width / htiles;
    if (*tileheight * vtiles != height || *tilewidth * htiles != width ||
	*tileheight < 0 || *tilewidth < 0)
	    return LFF_BADTILE;

    /* Set up the tile pointer table. Figure the number of tiles we need,
	then round up to a multiple of the image blocksize (including
	the image header).
    */
    ntiles = vtiles * htiles;
    needed = ntiles * sizeof(TILEPTR);
    *pixel_offset = needed + sizeof(LFF_HDR);
    if (*pixel_offset % LFF_BLOCKSIZE) {
	*pixel_offset += LFF_BLOCKSIZE;
	*pixel_offset -= *pixel_offset % LFF_BLOCKSIZE;
    }

    if (Tile_table != NULL)
	free(Tile_table);
    Tile_table = (TILEPTR *)malloc(*pixel_offset - sizeof(LFF_HDR));
    if (Tile_table == NULL)
	return LFF_ENOMEM;

    if (format != FORMAT_R &&
	format != FORMAT_RGB &&
	format != FORMAT_RGBA &&
	format != FORMAT_BINARY)
	    return LFF_BADFORMAT;
    if (storage != STORAGE_DUMP && storage != STORAGE_RLE)
	return LFF_BADSTORAGE;
    if (storage == STORAGE_RLE && format == FORMAT_BINARY)
	return LFF_BADSTORAGE;

    return LFF_OK;  /* All parameters are legal */
}

/*
    Dump the image (as specified by hdr block) onto file Lff_out.
    Return an error code which will be passed back up through
	write_lff().
*/
static int write_dump(hdr,pixsize,pixfun,vtiles,htiles)
LFF_HDR *hdr;
int (*pixfun)();
int vtiles, htiles; /* Number of tiles vertically and horizontally */
{
    int width,			    /* Line length in pixels */
	status = LFF_OK,	    /* Return code */
	i, j;			    /* Row and column # of tile being written */
    RGBPIXEL *buffer;		    /* Input scanline buffer */
    RPIXEL *encoded;		    /* Encoded output buffer */

    width = hdr->hdr_tilewidth;
    if ((buffer = (RGBPIXEL *)malloc(width * sizeof(RGBPIXEL))) == NULL ||
	(encoded = (RPIXEL *)malloc(width * sizeof(RGBPIXEL))) == NULL)
	return LFF_ENOMEM;

    for (i = 0; i < vtiles; i++) {	/* For each row of tiles */
	int yoff = i * hdr->hdr_tileheight + hdr->hdr_yoffset;

	for (j = 0; j < htiles; j++) {	/* For every tile in row */
	    int y,
		xoff = j * hdr->hdr_tilewidth + hdr->hdr_xoffset,
		tilenum = i * htiles + j;

	    /* Add (position,length) information to the header */
	    Tile_table[tilenum].tileptr = _tell_lff(Lff_out);
	    Tile_table[tilenum].tilelength = 0;   /* Should correct */

	    /* For each scanline in the tile */
	    for (y = 0; y < hdr->hdr_tileheight; y++) {
		int pixels;

		pixels = (*pixfun)(y+yoff,xoff,width,buffer);
		if (pixels != width) {
		    status = LFF_READ_ERR;
		    goto done;
		}

		if (hdr->hdr_format != FORMAT_BINARY) {
		    pack_pixels(buffer,encoded,pixsize,width);
		    if (_write_lff(Lff_out, encoded, width * pixsize) == -1) {
			status = LFF_WRITE_ERR;
			goto done;
		    }
		} else {    /* Just copy binary data directly to file */
		    if (_write_lff(Lff_out, (char *)buffer, hdr->hdr_tilepad) == -1) {
			status = LFF_WRITE_ERR;
			goto done;
		    }
		}
	    }
	}
    }
done:
    free(buffer);
    free(encoded);
    return status;
}

/*
    Pack  the  width  pixels  in  buffer[]  so	that  only  the  valid
	information is kept; for example, if format is	FORMAT_R,  the
	original buffer format (bytewise) is

	RxxxRxxxRxxx.....

	after packing, it will be

	RRRRRR.....

    The packed pixels are placed in the encoded[] character array.
    This does not handle binary data; it must be supplied as used.
*/
static int pack_pixels(buffer,encoded,pixsize,width)
register RGBPIXEL *buffer;
RPIXEL encoded[];
int pixsize, width;
{
    register RPIXEL *pixptr = encoded;
    register int i;
    register RGBPIXEL pix;

    switch (pixsize) {
	case 4:
	    for (i = 0; i < width; i++) {
		pix = *buffer++;

		*pixptr++ = GETRED(pix);
		*pixptr++ = GETGRN(pix);
		*pixptr++ = GETBLU(pix);
		*pixptr++ = GETALP(pix);
	    }
	    break;
	case 3:
	    for (i = 0; i < width; i++) {
		pix = *buffer++;

		*pixptr++ = GETRED(pix);
		*pixptr++ = GETGRN(pix);
		*pixptr++ = GETBLU(pix);
	    }
	    break;
	case 2:
	    for (i = 0; i < width; i++) {
		pix = *buffer++;

		*pixptr++ = GETRED(pix);
		*pixptr++ = GETGRN(pix);
	    }
	    break;
	case 1:
	    for (i = 0; i < width; i++)
		*pixptr++ = GETRED(*buffer++);
	    break;
	default:
	    return LFF_READ_ERR;    /* Should never get here */
    }
    return LFF_OK;
}

/*
    Return the length (in bytes) of the specified pixel format
*/
int pixel_length(format)
int format;
{
    int count = 0;

    if (format & FORMAT_R)
	count++;
    if (format & FORMAT_G)
	count++;
    if (format & FORMAT_B)
	count++;
    if (format & FORMAT_A)
	count++;
    return count;
}

/*
    Run-length encode the image (as specified by hdr block) onto file Lff_out.
    Return an error code which will be passed back up through
	write_lff().
*/
static RPIXEL Outblock[LFF_BLOCKSIZE],	/* Block to encode into */
	     *Blkptr;			/* Pointer into the block */
static int    Bytesleft,		/* Bytes left in block */
	      Pixsize;			/* Bytes/pixel */

static int write_rle(hdr,pixsize,pixfun,vtiles,htiles)
LFF_HDR *hdr;
int pixsize, (*pixfun)();
int vtiles, htiles; /* Number of tiles vertically and horizontally */
{
    int width,			    /* Line length in pixels */
	status = LFF_OK,	    /* Return code */
	i, j;			    /* Row and column # of tile being written */
    RGBPIXEL *buffer;		    /* Input scanline buffer */
    RPIXEL *encoded;		    /* Encoded output buffer */

    Pixsize = pixsize;
    width = hdr->hdr_tilewidth;
    if ((buffer = (RGBPIXEL *)malloc(width * sizeof(RGBPIXEL))) == NULL ||
	(encoded = (RPIXEL *)malloc(2 * width * sizeof(RGBPIXEL))) == NULL)
	return LFF_ENOMEM;

    for (i = 0; i < vtiles; i++) {	/* For each row of tiles */
	int yoff = i * hdr->hdr_tileheight + hdr->hdr_yoffset;

	for (j = 0; j < htiles; j++) {	/* For every tile in row */
	    int xoff = j * hdr->hdr_tilewidth + hdr->hdr_xoffset,
		tilenum = i * htiles + j;

	    /* Add (position,length) information to the header */
	    Tile_table[tilenum].tileptr = _tell_lff(Lff_out);
	    Tile_table[tilenum].tilelength = 0;   /* Should correct */

	    runlen(yoff,xoff,width,hdr->hdr_tileheight,
		   buffer,encoded,pixfun);
	    if (status != LFF_OK)
		goto done;
	}
    }
done:
    free(buffer);
    free(encoded);
    return status;
}

/*
    runlen(row,col,width,height,buffer,encoded,pixfun
	Run-length encode the tile of pixels with  upper  left	corner
	(row,col) of size (height,width) onto file Lff_out.  Return  an
	indication of success or failure. Buffer is used  to  read  in
	the unpacked (32-bit)  pixels,	and  encoded  to  collect  the
	encoded run-lengths. The function used to get pixels from each
	scanline is pixfun().
*/
static int runlen(row,col,width,height,buffer,encoded,pixfun)
int row, col, width, height;
RGBPIXEL *buffer;
RPIXEL *encoded;
int (*pixfun)();
{
    int y, dump_length, runs;

    Blkptr = Outblock;
    Bytesleft = LFF_BLOCKSIZE;
    dump_length = width * Pixsize;

    /* For each scanline in the tile */
    for (y = 0; y < height; y++) {
	int pixels, size;

	pixels = (*pixfun)(y+row,col,width,buffer);
	size = rle_encode(buffer,encoded,width,&runs);

	if (size > dump_length) {	/* A dumped packet */
	    (void)pack_pixels(buffer,encoded,Pixsize,width);
	    put_dump(encoded,width,dump_length);
	} else				/* An encoded packet */
	    put_run(encoded,runs,runs * (1 + Pixsize));
    }

    /* Flush the final block if needed */
    if (Bytesleft != LFF_BLOCKSIZE)
	flush_block();
    return LFF_OK;
}

/*
    rle_encode(buffer,encoded,length,runs)
	Run-length encoded the length  pixels  in  buffer[]  into  the
	array encoded[]. Return the byte length of the encoded[] array
	and place the number of runs generated into runs.
*/
static int rle_encode(buffer,encoded,length,runs)
register RGBPIXEL *buffer;
RPIXEL *encoded;
int length;
register int *runs;
{
    register RPIXEL count,  /* Run-length count */
		   *bufptr; /* Run count & pointer into encoded line */
    register
	RGBPIXEL lastpixel; /* Last pixel read */
    register int i;	    /* Index into buffer[] */

    lastpixel = *buffer++;  /* Set up for encoding - first pixel */
    *runs = 0;		    /* No runs so far */
    count = 0;		    /* 1 pixel in first run */
    bufptr = encoded;	    /* Start encoding at beginning of buffer */

#ifdef DEBUG
    fprintf(stderr, "rle_encode(length = %d)\n", length );
#endif

    for (i = 1; i < length; i++,buffer++) {
	/* If the pixel value changes, or 256 pixels have been
	    encoded, place a count-value run in the buffer
	*/
	if (*buffer != lastpixel) {
#ifdef DEBUG
	    fprintf(stderr, "    do_run(count = %3d, value = 0x%08x) [changed]\n",count,lastpixel );
#endif
	    bufptr = do_run(bufptr,count,lastpixel);
	    count = 0;
	    (*runs)++;
	    lastpixel = *buffer;
	} else
	    count++;

	if (count == 255 || i == length - 1) {
#ifdef DEBUG
	    fprintf(stderr, "    do_run(count = %3d, value = 0x%08x) [overrun, i = %d]\n",
		 count,lastpixel,i );
#endif
	    bufptr = do_run(bufptr,count,lastpixel);
	    (*runs)++;
	    lastpixel = *++buffer;
	    i++;
	    count = 0;
	}
    }

    /* If the loop exits with i == length, there was a max length +  1
	(257) pixels run at the end of	the  line,  requiring  another
	packet.
    */
    if (i == length) {
#ifdef DEBUG
	fprintf(stderr, "    do_run(count = %3d, value = 0x%08x) [end of rle_encode, i = %d]\n",
	    count,lastpixel,i );
#endif
	bufptr = do_run(bufptr,count,lastpixel);
	(*runs)++;
    }

#ifdef DEBUG
    fprintf(stderr, "    rle_encode returns %d bytes, %d runs\n", bufptr-encoded, *runs );
#endif
    return bufptr - encoded;
}

/*
    do_run(bufptr,count,pixel)
	Encode a run-length [count,pixel] combination into the	passed
	buffer, returning a new pointer to the buffer.

	In the Lucasfilm format, a run has the following order in memory:
	    count R [G B [A]]
*/
static RPIXEL *do_run(bufptr,count,pixel)
RPIXEL *bufptr, count;
RGBPIXEL pixel;
{
    *bufptr++ = count;

    switch (Pixsize) {
	case 4: *bufptr++ = GETRED(pixel);
		*bufptr++ = GETGRN(pixel);
		*bufptr++ = GETBLU(pixel);
		*bufptr++ = GETALP(pixel);
		break;
	case 3: *bufptr++ = GETRED(pixel);
		*bufptr++ = GETGRN(pixel);
		*bufptr++ = GETBLU(pixel);
		break;
	case 2: *bufptr++ = GETRED(pixel);
		*bufptr++ = GETGRN(pixel);
		break;
	case 1: *bufptr++ = GETRED(pixel);
		break;
    }

    return bufptr;
}

/*
    put_dump(buffer,width,dump_length)
	Write a dumped packet to Lff_out.  Buffer[] contains the dumped
	pixels,  width	is  the  number  of  pixels   contained,   and
	dump_length is the length of the array in bytes.
*/
static void put_dump(buffer,width,dump_length)
RPIXEL *buffer;
int width, dump_length;
{
    int needed,     /* Space needed for dump */
	avail,	    /* Space available */
	plength,    /* Space in pixels */
	blength;    /* Space in bytes */

    /* Compute space needed for the dump + header + end-block packet */
    needed = 2 * PACKET_SIZE + dump_length;

    /* If there's not enough space, split the dump into multiple parts */
    while (Bytesleft < needed) {
	avail = Bytesleft - 2 * PACKET_SIZE;
	plength = avail / Pixsize;
	blength = plength * Pixsize;

	if (plength > 0) {
	    add_packet(plength,blength,FLAG_DUMP,buffer);
	    buffer += blength;	    /* Increment dump pointer */
	    dump_length -= blength; /* Decrement dump length */
	    width -= plength;	    /* ... and pixel count */
	    needed -= blength;	    /* ... and bytes needed */
	}
	flush_block();
    }

    /* There is enough space, so write the remainder of the packet */
    add_packet(width,dump_length,FLAG_DUMP,buffer);
}

/*
    put_run(encoded,width,run_length)
	Write an encoded packet to  Lff_out.   Encoded[]  contains  the
	runs, width is the number  of  runs,  and  run_length  is  the
	length of the array in bytes.
*/
static void put_run(encoded,width,run_length)
RPIXEL *encoded;
int width, run_length;
{
    int needed,     /* Space needed for run */
	avail,	    /* Space available */
	plength,    /* Length available in runs */
	blength;    /* Length available in bytes */

    /* Compute space needed for the run + header + end-block packet */
    needed = 2 * PACKET_SIZE + run_length;

    /* If there's not enough space, split the runs */
    while (Bytesleft < needed) {
	avail = Bytesleft - 2 * PACKET_SIZE;
	plength = avail / (Pixsize + 1);
	blength = plength * (Pixsize + 1);

	if (plength > 0) {
	    add_packet(plength,blength,FLAG_RLE,encoded);
	    encoded += blength;     /* Increment run pointer */
	    run_length -= blength;  /* Decrement packet length */
	    width -= plength;	    /* ... and run count */
	    needed -= blength;	    /* ... and bytes needed */
	}
	flush_block();
    }

    /* There is enough space, so write the remainder of the packet */
    add_packet(width,run_length,FLAG_RLE,encoded);
}

/*
    add_packet(count,bytelength,flag,pixels)
	Add a new packet in the current block containing count	pixels
	of total length bytelength. The pixel data is in pixels[]  and
	the packet flag is as  specified.  Updates  internal  encoding
	block pointers as needed.
*/
/*static*/ void add_packet(count,bytelength,flag,pixels)
int count,
    bytelength,
    flag;
RPIXEL *pixels;
{
    PACKET_HDR packet;

    packet.flag = flag;
    packet.count = count - 1;

    packet_to_vax(&packet);

    *(PACKET_HDR *)Blkptr = packet;
    Blkptr += PACKET_SIZE;

    bcopy(pixels,Blkptr,bytelength);
    Blkptr += bytelength;

    Bytesleft -= (PACKET_SIZE + bytelength);
}

/*
    flush_block()
	Flush  the  internal  encoding	block  to   disk   (adding   a
	end-of-block packet) and reset pointers for the next block.
*/
static void flush_block()
{
    static PACKET_HDR end_of_block = { 0, FLAG_PAD };
    static int hdr_converted = FALSE;

    if (!hdr_converted) {
	packet_to_vax(&end_of_block);
	hdr_converted = TRUE;
    }

    *(PACKET_HDR *)Blkptr = end_of_block;
    if (_write_lff(Lff_out, Outblock, LFF_BLOCKSIZE) == -1)
	gr_error("I/O ERROR: _write_lff failed in flush_block\n");

    Blkptr = Outblock;
    Bytesleft = LFF_BLOCKSIZE;
}
