 /*
  * Khoros: $Id: graphics_3D.c,v 1.1 1991/05/10 15:56:56 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: graphics_3D.c,v 1.1 1991/05/10 15:56:56 khoros Exp $";
#endif

 /*
  * $Log: graphics_3D.c,v $
 * Revision 1.1  1991/05/10  15:56:56  khoros
 * Initial revision
 *
  */ 


/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 *
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "X3D.h"	


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	    file name: graphics_2D.c
   >>>>               
   >>>>   description: 3D Viewing System Control Utilities
   >>>>              
   >>>>      routines:
   >>>>               _X3D_get_graphics()		      <<<<              
   >>>>               _X3D_update_graphics()		      <<<<              
   >>>>                                                       <<<<
   >>>>               X3D_init_graphics()	      	      <<<<              
   >>>>               X3D_set_projection()	      	      <<<<              
   >>>>               X3D_close_graphics()	      	      <<<<              
   >>>>               X3D_set_viewpoint()	      	      <<<<              
   >>>>               X3D_set_view_distance()	      	      <<<<              
   >>>>               X3D_set_window()	    	   	      <<<<              
   >>>>               X3D_set_viewport()    	   	      <<<<              
   >>>>               X3D_set_wc_scale()	      	      <<<<              
   >>>>               X3D_set_line_type()	      	      <<<<              
   >>>>               X3D_set_line_width()	      	      <<<<              
   >>>>                                                       <<<<
   >>>> modifications:
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


static X3DList *graphics_head = NULL;

/************************************************************
*
*  MODULE NAME: _X3D_get_graphics
*
*      PURPOSE: Returns the X3D structure associated with a
*		particular graphics ID
*
*        INPUT: id - xvgraphics ID
*
*       OUTPUT: returns the X3D graphics structure associated with that id
*
*    CALLED BY: many routines
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

X3DGraphics *_X3D_get_graphics (id)

int	id;
{
	X3DList		*current;

	/*
	 *  Search through the linked graphics list for the desired graphics
	 *  id.
	 */
	if (graphics_head == NULL)
	   return(NULL);
	else if (graphics_head->id == id)
	   return(graphics_head->graphics);

	current = graphics_head;
	while (current->next != NULL && current->id != id)
	      current = current->next;

	if (current->id == id)
	{
	   /*
	    *  Stick the current graphics structure at the top of the list, 
	    *  since it will probably be used again soon.
	    */
	   current->prev->next = current->next;
	   if (current->next != NULL)
	   {
	      current->next->prev = current->prev;
	   }

	   current->next = graphics_head;
	   current->prev = NULL;

	   graphics_head->prev = current;
	   graphics_head = current;

	   return(current->graphics);
	}
	else
	   return(NULL);
}



/************************************************************
*
*  MODULE NAME: _X3D_update_graphics
*
*      PURPOSE: Recomputes the X3D Grapichs Matrix, used to
*		transform 3D world coordinates into device
*		coordinates.
*
*        INPUT: graphics - X3D graphics structure
*
*       OUTPUT: none
*
*    CALLED BY: X3D_set_viewpoint(), X3D_set_view_distance,
*		X3D_set_window(), X3D_set_viewport(), X3D_set_wc_min_max()
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

_X3D_update_graphics (graphics)

X3DGraphics	*graphics;
{
	Real	center_x, center_y, center_z, 
		xmin, xmax, ymin, ymax, zmin, dummy, w;
	Matrix	matrix1, matrix2;

	/*
	 *  Oh where is my maxima package.....
	 *
	 *  Mulitiply the matrix mess together to get the final
	 *  transform matrix.  In the future i am planning on
	 *  using maxima to get ONE final matrix equation, but since
	 *  the matrix operations are still not finalized i will
	 *  wait.
	 */
/*
 * Convert to unit cube.
 */
	_X3D_matrix_set_ndc(graphics->xv_min, graphics->xv_max,
			    graphics->yv_min, graphics->yv_max,
			    graphics->wc_min, graphics->wc_max, matrix1);

	center_x = center_y = center_z = 0.5;
	X3D_matrix_set_translate(-center_x, -center_y, -center_z, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);

	_X3D_matrix_set_viewing(graphics->theta, graphics->alpha,
			       graphics->eye_dist, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);


	/*
	 * Foley and Van Dam Style
	 */
	if (graphics->projection == Perspective)
	{
	   /*
	    *  Set the Perspective transformation
	    */
	   _X3D_matrix_set_perspective(graphics->view_dist, matrix2);
	   _X3D_matrix_mult(matrix1, matrix2, matrix1);
	}
	else
	{
	   /*
	    *  Set the Parallel transformation
	    */
	   _X3D_matrix_set_parallel(graphics->projection, matrix2);
	   _X3D_matrix_mult(matrix1, matrix2, matrix1);
	}

	/*
	 *  Set the translation to the origin
	 */
	X3D_matrix_set_translate(0.5, 0.5, 0.5, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);


	/*
	 *  Set the scale about the origin
	 */
	xmin = (graphics->xv_max - graphics->xv_min)/1.0;
	ymin = (graphics->yv_max - graphics->yv_min)/1.0;
	zmin = (graphics->yv_max - graphics->yv_min)/1.0;
	X3D_matrix_set_scale(xmin, ymin, zmin, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);

	X3D_matrix_set_translate(graphics->xv_min, graphics->yv_min, 0.0, 
				 matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);

	/*
	 *  Set the device coordinate transformation matrix
	 */
	_X3D_matrix_set_dc(graphics, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, graphics->matrix);

	/*
	 *  Update the 2D part of the graphics structure.  This is really
	 *  bad, but it  is the only way to allow the user to switch 
	 *  between drawing 3D and 2D graphics on the same id.
	 */
	_X2D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_init_graphics
*
*      PURPOSE: Gets the graphics ID number from the user, and
*	 	creates the X3DGraphics structure that will be 
*		associated with it, for future reference.
*
*        INPUT: 1) id		- xvgraphics ID
*		2) win_type	- Type of window (2D or 3D)
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

X3D_init_graphics (id, win_type)

int	id, win_type;
{
	X3DGraphics	*graphics;
	X3DList		*entry;

	entry = (X3DList *) malloc(sizeof(X3DList));
	if (entry == NULL)
	   return;

	graphics = (X3DGraphics *) calloc(1,sizeof(X3DGraphics));
	if (graphics == NULL)
	   return;

	entry->id	= id;
	entry->graphics	= graphics;
	entry->next	= graphics_head;
	entry->prev	= NULL;

	if (graphics_head != NULL)
	   graphics_head->prev	= entry;

	graphics_head = entry;

	graphics->id 	     = id;
	graphics->win_type   = win_type;
	graphics->device     = X11;
	graphics->line_type  = Solid;
	graphics->line_width = 0;
	graphics->fg	=
	graphics->bg	= ~0;

	/*
	 *  Set some defaults.  We really should not have to, but just to
	 *  be on the safe side.
	 */
	graphics->projection = Perspective;
	graphics->eye_dist = 1.0;
	graphics->view_dist = 0.5;
	graphics->alpha = 0.0;
	graphics->theta = 0.0;

	/*
	 *  Set scale type for the World Coordinates to be linear.
	 */
	graphics->scale.x = LINEAR;
	graphics->scale.y = LINEAR;
	graphics->scale.z = LINEAR;

	/*
	 *  Set the View plane normal to be down the Z axis.
	 */
	graphics->vpn.x = 0.0;
	graphics->vpn.y = 0.0;
	graphics->vpn.z = -1.0;

	/*
	 *  Set the graphics device coordinates to be a 100x100 window.
	 */
	graphics->X11_xmin = graphics->X11_ymin = 0;
	graphics->X11_xmax = graphics->X11_ymax = 100;

	graphics->POS_xmin = graphics->POS_ymin = 0;
	graphics->POS_xmax = 3247;
	graphics->POS_ymax = 5018;

	graphics->IMP_xmin = graphics->IMP_ymin = 0;
	graphics->IMP_xmax = graphics->IMP_ymax = 2000;

	/*
	 *  Set the graphics output for monochrome and output inversion
	 */
	graphics->POS_mono   = TRUE;
	graphics->POS_invert = TRUE;

	/*
	 *  Set the viewport minimums and maximums to be the entire
	 *  viewport.
	 */
	graphics->xv_min = graphics->yv_min = graphics->zv_min = 0.0;
	graphics->xv_max = graphics->yv_max = graphics->zv_max = 1.0;

	/*
	 *  Set the world coordinate minimums and maximums to be 
	 *  between 0.0 and 1.0
	 */
	graphics->wc_min.x = graphics->wc_min.y = graphics->wc_min.z = 0.0;
	graphics->wc_max.x = graphics->wc_max.y = graphics->wc_max.z = 1.0;
	graphics->smin = graphics->wc_min;
	graphics->smax = graphics->wc_max;

	/*
	 *  Set the center of object, the point of focus, to be
	 *  (0,0,0).
	 */
	graphics->center.x = graphics->center.y = graphics->center.z = 0.0;

	/*
	 *  Set the text defaults.
	 */
	graphics->orientation = LeftHanded;
	graphics->justify     = CenterJustify;
	graphics->spacing     = 0.0;
	graphics->aspect      = 0.75;

	/*
	 *  Set the default font to be the Default X3D font.
	 */
	X3D_set_font(id, DefaultFont);
}


/************************************************************
*
*  MODULE NAME: X3D_set_projection
*
*      PURPOSE: Sets the projection type
*
*        INPUT: 1) id           - X3D graphics structure ID
*               2) projection   - Type of projection:
*					Perspective     1
*					Orthographic    2
*					Cavalier        3
*					Cabinet         4
*					
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

X3D_set_projection (id, projection)

int     id, projection;
{
        X3DGraphics *graphics;

        if (!(graphics = _X3D_get_graphics(id)))
        {
           (void) fprintf (stderr,"X3D_set_projection:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
           return;
        }


	switch (projection)
	{
	   case	Perspective:	graphics->projection = Perspective;
				break;

	   case	Orthographic:	graphics->projection = Orthographic;
				break;

	   case	Cavalier:	graphics->projection = Cavalier;
				break;

	   case	Cabinet:	graphics->projection = Cabinet;
				break;

	   	default:	(void) fprintf (stderr,"X3D_set_projection:");
           			(void) fprintf (stderr,"\t unknown projection");				(void) fprintf (stderr," type %d\n",projection);
				break;
	}
	_X3D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_close_graphics
*
*      PURPOSE: Deletes the X3D graphics structure.
*
*        INPUT: id - X3D graphics structure ID
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

X3D_close_graphics (id)

int	id;
{
	X3DList		*current;

	/*
	 *  Search through the linked graphics list for the desired graphics
	 *  id.
	 */
	if (graphics_head == NULL)
	   return;

	current = graphics_head;
	while (current->next != NULL && current->id != id)
	      current = current->next;

	if (current->id == id)
	{
	   /*
	    */
	   if (current->next == NULL && current->prev == NULL)
	   {
	      graphics_head = NULL;
	   }
	   else if (current->prev == NULL)
	   {
	      current->next->prev = NULL;
	      graphics_head = current->next;
	   }
	   else if (current->next == NULL)
	   {
	      current->prev->next = NULL;
	   }
	   else
	   {
	      current->next->prev = current->prev;
	      current->prev->next = current->next;
	   }
	   free(current->graphics);
	   free(current);
	}
}




/************************************************************
*
*  MODULE NAME: X3D_set_viewpoint
*
*      PURPOSE: Sets the eye orientation for a given X3D graphics
*		structure.
*
*        INPUT: id        - X3D graphics structure ID
*		alpha     - the angle describing the position of the
*			    eye between the Z axis and the XY plane
*		theta     - Angle describing the position of the eye
*			    around Z axis with respect to the XY plane
*		eye_dist  - Distance of the eye from the object
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
************************************************************/

X3D_set_viewpoint (id, alpha, theta, eye_distance)

int	id;
Real	alpha, theta, eye_distance;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_viewpoint:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
	graphics->alpha = alpha;
	graphics->theta = theta;
	graphics->eye_dist = eye_distance;
	_X3D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_set_view_distance
*
*      PURPOSE: Sets the eye orientation for a given X3D graphics
*               structure.
*
*        INPUT: id        - X3D graphics structure ID
*               view_dist - Distance of the eye from the viewport
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
************************************************************/

X3D_set_view_distance (id, view_distance)

int     id;
Real    view_distance;
{
        X3DGraphics *graphics;

        if (!(graphics = _X3D_get_graphics(id)))
        {
           (void) fprintf (stderr,"X3D_set_view_distance:");
           (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
           return;
        }
        graphics->view_dist = view_distance;
        _X3D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_set_window
*
*      PURPOSE: Sets the minimum and maximum device coordinate
*		for a given X11 Window.
*
*        INPUT: id        - X3D graphics structure ID
*		x, y,
*		width,
*		height	  - the offset and size of the window (in device
*			    coordinates).
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
************************************************************/

X3D_set_window (id, x, y, width, height)

int	id;
short	x, y, width, height;
{
	X3DGraphics 	*graphics;


	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_window:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}

	if (graphics->device == X11)
	{
	   graphics->X11_xmin = x;
	   graphics->X11_ymin = y;
	   graphics->X11_xmax = width  + x;
	   graphics->X11_ymax = height + y;
	}
	else if (graphics->device == POSTSCR)
	{
	   graphics->POS_xmin = x;
	   graphics->POS_ymin = y;
	   graphics->POS_xmax = width  + x;
	   graphics->POS_ymax = height + y;
	}
	else if (graphics->device == IMPRESS)
	{
	   graphics->IMP_xmin = x;
	   graphics->IMP_ymin = y;
	   graphics->IMP_xmax = width  + x;
	   graphics->IMP_ymax = height + y;
	}
	else if (graphics->device == HPGL)
	{
	   graphics->HPGL_xmin = x;
	   graphics->HPGL_ymin = y;
	   graphics->HPGL_xmax = width  + x;
	   graphics->HPGL_ymax = height + y;
	}
	_X3D_update_graphics(graphics);
}




/************************************************************
*
*  MODULE NAME: X3D_set_viewport
*
*      PURPOSE: Sets the minimum and maximum viewport values
*		for a given X3D graphics structure.
*
*        INPUT: id     - X3D graphics structure ID
*		xmin   - A float containing the x minimum viewport 
*			 normalized device coordinate value.
*		xmax   - A float containing the x maximum world 
*			 normalized device coordinate value.
*		ymin   - A float containing the y minimum viewport 
*			 normalized device coordinate value.
*		ymax   - A float containing the y maximum world 
*			 normalized device coordinate value.
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
************************************************************/

X3D_set_viewport (id, xmin, xmax, ymin, ymax, zmin, zmax)

int	id;
float	xmin, xmax, ymin, ymax, zmin, zmax;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_viewport:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
	graphics->xv_min = xmin;
	graphics->xv_max = xmax;
	graphics->yv_min = ymin;
	graphics->yv_max = ymax;
	graphics->zv_min = zmin;
	graphics->zv_max = zmax;
	if (graphics->xv_min == graphics->xv_max)
	{
	   graphics->xv_min -= 0.5;
	   graphics->xv_max += 0.5;
	}
	if (graphics->yv_min == graphics->yv_max)
	{
	   graphics->yv_min -= 0.5;
	   graphics->yv_max += 0.5;
	}
	if (graphics->zv_min == graphics->zv_max)
	{
	   graphics->zv_min -= 0.5;
	   graphics->zv_max += 0.5;
	}
	_X3D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_set_wc_min_max
*
*      PURPOSE: Sets the minimum and maximum world coordinate
*		points for a given X3D graphics structure.
*
*        INPUT: id     - X3D graphics structure ID
*		wc_min - A coord containing the minimum world 
*			 coordinate values
*		wc_max - A coord containing the maximum world 
*			 coordinate values
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

X3D_set_wc_min_max (id, wc_min, wc_max)

int	id;
Coord	wc_min, wc_max;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_wc_min_max:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}

	if (wc_min.x == wc_max.x)
	{
	   wc_min.x -= 0.5;
	   wc_max.x += 0.5;
	}
	if (wc_min.y == wc_max.y)
	{
	   wc_min.y -= 0.5;
	   wc_max.y += 0.5;
	}
	if (wc_min.z == wc_max.z)
	{
	   wc_min.z -= 0.5;
	   wc_max.z += 0.5;
	}

	graphics->smin = wc_min;
	graphics->smax = wc_max;

	if (!(_X3D_scale_wc_coord(graphics->scale, wc_min, &graphics->wc_min)))
	   graphics->wc_min = wc_min;

	if (!(_X3D_scale_wc_coord(graphics->scale, wc_max, &graphics->wc_max)))
	   graphics->wc_max = wc_max;

	_X3D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_set_wc_scale
*
*      PURPOSE: Sets the minimum and maximum world coordinate
*		points for a given X3D graphics structure.
*
*        INPUT: id     - X3D graphics structure ID
*		scale - A coord containing the maximum world 
*			 coordinate scale types for the x, y,
*			 & z axes.
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young & Mike Lang
*
*
*************************************************************/

X3D_set_wc_scale(id, scale)

int	id;
Coord	scale;
{
	X3DGraphics *graphics;
	Coord wc_min, wc_max, smin, smax;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_wc_scale:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}

	wc_min = graphics->wc_min;
	wc_max = graphics->wc_max;
        smin   = graphics->smin;
        smax   = graphics->smax;

	graphics->scale = scale;
	if (!(_X3D_scale_wc_coord(graphics->scale, smin, &graphics->wc_min)))
	   graphics->wc_min = wc_min;

	if (!(_X3D_scale_wc_coord(graphics->scale, smax, &graphics->wc_max)))
	   graphics->wc_max = wc_max;

	_X3D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X3D_set_line_type
*
*      PURPOSE: Sets the line type for a given graphics context
*
*        INPUT: id        - X3D graphics structure ID
*		line_type - The new line type.
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young & Tom Sauer
*
*
*************************************************************/

X3D_set_line_type(id, line_type)

int	id, line_type;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_line_type:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}

        X3D_draw[graphics->device].line_type(graphics, line_type);
}


/************************************************************
*
*  MODULE NAME: X3D_set_line_width
*
*      PURPOSE: Sets the line width for a given graphics context
*
*        INPUT: id        - X3D graphics structure ID
*		line_width - The new line width.
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young & Tom Sauer
*
*
*************************************************************/

X3D_set_line_width(id, line_width)

int	id; 
int     line_width;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_line_width:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
        X3D_draw[graphics->device].line_width(graphics, line_width);
}



/************************************************************
*
*  MODULE NAME: X3D_set_draw
*
*      PURPOSE: Sets the foreground draw color
*
*        INPUT: id - X3D graphics structure ID
*		fg - The new foreground color
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young & Tom Sauer
*
*
*************************************************************/

X3D_set_draw(id, fg)

int	id;
XColor  *fg;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_draw:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
        X3D_draw[graphics->device].draw_color(graphics, fg);
}



/************************************************************
*
*  MODULE NAME: X3D_set_fill
*
*      PURPOSE: Sets the background fill color
*
*        INPUT: id - X3D graphics structure ID
*		bg - The new background color
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young & Tom Sauer
*
*
*************************************************************/

X3D_set_fill(id, bg)

int	id;
XColor  *bg;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_draw:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
        X3D_draw[graphics->device].draw_color(graphics, bg);
}
