 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>   Static:
   >>>>	            _kgh_get_prog_info()
   >>>>  Private:
   >>>>   Public:
   >>>>		    gw_read_progfile()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

static char *_kgh_get_prog_info PROTO((kfile *, char *, char *, int *));
static int check_status PROTO((int, int, char **keynames [2]));

/************************************************************
*
*  Routine Name: gw_read_progfile - read and parse a Khoros 1.0 prog file
*
*      Purpose:  This routine reads the program specification 
*		 provided in the .prog file.
*		 It fills out the global **prog_spec array of character blocks,
*		 each of which is referenced by an index with a #define'd name
*		 that indicates the key from which it came.
*
*     	         For instance, after _kgh_get_prog_info is called successfully,
*		 prog_spec[AUTHORS] will hold the block of text taken from
*	         the .prog file, between the begin key of "-authors" and the
*		 end key of "-authors_end";  prog_spec[Restrictions] will 
*		 have the text found between keys "-restrictions" and 
*		 "-restrictions_end", and so on.
*
*	  Input: file - open kfile pointer to read from
*	 Output:
*       Returns:  prog_spec - array of strings, each of which holds a block
*		 	     of text pulled from the .prog file.
*
*   Written By:  Danielle Argiro & Per Lysne
*
*************************************************************/

char **gw_read_progfile(
   kfile *file)
{
    int status = 0;
    char **prog_spec;
    char **keynames[2];

    /* 
     * check for premature end of file 
     */
    if (kfeof(file))
           return(NULL);

    /* 
     * allocate program specification array 
     */
    prog_spec = (char **) kcalloc(1,sizeof(char *)*MAX_PROGSPEC_SIZE);
    if (prog_spec == NULL)
    {
	   kfprintf(kstderr, "gw_read_progfile:\n");
           kfprintf(kstderr, "Out of memory! Cannot calloc prog_spec\n");
           return(NULL);
    }

    /* 
     * get text blocks for each key in the .prog file 
     * get new key begin and key end (which may have some
     * capital letters) and over-write the old keys with them
     */
    gw_init_keynames(keynames);

    prog_spec[AUTHORS]      = _kgh_get_prog_info(file, keynames[0][AUTHORS], 
			  	keynames[1][AUTHORS], &status);
    if (!(check_status(status, AUTHORS, keynames))) return(NULL);

    prog_spec[LIBRARY_SHORTDESC]= _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_SHORTDESC], 
				keynames[1][LIBRARY_SHORTDESC], &status);
    if (!(check_status(status, LIBRARY_SHORTDESC, keynames))) return(NULL);

    prog_spec[MAN1_LONGDESC] = _kgh_get_prog_info(file, 
				keynames[0][MAN1_LONGDESC], 
			 	keynames[1][MAN1_LONGDESC], &status);
    if (!(check_status(status, MAN1_LONGDESC, keynames))) return(NULL);
    
    prog_spec[MAN1_EXAMPLES] = _kgh_get_prog_info(file, 
				keynames[0][MAN1_EXAMPLES], 
			  	keynames[1][MAN1_EXAMPLES], &status);
    if (!(check_status(status, MAN1_EXAMPLES, keynames))) return(NULL);
    
    prog_spec[MAN1_RESTRICTIONS] = _kgh_get_prog_info(file, 
				keynames[0][MAN1_RESTRICTIONS], 
				keynames[1][MAN1_RESTRICTIONS], &status);
    if (!(check_status(status, MAN1_RESTRICTIONS, keynames))) return(NULL);
    
    prog_spec[MAN1_SEEALSO]   = _kgh_get_prog_info(file, 
				keynames[0][MAN1_SEEALSO], 
				keynames[1][MAN1_SEEALSO], &status);
    if (!(check_status(status, MAN1_SEEALSO, keynames))) return(NULL);

    prog_spec[LIBRARY_LONGDESC]  = _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_LONGDESC],
				keynames[1][LIBRARY_LONGDESC], &status);
    if (!(check_status(status, LIBRARY_LONGDESC, keynames))) return(NULL);
   
    prog_spec[LIBRARY_RESTRICT] = _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_RESTRICT], 
				keynames[1][LIBRARY_RESTRICT], &status);
    if (!(check_status(status, LIBRARY_RESTRICT, keynames))) return(NULL);
   
    prog_spec[MAN3_SEEALSO]   = _kgh_get_prog_info(file, 
				keynames[0][MAN3_SEEALSO], 
				keynames[1][MAN3_SEEALSO], &status);
    if (!(check_status(status, MAN3_SEEALSO, keynames))) return(NULL);

    prog_spec[USAGEADD]      = _kgh_get_prog_info(file, 
				keynames[0][USAGEADD], 
				keynames[1][USAGEADD], &status);
    if (!(check_status(status, USAGEADD, keynames))) return(NULL);

    prog_spec[INCLUDE_INCLUDES]= _kgh_get_prog_info(file, 
				keynames[0][INCLUDE_INCLUDES], 
				keynames[1][INCLUDE_INCLUDES], &status);
    if (!(check_status(status, INCLUDE_INCLUDES, keynames))) return(NULL);


    prog_spec[INCLUDE_ADDITIONS]    = _kgh_get_prog_info(file, 
				keynames[0][INCLUDE_ADDITIONS], 
				keynames[1][INCLUDE_ADDITIONS], &status);
    if (!(check_status(status, INCLUDE_ADDITIONS, keynames))) return(NULL);

    prog_spec[INCLUDE_MACROS] = _kgh_get_prog_info(file, 
				keynames[0][INCLUDE_MACROS], 
				keynames[1][INCLUDE_MACROS], &status);
    if (!(check_status(status, INCLUDE_MACROS, keynames))) return(NULL);


    prog_spec[MAIN_VARIABLES] = _kgh_get_prog_info(file, 
				keynames[0][MAIN_VARIABLES], 
				keynames[1][MAIN_VARIABLES], &status);
    if (!(check_status(status, MAIN_VARIABLES, keynames))) return(NULL);

    prog_spec[MAIN_BEFORELIB] = _kgh_get_prog_info(file, 
				keynames[0][MAIN_BEFORELIB], 
				keynames[1][MAIN_BEFORELIB], &status);
    if (!(check_status(status, MAIN_BEFORELIB, keynames))) return(NULL);

    prog_spec[MAIN_LIBCALL]   = _kgh_get_prog_info(file, 
				keynames[0][MAIN_LIBCALL], 
				keynames[1][MAIN_LIBCALL], &status);
    if (!(check_status(status, MAIN_LIBCALL, keynames))) return(NULL);

    prog_spec[MAIN_AFTERLIB]  = _kgh_get_prog_info(file, 
				keynames[0][MAIN_AFTERLIB], 
				keynames[1][MAIN_AFTERLIB], &status);
    if (!(check_status(status, MAIN_AFTERLIB, keynames))) return(NULL);

    prog_spec[LIBRARY_INPUT]  = _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_INPUT], 
				keynames[1][LIBRARY_INPUT], &status);
    if (!(check_status(status, LIBRARY_INPUT, keynames))) return(NULL);

    prog_spec[LIBRARY_OUTPUT] = _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_OUTPUT], 
				keynames[1][LIBRARY_OUTPUT], &status);
    if (!(check_status(status, LIBRARY_OUTPUT, keynames))) return(NULL);

    prog_spec[LIBRARY_DEF]    = _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_DEF], 
				keynames[1][LIBRARY_DEF], &status);
    if (!(check_status(status, LIBRARY_DEF, keynames))) return(NULL);

    prog_spec[LIBRARY_CODE]   = _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_CODE], 
				keynames[1][LIBRARY_CODE], &status);
    if (!(check_status(status, LIBRARY_CODE, keynames))) return(NULL);

    prog_spec[LIBRARY_INCLUDES]= _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_INCLUDES], 
				keynames[1][LIBRARY_INCLUDES], &status);
    if (!(check_status(status, LIBRARY_INCLUDES, keynames))) return(NULL);

    prog_spec[LIBRARY_MODS]= _kgh_get_prog_info(file, 
				keynames[0][LIBRARY_MODS], 
				keynames[1][LIBRARY_MODS], &status);
    if (!(check_status(status, LIBRARY_MODS, keynames))) return(NULL);

    return(prog_spec);
}

/*-----------------------------------------------------------------------------
|
| Routine Name: _kgh_get_prog_info()
|
|      Purpose: This function reads a block of text between two keys from the
|               an open file and returns it in a contiguous block of memory.
|               This routine is a front end to the general parser, and services
|               the ghost routines specific needs.
|
|         Input: file      - open stream to a file
|                key_begin - begin key of desired character block
|                key_end   - end key of desired character block
|
|        Output: Returns a pointer to a block of characters that was read from
|                a file on success, or NULL on failure.
|
|                status - returns the status of the textblock search.
|                         may be one of:
|                         KPARSE_OK,       { valid text block found }
|                         KPARSE_NOKEY,    { begin key not found    }
|                         KPARSE_NOEND,    { end key not found    }
|                         KPARSE_PARTKEY   { only part of begin key found }
|                         KPARSE_PARTEND   { only part of end key found }
|                         KPARSE_SYNTAXKEY { syntax error on begin key }
|                         KPARSE_SYNTAXEND { syntax error on end key }
|                         KPARSE_DATAERR   { an error occurred while parsing}
|
|     Written By: Danielle Argiro & Steve Jorgensen
|           Date: Oct 12, 1992 16:47
|  Modifications: Converted from Khoros 1.0 (DA & SJ)
|
----------------------------------------------------------------*/

static char *_kgh_get_prog_info(
   kfile *file,
   char  *key_begin,
   char  *key_end,
   int   *status)
  {
    char *string_return;

    krewind (file);
    string_return = kparse_file_scan(file, key_begin, key_end,
                                         KIGNORE_CASE, NULL, NULL, status);

    if (kstrlen(string_return) < 2)
      {
        kfree(string_return);
        string_return = NULL;
      }
    return (string_return);
  }

static int check_status(
   int  status,
   int  key,
   char ***keynames)
{
    if (status == KPARSE_NOKEY)
    {
        kfprintf(kstderr, "ERROR in *.prog file:\n");
        kfprintf(kstderr, "Begin key '%s' not found.\n", keynames[0][key]);
        return(FALSE);
    }
    else if (status == KPARSE_NOEND)
    {
        kfprintf(kstderr, "ERROR in *.prog file:\n");
        kfprintf(kstderr, "End key '%s' not found.\n", keynames[1][key]);
        return(FALSE);
    }
    else if (status != KPARSE_OK)
    {
        kfprintf(kstderr, "ERROR in *.prog file:\n");
        kfprintf(kstderr, "Parser failure for keys '%s' and '%s'",
		 keynames[0][key], keynames[1][key]);
        return(FALSE);
    }
    return(TRUE);
}

