 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		    Routines to Read the UIS file             <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>		        kvf_copy_database()		      <<<<
   >>>>			kvf_read_database()		      <<<<
   >>>>   Static:                                             <<<<
   >>>>			kvf_read_subform()		      <<<<
   >>>>			kvf_read_pane()		              <<<<
   >>>>			kvf_realloc_database()	              <<<<
   >>>>			kvf_free_database()	              <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "uisupdate.h"

/* db_util.c */
static char **kvf_read_subform     PROTO((char *, char **, int *, int *, 
                                          int **, int **, char ***, int *));
static char **kvf_read_pane        PROTO((char *, char **, int *, int *, 
                                          int **, int **, char ***, int *));
static char **kvf_realloc_database PROTO((char **, int *, int, int **, int **));

static void kvf_free_database      PROTO((char **));

/*------------------------------------------------------------
|
|    Routine Name: kvf_read_database
|
|         Purpose: Reads in a UIS file and returns in the internal UIS 
|                  structure with the contents.  In some very special cases 
|                  (conductor and ghostwriter), this routine may be called 
|                  directly from the application in order to return the 
|                  internal representation of the UIS, as well as the 
|                  filename_lookup, linenum_lookup, and filecount arrays, so 
|                  that they may be passed to kvf_build_form.
|
|                  IMPORTANT NOTE: Use this routine at your own risk!
|                  In general, all applications should call kvf_create_form(),
|                  which calls this routine at the right time.
|
|           Input: fid       - Open stream to UIS file opened for reading
|		   filenames - The array of strings as a whole must be 
|                              allocated and the first element should be 
|                              initialized to the name of the UIS file before
|                              this argument is passed in.
|		   filecount - Must be initialized to 0 before passing.
|
|          Output: num  - Returns number of lines in the UIS
|		   size - Returns allocated size of the UIS ( >= num)
|
|		   filename_lookup - Allocates & returns array of integers where
|                       the index is the line number of the UIS, and the array 
|                       element indicates the number id of the file that the 
| 			line originated in.  Internally used to generate 
|                       correct error messages for UIS syntax & logic errors, 
|                       this array must be used as input to kvf_build_form.
|
|		   linenum_lookup - Allocates & returns array of integers where
|                       the index is the line number of the UIS, and the array 
|                       element indicates the line number in the file that the 
|                       line originated in.  Internally used to generate 
|                       correct error messages for UIS syntax & logic errors, 
|                       this array must be used as input to kvf_build_form.
|
|		   filenames - 
|                       Returns array of filenames, indexed by elements of 
|                       filename_lookup.   Internally used to generate correct 
|                       error messages for UIS syntax & logic errors, this 
|                       array must be used as input to kvf_build_form.
|
|		   filecount - 
|			Returns the total number of files opened, minus 1.
|
|       Returns: The array of strings which the internal representation of
|		 the UIS is stored
|
|    Written By: Danielle Argiro & Mark Young
|          Date: Jul 16, 1992 10:00
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

char **kvf_read_database(
   kfile *file,
   int   *num,
   int   *size,
   int   **filename_lookup,
   int   **linenum_lookup,
   char  ***filenames,
   int   *filecount)
{
	Line_Info line_info;
        char      temp[KLENGTH*20];
	int	  i, j, db_size, done, length, count, file_num; 
	char      **database;
	int       internal_linenum = 1, flag = -1;

	kvf_clear_line_info(&line_info);
	file_num = *filecount;

	/* premature end of file */
	if (kfeof(file))
	{
	   *num = 0;
	   return(NULL);
	}

	/* allocate internal database */
	db_size = KLENGTH;
	database = (char **) kcalloc(1,sizeof(char *)*KLENGTH);
	if (database == NULL)
	{
	    kerror("kforms", "kvf_read_database",
		   "Cannot allocate room for internal UIS");
	    *num = 0;
	    return(NULL);
	}

	*filename_lookup = (int *) kmalloc(sizeof(int) * KLENGTH);
	if (*filename_lookup == NULL)
        {
	    kerror("kforms", "kvf_read_database",
		   "Cannot allocate room for internal filename lookup table");
            *num = 0;
            return(NULL);
        }

	*linenum_lookup = (int *) kmalloc(sizeof(int) * KLENGTH);
	if (*linenum_lookup == NULL)
        {
	    kerror("kforms", "kvf_read_database",
		   "- Cannot allocate room for internal linenum lookup table");
            *num = 0;
            return(NULL);
        }

	done  = FALSE;
	count = -1;
	i = 0; 
	while (!done)
 	{
           j = 0;

	   if (kfgets(temp, KLENGTH*20, file) == NULL) 
	   {
	      errno = KUIS_SYNTAX;
	      kerror("kforms", "kvf_read_database", 
		     "Premature End of File encountered in UIS");
	      *num = 0;
	      kvf_free_database(database);
	      return(NULL);
	   }

	   while(temp[j] == ' ' || temp[j] == '\t') j++;

	   flag = kvf_get_line_type(&temp[j]);
	   switch (flag)
	   {
		 case KUIS_INCLUDEPANE:
		      kvf_oldparse_includepane_line(&temp[j], &line_info);
		      database = kvf_read_pane(line_info.filename, database, 
						&i, &db_size, filename_lookup, 
                                                linenum_lookup, filenames, 
						filecount);
		      kvf_free_line_info_strings(&line_info);
		      internal_linenum++;
		      if (database == NULL)
		      {
		         *num = 0;
		         kvf_free_database(database);
			
			 errno = KUIS_SYNTAX;
	                 kerror("kforms", "kvf_read_database",
				"Error found in included file specified by \
line '%s'", &temp[j]);
		         return(NULL);
		      }
		      break;

		 case KUIS_INCLUDESUBFORM:
		      kvf_oldparse_includesubform_line(&temp[j], &line_info);
		      database = kvf_read_subform(line_info.filename, database,
					          &i, &db_size, filename_lookup,
                                                  linenum_lookup, filenames, 
						  filecount);
		      kvf_free_line_info_strings(&line_info);
		      internal_linenum++;
		      if (database == NULL)
		      {
		         *num = 0;
		         kvf_free_database(database);
			
			 errno = KUIS_SYNTAX;
	                 kerror("kforms", "kvf_read_database",
				"Error found in included file specified by \
line '%s'", &temp[j]);
		         return(NULL);
		      }
		      break;

		 case KUIS_STARTFORM:
		      count = 1;
		      break;

		 case KUIS_STARTMASTER:
		 case KUIS_STARTSUBMENU:
		 case KUIS_STARTSUBFORM:
		 case KUIS_STARTGUIDE:
		 case KUIS_STARTPANE:
		 case KUIS_TOGGLE:
		 case KUIS_MUTEXCL:
		 case KUIS_MUTINCL:
		 case KUIS_GROUP:
		      count++;
		      break;

		 case KUIS_END:
		      count--;
		      break;

		 case -1:
		      *num = 0;
		      kvf_free_database(database);
		      return(NULL); 
		      /* break; */

	   }

	   if (flag != KUIS_INCLUDEPANE && flag != KUIS_INCLUDESUBFORM)
	   {
		 if (i == (db_size-1))
		 {
		    database = kvf_realloc_database(database, &db_size,
						  KLENGTH,
					          filename_lookup, 
						  linenum_lookup);
		    if (database == NULL) return(NULL);
		 }
	         length = kstrlen(&temp[j]);
	         database[i] = (char *) kcalloc(1, sizeof(char) * length);
		 if (database[i] == NULL)
		 {
		    kerror("kforms", "kvf_read_database",
			   "Unable to allocate line in internal UIS");
		    return(NULL);
		 }
	         (void) kstrncpy(database[i],&temp[j], length);
	         database[i][length-1] = '\0';
		 (*filename_lookup)[i] = file_num;
		 (*linenum_lookup)[i] = internal_linenum;
	         i++;
		 internal_linenum++;
	    }

	    if (count == 0) done = TRUE;
	}

	/*
	 * error check for missing control lines
	 */

	if (i < 3)
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_read_database", 
                   "possible missing control [-F, -S, -M, -G, -P, -T, or \
-C] line, or misplaced end [-E] line");
	    kvf_free_database(database);
	    *num = 0;
	    return(NULL);
	}
	flag = kvf_get_line_type(database[i-3]);
	if ((flag != KUIS_END) && (flag != KUIS_COMMENT))
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_read_database", 
                   "possible missing control [-F, -S, -M, -G, -P, -T, or \
-C] line, or misplaced end [-E] line");
	    kvf_free_database(database);
	    *num = 0;
	    return(NULL);
	}

	*num  = i;
	if (db_size < KLENGTH)
	{
	    *filename_lookup = (int *) krealloc(*filename_lookup, db_size);
	    if (*filename_lookup == NULL)
	    {
		kerror("kforms", "kvf_read_database",
		       "Unable to allocate internal filename_lookup array");
		return(NULL);
	    }

	    *linenum_lookup  = (int *) krealloc(*filename_lookup, db_size);
	    if (*linenum_lookup == NULL)
	    {
		kerror("kforms", "kvf_read_database",
		       "Unable to allocate internal linenum_lookup array");
		return(NULL);
	    }

	    database = (char **) krealloc(database, db_size);
	    if (database == NULL)
	    {
		kerror("kforms", "kvf_read_database",
		       "Unable to re-allocate internal UIS");
		return(NULL);
	    }
	}
	*size = db_size;

        return(database);
}



/*--------------------------------------------------------------
|
|   Routine Name: kvf_read_subform
|
|        Purpose: Reads in a UIS subform def from the specified UIS file
|		  into the supplied database.  This routine is called
|		  when kvf_read_database finds a KUIS_INCLUDESUBFORM user
|		  interface specification.  '-k filename'
|
|		  kvf_read_subform includes the subform specification into
|		  the existing database file.  If the subform file has
|		  a KUIS_STARTSUBFORM specification then the subform will be read
|		  in and the KUIS_STARTFORM will be discarded; otherwise the entire
|		  subform description is read.
|
|
|          Input: filename - name of the UIS file in which to get the
|			     subform definition.
|		  num      - number of current lines in the database
|		  db_size  - size of current database
|                 filename_lookup  - indexes into 'filenames' array 
|                 linenum_lookup   - holds actual line numbers associated with
|                                    UIS lines, as opposed to the indices into
|                                    the internal UIS 'database'
|                 filenames        - holds names of all files read in as part of
|                                    UIS (used with -p and -k UIS lines).
|                 filecount        - size of filenames
|
|        Output: database - adds the subform def onto the array of strings.
|		 num      - returns the current number of lines in the database
|		 db_size  - returns the size of current database
|
|       Returns: TRUE on success, FALSE otherwise.
|
|          Date: Apr 19, 1992
|    Written By: Mark Young & Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


static char **kvf_read_subform(
   char *filename,
   char **database,
   int  *num,
   int  *db_size,
   int  **filename_lookup,
   int  **linenum_lookup,
   char ***filenames,
   int  *filecount)
{
	kfile	   *file;
	char	   **db; 
	char       *line = NULL;
	char       temp[KLENGTH];
	Line_Info  line_info;
	int        *file_lookup, *line_lookup;
	int 	   i, j, count, size, offset = 0; 

	/*
	 *  Open the database file that contains the subform specification.
	 */
	if ((file = kfopen(filename, "r")) == NULL)
        {
           ksprintf(temp, "Unable to open '%s' to read pane", filename);
	   kerror("kforms", "kvf_read_database", 
                  "Unable to open '%s' to read pane", filename);
           return(NULL);
        }

	/*
	 * add one more name to the filenames array, increment filecount - 
         * need to allocate another string for the filenames array
	 */
        (*filecount)++;
	*filenames = (char **) krealloc(*filenames,
					(*filecount+1)*sizeof(char*));
	if (*filenames == NULL)
	{
	    kerror("kforms", "kvf_read_subform",
		   "Unable to allocate internal filenames array");
	    return(NULL);
	}
	(*filenames)[*filecount] = kstrdup(filename);

	/*
         * read the pane specification into a temporary database;
         * read the filename lookup table into a temporary array
	 */
	db = kvf_read_database(file, &count, &size, &file_lookup, 
				&line_lookup, filenames, filecount);
	if (db == NULL)
	{
	   kfclose(file);
	   return(NULL);
	}

	/*
	 *  Check to see if the pane specification starts with
	 *  a KUIS_STARTFORM specification.
	 */
	kvf_clear_line_info(&line_info);
	(void) kvf_get_comments(db, &offset);
	if (kvf_get_line_type(db[offset]) == KUIS_STARTFORM)
	{
	   if (!(kvf_oldparse_startform_line(db[offset], &line_info))) 
		return(NULL);
	   kvf_deparse_startform_line(&line_info, &line);
	   ksprintf(temp, "@%s", line);
	   kfree(line);
	   kfree(db[offset]);
	   db[offset] = kstrdup(temp);
	   count--; offset++;
	}

	/*
	 *  Copy the rest of the database, which should be the subform,
	 *  into old database.  Copy the associated entries from filename
         *  array into old filename lookup table.
	 */
	for (j = *num, i = -1; i < count; i++, j++)
	{
	    if (j == (*db_size-1))
	    {
	       database = kvf_realloc_database(database, db_size, 
					       KLENGTH,
					       filename_lookup, 
					       linenum_lookup);
	       if (database == NULL)
	       {
		  kfclose(file);
	          return(NULL);
	       }
	    }
	    if (i == -1)
	    {
		ksprintf(temp, "@-k %s", filename);
		database[j] = kstrdup(temp);
	    }
	    else
	    {
	        database[j] = db[i];
	        (*filename_lookup)[j] = file_lookup[i];
	        (*linenum_lookup)[j] = line_lookup[i];
	        db[i] = NULL;
	    }
	}

	/*
	 *  Free the temporary database.
	 */
	kvf_free_database(db);
	kfree(file_lookup);
	kfree(line_lookup);

	*num = j;
        kfclose(file);

	return(database);
}



/*--------------------------------------------------------------
|
|   Routine Name: kvf_read_pane
|
|        Purpose: Reads in a UIS pane def from the specified UIS file
|		  into the supplied database.  This routine is called
|		  when kvf_read_database finds a KUIS_INCLUDEPANE user
|		  interface specification.  '-p filename'
|
|		  kvf_read_pane includes the pane specification into
|		  the existing database file.  If the pane file has
|		  a KUIS_STARTFORM and KUIS_STARTSUBFORM lines then the pane will be 
|                 read in and the 2 first lines will be discarded; otherwise 
|                 the entire pane description is read.
|
|          Input: filename - name of the UIS file in which to get the
|			     pane definition.
|		  num      - number of current lines in the database
|		  db_size  - size of current database
|                 filename_lookup  - indexes into 'filenames' array 
|                 linenum_lookup   - holds actual line numbers associated with
|                                    UIS lines, as opposed to the indices into
|                                    the internal UIS 'database'
|                 filenames        - holds names of all files read in as part of
|                                    UIS (used with -p and -k UIS lines).
|                 filecount        - size of filenames
|
|         Output: database - adds the pane def onto the array of strings.
|		  num      - returns the current number of lines in the database
|		  db_size  - returns the size of current database
|
|        Returns: TRUE on success, FALSE otherwise.
|
|          Date: Apr 19, 1992
|    Written By: Mark Young & Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static char **kvf_read_pane(
   char *filename,
   char **database,
   int  *num,
   int  *db_size,
   int  **filename_lookup,
   int  **linenum_lookup,
   char ***filenames,
   int  *filecount)
{
	kfile	  *file;
	char	  **db; 
	char      *line = NULL;
	char      temp[KLENGTH];
	Line_Info line_info;
	int       *file_lookup, *line_lookup;
	int 	  i, j, count, size, offset = 0; 

	/*
	 *  Open the database file that contains the pane specification.
	 */
	if ((file = kfopen(filename, "r")) == NULL)
        {
	   kerror("kforms", "kvf_read_database", 
                  "Unable to open '%s' to read pane", filename);
           return(NULL);
        }

	/*
	 * add one more name to the filenames array, increment filecount - 
         * need to allocate another string for the filenames array
	 */
        (*filecount)++;
	*filenames = (char **) krealloc(*filenames, 
				(*filecount+1)*sizeof(char*));
	if (*filenames == NULL)
	{
	    kerror("kforms", "kvf_read_pane",
		   "Unable to allocate internal filenames array");
	    return(NULL);
	}
	(*filenames)[*filecount] = kstrdup(filename);
	(*filenames)[*filecount] = kstrdup(filename);

	/*
         * read the pane specification into a temporary database;
         * read the filename lookup table into a temporary array
	 */
	db = kvf_read_database(file, &count, &size, &file_lookup, 
				&line_lookup, filenames, filecount);
	if (db == NULL)
	{
	   kfclose(file);
	   return(NULL);
	}

	/*
	 *  Check to see if the pane specification starts with
	 *  a KUIS_STARTFORM specification.
	 */
	kvf_clear_line_info(&line_info);
	(void) kvf_get_comments(db, &offset);
	if (kvf_get_line_type(db[offset]) == KUIS_STARTFORM)
	{
	   if (!(kvf_oldparse_startform_line(db[offset], &line_info))) 
		return(NULL);
	   kvf_deparse_startform_line(&line_info, &line);
	   ksprintf(temp, "@%s", line);
           kfree(line);
           kfree(db[offset]);
           db[offset] = kstrdup(temp);
	   kvf_free_line_info_strings(&line_info);
	   count--; offset++;
	}

	/*
	 *  Check to see if the pane specification starts with
	 *  a KUIS_STARTSUBFORM specification.
	 */
	(void) kvf_get_comments(db, &offset);
	if (kvf_get_line_type(db[offset]) == KUIS_STARTSUBFORM)
	{
	   if (!(kvf_oldparse_startsubform_line(db[offset], &line_info)))
		return(NULL);
	   kvf_deparse_startsubform_line(&line_info, &line);
	   ksprintf(temp, "@%s", line);
           kfree(line);
           kfree(db[offset]);
           db[offset] = kstrdup(temp);
	   kvf_free_line_info_strings(&line_info);
	   count--; offset++;
	}

	/*
	 *  Copy the rest of the database, which should be the pane,
	 *  into old database.
	 */
	for (j = *num, i = -1; i < count; i++, j++)
	{
	    if (j == (*db_size-1))
	    {
	       database = kvf_realloc_database(database, db_size,
					       KLENGTH,
					       filename_lookup, 
					       linenum_lookup);
	       if (database == NULL)
	       {
		  kfclose(file);
	          return(NULL);
	       }
	    }
	    if (i == -1)
            {
                ksprintf(temp, "@-p %s", filename);
                database[j] = kstrdup(temp);
            }
	    else
	    {
	        database[j] = db[i];
	        (*filename_lookup)[j] = file_lookup[i];
	        (*linenum_lookup)[j] = line_lookup[i];
	        db[i] = NULL;
	    }
	}

	/*
	 *  Free the temporary database.
	 */
	kvf_free_database(db);
	kfree(file_lookup);
	kfree(line_lookup);
	*num = j;
        kfclose(file);

	return(database);
}



/*-----------------------------------------------------------------
|
|  Routine Name: kvf_copy_database
|
|	Purpose: copies a database, returns a pointer to the new database.
|
|	  Input: old_database     - pointer to the database to be copied.
|                filename_lookup  - indexes into 'filenames' array 
|                linenum_lookup   - holds actual line numbers associated with
|                                   UIS lines, as opposed to the indices into
|                                   the internal UIS 'database'
|                line_num         - returns size of copied database
|        Output: none
|       Returns: a pointer to the new database 
|          Date: Apr 19, 1992
|    Written By: Danielle Argiro and Mark Young
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------------*/

char **kvf_copy_database(
   char **old_database,
   int  **filename_lookup,
   int  **linenum_lookup,
   int  *line_num)
{
	char **database;
	int  db_size, balance = 0, flag, sub_type, index = 0,
	     subform_index = -1, pseudo_subform = FALSE;
	
	/* allocate new internal database */
	db_size   = KLENGTH;
	*line_num = 0;
        database  = (char **) kcalloc(1,sizeof(char *)*KLENGTH);
        if (database == NULL)
        {
	    kerror("kforms", "kvf_read_database",
		   "Cannot allocate internal UIS");
            return(NULL);
        }
 
        while (old_database[index] != NULL)
        {
	   if (index == (db_size-1))
	   {
	      database = kvf_realloc_database(database, &db_size, 
					      KLENGTH,
					      filename_lookup, linenum_lookup);
	      if (database == NULL)
		  return(NULL);
	   }

	   flag = kvf_get_line_type(old_database[index]);
	   if (flag  == KUIS_STARTSUBFORM)
	   {
	      /*
	       *  Find out if the subform is either a KUIS_PSEUDOSUBFORM or
	       *  a KUIS_SUBFORMBUTTON.  KUIS_PSEUDOSUBFORM can be shared among other
	       *  other databases.
	       */
	      do
	      {
		 subform_index++;
		 if (database[subform_index] == NULL)
		 {
		    sub_type = KUIS_SUBFORMBUTTON;
		    break;
		 }
		 else
		    sub_type = kvf_get_line_type(database[subform_index]);
	      } while (sub_type != KUIS_PSEUDOSUBFORM && sub_type != KUIS_SUBFORMBUTTON);

	      if (sub_type == KUIS_PSEUDOSUBFORM)
	      {
	         pseudo_subform = TRUE;
	         balance = 1;
	      }
	   }
	   else if (pseudo_subform)
	   {
	      switch (flag)
	      {
		 case KUIS_STARTPANE:	balance++; break;
		 case KUIS_STARTGUIDE:	balance++; break;
		 case KUIS_TOGGLE:	balance++; break;
		 case KUIS_MUTEXCL:	balance++; break;
		 case KUIS_MUTINCL:	balance++; break;
		 case KUIS_GROUP:	balance++; break;

		 case KUIS_END:		balance--; break;
	      }
	   }

	   if (pseudo_subform)
	      database[index] = old_database[index];
	   else
 	      database[index] = kstrdup(old_database[index]);

	   index++;

	   if (pseudo_subform && balance == 0)
	      pseudo_subform = FALSE;
	}
	*line_num = index;
	return(database);
}



/*-------------------------------------------------------------
|
|  Routine Name: kvf_free_database
|
|       Purpose: Frees internal representation of the UIS file.  
|
|         Input: database  - the form database to be freed.
|        Output: None
|       Returns: nothing
|          Date: Apr 19, 1992
|    Written By: Danielle Argiro & Mark Young
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static void kvf_free_database(
   char **database)
{
	int  balance = 0, flag, sub_type, save_index, index = 0,
	     subform_index = -1, pseudo_subform = FALSE;

	if (database == NULL) return;

	while (database[index] != NULL)
	{
	   flag = kvf_get_line_type(database[index]);
	   if (flag == KUIS_STARTSUBFORM)
	      break;
	   else
	      index++;
	}
	save_index = index;

        while (database[index] != NULL)
        {
	   flag = kvf_get_line_type(database[index]);
	   if (flag  == KUIS_STARTSUBFORM)
	   {
	      /*
	       *  Find out if the subform is either a KUIS_PSEUDOSUBFORM or
	       *  a KUIS_SUBFORMBUTTON.  KUIS_PSEUDOSUBFORM can be shared among other
	       *  other databases.
	       */
	      do
	      {
		 subform_index++;
		 if (subform_index >= save_index)
		 {
		    sub_type = KUIS_SUBFORMBUTTON;
		    break;
		 }
		 else
		    sub_type = kvf_get_line_type(database[subform_index]);
	      } while (sub_type != KUIS_PSEUDOSUBFORM && sub_type != KUIS_SUBFORMBUTTON);

	      if (sub_type == KUIS_PSEUDOSUBFORM)
	      {
	         pseudo_subform = TRUE;
	         balance = 1;
	      }
	   }
	   else if (pseudo_subform)
	   {
	      switch (flag)
	      {
		 case KUIS_STARTPANE:	balance++; break;
		 case KUIS_STARTGUIDE:	balance++; break;
		 case KUIS_TOGGLE:	balance++; break;
		 case KUIS_MUTEXCL:	balance++; break;
		 case KUIS_MUTINCL:	balance++; break;
		 case KUIS_GROUP:	balance++; break;

		 case KUIS_END:		balance--; break;
	      }
	   }

	   if (pseudo_subform == FALSE)
 	      kfree(database[index]);
	   else if (balance == 0)
	      pseudo_subform = FALSE;

	   database[index] = NULL;
	   index++;
	}

	for (index = 0; index < save_index; index++)
	   kfree(database[index]);

	kfree(database);
}



/*------------------------------------------------------------
|
|  Routine Name: kvf_realloc_database
|
|       Purpose: Reallocates the internal representation of the 
|                UIS file. Also reallocates the filename_lookup,
|                and linenum_lookup arrays.
|
|         Input: database - internal rep. of UIS file to be reallocated
|                request  - number of additional lines to be allocated
|                filename_lookup  - indexes into 'filenames' array 
|                linenum_lookup   - holds actual line numbers associated with
|                                   UIS lines, as opposed to the indices into
|                                   the internal UIS 'database'
|
|        Output: size             - returns new size of the database
|                filename_lookup  - reallocated to new size
|                linenum_lookup   - reallocated to new size
|
|       Returns: the database, reallocated to its new size
|
|          Date: Apr 19, 1992
|    Written By: Danielle Argiro & Mark Young
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static char **kvf_realloc_database(
   char **database,
   int  *size,
   int  request,
   int  **filename_lookup,
   int  **linenum_lookup)
{
	int  num_bytes;


	if (database == NULL)
	{
	   num_bytes = request * sizeof(char *);
	   database = (char  **) kcalloc(1, num_bytes);
	   if (database == NULL)
	   {
	        kerror("kforms", "kvf_realloc_database",
		       "Cannot allocate room for internal UIS");
	        return(NULL);
	   }
	   num_bytes = request * sizeof(int);

	   *filename_lookup = (int *) kmalloc(num_bytes);
	   if (*filename_lookup == NULL)
	   {
	        kerror("kforms", "kvf_realloc_database",
		       "Unable to allocate internal filename_lookup array");
	        return(NULL);
	   }
	   *linenum_lookup = (int *) kmalloc(num_bytes);
	   if (*linenum_lookup == NULL)
	   {
	        kerror("kforms", "kvf_realloc_database",
		       "Unable to allocate internal linenum_lookup array");
	        return(NULL);
	   }
	   *size = request;
	}
	else
	{
	   num_bytes = (*size + request) * sizeof(char *);
	   database = (char **) krealloc(database, num_bytes);
	   if (database == NULL)
	   {
	        kerror("kforms", "kvf_realloc_database",
		       "Cannot allocate room for internal UIS");
	        return(NULL);
	   }

	   /*
	    *  Need to put nulls in the new part of the allocated array.
	    */
	   num_bytes = request * sizeof(char *);
	   kmemset((char *) &database[*size], 0, num_bytes);

	   num_bytes = (*size + request) * sizeof(int);
	   *filename_lookup = (int *) krealloc(*filename_lookup, num_bytes);
	   if (*filename_lookup == NULL)
	   {
	        kerror("kforms", "kvf_realloc_database",
		       "Unable to allocate internal filename_lookup array");
	        return(NULL);
	   }

	   *linenum_lookup  = (int *) krealloc(*linenum_lookup, num_bytes);
	   if (*linenum_lookup == NULL)
	   {
	        kerror("kforms", "kvf_realloc_database",
		       "Unable to allocate internal linenum_lookup array");
	        return(NULL);
	   }

	   *size += request;
	}
	return(database);
}
