 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for mlud
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lmlud
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lmlud - compute LU decomposition
* 
*       Purpose: lmlud computes LU decomposition of a general matrix using
*                partial pivoting and row interchanges.
*              
*		 The L matrix is lower triangular with unit diagonal elements.
*                The U matrix is upper triangular. The pivot vector indicates
*                that the i'th row was interchanged with the  row given in the
*                i'th value in the vector.
*
*                lmlud used the DGETRF or ZGETRF routines from LAPACK to
*                perform the decomposition.
*
*         Input: kobject in_obj - matrix to be factored
*
*        Output: kobject lt_obj - object to receive the lower triangular factor
*                kobject ut_obj - object to receive the upper triangular factor
*                kobject pi_obj - object to receive the pivot vector
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lmlud (
	kobject in_obj,
	kobject lt_obj,
	kobject ut_obj,
	kobject pi_obj)
/* -library_def_end */

/* -library_code */
  {
        char *lib = "kmatrix", *rtn = "lmlud";
        double *A=NULL,*L=NULL,*U=NULL;
        kdcomplex *AZ=NULL,*LZ=NULL,*UZ=NULL;
        int i,j,m,n,num,nplanes,lda,*ipiv,info;
	int type, otype;
	int wc,hc,dc,tc,ec;
        int pi_len;
        klist *objlist=NULL;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (ut_obj == KOBJECT_INVALID && lt_obj == KOBJECT_INVALID &&
            pi_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Must have at least one output object");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* See if the data type of the input object is real or complex */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_DATA_TYPE,&type));
        if (type == KCOMPLEX || type == KDCOMPLEX) otype = KDCOMPLEX;
        else otype = KDOUBLE;

        /* Constrain the input data to double real or double complex for read */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, otype));

        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec));
	/* Set up to process input by planes */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wc,hc,1,1,1));
        nplanes = dc*tc*ec;
        pi_len = kmin(wc,hc);

        /* Set the output object sizes and data types */
        if (lt_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(lt_obj)) KCALL(kpds_create_value(lt_obj));
            KCALL(kpds_set_attributes(lt_obj,
                                 KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,otype,NULL));
            KCALL((lt_obj = kpds_reference_object(lt_obj)));
	    objlist = klist_add(objlist,lt_obj,"KOBJECT");
            KCALL(kpds_set_attribute(lt_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,hc,1,1,1));
          }
        if (ut_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(ut_obj)) KCALL(kpds_create_value(ut_obj));
            KCALL(kpds_set_attributes(ut_obj,
                                 KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,otype,NULL));
            KCALL((ut_obj = kpds_reference_object(ut_obj)));
	    objlist = klist_add(objlist,ut_obj,"KOBJECT");
            KCALL(kpds_set_attribute(ut_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,hc,1,1,1));
          }
        if (pi_obj != KOBJECT_INVALID) 
          {
            if (!kpds_query_value(pi_obj)) KCALL(kpds_create_value(pi_obj));
            KCALL(kpds_set_attributes(pi_obj,
                                 KPDS_VALUE_SIZE,1,pi_len,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,KINT,NULL));
            KCALL((pi_obj = kpds_reference_object(pi_obj)));
	    objlist = klist_add(objlist,pi_obj,"KOBJECT");
            KCALL(kpds_set_attribute(pi_obj,KPDS_VALUE_REGION_SIZE,
                                            1,pi_len,1,1,1));
          }

        /* Allocate plane buffers */
        if (otype == KDOUBLE)
          {
            KCALL(!((A=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,A,"KMALLOC");
            KCALL(!((L=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,L,"KMALLOC");
            KCALL(!((U=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
	    objlist = klist_add(objlist,U,"KMALLOC");
          }
        else
          {
            KCALL(!((AZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,AZ,"KMALLOC");
            KCALL(!((LZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,LZ,"KMALLOC");
            KCALL(!((UZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,UZ,"KMALLOC");
          } 

        /* Allocate IPVT array */
        KCALL(!((ipiv=(int *)kmalloc(pi_len*sizeof(int))) == NULL));
        objlist = klist_add(objlist,ipiv,"KMALLOC");

if (otype == KDOUBLE) kinfo(KVERBOSE,"Type is KDOUBLE\n");
else if (otype == KDCOMPLEX) kinfo(KVERBOSE,"Type is KDCOMPLEX\n");
        for (num=0; num<nplanes; num++)
          {
kinfo(KVERBOSE,"nplanes: %d wc: %d hc: %d num:%d\n",nplanes,wc,hc,num);
            /* Get a plane from the input object */
            if (otype == KDOUBLE)
              A = (double *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)A);
            else
              AZ=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)AZ);
kinfo(KVERBOSE,"Got a plane...\n");

            if (otype == KDOUBLE)
              {
                kbzero(L,wc*hc*sizeof(double));
                kbzero(U,wc*hc*sizeof(double));
              }
            else
              {
                kbzero(LZ,wc*hc*sizeof(kdcomplex));
                kbzero(UZ,wc*hc*sizeof(kdcomplex));
              }
            kbzero(ipiv,pi_len*sizeof(int));
kinfo(KVERBOSE,"Zeroed plane buffers...\n");

            /* Transpose so that F77 will like it */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    L[j*hc+i] = A[i*wc+j];
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    LZ[j*hc+i] = AZ[i*wc+j];
              }
kinfo(KVERBOSE,"Transposed into L{Z}...\n");

            /* Obtain LU factorization */
	    if (otype == KDOUBLE) DGETRF(&hc,&wc,L,&hc,ipiv,&info);
            else ZGETRF(&hc,&wc,LZ,&hc,ipiv,&info);

            if (info != 0)
              {
                kinfo(KSTANDARD,"lmlud: LAPACK {dz}getrf() returned with INFO=%d\n\
where INFO < 0 -> for INFO = -i, the i'th argument had an illegal value and\n\
      INFO > 0 -> for INFO = i, U(i,i) is exactly zero. The factorization\n\
                  has been completed, but the factor U is exactly singular\n\
                  and division by zero will occur during backsubstitution.",\
                  info);
              }
kinfo(KVERBOSE,"Factored...\n");

            /* Untranspose so that C will like it! */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    A[i*wc+j] = L[j*hc+i];
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    AZ[i*wc+j] = LZ[j*hc+i];
              }
kinfo(KVERBOSE,"Untransposed into A{Z}...\n");

            /* Split the factorized packed matrix into L and U parts */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  {
                    for (j=i; j<wc; j++)
                      {
                        U[i*wc+j] = A[i*wc+j];
                        L[i*wc+j] = 0.0;
                      }
                  }
                for (i=0; i<hc; i++) L[i*wc+i] = 1.0;
                for (i=1; i<hc; i++)
                  for (j=0; j<i; j++) L[i*wc+j] = A[i*wc+j];
              }
            else
              {
                for (i=0; i<hc; i++)
                  {
                    for (j=i; j<wc; j++)
                      {
                        UZ[i*wc+j] = AZ[i*wc+j];
                        LZ[i*wc+j].r = 0.0;
                        LZ[i*wc+j].i = 0.0;
                      }
                  }
                for (i=0; i<hc; i++)
                  {
                    LZ[i*wc+i].r = 1.0;
                    LZ[i*wc+i].i = 1.0;
                  }  
                for (i=1; i<hc; i++)
                  for (j=0; j<i; j++) LZ[i*wc+j] = AZ[i*wc+j];
              }
kinfo(KVERBOSE,"Split...\n");

            /* Write the data to the output objects */
            if (otype == KDOUBLE)
              {
                if (lt_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(lt_obj,KPDS_VALUE_REGION,(kaddr)L));
                if (ut_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(ut_obj,KPDS_VALUE_REGION,(kaddr)U));
              }
            else
              {
                if (lt_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(lt_obj,KPDS_VALUE_REGION,(kaddr)LZ));
                if (ut_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(ut_obj,KPDS_VALUE_REGION,(kaddr)UZ));
              }
            if (pi_obj != KOBJECT_INVALID)
              KCALL(kpds_put_data(pi_obj,KPDS_VALUE_REGION,(kaddr)ipiv));
           
kinfo(KVERBOSE,"Written...\n");
          }

        (void)klist_free(objlist, (kfunc_void)lkcall_free);

        return(TRUE);
  }
/* -library_code_end */
