 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for irotate
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lirotate
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lirotate - rotate image plane by arbitrary angle about an arbitrary point
* 
*       Purpose: rotate image plane about an arbitrary point and an
*                aribitrary angle. Pixel values are obtained by bilinear
*                interpolation.
*
*         Input: in_obj - input object to be processed
*                wctr - W coordintae of center of rotation
*                hctr - H coordintae of center of rotation
*                resize - if non-zero, resize output to fit data
*                planes - if non-zero, process date by full planes
*                             instead of prisms. Can be MUCH faster by planes
*                             than prisms, but is less well adapted for large 
*                             data sets.
*                angle - rotation angle in degrees. Positive is CW.
*
*        Output: out_obj - output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: *
*   Declaration: int lirotate(
*		 !   kobject in_obj,
*		 !   double wctr,
*		 !   double hctr,
*		 !   int resize,
*		 !   int planes,
*		 !   double angle,
*		 !   kobject out_obj)
****************************************************************/
/* -library_def */
int lirotate(
  kobject in_obj,
  double wctr,
  double hctr,
  int resize,
  int planes,
  double angle,
  kobject out_obj
)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kimage_proc", *rtn = "lirotate";
        klist *objlist=NULL;
        unsigned int i,j,k;
        int map_exists,mask_exists;
	unsigned int wc,hc,dc,tc,ec; /* Dimensions of input data set */
	unsigned int wout,hout,dout,tout,eout; /* Dimensions of output data */
	unsigned int wro,hro,dro; /* Dimensions of output region */
	unsigned int wri,hri,dri; /* Dimensions of input region */
        double *bd=NULL,*bm=NULL,*bmask=NULL,*bdata=NULL;
        double dang,sine,csine,in,kn,z1,z2,z3,z4,za,zb,zp;
        int jl,jh,il,ih;
        int beg[5],end[5];
        float wctri,hctri,wctro,hctro,xhi,xlo,yhi,ylo;
        float x1out,x2out,x3out,x4out;
        float y1out,y2out,y3out,y4out;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID || out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input or output object");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* Create the output value segment if it's not there already and
           copy it's size and data type attributes from the input obj.  */
        KCALL(kpds_copy_object(in_obj,out_obj));

        mask_exists=kpds_query_mask(in_obj);
        map_exists =kpds_query_map(in_obj);

        /* If map exists, push all the data thru the map */
        if (map_exists)
          {
            KCALL(kpds_set_attribute(in_obj,KPDS_MAPPING_MODE,KMAPPED));
            KCALL(kpds_destroy_map(out_obj));
          }

        /* Constrain the input data to double real for reading */
        KCALL(kpds_set_attributes(in_obj,
                                 KPDS_VALUE_DATA_TYPE, KDOUBLE,
                                 KPDS_VALUE_COMPLEX_CONVERT, KREAL,
                                 KPDS_VALUE_INTERPOLATE,KPAD,NULL));

        if (mask_exists) KCALL(kpds_set_attributes(in_obj,
                                 KPDS_MASK_DATA_TYPE,KDOUBLE,
                                 KPDS_VALUE_INTERPOLATE,KPAD,NULL));
 
        /* See how big the data is */
        KCALL(kpds_get_attributes(in_obj,
                                 KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec,NULL));

        /* Constrain to 3D spatial coordinates */
        if (ec != 1 || tc != 1)
          {
            kerror(lib, rtn, "no support for rotation of data on E or T dimensions");
            (void)klist_free(objlist, (kfunc_void)lkcall_free); \
            return(0);
          }

        dang = angle*KPI/180.0; /* Convert to radians */
        csine = cos(dang);
        sine = sin(dang);
        if (angle == 90.0) { csine = 0.0; sine = 1.0; }
        else if (angle == -90.0) { csine = 0.0; sine = -1.0; }
        else if (angle == 180.0 ||
                 angle == -180.0) { csine = -1.0; sine = 0.0; }
        else if (angle == 360.0 ||
                 angle == -360.0) { csine = 1.0; sine = 0.0; }

        if (resize)
          {
            /* Look at all four corners of the input object to see where they
               will map to. Take the largest set of X and Y coordinates for
               the size of the output value region */
            x1out = wctr + csine*(0-wctr) - sine*(0-hctr);
            y1out = hctr + sine*(0-wctr) + csine*(0-hctr);
            x2out = wctr + csine*(wc-wctr) - sine*(0-hctr);
            y2out = hctr + sine*(wc-wctr) + csine*(0-hctr);
            x3out = wctr + csine*(wc-wctr) - sine*(hc-hctr);
            y3out = hctr + sine*(wc-wctr) + csine*(hc-hctr);
            x4out = wctr + csine*(0-wctr) - sine*(hc-hctr);
            y4out = hctr + sine*(0-wctr) + csine*(hc-hctr);

            xlo = x1out;
            if (x2out < xlo) xlo = x2out;
            if (x3out < xlo) xlo = x3out;
            if (x4out < xlo) xlo = x4out;
            ylo = y1out;
            if (y2out < ylo) ylo = y2out;
            if (y3out < ylo) ylo = y3out;
            if (y4out < ylo) ylo = y4out;
            xhi = x1out;
            if (x2out > xhi) xhi = x2out;
            if (x3out > xhi) xhi = x3out;
            if (x4out > xhi) xhi = x4out;
            yhi = y1out;
            if (y2out > yhi) yhi = y2out;
            if (y3out > yhi) yhi = y3out;
            if (y4out > yhi) yhi = y4out;

            /* Figure out size for output image value seg */
            wout = (xhi-xlo)+1;
            hout = (yhi-ylo)+1;
            dout = dc;
            tout = 1;
            eout = 1;

            wctri = wctr;
            wctro = wctr+0.5*(wout-wc);
            hctri = hctr;
            hctro = hctr+0.5*(hout-hc);
          }
        else
          {
            wout = wc;
            hout = hc;
            dout = dc;
            tout = 1;
            eout = 1;
            wctri = wctr;
            wctro = wctr;
            hctri = hctr;
            hctro = hctr;
          }

        if (!planes)
          {
            /* Set the region sizes */
            wri = 2;  hri = 2;  dri = dc;
            wro = wout; hro = 1;  dro = dout;
          }
        else
          {
            /* Set the region sizes */
            wri = wc; hri = hc; dri = 1;
            wro = wout; hro = hout; dro = 1;
          }

        /* Set up the output object, reference it, and set presentation vals */
        KCALL(kpds_set_attributes(out_obj,
                                 KPDS_VALUE_SIZE,wout,hout,dout,1,1,NULL));
        if (mask_exists)
          KCALL(kpds_set_attributes(out_obj,
                                 KPDS_MASK_SIZE,wout,hout,dout,1,1,NULL));
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");
        KCALL(kpds_set_attributes(out_obj,
                                 KPDS_VALUE_DATA_TYPE, KDOUBLE,NULL));
        if (mask_exists)
          {
            KCALL(kpds_set_attributes(out_obj,
                                 KPDS_MASK_DATA_TYPE,KDOUBLE,NULL));
            KCALL(kpds_set_attributes(out_obj,
                                 KPDS_MASK_SIZE,wout,hout,dout,1,1,NULL));
          }

        /* Set the region sizes on the input and output objects */
        KCALL(kpds_set_attribute(in_obj,
                                  KPDS_VALUE_REGION_SIZE,wri,hri,dri,1,1));
        KCALL(kpds_set_attribute(out_obj,
                                  KPDS_VALUE_REGION_SIZE,wro,hro,dro,1,1));
        if (mask_exists)
          { 
            KCALL(kpds_set_attribute(in_obj,
                                   KPDS_MASK_REGION_SIZE,wri,hri,dri,1,1));
            KCALL(kpds_set_attribute(out_obj,
                                   KPDS_MASK_REGION_SIZE,wro,hro,dro,1,1));
          }

        /* Set up output buffer */
        bd = (double *)kmalloc(wro*hro*dro*sizeof(double));
        objlist = klist_add(objlist,bd,"KMALLOC");
        if (mask_exists)
          {
            bm = (double *)kmalloc(wro*hro*dro*sizeof(double));
            objlist = klist_add(objlist,bm,"KMALLOC");
          }

        if (!planes)
          {
            for (i=0; i<hout; i++)
              {
                for (j=0; j<wout; j++)
                  {
                    kn = wctri + csine*(j-wctro) + sine*(i-hctro);
                    in = hctri - sine*(j-wctro)  + csine*(i-hctro);
                    jl = floor(kn); jh = jl+1;
                    il = floor(in); ih = il+1;
                    /* Go under KPDS to KDMS to speed the large data set process
                    ing case. This is NOT the way to do this in general, but is
                    a last grasp for speed and works by avoiding the overhead
                    imposed by KPDS. This will only work if the index order is
                    whdte, a condition that can be imposed by a kpds_open_object
                    call. This is normally the case for use by a kroutine
                    driver, but could be a problem for folks using this routine
                    in thier own processing code. */
                    beg[0] = jl; beg[1]=il; beg[2]=0;    beg[3]=0; beg[4]=0;
          /*  region size in time and elements direction should be 1, not 2 
                    end[0] = jh; end[1]=ih; end[2]=dc-1; end[3]=1; end[4]=1;
	   */
                    end[0] = jh; end[1]=ih; end[2]=dc-1; end[3]=0; end[4]=0;
                    bdata=kdms_get_data(in_obj,KDMS_SEGMENT_VALUE,beg,end,(kaddr)bdata);
                    if (mask_exists)
                      bdata=kdms_get_data(in_obj,KDMS_SEGMENT_MASK,beg,end,(kaddr)bmask);
                    for (k=0; k<dro; k++)
                      {
                        z1 = bdata[4*k+0]; z2 = bdata[4*k+1];
                        z4 = bdata[4*k+3]; z3 = bdata[4*k+2];
                        za = z1*(ih-in)+z4*(in-il);
                        zb = z2*(ih-in)+z3*(in-il);
                        zp = za*(jh-kn)+zb*(kn-jl);
                        bd[k*wro+j] = zp;
                        if (mask_exists)
                          if (bmask[4*k+0] != 0 && bmask[4*k+1] != 0 &&
                            bmask[4*k+3] != 0 && bmask[4*k+2] != 0)
                              bm[k*wro+j] = 1;
                      }
                  }
                KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)bd));
                if (mask_exists)
                  KCALL(kpds_put_data(out_obj,KPDS_MASK_REGION,(kaddr)bm));
              }
          }
        else /* Process data by full planes instead of prisms */
          {
            for (k=0; k<dout; k++)
              {
                bdata=kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)bdata);
                if (mask_exists)
                  bmask=kpds_get_data(in_obj,KPDS_MASK_REGION,(kaddr)bmask);
                for (i=0; i<hout; i++)
                  {
                    for (j=0; j<wout; j++)
                      {
                        kn = wctri + csine*(j-wctro) + sine*(i-hctro);
                        in = hctri - sine*(j-wctro)  + csine*(i-hctro);
                        jl = floor(kn); jh = jl+1;
                        il = floor(in); ih = il+1;
                        if (in >= 0 && in < hc &&
                            kn >= 0 && kn < wc &&
                            jl >= 0 && jl < wc &&
                            jh >= 0 && jh < wc &&
                            il >= 0 && il < hc &&
                            ih >= 0 && ih < hc)
                          {
                            z1 = bdata[il*wc+jl]; z2 = bdata[il*wc+jh];
                            z4 = bdata[ih*wc+jl]; z3 = bdata[ih*wc+jh];
                            za = z1*(ih-in)+z4*(in-il);
                            zb = z2*(ih-in)+z3*(in-il);
                            zp = za*(jh-kn)+zb*(kn-jl);
                            bd[i*wout+j] = zp;
                          }
                        else
                          {
                            bd[i*wout+j] = 0;
                          } 
                        if (mask_exists)
                          if (bmask[il*wc+jl] != 0 && bmask[il*wc+jh] != 0 &&
                            bmask[ih*wc+jl] != 0 && bmask[ih*wc+jh] != 0)
                              bm[i*wout+j] = 1;
                      }
                  }
                KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)bd));
                if (mask_exists)
                  KCALL(kpds_put_data(out_obj,KPDS_MASK_REGION,(kaddr)bm));
              }
          }

        /* Free up everything that has been allocated */
        kfree(bdata);
        if (mask_exists) kfree(bmask);
        (void)klist_free(objlist, (kfunc_void)lkcall_free);
 
        return TRUE;
  }
/* -library_code_end */
