
/*********************************************************************

Copyright (C) 1993, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus it is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

1/1/95, wes.  made clip_line_persp() and clip_line_parallel() routines
    public (instead of static) for use by code in rmonster_mouse.c.

*********************************************************************/

#include <design.h>
#include <geometry.h>
#include "rmonster_objs.h"
#include "clip.h"

/**
  * this file contains the various routines which are used for
  * performing clipping operations.  part of the rmonster_obj
  * struct is a pointer to a routine to be used for clipping.
  *
**/

/**
  * code for line clipping algorithms is based upon
  * the Liang-Barsky as described in Foley, et. al, version 2
  * 1990 (the big white book).
**/
#if 0
int clip_line_persp PROTO((float *,float *,float *,float *,float *,float *,float *,int,int));
int clip_line_parallel PROTO((float *,float *,float *,float *,float *,float *,float *,int,int));
#endif

int
clip_nullfunc(rmonster_prim *rp,
	      Poly *in_p,
	      Poly *o_p,
	      int projection,
	      float *zmin,
	      int hither,
	      int yon)
{
    return(CHILL);
}

int
clip_polyline_con(rmonster_prim *rp,
		  Poly *in_p,
		  Poly *o_p,
		  int projection,
		  float *zmin,
		  int hither,
		  int yon)
{
    int status;
    float px0,px1,py0,py1,pz0,pz1;

#if 0
    kfprintf(kstderr," entering clip polyline con function.\n");
    kfprintf(kstderr," clipping from %f,%f,%f to %f,%f,%f \n",p->vert[0].x,p->vert[0].y,p->vert[0].z,p->vert[1].x,p->vert[1].y,p->vert[1].z);
#endif

    px0 = in_p->vert[0].x;
    py0 = in_p->vert[0].y;
    pz0 = in_p->vert[0].z;
    
    px1 = in_p->vert[1].x;
    py1 = in_p->vert[1].y;
    pz1 = in_p->vert[1].z;

    if (projection == KPROJECTION_PERSPECTIVE)
	status = clip_line_persp(&px0,&py0,&pz0,&px1,&py1,&pz1,zmin,hither,yon);
    else
	status = clip_line_parallel(&px0,&py0,&pz0,&px1,&py1,&pz1,zmin,hither,yon);

    if (status == CHILL)
    {
	o_p->vert[0].x = px0;
	o_p->vert[0].y = py0;
	o_p->vert[0].z = pz0;
    
	o_p->vert[1].x = px1;
	o_p->vert[1].y = py1;
	o_p->vert[1].z = pz1;
    }
    
    return(status);
}

int
clip_polyline_dis(rmonster_prim *rp,
		  Poly *in_p,
		  Poly *o_p,
		  int projection,
		  float *zmin,
		  int hither,
		  int yon)
{
    int status;
    float px0,px1,py0,py1,pz0,pz1;

#if 0
    kfprintf(kstderr," entering clip polyline dis function.\n");
    kfprintf(kstderr," clipping from %f,%f,%f to %f,%f,%f \n",p->vert[0].x,p->vert[0].y,p->vert[0].z,p->vert[1].x,p->vert[1].y,p->vert[1].z);
#endif

    px0 = in_p->vert[0].x;
    py0 = in_p->vert[0].y;
    pz0 = in_p->vert[0].z;
    
    px1 = in_p->vert[1].x;
    py1 = in_p->vert[1].y;
    pz1 = in_p->vert[1].z;

    if (projection == KPROJECTION_PERSPECTIVE)
	status = clip_line_persp(&px0,&py0,&pz0,&px1,&py1,&pz1,zmin,hither,yon);
    else
	status = clip_line_parallel(&px0,&py0,&pz0,&px1,&py1,&pz1,zmin,hither,yon);

    if (status == CHILL)
    {
	o_p->vert[0].x = px0;
	o_p->vert[0].y = py0;
	o_p->vert[0].z = pz0;
    
	o_p->vert[1].x = px1;
	o_p->vert[1].y = py1;
	o_p->vert[1].z = pz1;
    }
    
    return(status); /* either CHILL or WHACKED */
}

int
clip_polygon_cheap(Poly *in_p,
		   int projection,
		   float *zmin,
		   int hither,
		   int yon)
/**
  * this routine is cheap because it doesn't do *real* polygon
  * clipping.
  * it will clip polygons which
  * a. lie entirely outside the view volume
  * b. have _any_ points that lie closer than zmin to the viewer
  *
  * polygons which straddle the view volume will be accepted
  * without modification.
  *
  * april 1995
**/
{
    int i,triv_reject;
    float loc_zmin;
    float t;

    if (projection == KPROJECTION_PERSPECTIVE)
	loc_zmin = *zmin;
    else
	loc_zmin = 0.;

    /* do zmin checks first */

    for (i=0;i<in_p->n;i++)
    {
	if (in_p->vert[i].z < loc_zmin) 
	{
#ifdef DEBUG
	    fprintf(stderr," rejecting a polygon because it's too close. \n");
#endif
	    return(WHACKED);
	}
    }

#if 0
    /* check for all vertices off to the left */
    if (projection == KPROJECTION_PERSPECTIVE)
    {

	/* check for all verts off to the right or left*/
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    t = FABS(in_p->vert[i].x)*in_p->vert[i].z;
	    if (t <= 1.)
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);
#if 0
	/* test for off to the left.  this works because we've
	   already tested against zmin. */
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    if (in_p->vert[i].x >= (-1. * in_p->vert[i].z))
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);
#endif	
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    t = FABS(in_p->vert[i].y) * in_p->vert[i].z;
	    if (t <= 1.)
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);
#if 0
	/* test for off to the bottom.  this works because we've
	   already tested against zmin. */
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    if (in_p->vert[i].y >= (-1. * in_p->vert[i].z))
	    {
		triv_reject = 0;
		break;
	    }
	}
#endif
    }
    else			/* parallel projection */
    {
#endif
	/* check for all verts off to the right */
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    if (in_p->vert[i].x <= 1.0)
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);

	/* test for off to the left.  this works because we've
	   already tested against zmin. */
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    if (in_p->vert[i].x >= -1.)
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);
	
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    if (in_p->vert[i].y <= 1.0)
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);

	/* test for off to the bottom.  this works because we've
	   already tested against zmin. */
	triv_reject = 1;
	for (i=0;i<in_p->n;i++)
	{
	    if (in_p->vert[i].y >= -1.)
	    {
		triv_reject = 0;
		break;
	    }
	}
	if (triv_reject)
	    return(WHACKED);
/*  } */
    /* right now, no checks against zmax */
    return(CHILL);
}

static int clipt PROTO((float,float,float *,float *));

/* the following is for clipping against the canonical view
   volume for perspective projection. */
int
clip_line_persp(float *px0,
		float *py0,
		float *pz0,
		float *px1,
		float *py1,
		float *pz1,
		float *zmin,
		int hither,
		int yon)
{
    int accept=WHACKED;
    float tmin,tmax,dx,dy,dz,temp;

    tmin = 0.;
    tmax = 1.;
    dx = *px1 - *px0;
    dz = *pz1 - *pz0;
    temp = -(*px0) - (*pz0);
    if (clipt(dx+dz,temp,&tmin,&tmax))  /* right side */
    {
	temp = *px0 - *pz0;
	if (clipt(-dx+dz,temp,&tmin,&tmax)) /* left side */
	{
	    dy = *py1 - *py0;
	    temp = *py0 - *pz0;
	    if (clipt(-dy+dz,temp,&tmin,&tmax)) /* bottom */
	    {
		temp = -(*py0) - *pz0;
		if (clipt(dy+dz,temp,&tmin,&tmax)) /* top */
		{

		    temp = *zmin - *pz0;
		    
		    if ((!hither) || (clipt(dz,temp,&tmin,&tmax))) /* front */
		    {
			temp = *pz0 - 1.;
			if ((!yon) || (clipt(-dz,temp,&tmin,&tmax)))  /* back */
			{

			    /* if got here, then we have:
			       -z <= x <= z;
			       -z <= y <= z;
			       zmin <= z <= 1 */
			    
			    accept = CHILL;
			    if (tmax < 1.)
			    {
				*px1 = *px0 + tmax * dx;
				*py1 = *py0 + tmax * dy;
				*pz1 = *pz0 + tmax * dz;
			    }
			    if (tmin > 0.)
			    {
				*px0 = *px0 + tmin * dx;
				*py0 = *py0 + tmin * dy;
				*pz0 = *pz0 + tmin * dz;
			    }
			} 
		    }
		}
	    }
	}
    }
    return(accept);
}

int
clip_line_parallel(float *px0,
		   float *py0,
		   float *pz0,
		   float *px1,
		   float *py1,
		   float *pz1,
		   float *zmin,
		   int hither,
		   int yon)
{
    int accept=WHACKED;
    float tmin,tmax,dx,dy,dz,temp;

    tmin = 0.;
    tmax = 1.;
    dx = *px1 - *px0;
    dz = *pz1 - *pz0;
    temp = *px0 - 1.;
    if (clipt(-dx,temp,&tmin,&tmax))  /* right side */
    {
	temp = -1. - *px0;
	if (clipt(dx,temp,&tmin,&tmax)) /* left side */
	{
	    dy = *py1 - *py0;
	    temp = -1. - *py0;
	    if (clipt(dy,temp,&tmin,&tmax)) /* bottom */
	    {
		temp = *py0 - 1.;
		if (clipt(-dy,temp,&tmin,&tmax)) /* top */
		{
/*		    temp = *zmin - *pz0; */
		    temp = -(*pz0);  /* zmin in parallel proj. is zero */
		    if ((!hither) || (clipt(dz,temp,&tmin,&tmax))) /* front */
		    {
			temp = *pz0 - 1.;
			if ((!yon) || (clipt(-dz,temp,&tmin,&tmax))) /* back */
			{

			    /* if got here, then we have:
			       -z <= x <= z;
			       -z <= y <= z;
			       zmin <= z <= 1 */
			    
			    accept = CHILL;
			    if (tmax < 1.)
			    {
				*px1 = *px0 + tmax * dx;
				*py1 = *py0 + tmax * dy;
				*pz1 = *pz0 + tmax * dz;
			    }
			    if (tmin > 0.)
			    {
				*px0 = *px0 + tmin * dx;
				*py0 = *py0 + tmin * dy;
				*pz0 = *pz0 + tmin * dz;
			    }
			} 
		    }
		}
	    }
	}
    }
    return(accept);
}

static int
clipt(float denom,
      float num,
      float *tE,
      float *tL)
{
    float t;
    int accept;

    accept = 1;
    if (denom > 0.)
    {
	t = num/denom;
	if (t > *tL)
	    accept = 0;
	else if (t > *tE)
	    *tE = t;
    }
    else if (denom < 0.)
    {
	t = num/denom;
	if (t < *tE)
	    accept = 0;
	else if (t < *tL)
	    *tL = t;
    }
    else
    {
	if (num > 0.)
	    accept = 0;
    }
    return(accept);
}


int cheap_sphere_clip(vertex_3d *v,
		      float *zmin, 
		      int projection)
{
    /**
      * this is the cheapest of the cheap - will return TRUE if all the
      * vertex (representing the center of the sphere) lies within the
      * canonical view volume.
    **/

    register vertex_3d *rv;
    int i,status;
    float xsign,ysign;

    status = CHILL;
    rv = v;

        if ((rv->v[2] >=  *zmin) && (rv->v[2] <= 1.0))
	{
	  if (rv->v[0] < 0)
	     xsign = -1.;
	  else
	     xsign = 1.;
	  if (rv->v[1] < 0)
	     ysign = -1.;
	  else
	     ysign = 1.;

            if (projection == KPROJECTION_PERSPECTIVE)
	    {
	        if (xsign * rv->v[0] <= rv->v[2])
		   if (ysign * rv->v[1] <= rv->v[2])
		      return(TRUE);
	    }
	    else
	    {
	        if (xsign * rv->v[0] < 1.)
		   if (ysign * rv->v[1] < 1.)
		      return(TRUE);
	    }
	}

    return(FALSE);
}
