 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         Functionality routines for input of plot functions
   >>>>
   >>>>  Private:
   >>>>           input_function
   >>>>   Static:
   >>>>           create_data_object_from_function
   >>>>           get_range
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "xprism.h"

static kobject 
create_data_object_from_function PROTO((int, char *, char *, char *, char *));

static int get_range PROTO((char *, float *, float *, int *));

/*-----------------------------------------------------------
|
|  Routine Name: input_function
|
|       Purpose: routine which is called to input a new file
|         Input: 
|        Output: None
|    Written By: Danielle Argiro & Mark Young
|          Date: June 28, 1994
| Modifications:
|
------------------------------------------------------------*/
void input_function(void)
{
        kobject object;
	char    function[KLENGTH];
	char    temp[KLENGTH];
	char    errormesg[KLENGTH];
	char    xrange[KLENGTH];
	char    yrange[KLENGTH];

        Plot_function *func_info = gui_info->Plot->function;
	errormesg[0] = '\0';

	/* 2D plot */
	if (func_info->plot_dim == 2)
	{
	    kstring_cleanup(func_info->func_2D, temp);
	    ksprintf(function, "f(x) = %s", temp);
	    ksprintf(xrange,   "x = (%s,%s,%s)", 
		     func_info->x_min,
		     func_info->x_max,
		     func_info->xnum_pts);

            object = create_data_object_from_function(2, function, xrange, 
						      NULL, errormesg);
	    if (object == NULL)
            {
                kerror(NULL, "input_function", 
		       "Failed to create plot from 2D function '%s'.\n%s", 
			function, errormesg);
                return;
            }
	}

	/* 3D plot */
	else if (func_info->plot_dim == 3)
        {
	    kstring_cleanup(func_info->func_3D, temp);
            ksprintf(function, "f(x,y) = %s", temp);
            ksprintf(xrange,   "x = (%s,%s,%s)",
                     func_info->x_min,
                     func_info->x_max,
                     func_info->xnum_pts);
            ksprintf(yrange,   "y = (%s,%s,%s)",
                     func_info->y_min,
                     func_info->y_max,
                     func_info->ynum_pts);
            object = create_data_object_from_function(3, function, xrange, 
						      yrange, errormesg);
            if (object == NULL)
            {
                kerror(NULL, "input_function", 
                       "Failed to create plot from 3D function '%s'.\n%s", 
			function, errormesg);
                return;
            }
        }
        create_plot(object, func_info->plot_dim, func_info->where_plot_val);
        kpds_close_object(object);
}

/*-----------------------------------------------------------
|
|  Routine Name: create_data_object_from_function
|
|       Purpose: Takes the plot type, a function, and intervals;
|                uses the kexpr parser to evaluate the function,
|                creates a kobject with the data points defined by 
|                the function, and returns it.
|
|         Input: plot_type - 2 for 2D, or 3 for 3D
|                function  - the function to plot, eg "sin(x)"
|                xrange    - range, eg, "
|        Output:
|    Written By: Danielle Argiro & Mark Young
|          Date: Jun 28, 1994
| Modifications:
|
------------------------------------------------------------*/

static kobject create_data_object_from_function(
    int  plot_type,
    char *function,
    char *xrange,
    char *yrange,
    char *errormesg)
{
	char    error[KLENGTH];
        float  xbegin, xend, xincr, xval; 
	float  ybegin, yend, yincr, yval; 
	float  value;
	kobject object;
	kaddr  func;
	float  *ypts, *zpts;
	
        char    temp[KLENGTH];
        int     i, j, xsize, ysize, ptnum;


        if ((func = kexpr_compile_function(0, function, errormesg)) == NULL)
           return(NULL);

	/*
	 * set the X range 
 	 */
        if (!(get_range(xrange, &xbegin, &xend, &xsize)))
           return(NULL);

        if (xend <= xbegin)
        {
           kerror(NULL, "get_function", "The beginning point is greater than the ending point. Please check the X-interval values, and try again");
           return(NULL);
        }

        if (xsize > 1)
           xincr = (xend - xbegin)/((float) (xsize - 1));
        else xincr = (xend - xbegin);

        if (plot_type == 3)
        {
           if (!(get_range(yrange, &ybegin, &yend, &ysize)))
                return(NULL);

           if (yend <= ybegin)
           {
               kerror(NULL, "get_function", "The beginning point is greater than the ending point. Please check the Y-interval values, and try again");
               return(NULL);
           }
           if (ysize > 1)
             yincr = (yend - ybegin)/((float) (ysize - 1));
           else yincr = (yend - ybegin);
        }
        else ysize = 1;

        if (xsize*ysize <= 0)
        {
           kerror(NULL, "get_function", "The number of data points specified is less than or equal to zero. Please specify the number of points to be greater than zero");
           return(NULL);
        }

	/*
	 * evaluate the plot points
	 */

        if (plot_type == 2)
        {
	   ypts = (float *) kmalloc(xsize * sizeof(float));

	   ptnum = 0;
           for (i = 0; i < xsize; i++)
           {
              xval = i*xincr + xbegin;
              if (!kexpr_execute_function(func, KFLOAT, &value, errormesg, xval))
                 return(NULL);

              ypts[ptnum] = value;
	      ptnum++;
           }
        }
        else
        {
           ypts = (float *) kmalloc(xsize * ysize * sizeof(float));
           zpts = (float *) kmalloc(xsize * ysize * sizeof(float));

	   ptnum = 0;
           for (j = 0; j < ysize; j++)
           {
              yval = j*yincr + ybegin;
              for (i = 0; i < xsize; i++)
              {
                 xval = i*xincr + xbegin;
                 ksprintf(temp, "f(%g,%g)", xval, yval);
                 if (!kexpr_execute_function(func, KFLOAT, &value, error,
					     xval, yval))
		 {
                    return(NULL);
		 }
                 zpts[ptnum] = value;
                 ptnum++;
              }
           }
        }

	/*
	 *  create the data object; points will be stored in data object
	object = kpds_open_output_object("./tmp.viff");
	 */
	object = kpds_create_object();
	kpds_create_value(object);
	kpds_set_attribute(object, KPDS_VALUE_DATA_TYPE, KFLOAT);

	if (plot_type == 2)
	{
	    /* 
	     * x values are implicit in 2D plot data storage;
	     * y values are explicitly stored along width in value segment
	     */
	    kpds_set_attribute(object, KPDS_VALUE_SIZE, ptnum, 1, 1, 1, 1);
	    kpds_put_data(object, KPDS_VALUE_LINE, ypts);
	    kpds_set_attribute(object, KPDS_LOCATION_GRID, KUNIFORM);
	    kpds_create_location(object);
	    kpds_set_attributes(object,
			KPDS_LOCATION_BEGIN, xbegin, 0.0, 0.0, 
			KPDS_LOCATION_END,  xend, 0.0, 0.0, 
			NULL);
	}
	else if (plot_type == 3)
	{
	    /* 
	     * x & y values are implicit in 3D plot data storage;
	     * z values are explicitly stored along width & height 
             * in value segment
	     */
	    kpds_set_attribute(object, KPDS_VALUE_SIZE, xsize, ysize, 1, 1, 1);
	    kpds_put_data(object, KPDS_VALUE_PLANE, zpts);
/*
	    kpds_set_attribute(object, KPDS_LOCATION_GRID, KUNIFORM);
	    kpds_create_location(object);
	    kpds_set_attributes(object,
			KPDS_LOCATION_BEGIN, xbegin, ybegin, 0.0, 
			KPDS_LOCATION_END,  xend, yend, 0.0, 
			NULL);
*/
	}
        return(object);
}


/*-----------------------------------------------------------
|
|  Routine Name: get_range
|
|       Purpose: Takes the plot type, a function, and intervals;
|                uses the kexpr parser to evaluate the function,
|                returns begin point, end point, and size for function range.
|
|         Input: interval - interval string, of form:
|
|        Output: begin - passes back begin point
|                end   - passes back end point
|                size  - passes size
|
|    Written By: Danielle Argiro & Mark Young
|          Date: Jun 28, 1994
| Modifications:
|
------------------------------------------------------------*/

static int get_range(
    char   *interval,
    float *begin,
    float *end,
    int    *size)
{
        char  dummy[KLENGTH]; 
	char  init[KLENGTH]; 
	char  last[KLENGTH]; 
	char  num_points[KLENGTH];
	char  error[KLENGTH];

        if (sscanf(interval," %[^=]%*[ =\t\n](%[^,],%[^,],%[^)])", 
		             dummy, init, last, num_points) != 4)
        {
           kerror(NULL, "get_function", "Unable to parse interval string '%s'", 
	          interval);
           return(FALSE);
        }

        if (kexpr_evaluate_float(0, init, begin, error) == FALSE)
           return(FALSE);

        if (kexpr_evaluate_float(0, last, end, error) == FALSE)
           return(FALSE);

        if (kexpr_evaluate_int(0, num_points, size, error) == FALSE)
           return(FALSE);

        return(TRUE);
}


