 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>  		        Legend I/O Routines
   >>>>
   >>>>  Private:
   >>>>			spc_read_legend_file()
   >>>>			spc_write_legend_file()
   >>>>			spc_construct_legend_info()
   >>>>   Static:
   >>>>			parse_legend_info()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

static int parse_legend_info      PROTO((char *, int *, int *, 
				         int *, char **, int *));
/*-----------------------------------------------------------
|
|  Routine Name: spc_read_legend_file
|
|       Purpose: Reads an ascii legend file
|
|         Input: filename - name of file to read
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Nov 3, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int spc_read_legend_file(
    char *filename)
{
	int         i, j, done, class_id; 
	int         max_classnum = 0;
	char        *classname; 
	char        temp[KLENGTH];
	int         red, green, blue;
	kfile       *file;
	LegendEntry *legend_ptr;
        

	/* open legend file */
	if ((file = kfopen(filename, "r")) == NULL)
        {
	   kerror(NULL, "read_legend_file", 
	          "Unable to open '%s' to read legend.", filename);
	   kinfo(KHOSTILE, "Get it right!");
           return(FALSE);
        }

	/* premature end of file */
        if (kfeof(file))
           return(FALSE);

	/* while not at end of file */
	done = FALSE; i = 0; 
	while (!done)
	{
           j = 0;
	   i++;
    	        /* get a line of information */
           if (kfgets(temp, KLENGTH, file) == NULL) 
		done = TRUE;
	   else
	   {
	   	/* skip over any leading whitespace */
		while(temp[j] == ' ' || temp[j] == '\t') j++;

	   	/* skip over any blank lines;  comments have # symbol */
		if ((temp[j] != '\n') && (temp[j] != '#'))
		{
		    /* get legend information */
	 	    if (!(parse_legend_info(&temp[j], &red, &green, &blue, 
					    &classname, &class_id)))
			return(FALSE);

	            if (class_id > max_classnum) max_classnum = class_id;

		    /* add a node to the legend structure */
		    legend_ptr = spc_find_class_from_id(class_id);
		    if (legend_ptr == NULL)
		    {
		        spc_add_legend_entry(class_id, red, green, blue, 
		    	 		     classname, NULL, 0, SPC_LGD_CREATED);  
		    }
		    else
		    {
		        legend_ptr->redval     = red;
			legend_ptr->greenval   = green;
			legend_ptr->blueval    = blue;
			kfree(legend_ptr->classname);
			legend_ptr->classname       = kstrdup(classname);
			legend_ptr->classname_token = kstring_to_token(classname);
			ksprintf(temp, " RGB values:  %d %d %d",
				red, green, blue);
			xvw_set_attribute(legend_ptr->RGBobj, XVW_LABEL,
					  temp);
			xvw_set_attribute(legend_ptr->classnamebutton, XVW_LABEL,
					  legend_ptr->classname);
			xvw_set_attributes(legend_ptr->colorbox,
                               XVW_COLORCELL_REDVAL,   legend_ptr->redval,
                               XVW_COLORCELL_GREENVAL, legend_ptr->greenval,
                               XVW_COLORCELL_BLUEVAL,  legend_ptr->blueval,
                               XVW_COLORCELL_UPDATE,   TRUE,
                               NULL);
		    }

		}
	    }
	}

	/*
         * update GUI to reflect filename
         */
        xvf_set_attribute(gui_info->Files->files->in_legend_struct,
                          XVF_FILE_NAME, filename);
	kfree(gui_info->Files->files->in_legend);
	gui_info->Files->files->in_legend = kstrdup(filename);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: parse_legend_info
|
|       Purpose: Parses one line of info from the ascii legend file
|
|         Input: line - line of info of the legend file
|
|        -------------------------------------------
|        |    RGB   |   Text       |  Class Number |
|        |    .     |      .       |    .          |
|        |    .     |      .       |    2          |
|        | 80 255 0 |"CLASS 1"     |    3          |
|        |    .     |      .       |    4          |
|
|        Output: red        - red value of RGB triple
|                green      - green value of RGB triple
|                blue       - blue value of RGB triple
|                classname  - name assoc. w/ class
|		 class_id   - ID number of class
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Nov 3, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


char *legend_scan = "%d%d%d%*[ ]%*['`]%[^'`]%*['`]%*[ ]%d";
#define MAXCLUSTERNUM 4096

static int parse_legend_info(
    char   *line,
    int    *red, 
    int    *green, 
    int    *blue,
    char  **classname,
    int    *class_id)
{
	char tmp_text[KLENGTH]; 
	int  status;
	int  r, g, b;

	status = sscanf(line, legend_scan, &r, &g, &b, tmp_text, class_id);
	*red   = (int) r;
	*green = (int) g;
	*blue  = (int) b;

	if (status < 5)
	{
	    kerror(NULL, "parse_legend_info", 
		   "Invalid line (missing information) in legend file");
	    kinfo(KHOSTILE, "What are you trying to do, mess me up?");
	    return(FALSE);
	}

	if ((kstrcmp(tmp_text," ")) == 0) *classname = NULL;
	else *classname = kstrdup(tmp_text);

	if (*class_id < 1)
	{
	    kerror(NULL, "parse_legend_info", 
		   "Invalid class ID (less than 0) in legend file");
	    kinfo(KHOSTILE, "Deliberately trying to confuse me?");
            return(FALSE);
	}
	if (*red < 0)
	{
	    kerror(NULL, "parse_legend_info", 
		   "Invalid (negative) Red value in legend file");
	    kinfo(KHOSTILE, "Suppose you thought I'd let you get away with that.");
            return(FALSE);
	}
	if (*green < 0)
	{
	    kerror(NULL, "parse_legend_info", 
		   "Invalid (negative) Green value in legend file");
	    kinfo(KHOSTILE, "What do you think this is?  NO BOGUS LEGEND FILES");
            return(FALSE);
	}
	if (*blue < 0)
	{
	    kerror(NULL, "parse_legend_info", 
		   "Invalid (negative) Blue value in legend file");
	    kinfo(KHOSTILE, "Get serious.  I can't do anything with legend files that are all SNAFU'd");
            return(FALSE);
	}

	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: spc_construct_legend_info
|
|       Purpose: When there are more classes defined in the image's class
|                column than are included in the current legend, we must 
|                construct the legend info (as much as possible) from the 
|                class column.  That is, we can get the class ID number,
|                and the clusters in the class. The color will be black,
|                and the classname will be assigned, "Un-named category %d"
|
|         Input: class_id   - class ID from spc_classes array
|        Output: returns an array of cluster numbers of size clusternum
|                clusternum - number of clusters in the class_id
|                red        - red value associated w/ class_id color
|                green      - green value associated w/ class_id color
|                blue       - blue value associated w/ class_id color
|                classname  - name associated w/ class_id category
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Nov 3, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int *spc_construct_legend_info(
    int  class_id,
    int  *clusternum,
    int  *red, 
    int  *green, 
    int  *blue,
    char **class)
{
	int  i, j; 
	int  clusters[MAXCLUSTERNUM]; 
        int  *return_clusters; 
	char temp[KLENGTH];

	ksprintf(temp, "Un-named class_id category %d", class_id);
	*class = kstrdup(temp);
	j = 0;
	for (i = 0; i < spc_map_rownum; i++)
	{
	    if (spc_classes[i] == class_id) 
		clusters[j++] = i;
	}
	*clusternum = j;
	if (j == 0) return(NULL);
	return_clusters = (int *) kmalloc (j * sizeof(int));
	for (i = 0; i < *clusternum; i++)
	    return_clusters[i] = clusters[i];

	*red    =  0;
	*green  =  0;
	*blue   =  0;

	return(return_clusters);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_write_legend_file
|
|       Purpose: Writes out an ascii legend file according to
|                current legend information
|
|         Input: filename - name of the legend file to create
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Nov 3, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int spc_write_legend_file(
    char *filename)
{
	kfile       *file;
	LegendEntry *legend_ptr; 

	legend_ptr = spc_legend_list;
	if (legend_ptr == NULL)
	{
	    kerror(NULL, "spc_write_legend_file",
		   "No current legend information to write!");
	    kinfo(KHOSTILE, "What did you expect me to do, dweeb?");
	    return(FALSE);
	}

	/* see if legend file already exists/prompt to over-write*/
        if ((file = kfopen(filename, "r")) != NULL)
        {
	   if (!(koverwrite(KSTANDARD, filename)))
	   {
	       kfclose(file);
               return(FALSE);
	   }
	   kfclose(file);
        }


	/* open legend file */
	if ((file = kfopen(filename, "w")) == NULL)
        {
	    kerror(NULL, "write_legend_file",
		   "Unable to open '%s' to write legend.", filename);
	    kinfo(KHOSTILE, "Give me a break.");
            return(FALSE);
        }

	while (legend_ptr != NULL)
	{
	     kfprintf(file, "%d %d %d '%s' %d\n", 
		      legend_ptr->redval, legend_ptr->greenval,
		      legend_ptr->blueval, legend_ptr->classname, 
		      legend_ptr->class);
	     legend_ptr = legend_ptr->next;
	}
	kfclose(file);
        return(TRUE);
}

